/************************************************************************
** MODULE INFORMATION*
**********************
**     FILE     NAME:       DPATFTP.C
**     SYSTEM   NAME:       IP
**     ORIGINAL AUTHOR(S):  Wim van Campen
**     VERSION  NUMBER:     1.00
**     CREATION DATE:       1990/11/4
**
** DESCRIPTION: DP application file for the tftp server
**              
*************************************************************************
** CHANGES INFORMATION **
*************************
** REVISION:    $Revision:   1.0  $
** WORKFILE:    $Workfile:   TFTP.C  $
** LOGINFO:     $Log:   I:/ETSTJAN/CPROG/BEHOLDER/BAPS/TFTP/VCS/TFTP.C_V  $
**              
**                 Rev 1.0   01 Feb 1991 15:08:30   etstjan
**              No explicit note
**              
**                 Rev 1.1   21 Nov 1990 13:34:08   etstjan
**              No explicit note
**              
**                 Rev 1.0   21 Nov 1990 12:46:50   etstjan
**              No explicit note
*************************************************************************/
#if ! defined(PRD)
static char _pvcs_hdr[] =
"$Header:   I:/ETSTJAN/CPROG/BEHOLDER/BAPS/TFTP/VCS/TFTP.C_V   1.0   01 Feb 1991 15:08:30   etstjan  $";
#endif

#include        <stdio.h>                /* for NULL                          */
#include        <stdlib.h>
#include        <string.h>
#include        <dos.h>                  /* for _enable, _disable             */

#include        <beholder.h>
#include        <power3.h>               /* for POWER screen functions        */


/* prototyping of static functions                                          */
static int  ProcessEvents(DPEVENT Event);
static void FreeTime(void);
static void EverySecond(void);
static void TftpStart(void);             /* Tftp application start          */
static void TftpStop(void);              /* Tftp application stop           */
static void TftpShow(void);              /* Tftp application show           */
static void TftpHide(void);              /* Tftp application hide           */

#define   TIMEOUT      10              /* wait time for retransmission      */
#define   RETRYTIMES   5               /* max number of retries             */

#define   TFTPPORT     69              /* default tftp server port          */

#define   MAXRECSIZE   1000            /* maximum size of receive packet    */
#define   MAXSNDSIZE   516             /* maximum size of send packet       */
#define   MAXDATA      512             /* maximum data area                 */

#define   OPREAD       1               /* read operation code               */ 
#define   OPWRITE      2               /* write operation code              */
#define   OPDATA       3               /* data transfer operation code      */
#define   OPACK        4               /* acknowledge operation code        */
#define   OPERROR      5               /* error operation code              */

#define   FILEBINARY   0               /* binary file transfer mode         */
#define   FILEASCII    1               /* ascii file transfer mode          */

#define   FILEREAD     0               /* read from file                    */
#define   FILEWRITE    1               /* write to file                     */

/* defines for error codes */
#define   NOTDEF       0               /* error code 'Not Defined'          */
#define   FNOTFOUND    1               /* 'File not Found'                  */
#define   ACCESSVIO    2               /* 'Access Violation'                */
#define   DISKFULL     3               /* 'Disk Full / Allocation exceeded' */
#define   ILLOP        4               /* 'Illegal TFTP operation'          */
#define   NOPORT       5               /* 'Unknown Port Number'             */
#define   FILEEXISTS   6               /* 'File already exists'             */
#define   NOUSER       7               /* 'No such User'                    */

/* defines for connection state */
#define   INTERM       0               /* intermediate connection state     */
#define   FIRSTREAD    1               /* waiting for first read operation  */
#define   FIRSTWRITE   2               /* waiting for first write operation */
#define   OTHERREAD    3               /* other read operations             */
#define   OTHERWRITE   4               /* other write operations            */

#define   SOCKADDR     struct sockaddr_in

typedef struct _tftpconn {                /* structure of a tftp connection */
  SOCKET             ThisSocket;          /* socket for conn. with client   */
  FILE               *ThisFile;           /* file for this socket           */
  BYTE               ConState;            /* connection state               */
  BYTE               FileMode;            /* mode: binary or ascii          */
  USHORT             AckSeqNr;            /* number to be acknowledged      */
  int                NextChar;            /* if > 0, first in new packet    */
  BYTE               LastCr;              /* last char written == '\r' ?    */
  BYTE               SaveBuf[MAXSNDSIZE]; /* last packet buffer             */
  int                SaveSize;            /* size of save buffer            */
  USHORT             RetransTimer;        /* retransmit timer for sending   */
  USHORT             RetryCounter;        /* Number of retries              */
  struct _tftpconn   *PrevConn;           /* maintain double linked list    */ 
  struct _tftpconn   *NextConn;           /* maintain double linked list    */ 
  } TFTPCONN;

PWWIN *TftpWin;                           /* Tftp messages window pointer   */

DPAPPS DpaTftp = {
    "Tftp Server",                          /* Application name             */
    ProcessEvents,                          /* Event processing function    */
    0,                                      /* Status, must be 0            */
    DPE_FREETIME | DPE_START | DPE_SHOW | DPE_HIDE |
                   DPE_EVERYSECOND | DPE_STOP | DPE_RESET,
    DPE_START | DPE_RESET,
    0,
    -1,
    0,
    0,
    0,
    0,                                      /* Count of active filters      */
    {0}                                     /* List of active Filters       */
    };

/* global variables */
SOCKET              TftpServer;
TFTPCONN            *ConnList;
BYTE                RecPacket[MAXRECSIZE];
ULONG               NrReadReq = 0;
ULONG               NrWriteReq = 0;
ULONG               ReadBytes = 0;
ULONG               WriteBytes = 0;
ULONG               NrTftpErr = 0;
ULONG               NrSpaceErr = 0;
USHORT              TimeOutVal = TIMEOUT;
USHORT              RetryVal = RETRYTIMES;
int                 TftpVis = 0;
int                 TftpAct = 0;

/* global strings */
char   AscMode[]      = "NETASCII";
char   BinMode[]      = "OCTET";
char   ConnError[]    = "\x0\x5\x0\x0Error creating connection";
int    ConnErrLen     = 30;
char   IllOpStr[]     = "Illegal operation code";
char   IllModeStr[]   = "Illegal mode specified";
char   IllSeqStr[]    = "Illegal sequence number received";
char   WriteErrStr[]  = "Error when writing to disk";
char   ReadErrStr[]   = "Error when reading from disk";
char   IllFormStr[]   = "Illegal format in ack packet";
char   SockRecErr[]   = "Error in socket receive";

/* prototypes of local functions */
static void     RemConn(TFTPCONN *ThisConn);
static TFTPCONN *AddConn(void);          
static void     NetError(TFTPCONN *Conn, int ECode, char *ErrString);
static void     AddNewConn(SOCKADDR *ConnAddr, char *Packet, int PackSize);
static void     ProcessConn(TFTPCONN *ThisConn);
static void     WriteAck(TFTPCONN *ThisConn, USHORT Number);
static void     RecvData(TFTPCONN *ThisConn, BYTE *Buffer, int BufSize);
static void     RecvAck(TFTPCONN *ThisConn, char *Buffer, int BufSize);
static int      FileRead(TFTPCONN *ThisConn);
static int      FileWrite(TFTPCONN *ThisConn, BYTE *Buffer, int Size);
static void     RestartTimer(TFTPCONN *ThisConn);
static int      CheckTimer(TFTPCONN *ThisConn);
static void     CloseFile(TFTPCONN *ThisConn);


/**************************************************************
** NAME:        ProcessEvents
** SYNOPSIS:    static int ProcessEvents(DPEVENTS Event);
**
** DESCRIPTION: Processes dispatcher events.
**
** RETURNS:     0 -->   no error
**              else    error code
**************************************************************/
static int ProcessEvents(DPEVENT Event)
{
  int ret = 0;

  switch (Event) {
    case DPE_START:         TftpStart();          break;
    case DPE_STOP:          TftpStop();           break;
    case DPE_FREETIME:      FreeTime();           break;
    case DPE_RESET:         NrReadReq = NrWriteReq =
                            ReadBytes = WriteBytes =
                            NrTftpErr = NrSpaceErr = 0;
                                                  break;
    case DPE_EVERYSECOND:   EverySecond();        break;
    case DPE_SHOW:          TftpShow();           break;
    case DPE_HIDE:          TftpHide();           break;
    }
  return ret;
}

/**************************************************************
** NAME:        FreeTime
** SYNOPSIS:    static void FreeTime(void);
** 
** DESCRIPTION: Handles the reception of Ping Reply packets,
**              if waiting for a reply.
** RETURNS:  
**************************************************************/
static void FreeTime(void)
{
  SOCKADDR   FromAddr;
  int        FromSize = sizeof(SOCKADDR);
  int        RetCode;
  TFTPCONN   *ThisConn, *SaveConn = NULL;

  if (TftpAct) {
    while ((RetCode = recvfrom(TftpServer, RecPacket, MAXRECSIZE, 0,
                               (struct sockaddr *)&FromAddr, &FromSize)) > 0) {
      AddNewConn(&FromAddr, RecPacket, RetCode);
      }
    if (RetCode != NOMESSAGE) {
      NrTftpErr++;
      }

    for (ThisConn = ConnList; ThisConn != NULL; ThisConn = SaveConn) {
      SaveConn = ThisConn->NextConn;           /* connection may be deleted */
      ProcessConn(ThisConn);
      }

    if (TftpVis) {
      pw_cursor(TftpWin, 0, 25);
      pw_printf(TftpWin, "%8lu", NrWriteReq);
      pw_cursor(TftpWin, 1, 25);
      pw_printf(TftpWin, "%8lu", NrReadReq);
      pw_cursor(TftpWin, 2, 25);
      pw_printf(TftpWin, "%8lu", WriteBytes);
      pw_cursor(TftpWin, 3, 25);
      pw_printf(TftpWin, "%8lu", ReadBytes);
      pw_cursor(TftpWin, 4, 25);
      pw_printf(TftpWin, "%8lu", NrSpaceErr);
      pw_cursor(TftpWin, 5, 25);
      pw_printf(TftpWin, "%8lu", NrTftpErr);
      }
    }
}

/**************************************************************
** NAME:        EverySecond
** SYNOPSIS:    static void EverySecond(void);
**  
** DESCRIPTION: Handles the Dispatcher EverySecond Event.
** RETURNS: 
**************************************************************/
static void EverySecond(void)
{
  TFTPCONN  *TmpConn;

  for (TmpConn = ConnList; TmpConn != NULL; TmpConn = TmpConn->NextConn) {
    if ((TmpConn->ConState != INTERM) && (TmpConn->RetransTimer > 0)) 
      TmpConn->RetransTimer--;
    }
}

/**************************************************************
** NAME:        TftpStart
** SYNOPSIS:    static void TftpStart(void);
**           
** DESCRIPTION: Starts the Tftp server.
** RETURNS:  
**************************************************************/
static void TftpStart(void)
{
  SOCKADDR  BindAdd;

  ConnList = NULL;
  if ((TftpServer = socket(AF_INET, SOCK_DGRAM, 0)) == NULL) {
    NrTftpErr++;
    return;
    }
  BindAdd.sin_family = AF_INET;
  BindAdd.sin_port = htons(TFTPPORT);
  BindAdd.sin_addr.s_addr = INADDR_ANY;
  if (bind(TftpServer, (struct sockaddr *)&BindAdd, sizeof(BindAdd)) != 0) {
    NrTftpErr++;
    closesocket(TftpServer);
    return;
    }
  TftpAct = 1;
}  

/**************************************************************
** NAME:        TftpStop
** SYNOPSIS:    static void TftpStop(void);
**           
** DESCRIPTION: Stops the Tftp server.
** RETURNS:  
**************************************************************/
static void TftpStop(void)
{
  TFTPCONN  *TmpSock;
  
  if (TftpAct) {
    for (TmpSock = ConnList; TmpSock != NULL; TmpSock = TmpSock->NextConn) {
      RemConn(TmpSock);
      }
    closesocket(TftpServer);
    TftpAct = 0;
    }
}

/**************************************************************
** NAME:        TftpShow
** SYNOPSIS:    static void TftpShow(void);
**           
** DESCRIPTION: Starts displaying of the Tftp server.
** RETURNS:  
**************************************************************/
static void TftpShow(void)
{
  if (!TftpVis) {
    if ((TftpWin = pw_open(7, 1, 6, 35, "Tftp Server",
                           PWM_DEFAULT, PWW_NORMAL)) == NULL) {
      return;
      }
    pw_printf(TftpWin, "#Tftp Put Requests:");
    pw_printf(TftpWin, "\n#Tftp Get Requests:");
    pw_printf(TftpWin, "\n#Bytes Put (to disk):");
    pw_printf(TftpWin, "\n#Bytes Get (from disk):");
    pw_printf(TftpWin, "\n#Space Errors:");
    pw_printf(TftpWin, "\n#Other Errors:");
    TftpVis = 1;
    }
}

/**************************************************************
** NAME:        TftpHide
** SYNOPSIS:    static void TftpHide(void);
**           
** DESCRIPTION: Stops displaying of the Tftp server.
** RETURNS:  
**************************************************************/
static void TftpHide(void)
{
  if (TftpVis) {
    TftpVis = 0;
    pw_close(TftpWin);
    }
}

/**************************************************************
** NAME:        RemConn
** SYNOPSIS:    static void RemConn(TFTPCONN *ThisConn);
** 
** DESCRIPTION: Removes 'ThisConn' from the connection list
**              'ConnList'. The socket in 'ThisConn' is
**              closed. The file in 'ThisConn' is closed,
**              if necessary. The buffer space for 'ThisConn'
**              is freed.
** RETURNS:   
**************************************************************/
static void RemConn(TFTPCONN *ThisConn)
{
  _disable();                                        /* protect linked list */
  if (ThisConn->PrevConn == NULL) {
    ConnList = ThisConn->NextConn;
    }
  else {
    ThisConn->PrevConn->NextConn = ThisConn->NextConn;
    }
  if (ThisConn->NextConn != NULL) {
    ThisConn->NextConn->PrevConn = ThisConn->PrevConn;
    }
  _enable();                                        /* end critical section */
  CloseFile(ThisConn);
  closesocket(ThisConn->ThisSocket);
  IPBufFree(ThisConn);
}

/**************************************************************
** NAME:        AddConn
** SYNOPSIS:    static TFTPCONN *AddConn(void);
**              
** DESCRIPTION: Adds a connection to the connection list
**              'ConnList'. Space is allocated and a socket
**              is created. The connection structure is
**              inserted at the head of the connection list.
** RETURNS:     NULL  -->   no space available
**              else  -->   pointer to the connection struct
**************************************************************/
static TFTPCONN *AddConn(void)
{
  TFTPCONN *TmpConn;

  if ((TmpConn = IPBufGet(sizeof(TFTPCONN))) == NULL) {
    NrSpaceErr++;
    return NULL;
    }
  if ((TmpConn->ThisSocket = socket(AF_INET, SOCK_DGRAM, 0)) == NULL) {
    IPBufFree(TmpConn);
    NrSpaceErr++;
    return NULL;
    }
  TmpConn->PrevConn = NULL;
  TmpConn->ThisFile = NULL;                           /* clear file pointer */
  TmpConn->ConState = INTERM;                   /* set to intermediate mode */
  _disable();                                        /* protect linked list */
  if ((TmpConn->NextConn = ConnList) != NULL) {
    TmpConn->NextConn->PrevConn = TmpConn;
    }
  ConnList = TmpConn;
  _enable();                                        /* end critical section */
  return TmpConn;
}

/**************************************************************
** NAME:        AddNewConn
** SYNOPSIS:    static void AddNewConn(SOCKADDR *ConnAddr,
**                          char *Packet, int PackSize);
**     
** DESCRIPTION: Adds a new connection to the connection
**              list. 'Packet' should be the packet received
**              from the client. The filename and filemode are
**              extracted from the packet. The file is opened
**              for reading or writing in the desired mode.
**              The connection structure is initialized.
**              The status is set so that the next processing
**              of the connection structure will result
**              in the desired operation.
**              If an error occures, an error message is sent
**              and the connection is closed.
** RETURNS:     void
**************************************************************/
static void AddNewConn(SOCKADDR *ConnAddr, char *Packet, int PackSize)
{
  TFTPCONN *NewConn;
  int      RetCode, i;
  char     ModeSpec[] = "rt";
  char     *ModeStr, *FileStr;
  USHORT   OpCode;

  if ((NewConn = AddConn()) == NULL) {
    sendto(TftpServer, ConnError, ConnErrLen, 0,
           (struct sockaddr *)ConnAddr, sizeof(SOCKADDR));
    return;
    }
  else {
    if ((RetCode = connect(NewConn->ThisSocket, (struct sockaddr *)ConnAddr,
                           sizeof(SOCKADDR))) < 0) {
      NrTftpErr++;
      RemConn(NewConn);               /* if connect fails, sending an error */
      return;                         /* message doesn't make sense         */
      }
    else {
      OpCode = ntohs(*(USHORT *)Packet);
      if ((OpCode == OPREAD) || (OpCode == OPWRITE)) {
        /* check validity of filename and mode */
        for (i = 2; i < PackSize; i++) 
          if (Packet[i] == '\0')
            break;
        if (i == PackSize) {
          NetError(NewConn, ILLOP, "Illegal filename");
          return;
          }
        ModeStr = Packet + ++i;
        for (; i < PackSize; i++)
          if (Packet[i] == '\0')
            break;
        if (i == PackSize) {
          NetError(NewConn, ILLOP, IllModeStr);
          return;
          }
        StrUpper(ModeStr, ModeStr);
        if (strcmp(ModeStr, AscMode) == 0) 
          NewConn->FileMode = FILEASCII;
        else 
          if (strcmp(ModeStr, BinMode) == 0) {
            NewConn->FileMode = FILEBINARY;
            ModeSpec[1] = 'b';
            }
          else {
            NetError(NewConn, ILLOP, IllModeStr);
            return;
            }

        FileStr = Packet + 2;
        if ((FileStr[0] != '\\') && (FileStr[0] != '/')) {
          NetError(NewConn, ACCESSVIO, "Filename must begin with '\\' or '/'");
          return;
          }
        
        /* extract drive specification, if present */
        if ((FileStr[1] == '\\') || (FileStr[1] == '/')) {
          FileStr++;
          FileStr[0] = FileStr[1];
          FileStr[1] = ':';
          }

        RestartTimer(NewConn);
        NewConn->AckSeqNr = 0;
        NewConn->SaveSize = MAXSNDSIZE;
        NewConn->NextChar = -1;
        NewConn->LastCr = 0;

        if (OpCode == OPREAD) {
          NrReadReq++;
          NewConn->ConState = FIRSTREAD;
          }
        else {
          ModeSpec[0] = 'w';
          NrWriteReq++;
          NewConn->ConState = FIRSTWRITE;
          }

        if ((NewConn->ThisFile = fopen(FileStr, ModeSpec)) == NULL) {
          NetError(NewConn, (OpCode == OPREAD) ? FNOTFOUND : ACCESSVIO,
                   FileStr);
          return;
          }
        }
      else {
        NetError(NewConn, ILLOP, IllOpStr);
        return;
        }
      }
    }
}

/**************************************************************
** NAME:        ProcessConn
** SYNOPSIS:    static void ProcessConn(TFTPCONN *ThisConn);                          char *Packet, int PackSize);
**     
** DESCRIPTION: Handles the Tftp service for 'ThisConn'.
**              If the connection state is 'FIRSTWRITE' or
**              'FIRSTREAD', the transfer is initiated.
**              Else, a packet is read from the connection's
**              socket and processed accordingly. If an
**              error is found in the packet, an error
**              message is sent and the connection is closed.
**              If no message is received on the socket, the
**              timers are checked. If a file is being read,
**              a fragment is retransmitted if timed out.
**              After a number of times, the connection is
**              closed.
**              If a file is written to disk, the connection
**              is closed after a time out.
** RETURNS:     void
**************************************************************/
static void ProcessConn(TFTPCONN *ThisConn)
{
  int    RetCode;
  BYTE   DummyAck[] = "\x0\x4\x0\x0";

  switch (ThisConn->ConState) {
    case FIRSTREAD:
      RecvAck(ThisConn, DummyAck, 4);
      ThisConn->ConState = OTHERREAD;
      break;

    case OTHERREAD:
      if ((RetCode = recv(ThisConn->ThisSocket, RecPacket, MAXRECSIZE, 0)) > 0) {
        RecvAck(ThisConn, RecPacket, RetCode);
        }
      else {
        if (RetCode != NOMESSAGE) {
          NetError(ThisConn, NOTDEF, SockRecErr);
          }
        else {
          switch (CheckTimer(ThisConn)) {
            case 1: send(ThisConn->ThisSocket, ThisConn->SaveBuf,
                         ThisConn->SaveSize, 0);
                    break;
            case 2: NetError(ThisConn, NOTDEF, "Retransmission counted down");
                    break;
            }
          }
        }
      break;

    case FIRSTWRITE:
      WriteAck(ThisConn, 0);
      ThisConn->AckSeqNr++;
      ThisConn->ConState = OTHERWRITE;
      break;

    case OTHERWRITE:
      if ((RetCode = recv(ThisConn->ThisSocket, RecPacket, MAXRECSIZE, 0)) > 0) {
        RecvData(ThisConn, RecPacket, RetCode);
        }
      else {
        if (RetCode != NOMESSAGE) {
          NetError(ThisConn, NOTDEF, SockRecErr);
          }
        else {
          if (CheckTimer(ThisConn) == 2) {
            NetError(ThisConn, NOTDEF, "Connection idle too long");
            }
          }
        }
      break;
    }
}

/**************************************************************
** NAME:        WriteAck
** SYNOPSIS:    static void WriteAck(TFTPCONN *ThisConn,
**                                   USHORT Number);
**     
** DESCRIPTION: Writes an acknowledge message with blocknr
**              'Number' to socket 'ThisSocket'.
** RETURNS:  
**************************************************************/
static void WriteAck(TFTPCONN *ThisConn, USHORT Number)
{
  BYTE  AckPacket[4];

  *(USHORT *)AckPacket = htons(OPACK);
  *(USHORT *)(AckPacket + 2) = htons(Number);
  send(ThisConn->ThisSocket, AckPacket, 4, 0);
}

/**************************************************************
** NAME:        RecvData
** SYNOPSIS:    static void RecvData(TFTPCONN *ThisConn,
**                                   BYTE *Buffer,
**                                   int BufSize);
**         
** DESCRIPTION: Processes putting of files to local disk.
**              If the expected block number is received,
**              the block is written to disk and an
**              acknowledgement is sent to the client.
**              The timer is restarted.
**              If an invalid packet is received, an error
**              is sent and the connection is closed.
** RETURNS:     void
**************************************************************/
static void RecvData(TFTPCONN *ThisConn, BYTE *Buffer, int BufSize)
{
  int     RetCode = MAXDATA;
  USHORT  BlockNr;

  if ((BufSize >= 4) && (BufSize <= MAXSNDSIZE) &&
      (ntohs(*(USHORT *)Buffer) == OPDATA)) {
    BlockNr = ntohs(*(USHORT *)(Buffer + 2));
    if (BlockNr == ThisConn->AckSeqNr) {
      RestartTimer(ThisConn);
      if ((RetCode = FileWrite(ThisConn, Buffer + 4, BufSize - 4)) < 0) {
        return;
        }
      WriteBytes += RetCode;
      ThisConn->AckSeqNr++;
      }
    else {
      if (BlockNr != ThisConn->AckSeqNr - 1) {
        NetError(ThisConn, NOTDEF, IllSeqStr);
        return;
        }
      }
    WriteAck(ThisConn, BlockNr);
    if (RetCode < MAXDATA) {
      RemConn(ThisConn);
      }
    }
  else {
    NetError(ThisConn, ILLOP, "Illegal format data packet");
    return;
    }
}

/**************************************************************
** NAME:        RecvAck
** SYNOPSIS:    static void RecvAck(TFTPCONN *ThisConn,
**                                  BYTE *Buffer,
**                                  int BufSize);
**         
** DESCRIPTION: Processes getting of files from local disk.
**              If the expected block number is received in
**              an acknowledgment, the next the block is
**              read from disk and the block is sent to the
**              client.
**              The timer is restarted.
**              If an invalid packet is received, an error
**              is sent and the connection is closed.
** RETURNS:     void
**************************************************************/
static void RecvAck(TFTPCONN *ThisConn, char *Buffer, int BufSize)
{
  USHORT  BlockNr;
  int     RetCode;

  if ((BufSize == 4) && (ntohs(*(USHORT *)Buffer) == OPACK)) {
    BlockNr = ntohs(*(USHORT *)(Buffer + 2));
    if (BlockNr == ThisConn->AckSeqNr) {
      if (ThisConn->SaveSize < MAXSNDSIZE) {
        RemConn(ThisConn);
        return;
        }
      if ((RetCode = FileRead(ThisConn)) >= 0) {
        ReadBytes += RetCode;
        *(USHORT *)ThisConn->SaveBuf = htons(OPDATA);
        *(USHORT *)(ThisConn->SaveBuf + 2) = htons(++ThisConn->AckSeqNr);
        send(ThisConn->ThisSocket, ThisConn->SaveBuf, ThisConn->SaveSize, 0);
        RestartTimer(ThisConn);
        }
      }
    else {
      if (BlockNr != ThisConn->AckSeqNr - 1) {
        NetError(ThisConn, NOTDEF, IllSeqStr);
        }
      }
    }
  else {
    NetError(ThisConn, ILLOP, IllFormStr);
    }
}

/**************************************************************
** NAME:        FileRead
** SYNOPSIS:    static int FileRead(TFTPCONN *ThisConn);
**             
** DESCRIPTION: Reads the next block for 'ThisConn'.
**              The file descriptor, buffer and size are
**              and fetched from the connection structure.
**              'FileMode' determines if data is read in
**              Ascii or in Binary format. In Ascii mode,
**              the data is converted to tftp format.
**              If a read error occures, a message is sent
**              to the client and the connection is aborted.
**              'ThisConn->NextChar' contains a character to
**              be written first on the next packet. If -1,
**              'NextChar' is empty.
**
** RETURNS:     -1    --> error occured, connection abandoned
**              else  --> number of bytes read
**************************************************************/
static int FileRead(TFTPCONN *ThisConn)
{
  int   Count, Ch;
  BYTE  *StrPtr = ThisConn->SaveBuf + 4;

  if (ThisConn->FileMode == FILEBINARY) {
    Count = fread(StrPtr, 1, MAXDATA, ThisConn->ThisFile);
    if (Count < 0) {
      NetError(ThisConn, NOTDEF, ReadErrStr);
      return -1;
      }
    else {
      ThisConn->SaveSize = Count + 4;
      return Count;
      }
    }
  else {
    if (ThisConn->FileMode == FILEASCII) {
      for (Count = 0; Count < MAXDATA; Count++) {
        if (ThisConn->NextChar >= 0) {
          *StrPtr++ = (BYTE)ThisConn->NextChar;
          ThisConn->NextChar = -1;
          continue;
          }
        Ch = fgetc(ThisConn->ThisFile);
        if (Ch == EOF) {
          if (ferror(ThisConn->ThisFile)) {
            NetError(ThisConn, NOTDEF, ReadErrStr);
            return -1;
            }
          else {
            ThisConn->SaveSize = Count + 4;
            return Count;
            }
          }
        else {
          if (Ch == '\n') {
            Ch = '\r';
            ThisConn->NextChar = '\n';
            }
          else {
            if (Ch == '\r') {
              ThisConn->NextChar = '\0';
              }
            }
          }
        *StrPtr++ = (BYTE)Ch;
        }
      ThisConn->SaveSize = Count + 4;
      return Count;
      }
    else {
      NetError(ThisConn, NOTDEF, IllModeStr);
      return -1;
      }
    }
}

/**************************************************************
** NAME:        FileWrite
** SYNOPSIS:    static int FileWrite(TFTPCONN *ThisConn,
**                                   BYTE *Buffer, int Size);
**             
** DESCRIPTION: Writes 'Size' characters from 'Buffer' to the
**              file specified in 'ThisConn'.
**              The file mode is determined from 'ThisConn'.
**              In Ascii mode, data is converted from the
**              standard tftp format to the local format.
**              If an error occures, in the data format or
**              in writing to the file, an error is sent to
**              the client and the connection is closed.
**              'ThisConn->LastCr' is a boolean for the last
**              character of the previous packet being a
**              '\r'.
**
** RETURNS:     -1    --> error occured, connection abandoned
**              else  --> number of bytes written
**************************************************************/
static int FileWrite(TFTPCONN *ThisConn, BYTE *Buffer, int Size)
{
  int   Count, Ch;

  if (ThisConn->FileMode == FILEBINARY) {
    Count = fwrite(Buffer, 1, Size, ThisConn->ThisFile);
    if (Count < Size) {
      NetError(ThisConn, DISKFULL, WriteErrStr);
      return -1;
      }
    else {
      return Count;
      }
    }
  else {
    if (ThisConn->FileMode == FILEASCII) {
      for (Count = 0; Count < Size; Count++) {
        Ch = *Buffer++;
        if (ThisConn->LastCr) {
          if (Ch == '\0') {
            Ch = '\r';
            }
          else {
            if (Ch != '\n') {
              NetError(ThisConn, NOTDEF, "CR not followed by '\\0' or '\\n'");
              return -1;
              }
            }
          ThisConn->LastCr = 0;
          }
        else {
          if (Ch == '\r') {
            ThisConn->LastCr = 1;
            continue;
            }
          }
        if (fputc(Ch, ThisConn->ThisFile) == EOF) {
          NetError(ThisConn, DISKFULL, WriteErrStr);
          return -1;
          }
        }
      return Count;
      }
    else {
      NetError(ThisConn, NOTDEF, IllModeStr);
      return -1;
      }
    }
}

/**************************************************************
** NAME:        NetError
** SYNOPSIS:    static void NetError(TFTPCONN *Conn,
**                                   int ECode,
**                                   char *ErrString);
**           
** DESCRIPTION: Sends an error message with code 'ECode' and
**              message 'ErrString' to the host at the other
**              end of the connection 'Conn'.
**              'ErrString' is truncated to 507 bytes.
**              The socket in 'Conn' must be connected before
**              calling this function.
**              'Conn' removed after sending the message.
** RETURNS:     
**************************************************************/
static void NetError(TFTPCONN *Conn, int ECode, char *ErrString)
{
  USHORT PackLen;
    
  NrTftpErr++;
  /* prepare packet in connection's send buffer */
  PackLen = min(strlen(ErrString), 507) + 1;
  *(USHORT *)Conn->SaveBuf = htons(OPERROR);
  *(USHORT *)(Conn->SaveBuf + 2) = htons(ECode);
  strncpy(Conn->SaveBuf + 4, ErrString, PackLen);
  PackLen += 4;
  Conn->SaveBuf[PackLen] = '\0';
  send(Conn->ThisSocket, Conn->SaveBuf, PackLen, 0);
  RemConn(Conn);
}

/**************************************************************
** NAME:        RestartTimer
** SYNOPSIS:    static void RestartTimer(TFTPCONN *ThisConn);
**             
** DESCRIPTION: Restarts the timer in 'ThisConn' by setting
**              new values as stored in 'TimeOutVal' and
**              'RetryVal'.
** RETURNS:    
**************************************************************/
static void RestartTimer(TFTPCONN *ThisConn)
{
  _disable();
  ThisConn->RetransTimer = TimeOutVal;
  ThisConn->RetryCounter = RetryVal;
  _enable();
}

/**************************************************************
** NAME:        CheckTimer
** SYNOPSIS:    static int CheckTimer(TFTPCONN *ThisConn);
**             
** DESCRIPTION: Checks the timer in 'ThisConn'. If
**              'RetransTimer' == 0, 'RetryCounter' is
**              decremented if it isn't 0 yet. In the latter
**              case, the timer is expired. Otherwise,
**              'RetransTimer' is initialized again.
** RETURNS:     0   -->  timer not expired yet
**              1   -->  timer value = 0, new cycle started
**              2   -->  timer expired
**************************************************************/
static int CheckTimer(TFTPCONN *ThisConn)
{
  int RetValue = 0;

  _disable();
  if (ThisConn->RetransTimer == 0) {
    if (ThisConn->RetryCounter == 0) {
      RetValue = 2;
      }
    else {
      ThisConn->RetryCounter--;
      ThisConn->RetransTimer = TimeOutVal;
      RetValue = 1;
      }
    }
  _enable();
  return RetValue;
}

/**************************************************************
** NAME:        CloseFile
** SYNOPSIS:    static void CloseFile(TFTPCONN *ThisConn);
**  
** DESCRIPTION: Closes 'ThisConn->ThisFile' if the file
**              pointer is unequal to NULL. The file pointer
**              is set to zero.
** RETURNS:     
**************************************************************/
static void CloseFile(TFTPCONN *ThisConn)
{
  if (ThisConn->ThisFile != NULL) {
    fclose(ThisConn->ThisFile);
    }
  ThisConn->ThisFile = NULL;
}
