/************************************************************************
** MODULE INFORMATION*
**********************
**     FILE     NAME:       pkt.c
**     SYSTEM   NAME:       PKT
**     ORIGINAL AUTHOR(S):  Ling Thio
**     VERSION  NUMBER:     v2.00
**     CREATION DATE:       1990/4/12
**
** DESCRIPTION: Main file for the PDS functions
**              
*************************************************************************
** CHANGES INFORMATION **
*************************
** REVISION:    $Revision:   1.0  $
** WORKFILE:    $Workfile:   PKT.C  $
** LOGINFO:     $Log:   I:/ETSTJAN/C600/BEHOLDER/NPD/PKT/VCS/PKT.C_V  $
**              
**                 Rev 1.0   25 Oct 1990 14:34:12   etstjan
**              Initial revision.
*************************************************************************/
#if ! defined(PRD)
static char _pvcs_hdr[] =
"$Header:   I:/ETSTJAN/C600/BEHOLDER/NPD/PKT/VCS/PKT.C_V   1.0   25 Oct 1990 14:34:12   etstjan  $";
#endif

#include <stdio.h>                          /* for NULL */
#include <memory.h>                         /* for memcmp() */
#include <dos.h>
#include <error.h>
#include <bufm.h>
#include "pktinc.h"                         /* main include file */

/* Disable debug statements */

PKTDEV *PktDevTable[PKTMAXINT];

/*************************************************************************
** NAME:        PktInit                                 [API]
** SYNOPSIS:    int PktInit(AddDevFunc)
**              ADDDEVFUNC AddDevFunc;
** DESCRIPTION: Initializes the PKT library functions.
**              int AddDevFunc(int PktInt);
**              <pAddDevFunc> will be called for each driver.
** RETURNS:     PKT_OK = 0      Successfull call
**              PKT_NO_SPACE    Insufficient memory
**              Error Code      Return code of <pAddDevFunc>
** SEE ALSO:    PktEnd
*************************************************************************/
int PktInit(ADDDEVFUNC AddDevFunc)
{
    int i, Index, ret;
    int PktInt;
    PKTDEV *pDev;
    
    ERR_DEB(ERR_NPD, 5, "PktInit enter");
    for (Index=0; Index<PKTMAXINT; Index++) 
        PktDevTable[Index]=NULL;
    for (Index=0; Index<PKTMAXINT; Index++) /* Scan INT 0x60-0x80 */
    {
        PktInt = Index + PKTINTOFFS;
        if (PdsCheckDriver(PktInt))
        {                                   /* PDS driver present */
            ERR_DEB( ERR_NPD, 5, "PktInit found driver at 0x%x",PktInt);
            pDev = PktDevTable[Index] = IPBufGet(sizeof(PKTDEV));
            if (pDev)
            {                  
                for (i=0; i<PKTMAXHANDLES; i++)
                    pDev->usHandleTable[i] = 0;
                PktGetInfo(PktInt, &(pDev->Info));
                if (AddDevFunc)
                    if (ret=AddDevFunc(PktInt))
                        return ret;             /* Oops, break up */
            }
            else
            {
                ERR_ERR( ERR_NPD, ERR_WARNING,"PktInit no memory for 0x%x",PktInt);
                return PktError(PktInt, AH_INIT, PKT_NO_SPACE);
            }
        }
    }
    ERR_DEB( ERR_NPD, 5, "PktInit okay");
    return PKT_OK;
}

/*************************************************************************
** NAME:        PktEnd                                  [API]
** SYNOPSIS:    int PktEnd(void)
** DESCRIPTION: Terminates the PKT library functions.
** RETURNS:     PKT_OK = 0      Successfull call
** SEE ALSO:    PktInit
*************************************************************************/
int PktEnd(void)
{
    int i;

    ERR_DEB( ERR_NPD, 5, "PktEnd");
    for (i=0; i<PKTMAXINT; i++)
        if (PktDevTable[i])
            IPBufFree(PktDevTable[i]);
    return PKT_OK;
}


/*************************************************************************
** NAME:        PktGetAddress                           [API]
** SYNOPSIS:    int PktGetAddress(PktInt, fpBuf, BufSize)
**              int handle          Handle number
**              char far *fpBuf     Ptr to reply buffer
**              int BufSize    Size of reply buf
** DESCRIPTION: Gets local network address of device.
**              Copies the local network address of device
**                <PktInt> to buffer <fpBuf> of size <BufSize>
** RETURNS:     PKT_OK = 0      Successfull call
**              PKT_BAD_HANDLE  Invalid handle number
**              PKT_NO_SPACE    Buffer size too small
*************************************************************************/
int PktGetAddress(int PktInt, char far *fpBuf, int * BufSize)
{
    int usHandle;
    ERR_DEB( ERR_NPD, 5, "PktGetAddress 0x%x",PktInt);
    usHandle=PktGetHandle(PktInt);
    if (usHandle)
    {
        ERR_DEB( ERR_NPD, 5, "Using handle %d",usHandle);
        return PdsGetAddress(PktInt, usHandle, fpBuf, BufSize);
    }
    else
    {
        int ret;
        _disable();                             /* just to get a valid handle */
        ret = PktAccessType(PktInt, "\xff\xff", 2, NULL, &usHandle);
        if (!ret)
        {
            ERR_DEB( ERR_NPD, 5, "Created handle %d",usHandle);
            ret = PdsGetAddress(PktInt, usHandle, fpBuf, BufSize);
            ret |= PktReleaseType(PktInt, usHandle);
        }
        _enable();
        ERR_DEB( ERR_NPD, 5, "Released handle %d",usHandle);
        return ret;
    }
}


/*************************************************************************
** NAME:        PktResetInterface                       [API]
** SYNOPSIS:    int PktResetInterface(PktInt)
**              int PktInt          Int number [0x60-0x80]
** DESCRIPTION: Resets device driver.
**                Aborts any transmits in process.
**                Reinitializes the receiver.
** RETURNS:     PKT_OK = 0      Successfull call
**              PKT_BAD_HANDLE  Invalid handle number
*************************************************************************/
int PktResetInterface(int PktInt)
{
    ERR_DEB( ERR_NPD, 5, "PktResetInterface");
    return PdsResetInterface(PktInt, PktGetHandle(PktInt));
}


/*************************************************************************
** NAME:        PktSetRcvMode                           [API]
** SYNOPSIS:    int PktSetRcvMode(PktInt, Handle, Mode)
**              int PktInt          Int number [0x60-0x80]
**              int Handle          Handle for type filter
**              int Mode            See below
** DESCRIPTION: Selects receive mode of type filter.
**                1  turn off receiver
**                2  receive IF address packets only
**                3  mode 2 + broadcasts
**                4  mode 3 + limited multicasts
**                5  mode 3 + all multicasts
**                6  all packets
**              note: most PDS drivers support mode 3 only.
** RETURNS:     PKT_OK = 0      Successfull call
**              PKT_BAD_HANDLE  Invalid handle number
**              PKT_BAD_MODE    Invalid mode number
*************************************************************************/
int PktSetRcvMode(int PktInt, int Handle, int Mode)
{
    ERR_DEB(ERR_NPD, 5, "PktSetRcvMode PktInt: %d Mode: %d Handle: %d", PktInt, Mode, Handle);
    return PdsSetRcvMode(PktInt, Handle, Mode);
}

/*************************************************************************
** NAME:        PktGetInfo                              [API]
** SYNOPSIS:    int PktGetInfo(PktInt, pInfo)
**              int PktInt          Int number [0x60-0x80]
**              PKTINFO *pInfo      Ptr to info struct
** DESCRIPTION: Fills an info structure.
** RETURNS:     PKT_OK          Successfull call
**              PKT_BAD_HANDLE  Invalid handle number
*************************************************************************/
int PktGetInfo(int PktInt, PKTINFO *pInfo)
{
    ERR_DEB( ERR_NPD, 5, "PktGetInfo");
    return PdsGetInfo(PktInt, pInfo);   /* No handles defined yet */
}                                       /* When calling this */


/*************************************************************************
** NAME:        PktSendPkt                              [API]
** SYNOPSIS:    int PktSendPkt(PktInt, fpBuf, BufSize)
**              int PktInt          Int number [0x60-0x80]
**              char far *fpBuf     Ptr to send data buffer
**              int BufSize         Bytes to send
** DESCRIPTION: Sends data.
**              Transmits <BufSize> bytes of data, starting at
**                <fpBuf>.  The application must supply the
**                entire packet, including network headers.
**                For example Ethernet:
**                  12 bytes Dest,12 bytes Src,2 bytes type.
** RETURNS:     PKT_OK = 0      Successfull call
**              PKT_CANT_SEND   Packet couldn't be sent
*************************************************************************/
int PktSendPkt(int PktInt, char far *fpBuf, int BufSize)
{
    ERR_DEB( ERR_NPD, 5, "PktSendPacket");
    return PdsSendPkt(PktInt, fpBuf, BufSize);
}

#ifdef FUTURE
/*************************************************************************
** NAME:        PktTerminate                            [API]
** SYNOPSIS:    int PktTerminate(PktInt)
**              int PktInt          Int number [0x60-0x80]
** DESCRIPTION: Terminates the device driver.
**              If possible, the driver will exit and allow
**                MS_DOS to reclaim its memory.
** RETURNS:     PKT_OK = 0      Successfull call
**              PKT_BAD_HANDLE  Invalid handle number
**              PKT_CANT_TERMINATE Could not terminate IF
*************************************************************************/
int PktTerminate(int PktInt)
{
    ERR_DEB( ERR_NPD, 5, "PktTerminate");
    return PdsTerminate(PktInt,PktGetHandle(PktInt));
}
#endif

/**************************************************************
** NAME:        PktDriverInfo                           [API]
** SYNOPSIS:    PKTINFO *PktDriverInfo(PktInt, ppInfo)
**              int PktInt          Int number [0x60-0x80]
** DESCRIPTION: Gets pointer to device driver's info structure.
** RETURNS:     the address of Info structure.
**************************************************************/
PKTINFO *PktDriverInfo(int PktInt)
{
    return &(PktDevTable[PktInt-PKTINTOFFS]->Info);
}


/*
Manufactor  Type    Compatible  Driver  Proc    Description
-----------------------------------------------------------------
WD          WD8003E             WD8003E 8390    8 bit, 8K
WD          WD8003EBT           WD8003E 8390    8 bit, 32K
WD          WD8003ET/A          WD8003E         16 bit mca, 16K
WD          WD8003E/16          WD8003E         16 bit 32K
3com        3c503               3c503   8390    8 bit, 8K
3com        3c523               3c523           16 bit, mca, 8K
SC&C        NE-1    NE1000      NE1000  8390    8 bit
SC&C        NE-2    NE2000                      16 bit
SC&C        NE-3    WD8003EBT   WD8003E         8 bit
Perri-Net   PE-300  NE1000      NE1000  8390    8 bit
-------------------------------------------------------
3c503, PE-300 show problems in high traffic promiscuous mode
*/




