/*******************************************************

    The CalcPlus Class Library Version 1.0,
    Author: Vladimir Schipunov, Copyright (C) 1996

    This library is free software. Permission to use,
    copy, modify and redistribute the CalcPlus library
    for any purpose is hereby granted without fee,
    provided that the above copyright notice appear
    in all copies.

*******************************************************/

#include <iostream.h>
#include <stdlib.h>
#include "yycalc.h"
#include "calcexpr.h"
#include "calctype.h"

//
//  Show statistics about used memory,
//  class MemoryUsage is defined below.
//  Uncomment if you want to see how much
//  dynamic memory application needs.
//
//  #define MEMORY_USAGE

main( int argc, const char** argv )
{
    int i;
    for( i = 1; i<argc; i++ )
    {
        char c0 = argv[i][0];
        char c1 = argv[i][1];
        if((c0=='-'||c0=='/')&&(c1=='d'||c1=='D'))
            Expression::Debug = 1;
    }
    XUserFunction::Argc = argc;
    XUserFunction::Argv = argv;

    if( argc<2 )
    {
        cout << "Usage: CALC File[s] [Args] [/d]" << endl;
        return 1;
    }
    CalcPlus calc;
    for( i = argc-1; i > 0; i-- )
        if( i==1 || argv[i][0]=='@' )
            calc.File( argv[i] );
    if( calc.Run())
        return 1;
    if( !calc.Global->funcs( "atexit" ))
        return 0;
    CString s("exiting... ");
    CLong* p = (CLong*)(calc.Call( "atexit", 1, &s ));
    if( !p )
        return 1;
    return 0;
}

#ifdef MEMORY_USAGE
//
//  Simple control of memory usage.
//  We are trying to detect out of memory error,
//  and also collect statistics about amount of
//  memory requested by application.
//
//  Constructor of the class saves count and number of bytes
//  of allocated memory. Destructor checks if all the allocated
//  memory freed. Note, that order of constructing objects is
//  significant here. For example, this module must be linked first
//  with Borland or last with Watcom.
//

class MemoryUsage
{
public:
    struct meminfo { long count, bytes; };
    meminfo sav, hit;

    MemoryUsage();
    ~MemoryUsage();
    void print( const char* title, const meminfo& ) const;
    void statistics( const meminfo& );
};

static MemoryUsage::meminfo info;
MemoryUsage MemoryCheck;

MemoryUsage::MemoryUsage() : sav( info )
{
    hit.count = hit.bytes = 0;
}

MemoryUsage::~MemoryUsage()
{
#ifdef ON_MEMORY_ERROR_ONLY
    if( !memcmp( &sav, &info, sizeof( meminfo )))
        return;
#endif
    cout << endl
        << "Summary of memory usage" << endl
        << "-----------------------" << endl
        << "\tCount\tBytes"          << endl;
    print( "Entry", sav );
    print( "Exit", info );
    print( "Max",  hit );
    cout << endl;
}

void MemoryUsage::print
(
    const char* title,
    const meminfo& info
)   const
{
    cout << title << "\t" << info.count
         << "\t" << info.bytes << endl;
}

void MemoryUsage::statistics( const meminfo& info )
{
    if( hit.count < info.count )
        hit.count = info.count;
    if( hit.bytes < info.bytes )
        hit.bytes = info.bytes;
}

void* operator new( unsigned n )
{
    if( !n )
        return 0;
    void *p = malloc( n + sizeof( long ));
    if( !p )
    {
        cerr << "Not enough memory!" << endl;
        exit( 1 );
    }
    *(long*)p = n;
    info.count++;
    info.bytes += n;
    MemoryCheck.statistics( info );
    return (char*)p + sizeof( long );
}

void operator delete( void* p )
{
    if( !p )
        return;
    char *q = (char*)p - sizeof( long );
    info.count--;
    info.bytes -= *(long*)q;
    free( q );
}

#endif
