  {--------------------------------------------------------------------------
     UNIT :          XMM                           INITIAL : 19891127 v 1.00
     AUTHOR :        Jeroen W. Pluimers            UPDATE :  19920705 v 1.20

     DESCRIPTION :   eXpanded Memory Manager

     HISTORY :       19891127 - 1.00 - jwp

                         initial translation from XMS 2.0 specification

                     19909322 - 1.11 - jwp

                         final XMS 2.0 implementation

                     19920705 - 1.20 - jwp

                         incorporation of XMS 3.0 specification

     COMPUTER :      NEAT-AT, ERC 386/25
     COMPILER :      TURBO PASCAL 5.0, 5.5 and 6.0

     COPYRIGHT :     (c) 1989-1992 Pluimers Software Ontwikkeling.
   --------------------------------------------------------------------------}

Unit XMM;

{$I Directive.Inc }
 
Interface

Type
  XMMMoveRec = Record Case Boolean of
    False : (Length       : Longint;  { 32-bit number of bytes to transfer   }
             SourceHandle : Word;     { Handle of source block               }
             SourceOffset : Longint;  { 32-bit offset into source            }
             DestHandle   : Word;     { Handle of destination block          }
             DestOffset   : Longint;  { 32-bit offset into destination block }
            );
   True   : (TheLenght    : Longint;  { note that the lenght must be even !  }
             ZeroSource   : Word;     { zero sourcehandle means a seg:ofs    }
             SourcePtr    : Pointer;  { pointer to "normal" memory           }
             ZeroDest     : Word;     { zero dest handle means a seg:ofs     }
             DestPtr      : Pointer;  { pointer to "normal" memory           }
            );
  End; { XMMMoveRec }

Const

  XMMOk              = $00;{ function succeeded.                              }

  XMMNotImplemented  = $80;{ the function is not implemented.                 }
  XMMVDiskFound      = $81;{ a VDISK device is detected.                      }
  XMMA20Err          = $82;{ an A20 error occurs.                             }
  XMMGenErr          = $8E;{ a general driver error occurs.                   }
  XMMUnrecovErr      = $8F;{ an unrecoverable driver error occurs.            }

  XMMHMAnotExist     = $90;{ the HMA does not exist.                          }
  XMMHMAInUse        = $91;{ the HMA is already in use.                       }
  XMMHMAminSize      = $92;{ DX is less than the /HMAMIN= parameter.          }
  XMMHMANotAlloced   = $93;{ the HMA is not allocated.                        }
  XMMA20StillEnabled = $94;{ the A20 line is still enabled.                   }

  XMMOutOfMemory     = $A0;{ all extended memory is allocated.                }
  XMMOutOfHandles    = $A1;{ all available extended memory handles are in use.}
  XMMInvalidHandle   = $A2;{ the handle is invalid.                           }
  XMMSourceHanldeInv = $A3;{ the SourceHandle is invalid.                     }
  XMMSourceOffsetInv = $A4;{ the SourceOffset is invalid.                     }
  XMMDestHanleInv    = $A5;{ the DestHandle is invalid.                       }
  XMMDestOffset      = $A6;{ the DestOffset is invalid.                       }
  XMMLenInv          = $A7;{ the Length is invalid.                           }
  XMMOverlap         = $A8;{ the move has an invalid overlap.                 }
  XMMParity          = $A9;{ a parity error occurs.                           }
  XMMEMBUnlocked     = $AA;{ the block is not locked.                         }
  XMMEMBLocked       = $AB;{ the block is locked.                             }
  XMMLockOverflow    = $AC;{ the block's lock count overflows.                }
  XMMLockFail        = $AD;{ the lock fails.                                  }

Function XMMInstalled : Boolean;
  {---------------------------------------------------------------------------
     ROUTINE :     XMMInstalled

     DESCRIPTION : THIS ROUTINE MUST BE CALLED BEFORE ANY OTHER ROUTINE
                   OR ALL THE OTHER ROUTINES WILL FAIL WITH ERROR CODE $80.

     RETURNS :     False - No XMM driver found.
                   True  - An XMM driver has been found.

     POST :        Internal pointer to XMM driver is established.
   ---------------------------------------------------------------------------}

Function XMMVersion(Var XMSversion,
                        XMSrevision  : Word;
                    Var HMAAvailable : Boolean) : Byte;
  {---------------------------------------------------------------------------
     ROUTINE :     XMMVersion

     DESCRIPTION : Get version numbers. Both the XMS version number and the
                   driver internal revision number are encoded as follows :
                     Low byte  = minor part.
                     High byte = major part.
                   e.g. XMSversion $0277 would mean version 2.77.
                   The HMAAvailable indicates the existence of the HMA (not
                   its availability) and is intended mainly for installation
                   programs.

     OUT :         XMSversion   - XMS version number.
                   XMSrevision  - Driver internal revision number.
                   HMAAvailable - False - no HMA detected.
                                  True  - HMA has been detected.

     RETURNS :     XMMOk              - Always if XMMInstalled has been called.
                   XMMNotImplemented  - XMMInstalled has not been called.

     PRE :         XMMInstalled must have been called first.
   ---------------------------------------------------------------------------}

Function XMMRequestHMA (SpaceNeeded : Word) : Byte;
  {---------------------------------------------------------------------------
     ROUTINE :     XMMRequestHMA

     DESCRIPTION : Attempts to reserve the whole 64K-16 byte high memory
                   area for the caller.
                   If the HMA is currently unused, the caller's size
                   parameter is compared to the /HMAMIN= parameter on
                   the driver's command line.
                   If the value passed by the caller is greater than or
                   equal to the amount specified by the driver's
                   parameter, the request succeeds.
                   This provides the ability to ensure that programs
                   which use the HMA efficiently have priority over
                   those which do not.
                   NOTE: See the sections "Prioritizing HMA Usage" and
                         "High Memory Area Restrictions" in the
                         documentation for more information.

     IN :          SpaceNeeded - Number of bytes in the HMA needed by caller.
                                 It is recommended that if the caller is
                                 an application program this value is
                                 set to 65535 and if the caller is a TSR
                                 the real number of bytes is used.

     RETURNS :     XMMOk              - the HMA is assigned to the caller.
                   XMMNotImplemented  - the function is not implemented.
                   XMMVDiskFound      - a VDISK device is detected.
                   XMMHMANotExist     - the HMA does not exist.
                   XMMHAMInUse        - the HMA is already in use.
                   XMMHAMMinSize      - SpaceNeeded is less than the /HMAMIN=
                                        environment parameter.

     PRE :         XMMInstalled must have been called first.
   ---------------------------------------------------------------------------}

Function XMMReleaseHMA : Byte;
  {---------------------------------------------------------------------------
     ROUTINE :     XMMReleaseHMA

     DESCRIPTION : Releases the high memory area and allows other
                   programs to use it.
                   Programs which allocate the HMA must release it
                   before exiting.
                   When the HMA has been released, any code or data
                   stored in it becomes invalid and should not be
                   accessed.

     RETURNS :     XMMOk              - the HMA is successfully released.
                   XMMNotImplemented  - the function is not implemented.
                   XMMVDiskFound      - a VDISK device is detected.
                   XMMHMANotExist     - the HMA does not exist.
                   XMMNotAllocated    - the HMA was not allocated.

     PRE :         XMMInstalled must have been called first.
   ---------------------------------------------------------------------------}

Function XMMGlobalEnableA20 : Byte;
  {---------------------------------------------------------------------------
     ROUTINE :     XMMGlobalEnableA20

     DESCRIPTION : Attempts to enable the A20 line.
                   It should only be used by programs which have control
                   of the HMA.
                   The A20 line should be turned off via
                   XMMGlobalDisableA20 before a program releases control
                   of the system.
                   NOTE: On many machines, toggling the A20 line is a
                         relatively slow operation.

     RETURNS :     XMMOk              - the A20 line is enabled.
                   XMMNotImplemented  - the function is not implemented.
                   XMMVDiskFound      - a VDISK device is detected.
                   XMMA20Err          - and A20 error occurs.

     PRE :         XMMInstalled must have been called first.
   ---------------------------------------------------------------------------}

Function XMMGlobalDisableA20 : Byte;
  {---------------------------------------------------------------------------
     ROUTINE :     XMMGlobalDisableA20

     DESCRIPTION : Attempts to disable the A20 line.
                   It should only be used by programs which have control
                   of the HMA.
                   The A20 line should be disabled before a program
                   releases control of the system.
                   NOTE: On many machines, toggling the A20 line is a
                         relatively slow operation.

     RETURNS :     XMMOk              - the A20 line is disabled.
                   XMMNotImplemented  - the function is not implemented.
                   XMMVDiskFound      - a VDISK device is detected.
                   XMMA20Err          - and A20 error occurs.
                   XMMA20StillEnabled - the A20 line is still enabled.

     PRE :         XMMInstalled must have been called first.
   ---------------------------------------------------------------------------}

Function XMMLocalEnableA20 : Byte;
  {---------------------------------------------------------------------------
     ROUTINE :     XMMLocalEnableA20

     DESCRIPTION : Attempts to enable the A20 line.
                   It should only be used by programs which need direct
                   access to extended memory.
                   Programs which use this function should call
                   XMMLocalDisableA20 before releasing control of the
                   system.
                   NOTE: On many machines, toggling the A20 line is a
                         relatively slow operation.

     RETURNS :     XMMOk              - the A20 line is enabled.
                   XMMNotImplemented  - the function is not implemented.
                   XMMVDiskFound      - a VDISK device is detected.
                   XMMA20Err          - and A20 error occurs.

     PRE :         XMMInstalled must have been called first.
   ---------------------------------------------------------------------------}

Function XMMLocalDisableA20 : Byte;
  {---------------------------------------------------------------------------
     ROUTINE :     XMMLocalDisableA20

     DESCRIPTION : Cancels a previous call to XMMLocalEnableA20.
                   It should only be used by programs which need direct
                   access to extended memory.
                   Previous calls to XMMLocalEnableA20 must be canceled
                   before releasing control of the system.
                   NOTE: On many machines, toggling the A20 line is a
                         relatively slow operation.

     RETURNS :     XMMOk              - the A20 line is disabled.
                   XMMNotImplemented  - the function is not implemented.
                   XMMVDiskFound      - a VDISK device is detected.
                   XMMA20Err          - and A20 error occurs.
                   XMMA20StillEnabled - the A20 line is still enabled.

     PRE :         XMMInstalled must have been called first.
   ---------------------------------------------------------------------------}

Function XMMQueryA20 (Var A20IsOn : Boolean) : Byte;
  {---------------------------------------------------------------------------
     ROUTINE :     XMMQueryA20

     DESCRIPTION : Checks to see if the A20 line is physically enabled.
                   It does this in a hardware independent manner by
                   seeing if "memory wrap" occurs.

     OUT :         A20IsOn  - False - the A20 line is not physically enabled.
                              True  - the A20 line us physically enabled.

     RETURNS :     XMMOk              - the function succeeds.
                   XMMNotImplemented  - the function is not implemented.
                   XMMVDiskFound      - a VDISK device is detected.

     PRE :         XMMInstalled must have been called first.
   ---------------------------------------------------------------------------}

Function XMMQueryFree (Var LargestFree,
                           TotalFree : Word): Byte;
  {---------------------------------------------------------------------------
     ROUTINE :     XMMQueryFree

     DESCRIPTION : Returns the size of the largest available extended
                   memory block and total free memory in the system.
                   NOTE: The 64K HMA is not included in the returned
                         value even if it is not in use.

     OUT :         LargestFree - Size of the largest free extended
                                 memory block in K-bytes.
                   TotalFree   - Total amount of free extended memory in
                                 K-bytes.

     RETURNS :     XMMOk              - the function succeeds.
                   XMMNotImplemented  - the function is not implemented.
                   XMMVDiskFound      - a VDISK device is detected.
                   XMMOutOfMemory     - all available extended memory is
                                        allocated.

     PRE :         XMMInstalled must have been called first.
   ---------------------------------------------------------------------------}

Function XMMAllocateEMB (    BlockSize : Word;
                         Var Handle    : Word) : Byte;
  {---------------------------------------------------------------------------
     ROUTINE :     XMMAllocateEMB

     DESCRIPTION : Attempts to allocate a block of the given size out of
                   the pool of free extended memory.
                   If a block is available, it is reserved for the
                   caller and a 16-bit handle to that block is returned.
                   The handle should be used in all subsequent extended
                   memory calls.
                   If no memory was allocated, the returned handle is
                   null.

                   NOTE: Extended memory handles are scarce resources.
                         Programs should try to allocate as few as
                         possible at any one time.
                         When all of a driver's handles are in use, any
                         free extended memory is unavailable.

     IN :          BlockSize - Amount of extended memory being requested
                               in K-bytes.

     OUT :         Handle    - 16-bit handle to the allocated block.

     RETURNS :     XMMOk              - the block is allocated.
                   XMMNotImplemented  - the function is not implemented.
                   XMMVDiskFound      - a VDISK device is detected.
                   XMMOutOfMemory     - all available extended memory is
                                        allocated.
                   XMMOutOfHandles    - all available extended memory
                                        handles are in use.

     PRE :         XMMInstalled must have been called first.
   ---------------------------------------------------------------------------}

Function XMMFreeEMB (Handle : Word) : Byte;
  {---------------------------------------------------------------------------
     ROUTINE :     XMMFreeEMB

     DESCRIPTION : This function frees a block of extended memory which
                   was previously allocated using XMMAllocateEMB.
                   Programs which allocate extended memory should free
                   their memory blocks before exiting.
                   When an extended memory buffer is freed, its handle
                   and all data stored in it become invalid and should
                   not be accessed.

     IN :          Handle - 16-bit handle to the allocated block which
                            should be freed.

     RETURNS :     XMMOk              - the block is allocated.
                   XMMNotImplemented  - the function is not implemented.
                   XMMVDiskFound      - a VDISK device is detected.
                   XMMInvalidHandle   - the handle is invalid.
                   XMMEMBLocked       - the block is locked.

     PRE :         XMMInstalled must have been called first.
   ---------------------------------------------------------------------------}

Function XMMMoveEMB (MoveRec : XMMMoveRec) : Byte;
  {---------------------------------------------------------------------------
     ROUTINE :     XMMMoveEMB

     DESCRIPTION : This function attempts to transfer a block of data
                   from one location to another.
                   It is primarily intended for moving blocks of data
                   between conventional memory and extended memory,
                   however it can be used for moving blocks within
                   conventional memory and within extended memory.

                   NOTE: If SourceHandle is set to 0000h, the
                         SourceOffset is interpreted as a standard
                         segment:offset pair which refers to memory that
                         is directly accessible by the processor.
                         The segment:offset pair is stored in Intel
                         DWORD notation.  The same is true for
                         DestHandle and DestOffset.
          
                   SourceHandle and DestHandle do not have to refer to
                   locked memory blocks.
          
                   Length must be even.  Although not required,
                   WORD-aligned moves can be significantly faster on
                   most machines. DWORD aligned move can be even faster
                   on 80386 machines.
          
                   If the source and destination blocks overlap, only
                   forward moves (i.e. where the source base is greater ????
                   than the destination base) are guaranteed to work
                   properly.
          
                   Programs should not enable the A20 line before
                   calling this function.  The state of the A20 line is
                   preserved.

                   This function is guaranteed to provide a reasonable
                   number of interrupt windows during long transfers.

                   The MoveStruc is a structure of type ExtMemMoveRec which
                   is defined as follows :

                     ExtMemMoveRec = Record Case Boolean of
                       False :
                        (Length       : Longint; number of bytes to transfer
                         SourceHandle : Word;    Handle of source block
                         SourceOffset : Longint; offset into source
                         DestHandle   : Word;    Handle of destination block
                         DestOffset   : Longint; offset into dest block
                        );
                      True :
                        (TheLenght    : Longint; lenght must be even !
                         ZeroSource   : Word;    zero sourcehandle means a
                         SourcePtr    : Pointer; ptr to "normal" memory
                         ZeroDest     : Word;    zero dest handle means a
                         DestPtr      : Pointer; ptr to "normal" memory
                        );
                     End; (* ExtMemMoveRec *)

     IN :          MoveStruc - Pointer to an Extended Memory Move
                               Structure (see above).

     RETURNS :     XMMOk              - the move is successful.
                   XMMNotImplemented  - the function is not implemented.
                   XMMVDiskFound      - a VDISK device is detected.
                   XMMA20Err          - an A20 error occurs.
                   XMMSourceHanldeInv - the SourceHandle is invalid.
                   XMMSourceOffsetInv - the SourceOffset is invalid.
                   XMMDestHanleInv    - the DestHandle is invalid.
                   XMMDestOffset      - the DestOffset is invalid.
                   XMMLenInv          - the Length is invalid.
                   XMMOverlap         - the move has an invalid overlap.
                   XMMParity          - a parity error occurs.

     PRE :         XMMInstalled must have been called first.
   ---------------------------------------------------------------------------}

Function XMMLockEMB (    Handle : Word;
                     Var Linear : Longint) : Byte;
  {---------------------------------------------------------------------------
     ROUTINE :     XMMLockEMB

     DESCRIPTION : Locks an extended memory block and returns its base
                   address as a 32-bit linear address.
                   Locked memory blocks are guaranteed not to move.
                   The 32-bit pointer is only valid while the block is
                   locked because unlocked blocks may be moved by the
                   XMS driver.
                   Locked blocks should be unlocked as soon as possible.

                   NOTE: A block does not have to be locked before using
                         XMMMoveEMB.
          
                   "Lock counts" are maintained for EMBs.

     IN :          Handle - Extended memory block handle to lock.

     OUT :         Linear - 32-bit linear address of the locked block.

     RETURNS :     XMMOk              - the block is locked.
                   XMMNotImplemented  - the function is not implemented.
                   XMMVDiskFound      - a VDISK device is detected.
                   XMMInvalidHandle   - the handle is invalid.
                   XMMLockOverflow    - the block's lock count overflows.
                   XMMLockFail        - the lock fails.

     PRE :         XMMInstalled must have been called first.
   ---------------------------------------------------------------------------}

Function XMMUnLockEMB (Handle : Word) : Byte;
  {---------------------------------------------------------------------------
     ROUTINE :     XMMUnLockEMB

     DESCRIPTION : Unlocks a locked extended memory block.  Any 32-bit
                   pointers into the block become invalid and should no
                   longer be used.

     IN :          Handle - Extended memory block handle to unlock.

     RETURNS :     XMMOk              - the block is unlocked.
                   XMMNotImplemented  - the function is not implemented.
                   XMMVDiskFound      - a VDISK device is detected.
                   XMMInvalidHandle   - the handle is invalid.
                   XMMEMBUnlocked     - the block is not locked.

     PRE :         XMMInstalled must have been called first.
   ---------------------------------------------------------------------------}

Function XMMGetHandleInfo (    Handle      : Word;
                           Var LockCount,
                               FreeHandles : Byte;
                           Var BlockLength : Word) : Byte;
  {---------------------------------------------------------------------------
     ROUTINE :     XMMGetHandleInfo

     DESCRIPTION : Returns additional information about an extended
                   memory block to the caller.

                   NOTE: To get the block's base address, use XMMLockEMB.

     IN :          Handle      - Extended memory block handle.

     OUT :         LockCount   - The block's lock count.
                   FreeHandles - Number of free EMB handles in the system.
                   BlockLength - The block's length in K-bytes.


     RETURNS :     XMMOk              - the block's information is found.
                   XMMNotImplemented  - the function is not implemented.
                   XMMVDiskFound      - a VDISK device is detected.
                   XMMInvalidHandle   - the handle is invalid.

     PRE :         XMMInstalled must have been called first.
   ---------------------------------------------------------------------------}
          
Function XMMReallocateEMB (Handle,
                           NewSize : Word) : Byte;
  {---------------------------------------------------------------------------
     ROUTINE :     XMMReallocateEMB

     DESCRIPTION : Attempts to reallocate an unlocked extended memory
                   block so that it becomes the newly specified size.
                   If the new size is smaller than the old block's size,
                   all data at the upper end of the old block is lost.

     IN :          Handle  - Unlocked extended memory block handle to
                             reallocate.
                   NewSize - New size for the extended memory block in
                             K-bytes.

     RETURNS :     XMMOk              - the block is reallocated.
                   XMMNotImplemented  - the function is not implemented.
                   XMMVDiskFound      - a VDISK device is detected.
                   XMMOutOfMemory     - all available extended memory is
                                        allocated.
                   XMMOutOfHandles    - all available extended memory
                                        handles are in use.
                   XMMInvalidHandle   - the handle is invalid.
                   XMMEMBLocked       - the block is locked.

     PRE :         XMMInstalled must have been called first.
   ---------------------------------------------------------------------------}

Function XMMRequestUMB (    RequestSize : Word;
                        Var UMBSeg,
                            BlockSize   : Word) : Byte;
  {---------------------------------------------------------------------------
     ROUTINE :     XMMRequestUMB

     DESCRIPTION : Attempts to allocate an upper memory block to the
                   caller. If the function fails, the size of the
                   largest free UMB is returned in DX.

                   NOTE: By definition UMBs are located below the 1MB
                         address boundary.
                         The A20 Line does not need to be enabled before
                         accessing an allocated UMB.
                         This function does not need to be implemented
                         by an XMS driver.

                   UMBs are paragraph aligned.

                   To determine the size of the largest available UMB,
                   attempt to allocate one with a size of FFFFh.

                   UMBs are unaffected by EMS calls.

     IN :          RequestSize - Size of requested memory block in
                                 paragraphs (one paragraph is 16 bytes).

     OUT :         UMBSeg      - Segment number of the upper memory block.
                   BlockSize   - If the request is granted, actual size
                                 in paragraphs of the allocated block in
                                 paragraphs.
                                 Otherwise, size of the largest
                                 available UMB in paragraphs.

     RETURNS :     XMMOk              - the request is granted.
                   XMMNotImplemented  - the function is not implemented.
                   XMMUMBSizeTooBig   - a smaller UMB is available.
                   XMMNoUMBs          - no UMBs are available.

     PRE :         XMMInstalled must have been called first.
   ---------------------------------------------------------------------------}
  
Function XMMReleaseUMB (UMBSeg : Word) : Byte;
  {---------------------------------------------------------------------------
     ROUTINE :     XMMReleaseUMB

     DESCRIPTION : This function frees a previously allocated upper
                   memory block.
                   When an UMB has been released, any code or data
                   stored in it becomes invalid and should not be
                   accessed.

                   NOTE: This function does not need to be implemented
                         by an XMS driver.

     IN :          UMBSeg - Segment number of the upper memory block.

     RETURNS :     XMMOk              - the block was released.
                   XMMNotImplemented  - the function is not implemented.
                   XMMInvalidUMB      - the UMB segment number is invalid.

     PRE :         XMMInstalled must have been called first.
   ---------------------------------------------------------------------------}
    
Function XMMReallocateUMB (RequestSize : Word;
                           UMBSeg : Word) : Byte;
 { ---------------------------------------------------------------------------
     ROUTINE :     XMMReallocateUMB

     DESCRIPTION : This function attempts to reallocate an Upper Memory
                   Block to a newly specified size. If the new size is
                   smaller than the old block's size, all data at the
                   upper end of the block is lost.

                   NOTE: This function does not need to be implemented
                         by an XMS driver.

     IN :          UMBSeg - Segment number of the upper memory block.

     RETURNS :     XMMOk              - the block was released.
                   XMMNotImplemented  - the function is not implemented.
                   XMMUMBSizeTooBig   - a smaller UMB is available.
                   XMMInvalidUMB      - the UMB segment number is invalid.

     PRE :         XMMInstalled must have been called first.
   --------------------------------------------------------------------------- }

Function XMMQueryAnyFree (Var LargestFree,
                              LastByte,
                              TotalFree : LongInt): Byte;

 { ---------------------------------------------------------------------------
     ROUTINE :     XMMQueryAnyFree

     DESCRIPTION : This function uses 32-bit values to return the size of
                   available memory, thus allowing returns up to 4GByte.
                   Additionally, it returns the highest known physical
                   memory address, that is, the physical address of the
                   last byte of memory.  There may be discontinuities in
                   the memory map below this address.

                   The memory pool reported on is the same as that
                   reported on by the existing Query Free Extended Memory
                   function 08h.  If the highest memory address is not
                   more than 64 Mb, then these two functions will return
                   the same results.

                   Because of its reliance on 32-bit registers, this
                   function is only available on 80386 and higher
                   processors.  XMS drivers on 80286 machines should
                   return error code 80h if this function is called.

                   If error code 81h is returned, the value in ECX will
                   still be valid.  If error code A0h is returned, EAX and
                   EDX will be 0, and ECX will still be valid.

     OUT :         LargestFree - Size of the largest free extended
                                 memory block in K-bytes.
                   LastByte    - Physical address of the last byte in memory.
                   TotalFree   - Total amount of free extended memory in
                                 K-bytes.

     RETURNS :     XMMOk              - the function succeeds.
                   XMMNotImplemented  - the function is not implemented.
                   XMMVDiskFound      - a VDISK device is detected.
                   XMMOutOfMemory     - all available extended memory is
                                        allocated.

     PRE :         XMMInstalled must have been called first.
   --------------------------------------------------------------------------- }

Function XMMAllocateAnyEMB (    BlockSize : Longint;
                            Var Handle    : Word) : Byte;
 { ---------------------------------------------------------------------------
     ROUTINE :     XMMAllocateAnyEMB

     DESCRIPTION : This function is similar to the existing Allocate
                   Extended Memory 09h, except that it uses a 32-bit
                   instead of a 16-bit value to specify the amount of
                   memory requested.  It allocates from the same memory
                   and handle pool as the current function.

                   Since it requires a 32-bit register, this function can
                   be supported only on 80386 and higher processors, and
                   XMS drivers on 80286 machines should return error code
                   80h.

     IN :          BlockSize - Amount of extended memory being requested
                               in K-bytes.

     OUT :         Handle    - 16-bit handle to the allocated block.

     RETURNS :     XMMOk              - the block is allocated.
                   XMMNotImplemented  - the function is not implemented.
                   XMMVDiskFound      - a VDISK device is detected.
                   XMMOutOfMemory     - all available extended memory is
                                        allocated.
                   XMMOutOfHandles    - all available extended memory
                                        handles are in use.

     PRE :         XMMInstalled must have been called first.
   --------------------------------------------------------------------------- }

Function XMMGetExtendedHandleInfo (    Handle      : Word;
                                   Var LockCount   : Byte;
                                   Var FreeHandles : Word;
                                   Var BlockLength : Longint) : Byte;

 { ---------------------------------------------------------------------------
     ROUTINE :     XMMGetExtendedHandleInfo

     DESCRIPTION : This function is similar to the Get EMB Handle
                   Information function. Since it uses a 32-bit register
                   to report the block size, it can be used to get
                   information on blocks larger than 64 Mb.  It also uses
                   a 16-bit instead of 8-bit register to report the number
                   of free handles, allowing the handle pool to be
                   extended beyond 256 entries.

                   Because of its reliance on a 32-bit register, this
                   function is available on 80386 and higher processors.
                   XMS drivers on 80286 machines should return error code
                   80h if this function is called.

     IN :          Handle      - Extended memory block handle.

     OUT :         LockCount   - The block's lock count.
                   FreeHandles - Number of free EMB handles in the system.
                   BlockLength - The block's length in K-bytes.


     RETURNS :     XMMOk              - the block's information is found.
                   XMMNotImplemented  - the function is not implemented.
                   XMMVDiskFound      - a VDISK device is detected.
                   XMMInvalidHandle   - the handle is invalid.

     PRE :         XMMInstalled must have been called first.
   --------------------------------------------------------------------------- }

Function XMMReallocateAnyEMB (Handle,
                              NewSize : Longint) : Byte;

 { ---------------------------------------------------------------------------
     ROUTINE :     XMMReallocateAnyEMB

     DESCRIPTION : This function is similar to the existing Reallocate
                   Extended Memory, except that it uses a 32-bit instead
                   of a 16-bit value to specify the amount of memory
                   requested.  It allocates from the same memory and
                   handle pool as the current function.

                   Since it requires a 32-bit register, this function can
                   be supported only on 80386 and higher processors, and
                   XMS drivers on 80286 machines should return error code
                   80h.

     IN :          Handle  - Unlocked extended memory block handle to
                             reallocate.
                   NewSize - New size for the extended memory block in
                             K-bytes.

     RETURNS :     XMMOk              - the block is reallocated.
                   XMMNotImplemented  - the function is not implemented.
                   XMMVDiskFound      - a VDISK device is detected.
                   XMMOutOfMemory     - all available extended memory is
                                        allocated.
                   XMMOutOfHandles    - all available extended memory
                                        handles are in use.
                   XMMInvalidHandle   - the handle is invalid.
                   XMMEMBLocked       - the block is locked.

     PRE :         XMMInstalled must have been called first.
   --------------------------------------------------------------------------- }




{-----------------------------------------------------------------------------}
{                                                                             }
{                      Implementation                                         }
{                                                                             }
{-----------------------------------------------------------------------------}

Implementation


{$L XMM.OBJ }  { Link in the assembler part }



Function XMMInstalled;                  External;

Function XMMVersion;                    External;

Function XMMRequestHMA;                 External;
Function XMMReleaseHMA;                 External;

Function XMMGlobalEnableA20;            External;
Function XMMGlobalDisableA20;           External;
Function XMMLocalEnableA20;             External;
Function XMMLocalDisableA20;            External;
Function XMMQueryA20;                   External;

Function XMMQueryFree;                  External;

Function XMMAllocateEMB;                External;
Function XMMFreeEMB;                    External;

Function XMMMoveEMB;                    External;

Function XMMLockEMB;                    External;
Function XMMUnLockEMB;                  External;

Function XMMGetHandleInfo;              External;

Function XMMReallocateEMB;              External;

Function XMMRequestUMB;                 External;
Function XMMReleaseUMB;                 External;
Function XMMReallocateUMB;              External;

Function XMMQueryAnyFree;               External;

Function XMMAllocateAnyEMB;             External;

Function XMMGetExtendedHandleInfo;      External;

Function XMMReallocateAnyEMB;           External;

End.


----------------------------------------------------------------------------

   Some notes about HMA usage from the official Microsoft documentation.

----------------------------------------------------------------------------


PRIORITIZING HMA USAGE:
-----------------------

    For DOS users to receive the maximum benefit from the High Memory Area,
programs which use the HMA must store as much of their resident code in it as
is possible.  It is very important that developers realize that the HMA is
allocated as a single unit. 

    For example, a TSR program which grabs the HMA and puts 10K of code into
it may prevent a later TSR from putting 62K into the HMA.  Obviously, regular
DOS programs would have more memory available to them below the 640K line if
the 62K TSR was moved into the HMA instead of the 10K one.

    The first method for dealing with conflicts such as this is to require 
programs which use the HMA to provide a command line option for disabling
this feature.  It is crucial that TSRs which do not make full use of the HMA
provide such a switch on their own command line (suggested name "/NOHMA").

    The second method for optimizing HMA usage is through the /HMAMIN=
parameter on the XMS device driver line.  The number after the parameter
is defined to be the minimum amount of HMA space (in K-bytes) used by any
driver or TSR.	For example, if "DEVICE=HIMEM.SYS /HMAMIN=48" is in a
user's CONFIG.SYS file, only programs which request at least 48K would be
allowed to allocate the HMA.  This number can be adjusted either by
installation programs or by the user himself.  If this parameter is not
specified, the default value of 0 is used causing the HMA to be allocated
on a first come, first served basis.

    Note that this problem does not impact application programs.  If the HMA
is available when an application program starts, the application is free to
use as much or as little of the HMA as it wants.  For this reason,
applications should pass FFFFh in DX when calling Function 01h.

HIGH MEMORY AREA RESTRICTIONS:
------------------------------

-   Far pointers to data located in the HMA cannot be passed to DOS.  DOS
    normalizes any pointer which is passed into it.  This will cause data
    addresses in the HMA to be invalidated.

-   Disk I/O directly into the HMA (via DOS, INT 13h, or otherwise) is not
    recommended.
       
-   Programs, especially drivers and TSRs, which use the HMA *MUST* use
    as much of it as possible.  If a driver or TSR is unable to use at
    least 90% of the available HMA (typically ~58K), they must provide
    a command line switch for overriding HMA usage.  This will allow
    the user to configure his machine for optimum use of the HMA.
       
-   Device drivers and TSRs cannot leave the A20 line permanently turned
    on.  Several applications rely on 1MB memory wrap and will overwrite the
    HMA if the A20 line is left enabled potentially causing a system crash.
        
-   Interrupt vectors must not point into the HMA.  This is a result of
    the previous restriction.  Note that interrupt vectors can point into
    any allocated upper memory blocks however.


The description of all functions and the notes about HMA usage are



              Copyright (c) 1988-1992 Microsoft Corporation.




Because this XMM is public domain (as is HIMEM) I am so free to use
that text and I won't accept any claims from anyone for it.
Furthermore I also use the same disclaimer as in XMS.TXT on the HIMEM
distribution disk :

Disclaimer of Warranty:
-----------------------

I, JEROEN W. PLUIMERS, FURTHER CALLED "THE AUTHOR" AND MY COMPANY
EXCLUDE ANY AND ALL IMPLIED WARRANTIES, INCLUDING WARRANTIES OF
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. NEITHER THE AUTHOR
NOR HIS COMPANY MAKE ANY WARRANTY OF REPRESENTATION, EITHER EXPRESS OR
IMPLIED, WITH RESPECT TO THIS EXTENDED MEMORY MANAGER, ITS QUALITY,
PERFORMANCE, MERCHANTABILITY, OR FITNESS FOR A PARTICULAR PURPOSE.
NEITHER THE AUTHOR NOR HIS COMPANY SHALL HAVE ANY LIABILITY FOR SPECIAL,
INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF OR RESULTING FROM
THE USE OR MODIFICATION OF THIS EXTENDED MEMORY MANAGER.

