/* MS-DOS SHELL - Main I/O Functions
 *
 * MS-DOS SHELL - Copyright (c) 1990,1,2 Data Logic Limited and Charles Forsyth
 *
 * This code is based on (in part) the shell program written by Charles
 * Forsyth and is subject to the following copyright restrictions:
 *
 * 1.  Redistribution and use in source and binary forms are permitted
 *     provided that the above copyright notice is duplicated in the
 *     source form and the copyright notice in file sh6.c is displayed
 *     on entry to the program.
 *
 * 2.  The sources (or parts thereof) or objects generated from the sources
 *     (or parts of sources) cannot be sold under any circumstances.
 *
 *    $Header: /usr/users/istewart/src/shell/sh2.1/RCS/sh5.c,v 2.3 1992/11/06 10:03:44 istewart Exp $
 *
 *    $Log: sh5.c,v $
 *	Revision 2.3  1992/11/06  10:03:44  istewart
 *	214 Beta test updates
 *
 *	Revision 2.3  1992/11/06  10:03:44  istewart
 *	214 Beta test updates
 *
 *	Revision 2.2  1992/09/03  18:54:45  istewart
 *	Beta 213 Updates
 *
 *	Revision 2.1  1992/07/10  10:52:48  istewart
 *	211 Beta updates
 *
 *	Revision 2.0  1992/04/13  17:39:09  Ian_Stewartson
 *	MS-Shell 2.0 Baseline release
 *
 *
 */

#include <sys/types.h>
#include <stdio.h>
#include <signal.h>
#include <errno.h>
#include <setjmp.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <limits.h>
#include <dirent.h>
#include <unistd.h>
#ifdef OS2
#define INCL_DOSSESMGR
#include <os2.h>
#endif
#include "sh.h"

/*
 * here documents
 */

typedef struct here {
    struct here		*h_next;	/* Link to next			*/
    char		*h_EoInputString;
    int			h_dosub;
    IO_Actions		*h_iop;		/* I/O options			*/
} Here_D;

/*
 * shell IO
 */

static IO_Buf		sharedbuf = {AFID_NOBUF};
static IO_Buf		mainbuf = {AFID_NOBUF};
static unsigned int	bufid = AFID_ID;	/* buffer id counter */
					/* list of hear docs while parsing */
static Here_D		*inhere = (Here_D *)NULL;
					/* list of active here documents */
static Here_D		*acthere = (Here_D *)NULL;

static int		WordList1_GetNextCharacter (IO_State *);
static void near	ReadHereFile (Here_D *);
static int		HereFile_GetNextCharacter (IO_State *);
static void near	SetLINENO (unsigned int);

/*
 * Read the next input character
 */

int GetNextCharacter (register int EndCharacter)
{
    register int	c;

    if (e.linep > e.eline)
    {
	while (((c = ReadCharacterFromIOStack ()) != CHAR_NEW_LINE) && c)
	    continue;

	ShellErrorMessage ("input line too long");
	ExpansionErrorDetected = TRUE;
	return c;
    }

    c = ReadCharacterFromIOStack ();

    if ((EndCharacter != CHAR_SINGLE_QUOTE) &&
	(EndCharacter != CHAR_BACKQUOTE) && (XGetTaskType () != X_FROM_STDOUT))
    {
	if (c == '\\')
	{
	    if (((c = ReadCharacterFromIOStack ()) == CHAR_NEW_LINE) &&
		(EndCharacter != CHAR_DOUBLE_QUOTE))
		return GetNextCharacter (EndCharacter);

	    c |= QUOTE;
	}
    }

    return c;
}

/*
 * Unget a character
 */

void ReturnGotCharacter (int c)
{
    if (IOBasePosition () & (IOSTACK_INSIDE | IOSTACK_FIRST))
	e.iop->PeekedChar = c;
}

/*
 * Check for end of file
 */

bool CheckForEndOfFile (void)
{
    return ((IOBasePosition () == IOSTACK_OUTSIDE) ||
	    ((e.iop->PeekedChar == 0) && (e.iop->PreviousChar == 0)))
	    ? TRUE : FALSE;
}

/* Read the next character */

int ReadCharacterFromIOStack (void)
{
    register int	c;
    char		s_SAProcessing = e.iop->StarAmpersandProcessing;

/* The StarAmpersandProcessing is transfered from the higher level to the lower
 * level at end of input at the higher level.  This is part of the
 * implementation of $* and $@ processing.
 */

    while (IOBasePosition () & (IOSTACK_INSIDE | IOSTACK_FIRST))
    {

/* Set up the current StarAmpersandProcessing */

	e.iop->StarAmpersandProcessing = s_SAProcessing;

/* If there is an unget character, use it */

	if ((c = e.iop->PeekedChar) != '\0')
	{
	    e.iop->PeekedChar = 0;
	    return c;
	}

/* Some special processing for multi-line commands */

	else
	{
	    if (e.iop->PreviousChar != 0)
	    {

/* Get the next character from the IO function */

		if ((c = (*e.iop->iofn)(e.iop)) != 0)
		{

/* End of current level, but continue at this level as another read
 * function has been put on the stack
 */

		    if (c == -1)
			continue;

/* If we are at the bottom - echo the character */

		    if ((IOStackPosition (0) & IOSTACK_FIRST) &&
			(FL_TEST ('v')))
			fputc (c, stderr);

/* Return the current character */

		    return (e.iop->PreviousChar = (char)c);
		}

		else if ((XGetTaskType () == X_FILE_IO) &&
			 (e.iop->PreviousChar != CHAR_NEW_LINE))
		{
		    e.iop->PreviousChar = 0;

		    if ((IOStackPosition (0) & IOSTACK_FIRST) &&
			(FL_TEST ('v')))
			fputc (CHAR_NEW_LINE, stderr);

		    return CHAR_NEW_LINE;
		}

		else
		    s_SAProcessing = e.iop->StarAmpersandProcessing;
	    }

	    if (XGetTaskType () == X_FILE_IO)
	    {
		if (AllowMultipleLines)
		    return e.iop->PreviousChar = 0;

/* If we are talking and about to read from the keyboard, check to see if
 * this is the first time.  If it is, we need to read the ENV file
 */

		if (InteractiveFlag &&
		    (IOStackPosition (1) & IOSTACK_FIRST) &&
		    !e.eof_p)
		{
		    if (FirstReadFromUser)
		    {
			FirstReadFromUser = FALSE;

			if (AddToStackForExection (
				GetVariableAsString (ENVVariable, FALSE)))
			    continue;
		    }

/* Output the prompt to read something */

		    LastUserPrompt = PS1;
		}
	    }
	}

/* Move down the IO stack */

	--(e.iop);
	SetLINENO (e.iop->LineCount);

/* If the reset X or R flag bit is set, set the -x or -r option flag and then
 * clear it.  This disables these options whilest the profiles and ENV files
 * are being loaded
 */

	if (IOStackPosition (0) & (IOSTACK_FIRST | IOSTACK_INSIDE))
	{
	    if (e.iop->TaskType & XRESET_XF)
	    {
		FL_SET ('x');
		e.iop->TaskType &= ~XRESET_XF;
	    }

	    if (e.iop->TaskType & XRESET_RF)
	    {
		RestrictedShellFlag = TRUE;
		e.iop->TaskType &= ~XRESET_RF;
	    }
	}
    }

/* End of file detected.  If more data on stack and the special EOF
 * processing is not enabled - return 0
 */

    if ((IOStackPosition (0) & (IOSTACK_INSIDE | IOSTACK_FIRST)) && !e.eof_p)
	return 0;

/* OK, we've about to leave - set the iop pointer to the first entry */

    (e.iop)++;
    SetLINENO (e.iop->LineCount);

/* Check for ignore */

    if (InteractiveFlag && (GlobalFlags & FLAGS_IGNOREEOF))
	return (e.iop->PreviousChar = CHAR_NEW_LINE);

    ExitTheShell (FALSE);
    /* NOTREACHED */
}

/* Add an Input channel to the input stack */

void AddToIOStack (IO_Args *argp, int (*fn)(IO_State *), char *name)
{
    ++e.iop;
    if (IOStackPosition (0) & IOSTACK_OUTSIDE)
    {
	e.iop--;
	ShellErrorMessage ("Shell input nested too deeply");
	ExpansionErrorDetected = TRUE;
	return;
    }

    e.iop->iofn = fn;

    if (argp->afid != AFID_NOBUF)
	e.iop->argp = argp;

    else
    {
	e.iop->argp  = ioargstack + (e.iop - iostack);
	*e.iop->argp = *argp;
	e.iop->argp->afbuf = (IOStackPosition (0) & IOSTACK_FIRST)
				? &mainbuf : &sharedbuf;

	if ((isatty (e.iop->argp->afile) == 0) &&
	    ((IOStackPosition (0) & IOSTACK_FIRST) ||
	     (lseek (e.iop->argp->afile, 0L, SEEK_CUR) != -1L)))
	{
	    if (++bufid == AFID_NOBUF)
		bufid = AFID_ID;

	    e.iop->argp->afid  = bufid;
	}
    }

    e.iop->PreviousChar = (char)(~CHAR_NEW_LINE);
    e.iop->PeekedChar = 0;
    e.iop->NextNonNLCharacter = 0;
    e.iop->NewLineCount = 0;
    SetLINENO ((e.iop->LineCount = 0));	/* Line Count */

/* Save the file name */

    e.iop->FileName = (name == null) ? null : StringSave (name);

/* Set up the task type */

    if ((fn == File_GetNextCharacter) || (fn == FileLine_GetNextCharacter))
	e.iop->TaskType = X_FILE_IO;

    else if ((fn == NonNLStdOut_GetNextCharacter) ||
	     (fn == StdOut_GetNextCharacter) ||
	     (fn == NonQuoteSO_GetNextCharacter))
	e.iop->TaskType = X_FROM_STDOUT;

    else
	e.iop->TaskType = X_ANYOTHER_IO;
}

/*
 * Input generating functions
 */

/*
 * Produce the characters of a string, then a newline, then EOF.
 */

int Line_GetNextCharacter (register IO_State *iop)
{
    register int	c;

    if (iop->argp->aword == (char *)NULL)
	return 0;

    if ((c = *iop->argp->aword++) == 0)
    {
	iop->argp->aword = (char *)NULL;
	return CHAR_NEW_LINE;
    }

    return c;
}

/*
 * Given a list of words, produce the characters
 * in them, with a space after each word.
 */

int WordList_GetNextCharacter (register IO_State *iop)
{
    register char	c;
    register char	**wl;

    if ((wl = iop->argp->awordlist) == (char **)NULL)
	return 0;

    if (*wl != (char *)NULL)
    {
	if ((c = *(*wl)++) != 0)
	    return (c & 0177);

	iop->argp->awordlist++;
	return CHAR_SPACE;
    }

    iop->argp->awordlist = (char **)NULL;
    return CHAR_NEW_LINE;
}

/*
 * Return the characters of a list of words, producing a space between them.
 */

int SpacedWordList_GetNextCharacter (IO_State *iop)
{
    register char	*wp;
    char		cflag;

    if ((wp = *(iop->argp->awordlist)++) != (char *)NULL)
    {
	if (*iop->argp->awordlist == (char *)NULL)
	    iop->StarAmpersandProcessing |= DSA_END;

	cflag = iop->StarAmpersandProcessing;
	PUSHIO (aword, wp, WordList1_GetNextCharacter, null);
	e.iop->StarAmpersandProcessing = cflag;
	return -1;
    }

    return 0;
}

/* Return next character from the word with a space at the end */

static int WordList1_GetNextCharacter (IO_State *iop)
{
    register int c;

    if ((iop->StarAmpersandProcessing & DSA_MODE) == DSA_AMP)
    {
	if (!(iop->StarAmpersandProcessing & DSA_START))
	    iop->StarAmpersandProcessing |= DSA_START;

/* Has the previous word ended */

	else if (iop->StarAmpersandProcessing & DSA_START1)
	{
	    iop->StarAmpersandProcessing &= ~DSA_START1;
	    return CHAR_DOUBLE_QUOTE;
	}
    }

    if (iop->argp->aword == (char *)NULL)
	return 0;

    if ((c = *iop->argp->aword) == '\0')
    {
	if ((iop->StarAmpersandProcessing & DSA_MODE) != DSA_AMP)
	{
	    iop->argp->aword = (char *)NULL;
	    return (iop->StarAmpersandProcessing & DSA_END) ? 0 : CHAR_SPACE;
	}

	if (!(iop->StarAmpersandProcessing & DSA_END1))
	{
	    iop->StarAmpersandProcessing |= DSA_END1;
	    return CHAR_DOUBLE_QUOTE;
	}

	iop->argp->aword = (char *)NULL;
	iop->StarAmpersandProcessing &= ~DSA_END1;
	iop->StarAmpersandProcessing |= DSA_START1;
	return (iop->StarAmpersandProcessing & DSA_END) ? 0 : CHAR_SPACE;
    }

    iop->argp->aword++;
/* if ((iop->StarAmpersandProcessing != DSA_NULL) &&
	any ((char)c, GetVariableAsString (IFS, FALSE))) */
	c |= QUOTE;

    return c;
}

/*
 * Produce the characters from a single word (string).
 */

int String_GetNextCharacter (IO_State *iop)
{
    register int	c;

    return ((iop->argp->aword == (char *)NULL) ||
	    ((c = *(iop->argp->aword++)) == 0)) ? 0 : c;
}

/*
 * Produce quoted characters from a single word (string).
 */

int QuotedString_GetNextCharacter (IO_State *iop)
{
    register int	c;

    return ((iop->argp->aword == (char *)NULL) ||
	    ((c = *(iop->argp->aword++)) == 0)) ? 0 : (c | QUOTE);
}

/*
 * Return the characters from a file.
 */

int File_GetNextCharacter (IO_State *iop)
{
    register IO_Args	*ap = iop->argp;
    register int	i;
    char		c;
    int			RetVal;
    IO_Buf		*bp = ap->afbuf;

    if (ap->afid != AFID_NOBUF)
    {

/* When we reread a buffer, we need to check to see if we have reached the
 * end.  If we have, we need to read the next buffer.  Hence, this loop for
 * the second read
 */

	while (((i = (ap->afid != bp->id)) || (bp->bufp == bp->ebufp)))
	{

/* Are we re-reading a corrupted buffer? */

	    if (i)
		lseek (ap->afile, ap->afpos, SEEK_SET);

/* No, filling so set offset to zero */

	    else
		ap->afoff = 0;

/* Save the start of the next buffer */

	    ap->afpos = lseek (ap->afile, 0L, SEEK_CUR);

/* Read in the next buffer */

	    if ((i = read (ap->afile, bp->buf, sizeof (bp->buf))) <= 0)
		return CloseUpIOStack (iop, TRUE);

/* Set up buffer id, start and end */

	    bp->id    = ap->afid;
	    bp->bufp  = bp->buf + ap->afoff;
	    bp->ebufp = bp->buf + i;
	}

/* Return the next character from the buffer */

	++(ap->afoff);
	RetVal = (int)(*(bp->bufp++) & 0177);
    }

/* If this is the terminal, there is special input processing */

#ifndef NO_HISTORY
    else if (IsConsole (ap->afile))
    {
        if ((RetVal = GetConsoleInput (ap)) == CHAR_NEW_LINE)
	    iop->LineCount = 0;
    }
#endif

    else if ((i = read (ap->afile, &c, sizeof(c))) != sizeof (c))
	return CloseUpIOStack (iop, TRUE);
    
    else
	RetVal = (int)(c & 0177);

/* Increment line count if appropriate */

    if (RetVal == CHAR_NEW_LINE)
	SetLINENO (++(iop->LineCount));
    
    return RetVal;
}

/*
 * Return the characters from a here temp file.
 */

static int HereFile_GetNextCharacter (register IO_State *iop)
{
    char			c;

    if (read (iop->argp->afile, &c, sizeof(c)) != sizeof(c))
	c = (char)CloseUpIOStack (iop, FALSE);

    return c;
}

/*
 * Return the characters produced by a process (`...`).
 * De-quote them if required.  Use in here documents.
 */

int NonQuoteSO_GetNextCharacter (IO_State *iop)
{
    return  StdOut_GetNextCharacter (iop) & ~QUOTE;
}

/*
 * Return the characters produced by a process (`...`).
 * De-quote them if required, and converting CHAR_NEW_LINE to space.
 */

int NonNLStdOut_GetNextCharacter (IO_State *iop)
{
    register int c;

    if ((c = StdOut_GetNextCharacter (iop) & ~QUOTE) == CHAR_NEW_LINE)
	c = CHAR_SPACE;

    return c;
}

/*
 * Process input from a `...` string
 */

int StdOut_GetNextCharacter (IO_State *iop)
{
    register int	c;

    if (iop->NextNonNLCharacter)
    {
	if (iop->NewLineCount)
	{
	    iop->NewLineCount--;
	    return (CHAR_NEW_LINE | QUOTE);
	}

	c = iop->NextNonNLCharacter;
	iop->NextNonNLCharacter = 0;
    }

    else if ((c = File_GetNextCharacter (iop)) == CHAR_NEW_LINE)
    {
	iop->NewLineCount = 1;

	while ((c = File_GetNextCharacter (iop)) == CHAR_NEW_LINE)
	    iop->NewLineCount++;

	iop->NextNonNLCharacter = (char)c;

	if (c == 0)
	    return c;

	iop->NewLineCount--;
	c = CHAR_NEW_LINE;
    }

    return (c != 0) ? (c | QUOTE) : 0;
}

/*
 * Return a single command (usually the first line) from a file.
 */

int FileLine_GetNextCharacter (IO_State *iop)
{
    register int	c;

    if (((c = File_GetNextCharacter (iop)) == CHAR_NEW_LINE) &&
	(!AllowMultipleLines))
	CloseUpIOStack (iop, TRUE);

    return c;
}

/*
 * Close all file handlers
 */

void CloseAllHandlers (void)
{
    register int	u;

    for (u = NUFILE; u < NOFILE;)
	S_close (u++, TRUE);
}

/*
 * remap fd into Shell's fd space
 */

int ReMapIOHandler (int fd)
{
    int		i;
    int		n_io = 0;
    int		map[NOFILE];
    int		o_fd = fd;

    if (fd < e.FirstAvailableFileHandler)
    {
	do
	{
	    map[n_io++] = fd;
	    fd = dup (fd);

	} while ((fd >= 0) && (fd < e.FirstAvailableFileHandler));

	for (i = 0; i < n_io; i++)
	    close (map[i]);

	S_Remap (o_fd, fd);
	S_close (o_fd, TRUE);

	if (fd < 0)
	    ShellErrorMessage ("too many files open");
    }

    return fd;
}

/*
 * here documents
 */

void SaveHereFileInfo (register char *s, IO_Actions *iop)
{
    register Here_D	*h, *lh;

    if ((h = (Here_D *) GetAllocatedSpace (sizeof(Here_D))) == (Here_D *)NULL)
	return;

    if ((h->h_EoInputString = evalstr (s, DOSUB)) == (char *)NULL)
	return;

    h->h_iop     = iop;
    iop->io_name = (char *)NULL;
    h->h_next    = (Here_D *)NULL;

    if (inhere == (Here_D *)NULL)
	inhere = h;

    else
    {
	for (lh = inhere; lh != (Here_D *)NULL; lh = lh->h_next)
	{
	    if (lh->h_next == (Here_D *)NULL)
	    {
		lh->h_next = h;
		break;
	    }
	}
    }

    iop->io_flag |= IOHERE|IOXHERE;

    for (s = h->h_EoInputString; *s; s++)
    {
	if (*s & QUOTE)
	{
	    iop->io_flag &= ~IOXHERE;
	    *s &= ~QUOTE;
	}
    }

    h->h_dosub = iop->io_flag & IOXHERE;
}

void GetAllHereFiles (void)
{
    register Here_D	*h, *hp;

/* Scan here files first leaving inhere list in place */

    for (hp = h = inhere; h != (Here_D *)NULL; hp = h, h = h->h_next)
	ReadHereFile (h);

/* Make inhere list active - keep list intact for scraphere */

    if (hp != (Here_D *)NULL)
    {
	hp->h_next = acthere;
	acthere    = inhere;
	inhere     = (Here_D *)NULL;
    }
}

static void near ReadHereFile (Here_D *h)
{
    int			tf;
    register int	c;
    int			EndInputCharacter = h->h_dosub ? 0 : CHAR_SINGLE_QUOTE;
    jmp_buf		ev;
    char		*line;
    char		*next;
    int			stop_len;
    int			c_len;
    FILE		*OutputFP;
    char		*EoInputString = h->h_EoInputString;

/* Create a temporary file and open it */

    h->h_iop->io_name = StringCopy (GenerateTemporaryFileName ());

    if ((tf = S_open (FALSE, h->h_iop->io_name, O_CMASK | O_NOINHERIT,
		      0600)) < 0)
	return;

    if (CreateNewEnvironment (setjmp (ErrorReturnPoint = ev)) == TRUE)
	S_Delete (tf);

    else
    {
	AddToIOStack (e.iop->argp, e.iop->iofn, h->h_iop->io_name);
	e.iobase = e.iop;

/* Strip leading tabs? */

	if (h->h_iop->io_flag & IOTHERE)
	{
	    while (*EoInputString && (*EoInputString == CHAR_TAB))
		++EoInputString;
	}

/* Open the Output buffer */

	line = GetAllocatedSpace ((stop_len = strlen (EoInputString) + 2) + 1);

	if ((OutputFP = fdopen (tf, "wt")) == (FILE *)NULL)
	{
	    ShellErrorMessage (Outofmemory1);
	    TerminateCurrentEnvironment ();
	}

/* Read in */

	while (1)
	{
	    next = line;
	    c_len = 0;

/* Set Prompt to next the next part of the command */

	    if (InteractiveFlag &&
		(IOStackPosition (1) & (IOSTACK_OUTSIDE | IOSTACK_FIRST)))
		LastUserPrompt = PS2;

/* Read the here document */

	    while (((c = GetNextCharacter (EndInputCharacter))
			!= CHAR_NEW_LINE) && c)
	    {

/* Strip leading tabs? */

		if ((h->h_iop->io_flag & IOTHERE) && (c == CHAR_TAB) &&
		    (next == line))
		    continue;

		if (EndInputCharacter == CHAR_SINGLE_QUOTE)
		    c &= ~QUOTE;

/* If not end of search string, add to search string */

		if ((++c_len) < stop_len)
		    *(next++) = (char)c;

/* If one greater that search string, buffer search string */

		else
		{
		    if (c_len == stop_len)
		    {
			*next = 0;
			fputs (line, OutputFP);
		    }

/* Buffer the current character */

		    fputc ((char)c, OutputFP);
		}
	    }

/* Check for end of document */

	    *next = 0;
	    if (strcmp (EoInputString, line) == 0 || c == 0)
		break;

	    if (c_len < stop_len)
	        fputs (line, OutputFP);

	    fputc (CHAR_NEW_LINE, OutputFP);
	}

	fclose (OutputFP);

	if (c == 0)
	    ShellErrorMessage ("here document `%s' unclosed\n", EoInputString);

	QuitCurrentEnvironment ();
    }

    S_close (tf, TRUE);
}

/*
 * open here temp file.
 * If unquoted here, expand here temp file into second temp file.
 */

int OpenHereFile (char *hname, int xdoll)
{
    register int	hf, tf;

/* If the here document is invalid or does not exist, return */

    if (hname == (char *)NULL)
	return -1;

    if ((hf = S_open (FALSE, hname, O_RDONLY)) < 0)
	return -1;

/* If processing for $, ` and ' is required, do it */

    if (xdoll)
    {
	char		c;
	char		*tname = StringCopy (GenerateTemporaryFileName ());
	FILE		*OutputFP;
	jmp_buf		ev;

	if ((tf = S_open (FALSE, tname, O_CMASK | O_NOINHERIT, 0600)) < 0)
	    return -1;

	if (CreateNewEnvironment (setjmp (ErrorReturnPoint = ev)) == FALSE)
	{
	    if ((OutputFP = fdopen (tf, "wt")) == (FILE *)NULL)
	    {
		ShellErrorMessage (Outofmemory1);
		TerminateCurrentEnvironment ();
	    }

	    PUSHIO (afile, hf, HereFile_GetNextCharacter, tname);
	    e.iobase = e.iop;

	    while ((c = (char)subgetc (0, 0)) != 0)
	    {

/* Determine how many characters to write.  If MSB set, write \x.
 * Otherwise, write x
 */

		if ((c & QUOTE) && !any ((char)(c & ~QUOTE), "$`\\"))
		    fputc ('\\', OutputFP);

		fputc ((char)(c & (~QUOTE)), OutputFP);
	    }

	    QuitCurrentEnvironment ();
	}

	else
	    S_Delete (tf);

	fclose (OutputFP);
	S_close (tf, TRUE);
	return S_open (TRUE, tname, O_RDONLY);
    }

    else
	return hf;
}

void ScrapHereList (void)
{
    register Here_D	*h;

    for (h = inhere; h != (Here_D *)NULL; h = h->h_next)
    {
	if ((h->h_iop != (IO_Actions *)NULL) &&
	    (h->h_iop->io_name != (char *)NULL))
	    unlink (h->h_iop->io_name);
    }

    inhere = (Here_D *)NULL;
}

/*
 * unlink here temp files before a ReleaseMemoryArea (area)
 */

void FreeAllHereFiles (int area)
{
    register Here_D	*h;
    register Here_D	*hl = (Here_D *)NULL;

    for (h = acthere; h != (Here_D *)NULL; hl = h, h = h->h_next)
    {
	if (GetMemoryAreaNumber ((void *)h) >= area)
	{
	    if (h->h_iop->io_name != (char *)NULL)
		unlink (h->h_iop->io_name);

	    if (hl == (Here_D *)NULL)
		acthere = h->h_next;

	    else
		hl->h_next = h->h_next;
	}
    }
}

/*
 * Generate a temporary filename
 */

char *GenerateTemporaryFileName (void)
{
    static char	tmpfile[FFNAME_MAX];
    char	*tmpdir;	/* Points to directory prefix of pipe	*/
    static int	temp_count = 0;
    char	*sep = DirectorySeparator;

/* Find out where we should put temporary files */

    if (((tmpdir = GetVariableAsString ("TMP", FALSE)) == null) &&
	((tmpdir = GetVariableAsString (HomeVariableName, FALSE)) == null) &&
	((tmpdir = GetVariableAsString ("TMPDIR", FALSE)) == null))
	tmpdir = ".";

    if (any (tmpdir[strlen (tmpdir) - 1], "/\\"))
	sep = null;

/* Get a unique temporary file name */

    while (1)
    {
	sprintf (tmpfile, "%s%ssht%.5u.tmp", tmpdir, sep, temp_count++);

	if (access (tmpfile, F_OK) != 0)
	    break;
    }

    return tmpfile;
}

/* Test to see if the current Input device is the console */

bool Interactive (void)
{
    return (InteractiveFlag && (XGetTaskType () == X_FILE_IO) &&
	    isatty (e.iop->argp->afile)) ? TRUE : FALSE;
}

/*
 * Close up a IO_Stack
 */

int CloseUpIOStack (IO_State *iop, bool CheckSTDIO)
{
    if ((!CheckSTDIO) || (iop->argp->afile > STDERR_FILENO))
	S_close (iop->argp->afile, TRUE);

    if (iop->FileName != null)
	SetMemoryAreaNumber ((void *)iop->FileName, MemoryAreaLevel + 1);

    iop->FileName = null;
    iop->argp->afile = -1;	/* illegal value */
    return 0;
}

/*
 * Open and add a file to the I/O stack for execution
 */

bool	AddToStackForExection (char *name)
{
    int		fp;

    if ((fp = OpenForExecution (name, (char **)NULL, (int *)NULL)) >= 0)
    {
	PUSHIO (afile, ReMapIOHandler (fp), File_GetNextCharacter, name);
	return TRUE;
    }

    return FALSE;
}

/*
 * Check IO stack position
 */

int	IOStackPosition (int offset)
{
    if (e.iop == iostack + offset)
	return IOSTACK_FIRST;

    return ((e.iop > iostack + offset) && (e.iop < &iostack[NPUSH]))
	    ? IOSTACK_INSIDE : IOSTACK_OUTSIDE;
}

int	IOBasePosition (void)
{
    if (e.iop == e.iobase)
	return IOSTACK_FIRST;

    return ((e.iop > e.iobase) && (e.iop < &iostack[NPUSH]))
	    ? IOSTACK_INSIDE : IOSTACK_OUTSIDE;
}

/*
 * Set up LINENO variable
 */

static void near SetLINENO (unsigned int Count)
{
    if (!(DisabledVariables & DISABLE_LINECOUNT))
	SetVariableFromNumeric (LineCountVariable, (long)Count);
}
