/*+
    Name:	hlgraph.c
    Date:	30-May-1988
    Author:	Kent J. Quirk
		(c) Copyright 1988 Ziff Communications Co.
    Abstract:	This program performs a graphics video benchmark.
    History:	09-Sep-88   kjq     Version 1.00
-*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <graph.h>
#include <conio.h>
#include <math.h>
#include "hl.h"
#include "hltimes.h"

#define MAXROWS 10
#define MAXCOLS 10

int nrows = 3;
int ncols = 3;
int vmax = 400;

int graphdata[MAXROWS][MAXCOLS] = {
    {  277, 251, 226, 246     } ,
    {  295, 295, 240, 170     } ,
    {  309, 298, 247, 145     } ,
    {  337, 290, 241, 139     } 
};

extern int *colorlist;
int base_color = 0;
struct videoconfig config;

TIME_REC tests[] = {
    {  0L, ""	  } ,
    {  0L, " 400 small areas   "     } ,
    {  0L, " 100 large areas   "     } ,
    {  0L, " 400 small ellipses"     } ,
    {  0L, " 200 large ellipses"     } ,
    {  0L, "4000 small lines   "     } ,
    {  0L, "2000 large lines   "     } ,
    {  0L, "General graphs     "     } ,
    { -1L, "HLGRAPH4"	  } 
};

#define NRECT    100
#define NELLIPSE 200
#define NLINE    2000
#define PIXFMT	 " %5ldK pels\000"

/**** f a t _ r e c t a n g l e ****
    Abstract:	Draws a rectangle with a drop shadow
    Parameters: short x1, y1, x2, y2 -- the corners of the rectangle
		short depth - the depth of the shadow (in pixels)
		int shadow, front -- the color of the shadow and front
		    (which is modified by base_color for visual interest)
    Returns:	nothing
****************************/
void fat_rectangle(x1, y1, x2, y2, depth, shadow, front)
short x1, y1, x2, y2, depth;
int shadow, front;
{
    int i;

    _setcolor(colorlist[(base_color+shadow) % 16]);
    for (i=depth; i>0; i--)
	_rectangle(_GBORDER, x1+i, y1-i, x2+i, y2-i);
    _setcolor(colorlist[(base_color+front) % 16]);
    _rectangle(_GFILLINTERIOR, x1, y1, x2, y2);
    _setcolor(0);
    _moveto(x2+1, y2);
    _lineto(x2+depth+1, y2-depth);
}

/**** b a r _ g r a p h ****
    Abstract:	Given the global graphdata[], this draws a bar graph
		with that data.
    Parameters: int x, y -- the position of the lower left corner
		int max_x, max_y -- the position of the upper right corner
    Returns:	Nothing
****************************/
void bar_graph(x, y, max_x, max_y)
int x, y, max_x, max_y;
{
    int barwidth;
    int yscale;
    int i, j;
    long t;
    int xl, hgt;
    int depth;

    _setlogorg(x, y);

    barwidth = max_x / (ncols * (nrows+1) - 1);
    depth = barwidth / 4;

    for (i=0; i<ncols; i++)
    {
	for (j=0; j<nrows; j++)
	{
	    xl = (i * (nrows+1) + j) * barwidth;
	    t = (long)graphdata[j][i] * (long)max_y;
	    hgt = (int)(t / (long)vmax);
	    fat_rectangle(xl+1, 0, xl + barwidth, hgt,
	    depth, j, j+8);
	}
    }

    _setcolor(0xFF);
    _moveto(0, max_y);
    _lineto(0, 0);
    _lineto(max_x, 0);
    if (++base_color >= 16)
	base_color = 0;
}

/**** p i e _ g r a p h ****
    Abstract:	In a manner similar to bar graph, this draws a pie chart.
****************************/
void pie_graph(x, y, wid, hgt)
int x, y, wid, hgt;
{
    _setlogorg(x,y);
    _setcolor(colorlist[base_color]);
    _pie(_GFILLINTERIOR, 0, 0, wid, hgt, wid, hgt/2, wid, hgt);
    _setcolor(colorlist[(base_color+1) % 16]);
    _pie(_GFILLINTERIOR, 0, 0, wid, hgt, wid, hgt, 0, hgt);
    _setcolor(colorlist[(base_color+2) % 16]);
    _pie(_GFILLINTERIOR, 0, 0, wid, hgt, 0, hgt, wid, hgt/2);
    _setcolor(0xFF);
    _pie(_GBORDER, 0, 0, wid, hgt, wid, hgt/2, wid, hgt);
    _pie(_GBORDER, 0, 0, wid, hgt, wid, hgt, 0, hgt);
    _pie(_GBORDER, 0, 0, wid, hgt, 0, hgt, wid, hgt/2);
    if (++base_color >= 16)
	base_color = 0;
}

/**** l i n e _ g r a p h ****
    Abstract:	In a manner similar to bar graph, this draws a line graph.
****************************/
void line_graph(x, y, wid, hgt)
int x, y, wid, hgt;
{
    int xstep = 4;
    int ystep = 4;
    int i;
    int x1, y1, x2, y2;

    y1 = y2 = hgt;                      /* realize that hgt is negative */
    x1 = xstep;
    x2 = wid;

    _setlogorg(x,y);
    _setcolor(colorlist[base_color]);
    _moveto(0, y1);
    while (x2 >= 0 && y2 <= 0)
    {
	_lineto(x1, 0);
	_lineto(x2, y2);
	_lineto(0, y1);
	x1 += xstep;
	x2 -= xstep;
	y1 += ystep/2;
	y2 += ystep;
    }
    if (++base_color >= 16)
	base_color = 0;
}

/**** l i n e s ****
    Abstract:	This tests line drawing speed of a video adapter
		by drawing a number of lines at random positions and 
		colors.
    Parameters: int npts -- the number of lines to draw
		int max_x, max_y -- the maximum x and y values to use
    Returns:	nothing
****************************/
void lines(npts, max_x, max_y)
int npts;
int max_x, max_y;
{
    int x, y, i;

    _moveto(0,0);
    srand(1);
    for (i=0; i<npts; i++)
    {
	x = rand() % max_x;
	y = rand() % max_y;
	_setcolor(colorlist[base_color]);
	_lineto(x, y);
	if (++base_color >= 16)
	    base_color = 0;
    }
}

/**** r e c t s ****
    Abstract:	Draws a number of rectangles at various positions and colors.
    Parameters: int npts -- the number of lines to draw
		int max_x, max_y -- the maximum x and y values to use
    Returns:	nothing
****************************/
void rects(n, max_x, max_y)
int n;
int max_x, max_y;
{
    int i, x, y, wid, hgt;

    srand(1);
    for (i=0; i<n; i++)
    {
	wid = rand() % max_x;
	hgt = rand() % max_y;
	x = rand() % (max_x - wid);
	y = rand() % (max_y - hgt);
	_setcolor(colorlist[base_color]);
	_rectangle(_GFILLINTERIOR, x, y, x+wid, y+hgt);
	if (++base_color >= 16)
	    base_color = 0;
    }
}

/**** e l l i p s e s ****
    Abstract:	Draws a number of ellipses at various positions and colors.
    Parameters: int npts -- the number of lines to draw
		int max_x, max_y -- the maximum x and y values to use
    Returns:	nothing
****************************/
void ellipses(n, max_x, max_y)
int n;
int max_x, max_y;
{
    int i, x, y, wid, hgt;

    srand(1);
    for (i=0; i<n; i++)
    {
	wid = rand() % max_x;
	hgt = rand() % max_y;
	x = rand() % (max_x - wid);
	y = rand() % (max_y - hgt);
	_setcolor(colorlist[base_color]);
	_ellipse(_GBORDER, x, y, x+wid, y+hgt);
	if (++base_color >= 16)
	    base_color = 0;
    }
}

/**** c a l c _ l i n e s ****
    Abstract:	Given the same parameters as lines(), this calculates the 
		number of pixels drawn.
****************************/
long calc_lines(npts, max_x, max_y)
int npts;
int max_x, max_y;
{
    int x, y, i;
    int ox=0, oy=0;
    double dx, dy;
    long npixels = 0L;

    srand(1);
    for (i=0; i<npts; i++)
    {
	x = rand() % max_x;
	y = rand() % max_y;
	dx = (double)(x-ox);
	dy = (double)(y-oy);
	npixels += (long)sqrt(dx*dx + dy*dy);
	oy = y;
	ox = x;
    }
    return(npixels/1000L);
}

/**** c a l c _ r e c t s ****
    Abstract:	Given the same parameters as rects(), this calculates the 
		number of pixels drawn.
****************************/
long calc_rects(n, max_x, max_y)
int n;
int max_x, max_y;
{
    int i, x, y, wid, hgt;
    long npixels = 0L;

    srand(1);
    for (i=0; i<n; i++)
    {
	wid = rand() % max_x;
	hgt = rand() % max_y;
	x = rand() % (max_x - wid);
	y = rand() % (max_y - hgt);
	npixels += (long)wid * hgt;
    }
    return(npixels/1000L);
}

/**** c a l c _ e l l i p s e s ****
    Abstract:	Given the same parameters as ellipses(), this calculates the 
		number of pixels drawn.
****************************/
long calc_ellipses(n, max_x, max_y)
int n;
int max_x, max_y;
{
    int i, x, y, wid, hgt;
    long npixels = 0L;
    long np1, np2;

    srand(1);
    for (i=0; i<n; i++)
    {
	wid = rand() % max_x;
	hgt = rand() % max_y;
	x = rand() % (max_x - wid);
	y = rand() % (max_y - hgt);
	np1 = (long)(wid + hgt) * 2L;
	np2 = 4L * (long)sqrt((double)wid*wid/4 + (double)hgt*hgt/4);
	npixels += (np1+np1+np2)/3L;
    }
    return(npixels/1000L);
}

/**** u s a g e ****
    Abstract:	Displays brief usage message and terminates.
    Returns:	Never returns.
****************************/
void usage()
{
    printf("Usage: HLGRAPH [-?] [-xXCOUNT] [-yYCOUNT]\n");
    printf("  XCOUNT and YCOUNT are the number of graph repetitions.\n");
    exit(1);
}


int main(argc, argv)
int argc;
char *argv[];
{
    int i, x, y, xp, yp, dx, dy;
    int xcount = 5;
    int ycount = 3;
    int batch = 0;
    int bench = 0;
    int wid, hgt;
    int test = 1;
    int program = -1;
    char *filename = NULL;

    for (i=1; i<argc; i++)
    {
	if (argv[i][0] != '-')
	    usage();

	switch (tolower(argv[i][1])) {
	case 'x':
	    xcount = atoi(argv[i]+2);
	    break;
	case 'y':
	    ycount = atoi(argv[i]+2);
	    break;
	case 'a':
	    batch = 1;
	    break;
	case 'b':
	    bench = 1;
	    break;
	case 'f':
	    filename = argv[i]+2;
	    break;
	case 'p':
	    program = atoi(argv[i]+2);
	    break;
	default:
	case '?':
	    usage();
	}
    }

    if (!init_video((struct videoconfig far *)&config))
    {
	printf("Couldn't set any graphics mode!\n");
	exit(1);
    }
    start_timer(0);

    start_timer(1);
    rects(NRECT*4, config.numxpixels/2, config.numypixels/2);
    stop_timer();
    sprintf(tests[test].desc+strlen(tests[test].desc), PIXFMT,
    calc_rects(NRECT*4, config.numxpixels/2, config.numypixels/2));
    tests[test++].ticks = get_timer(1);

    start_timer(1);
    rects(NRECT, config.numxpixels, config.numypixels);
    stop_timer();
    sprintf(tests[test].desc+strlen(tests[test].desc), PIXFMT,
    calc_rects(NRECT, config.numxpixels, config.numypixels));
    tests[test++].ticks = get_timer(1);

    _setcolor(0);
    _rectangle(_GFILLINTERIOR, 0, 0, config.numxpixels, config.numypixels);

    start_timer(1);
    ellipses(NELLIPSE*2, config.numxpixels/2, config.numypixels/2);
    stop_timer();
    sprintf(tests[test].desc+strlen(tests[test].desc), PIXFMT,
    calc_ellipses(NELLIPSE*2, config.numxpixels/2, config.numypixels/2));
    tests[test++].ticks = get_timer(1);

    start_timer(1);
    ellipses(NELLIPSE, config.numxpixels, config.numypixels);
    stop_timer();
    sprintf(tests[test].desc+strlen(tests[test].desc), PIXFMT,
    calc_ellipses(NELLIPSE, config.numxpixels, config.numypixels));
    tests[test++].ticks = get_timer(1);

    _setcolor(0);
    _rectangle(_GFILLINTERIOR, 0, 0, config.numxpixels, config.numypixels);

    start_timer(1);
    lines(NLINE * 2, config.numxpixels/2, config.numypixels/2);
    stop_timer();
    sprintf(tests[test].desc+strlen(tests[test].desc), PIXFMT,
    calc_lines(NLINE*2, config.numxpixels/2, config.numypixels/2));
    tests[test++].ticks = get_timer(1);

    start_timer(1);
    lines(NLINE, config.numxpixels, config.numypixels);
    stop_timer();
    sprintf(tests[test].desc+strlen(tests[test].desc), PIXFMT,
    calc_lines(NLINE, config.numxpixels, config.numypixels));
    tests[test++].ticks = get_timer(1);

    _setcolor(colorlist[0]);
    _rectangle(_GFILLINTERIOR, 0, 0, config.numxpixels, config.numypixels);

    start_timer(1);
    wid = div(config.numxpixels * 2, xcount * 3).quot;
    hgt = div(config.numypixels * -1, ycount * 2).quot;
    dx = config.numxpixels / (xcount + 1);
    dy = config.numypixels / (ycount + 1);

    _clearscreen(_GCLEARSCREEN);
    i=0;
    for (y=ycount; y>0; y--)
    {
	yp = y * dy + dy/2;
	for (x=0; x<xcount; x++)
	{
	    xp = x * dx + dx/2;
	    switch (++i % 3) {
	    case 1:
		line_graph(xp, yp, wid, hgt);
		break;
	    case 2:
		pie_graph(xp, yp, wid, hgt);
		break;
	    case 0:
		bar_graph(xp, yp, wid, hgt);
		break;
	    default:
		printf("Whoops!\n");
		break;
	    }
	}
    }
    stop_timer();
    tests[test++].ticks = get_timer(1);
    tests[0].ticks=0;		/* add times to get total time */
    for (i=1; i<test; i++)
	tests[0].ticks += tests[i].ticks;

    if (!batch)
    {
	_settextposition(1,1);
	_outtext("Press any key...");
	getch();
    }
    else
    {
	start_timer(2);
	do stop_timer(); while (get_timer(2) < 36);	/* 36 is 2 seconds */
    }
    _setvideomode(_DEFAULTMODE);

    sprintf(tests[0].desc, "Total: Graphics (%dx%d,%d col)",
    config.numxpixels, config.numypixels, config.numcolors);

    if ((!bench) && (!batch))
	for (i=0; i<test; i++)
	    printf("%s:  %s\n", time_secs(tests[i].ticks), tests[i].desc);

    if ((program != -1) && (filename != NULL))
    {
	opentime(filename);
	for (i=0; i<test+1; i++)
	    savetime(program, i, &tests[i]);
	closetime();
    }

    return(0);
}

