/*+
    Name:	HLSORT.C
    Author:     Kent J. Quirk
		(c) Copyright 1988 Ziff Communications Co.
    Date:       April 1988
    Abstract:   This program tests CPU speed by generating a random array of
		data and then sorting it.
    History:	09-Sep-88   kjq     Version 1.00
-*/

#include <stdio.h>
#include <stdlib.h>

#include "hl.h"
#include "hltimes.h"

#define DT_INT 0
#define DT_LONG 1
#define DT_FLOAT 2

/* this program can be converted to sort ints, longs, or floats simply 
    by changing the following definition */
#define DT_TYPE DT_INT

#if DT_TYPE == DT_INT
typedef int DATATYPE;
#elif DT_TYPE == DT_LONG
typedef long DATATYPE;
#elif DT_TYPE == DT_FLOAT
typedef float DATATYPE;
#endif


void usage()
{
    printf("Usage:  HLSORT [-?] [-nNPTS] [-p]\n");
    printf("   NPTS is the number of points in the array (default 30,000).\n");
    printf("   -r reverse-sorts data after forward sort.\n");
    printf("      This can take a LONG time!\n");
    printf("   -p sends the data to stdout when finished.\n");
    exit(1);
}

/**** r a n d d a t a ****
    Abstract:	Generate a random number of the appropriate type.
****************************/
DATATYPE randdata()
{
    DATATYPE x;
    x = (DATATYPE)rand();       /* change for non-numeric types */
    return(x);
}

/**** c o m p d a t a ****
    Abstract:	Compare two data elements.
****************************/
int compdata(elem1, elem2)
DATATYPE *elem1, *elem2;
{
    return((int)(*elem1 - *elem2));     /* change for non-numeric types */
}

/**** r c o m p d a t a ****
    Abstract:	Compare two data elements for qsort().
****************************/
int rcompdata(elem1, elem2)
DATATYPE *elem1, *elem2;
{
    return((int)(*elem2 - *elem1));     /* change for non-numeric types */
}

/**** p r i n t d a t a ****
    Abstract:	Prints an item of the appropriate data type. 
****************************/
void printdata(x)
DATATYPE x;
{
#if DT_TYPE == DT_INT
    printf("%d\n", x);
#elif DT_TYPE == DT_LONG
    printf("%ld\n", x);
#elif DT_TYPE == DT_FLOAT
    printf("%g\n", x);
#endif
}

TIME_REC timerecs[] = {
    {  0L, "Total: CPU/Sort" } ,
    {  0L, "Data Generation" } ,
    {  0L, "Data Sort" } ,
    { -1L, "HLSORT" } ,
};

int main(argc, argv)
int argc;
char *argv[];
{
    int i;
    DATATYPE *data;
    int npts = 30000;
    int printout = 0;
    int reverse = 0;
    int fix_microsoft = 0;
    int bench = 0;
    int program = -1;
    char *filename = NULL;

    for (i=1; i<argc; i++)
    {
	if (argv[i][0] != '-')
	{
	    printf("Invalid argument '%s'\n", argv[i]);
	    usage();
	}

	switch(tolower(argv[i][1])) {
	case 'a': 
	case 'b':
	    bench = 1;
	    break;
	case 'n':
	    npts = atoi(argv[i]+2);
	    break;
	case 'o':
	    printout = 1;
	    break;
	case 'r':
	    reverse = 1;
	    break;
	case 'm':
	    fix_microsoft = 1;
	    break;
	case 'p':
	    program = atoi(argv[i]+2);
	    break;
	case 'f':
	    filename = argv[i]+2;
	    break;
	default:
	    printf("Unknown argument '%s'\n", argv[i]);
	case '?':                       /* default will fall through */
	    usage();
	}
    }

    if ((data = calloc(npts, sizeof(DATATYPE))) == NULL)
    {
	printf("Unable to alloc enough space for %d %d-byte records.\n",
	npts, sizeof(DATATYPE));
	usage();
    }

    printf("Generating %d %d-byte records.\n", npts, sizeof(DATATYPE));

    start_timer(0);
    start_timer(1);
    srand(1);           /* initialize seed to known pattern */
    for (i=0; i<npts; i++)
	data[i] = randdata();
    stop_timer();
    timerecs[1].ticks = get_timer(1);

    start_timer(1);
    qsort(data, npts, sizeof(DATATYPE), compdata);
    stop_timer();
    timerecs[2].ticks = get_timer(1);
    timerecs[0].ticks = get_timer(0);

    if (reverse)
    {
	start_timer(1);
	if (fix_microsoft)
	{
	    DATATYPE t;
	    i = rand() % npts;	    /* get number in range */
	    t = data[0];	    /* and swap with first location */
	    data[0] = data[i];
	    data[i] = t;
	}
	qsort(data, npts, sizeof(DATATYPE), rcompdata);
	stop_timer();

	printf("Reverse sort of already-sorted data took %ld ticks.\n",
	    get_timer(1));
    }

    if (!bench)
	for (i=0; timerecs[i].ticks != -1; i++)
	    printf("%s:  %s\n", time_secs(timerecs[i].ticks),
		timerecs[i].desc);

    if ((program != -1) && (filename != NULL))
    {
	opentime(filename);
	for (i=0; ;i++)
	{
	    savetime(program, i, &timerecs[i]);
	    if (timerecs[i].ticks == -1)
		break;
	}
	closetime();
    }

    if (printout)
    {
	printf("Sorted data:\n");
	for (i=0; i<npts; i++)
	    printdata(data[i]);
    }
    return(0);
}

