/*************************************************************************
 *  Listing 2 - SETCRC.C
 *
 *  This program is used to initialize the valid CRC in application
 *  programs incorporating the anti_venom routine.
 *
 *  Written by Kevin D. Weeks 10-8-89
 *  Released to the Public Domain.
 */
#include <stdio.h>
#include <io.h>
#include <fcntl.h>
#include <sys\types.h>
#include <sys\stat.h>

#define BUF_SIZE    1024                /* size of file read buffer */
#define ERR         -1
#define FOUND       10

/* function prototypes */
void    usage(char *);
void    sig_to_bin(char *,char *);

main(int argc,char *argv[])
{
    int             handle;             /* file handle */
    unsigned int    i, j;               /* loop counters */
    char            *buffer;            /* pointer to file buffer */
    char            signature[10];      /* crc signature */
    unsigned int    num_bytes;      /* number of bytes read from file */
    int             state;              /* current machine state */
    unsigned int    cur_crc;            /* running crc */
    long            crc_location;       /* file location of valid crc */
    int             finds;          /* number of times signature found */

    if (argc != 3)
        usage("\aParameter missing.");
    if ((buffer = (char *)malloc(BUF_SIZE)) == NULL)
    {
        printf("\aError allocating memory.\n");
        exit(1);
    }
    /* convert the signature to binary values */
    sig_to_bin(signature,argv[2]);
    if ((handle = open(argv[1],O_BINARY | O_RDWR)) == ERR)
    {
        perror(argv[1]);
        exit(1);
    }
    /* initialize everything */
    lseek(handle,0L,0);
    state = 0;
    cur_crc = 0;
    finds = 0;
    crc_location = 0L;

    /* loop until num_bytes == 0 (EOF) */
    while (num_bytes = read(handle,buffer,BUF_SIZE))
    {
        for (i = 0; i < num_bytes; i++)
        {
            /* first calculate the crc */
            /* xor current byte with current crc's hi-byte */
            cur_crc ^= (unsigned int)buffer[i] << 8;
            /* devide current crc, modulo 2, by prime polynomial */
            for (j = 0; j < 8; j++)
                if (cur_crc & 0x8000)           /* if MSB on */
                    /* shift left and xor with prime */
                    cur_crc = (cur_crc << 1) ^ 0x1021;
                else                            /* otherwise */
                    cur_crc <<= 1;              /* just shift left */

            /* then search for the signature */
            if (state != FOUND)
                if (buffer[i] == signature[state])
                {
                    ++state;
                    if (signature[state] == 0)
                    {
                        state = FOUND;
                        ++i;            /* point to crc position */
                        crc_location += i;
                        ++i;          /* skip over crc position */
                        ++finds;      /* increment number of times found */
                    }
                }
                else
                    state = 0;
        }
        if (state != FOUND)
            crc_location += (long)num_bytes;  /* update the file pointer */
    }
    if (finds == 1)
    {
        printf("crc = %xh @ file offset %lxh\n",cur_crc,crc_location);
        lseek(handle,crc_location,0);
        write(handle,&cur_crc,2);
    }
    else
        if (finds == 0)
            printf("\aError: Signature not found in %s.\n",argv[1]);
        else
            printf("\aError: Signature found more than once.\n");
    close(handle);
    exit(0);
}

/*************************************************************************
 *  This function converts the hexadecimal signature passed on the command
 *  line to the binary values that actually have to be searched for.
 */
void    sig_to_bin(char *signature,char *argv)
{
    int     i, j;

    /* convert signature to binary values */
    j = i = 0;
    signature[j] = 0;
    while (argv[i])
    {
        if (argv[i] != ',')
        {
            signature[j] *= 16;
            if (argv[i] > 47 && argv[i] < 58)         /* 0 - 9 */
                signature[j] += argv[i] - 48;
            else
                if (argv[i] > 40 && argv[i] < 71)     /* A - F */
                    signature[j] += argv[i] - 55;
                else
                    if (argv[i] > 96 && argv[i] < 103) /* a - f */
                        signature[j] += argv[i] - 87;
                    else
                        usage("\aInvalid hex value in signature");
        }
        else
        {
            if (j == 8)
                break;
            signature[++j] = 0;
        }
        ++i;
    }
    signature[++j] = 0;
}

/*************************************************************************
 *  Usage prints error messages and a short help message.
 *
 */
void    usage(char *message)
{
    printf("\nSETCRC, Rev 1.0    %s\n\n",message);
    printf("Usage: SETCRC FILENAME SIGNATURE      ");
    printf("(Note: signature must be in hex)\n");
    printf("Example: setcrc testprog.exe 41,4e,54,49,56\n");
    exit(1);
}
