/*
	dqueue.c -- Data queue manipulation

  Poor Man's Packet (PMP)
  Copyright (c) 1991 by Andrew C. Payne    All Rights Reserved.

  Permission to use, copy, modify, and distribute this software and its
  documentation without fee for NON-COMMERCIAL AMATEUR RADIO USE ONLY is hereby
  granted, provided that the above copyright notice appear in all copies.
  The author makes no representations about the suitability of this software
  for any purpose.  It is provided "as is" without express or implied warranty.

	Andrew C. Payne
	12/03/89
*/

#include <stdio.h>
#include <stdlib.h>
#include <alloc.h>
#include <mem.h>
#include "pmp.h"

/* DQInit(*dq)
	Initializes a data queue.
*/
void DQInit(DQUEUE *dq)
{
	dq->next = NULL;
	dq->len = 0;
}

/* DQAdd(*dq, data, len)
	Adds data to a data queue.  Returns TRUE if error.
*/
int DQAdd(DQUEUE *dq, byte *d, int len)
{
	struct dqentry	*p;

/* don't add zero length items */
	if(len == 0)
		return FALSE;

/* allocate a data queue entry */
	p = malloc(sizeof(struct dqentry) + len);
	if(p == NULL)
		return TRUE;

/* initialize entry */
	p->next = dq->next;
	p->len = len;
	memcpy(p->data,d,len);		/* copy in data */

/* add entry to queue */
	dq->next = p;
	dq->len += len;
	return FALSE;
}

/* DQFirst(*dq)
	Returns a pointer to the first queue entry.
*/
struct dqentry *DQFirst(DQUEUE *dq)
{
	struct	dqentry	*p;

	p = dq->next;
	if(p == NULL)		/* empty */
		return NULL;

	while(p->next != NULL)	/* find last */
		p = p->next;

	return p;
}

/* DQRemoveFirst(*dq)
	Removes the first entry from the data queue.
*/
void DQRemoveFirst(DQUEUE *dq)
{
	struct	dqentry	*p,*q;

	p = dq->next;
	if(p == NULL)		/* empty queue */
		return;		/* nothing read */

/* find the last entry in the queue */
	q = dq;
	while(p->next != NULL) {
		q = p;
		p = p->next;
	}

/* remove the entry */
	dq->len -= p->len;	/* maintain size count */
	free(p);
	q->next = NULL;
}

/* DQExtract(dq, dest, len)
	Extracts the specified number of bytes from the DQUEUE into the
	destination buffer specified.

	Will extract at most the number of bytes in the buffer.
*/
void DQExtract(DQUEUE *dq, byte *dest, int len)
{
	struct dqentry	*p;

	len = min(len, dq->len);

	while(len) {
		p = DQFirst(dq);
		if(p->len > len) {		/* use partial */
			memcpy(dest,p->data,len);
			memcpy(p->data, p->data+len, p->len-len);
			p->len -= len;
			dq->len -= len;
			break;
		} else {			/* use whole */
			memcpy(dest,p->data,p->len);
			len -= p->len;
			dest += p->len;
			DQRemoveFirst(dq);
		}
	}
}






