/*
	param.c -- Reads paramter file when program is started

  Poor Man's Packet (PMP)
  Copyright (c) 1991 by Andrew C. Payne    All Rights Reserved.

  Permission to use, copy, modify, and distribute this software and its
  documentation without fee for NON-COMMERCIAL AMATEUR RADIO USE ONLY is hereby
  granted, provided that the above copyright notice appear in all copies.
  The author makes no representations about the suitability of this software
  for any purpose.  It is provided "as is" without express or implied warranty.

	August, 1989
	Andrew C. Payne
*/

/* ---- Includes ----- */
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <conio.h>
#include <dos.h>
#include "pmp.h"
#include "ports.h"

static int 	domycall(char *p),dotxdelay(char *p),dopthresh(char *p),
		dofrack(char *p), doretry(char *p), donormal(char *p),
		doinverse(char *p), dobright(char *p), dostatus(char *p),
		dosound(char *p), doctext(char *p), dobtext(char *p),
		dopttport(char *p), dotxport(char *p), docdport(char *p),
		dorxport(char *p),docheck(char *p), domaxframe(char *p),
		domessage(char *p), docapture(char *p), dodefine(char *p),
		doinit(char *p), dobeacon(char *p), dobaddr(char *p),
		doautowrap(char *p), dopaclen(char *p), doclock(char *p);

extern int SetAX25Addr(struct ax25_addr *p, char *c);

/* ----- Command list Structure ----- */

static struct param_cmd pcmds[] = {
	{ "mycall",	domycall },
	{ "txdelay",	dotxdelay },
	{ "pthresh",	dopthresh },
	{ "frack",	dofrack },
	{ "retry",	doretry },
	{ "normal",	donormal },
	{ "inverse",	doinverse },
	{ "bright",	dobright },
	{ "status",	dostatus },
	{ "message",	domessage },
	{ "sound",	dosound },
	{ "ctext",	doctext },
	{ "btext", 	dobtext },
	{ "pttport",	dopttport },
	{ "txport",	dotxport },
	{ "cdport",	docdport },
	{ "rxport",	dorxport },
	{ "check",	docheck },
	{ "maxframe",	domaxframe },
	{ "capture", 	docapture },
	{ "define",	dodefine },
	{ "init",	doinit },
	{ "beacon",	dobeacon },
	{ "baddr",	dobaddr },
	{ "autowrap",	doautowrap },
	{ "paclen",	dopaclen },
	{ "clockskew",	doclock },
	{ NULL,		NULL } };

static int	line;		/* current line number */

/* error()
	Show error message.
*/
void error(char *format, ...)
{
	char	s[100];
	va_list	argptr;

	cprintf("Error on line %d in configuration file PMP.CFG :\r\n      ",line);
	va_start(argptr, format);
	vsprintf(s, format, argptr);
	va_end(argptr);
	cprintf(s);
}

/* InitParameters()
	Sets initial parameters.
*/
void InitParameters()
{
	int	i;

	SetAX25Addr(&MyCall,"NOCALL");
	TXStartFlags = 50;
	TXEndFlags = 2;
	Sound = FALSE;
	Pthresh = 0;
	Frack = 3;
	Retry = 7;
	MaxFrame = 7;
	Sound = TRUE;
	*btext = '\0';
	strcpy(baddr, "CQ");
	BeaconInt = 0;
	nctexts = 0;
	CaptureBuffer = 2048;
	AutoWrap = 0;
	Paclen = 128;
	for(i=1; i<=10; i++)
		*fkeys[i] = '\0';

	Ppersist = 12000;
	Slottime = 2;
	Pwait = 0;
	ClockSkew = FALSE;

/* default I/O ports */
	PTTPort = TXPort = 0x378;
	TXBit = 1;
	PTTBit = 2;

	CDPort = RXPort = 0x379;
	RXBit = 8;
	CDBit = 0x80;
	CDLevel = 0;
}

/* ReadParameters()
	Opens and reads the parameter file.

	Returns TRUE if error.
*/
int ReadParameters()
{
	FILE	*infile;
	char	s[100];
	char	cmd[80];
	char	*p;
	int	i;

/* open the file for input */
	if((infile = fopen(ParamFname,"r")) == NULL) {
		cprintf("Error -- Can't find configuration file '%s'   ",ParamFname);
		return TRUE;
	}

/* read through lines till end of file */
	line = 0;
	while(fgets(s,100,infile) != NULL) {
		line++;
		p = sob(s);
		if(*p && *p != '#') {
			i = dispatch(p, pcmds);
			if(i == FAIL) {
				p = extract(p, cmd);
				error("Unknown command '%s'",cmd);
			}
			if(i) {
				fclose(infile);
				return TRUE;
			}
		}
	}
	fclose(infile);
	return FALSE;
}

/* ----- Command Handlers ----- */

/* domycall(s)
	Sets the callsign of this node.
*/
static int domycall(char *s)
{
	if(SetAX25Addr(&MyCall,s)) {
		error("Invalid callsign");
		return TRUE;
	} else
		return FALSE;
}

/* dotxdelay(s)
	Sets the transmitter delay.  This is the wait after the TX is keyed
	and before data is sent.  The time is given in milliseconds.
*/
static int dotxdelay(char *s)
{
/* convert time in ms to # of leading syncs */
	TXStartFlags = max(2,(long)atoi(s) * 1200 / 8000);
	return FALSE;
}

/* dopthresh(s)
	This parameter is a slight modification to AX.25 Version 2.0.  If
	an I-packet has a data field smaller than this parameter, it is
	always resent w/ a poll after a timeout, instead of polling w/ RR.
*/
static int dopthresh(char *s)
{
	Pthresh = atoi(s);
	return FALSE;
}

/* dofrack(s)
	This parameter is used to calculate the timeout time before retrying.
*/
static int dofrack(char *s)
{
	Frack = atoi(s);
	return FALSE;
}

/* docheck(s)
	This parameter is used as a keep-alive timer (in seconds).
*/
static int docheck(char *s)
{
	Check = atoi(s);
	return FALSE;
}

/* doretry(s)
	This parameter specifies the maximum number of retries before a
	connection is dropped.
*/
static int doretry(char *s)
{
	Retry = atoi(s);
	return FALSE;
}

/* domaxframe(s)
	This parameter specifies the maximum number of frames allowed
	outstanding (e.g. unACKed).
*/
static int domaxframe(char *s)
{
	MaxFrame = atoi(s);
	if(MaxFrame < 1 || MaxFrame > 7) {
		error("MAXFRAME must be between 1 and 7");
		return TRUE;
	} else
		return FALSE;
}

/* donormal(s)
	Sets the normal screen attribute.
*/
static int donormal(char *s)
{
	NormalAttr = atoi(s);
	return FALSE;
}

/* doinverse(s)
	Sets the inverse screen attribute.
*/
static int doinverse(char *s)
{
	InvAttr = atoi(s);
	return FALSE;
}

/* dobright(s)
	Sets the bright screen attribute
*/
static int dobright(char *s)
{
	BrightAttr = atoi(s);
	return FALSE;
}

/* dostatus(s)
	Sets the status line attribute
*/
static int dostatus(char *s)
{
	StatusAttr = atoi(s);
	return FALSE;
}

/* domessage(s)
	Sets the message line attribute
*/
static int domessage(char *s)
{
	MsgAttr = atoi(s);
	return FALSE;
}

/* dosound(s)
	Enables/disables sound.
*/
static int dosound(char *s)
{
	Sound = atoi(s);
	return FALSE;
}

/* doctext(s)
	Sets CTEXT lines, the connect welcome string.
*/
static int doctext(char *s)
{
	if(nctexts >= MAXCTEXTS) {	/* check for too many lines */
		error("Too many CTEXT lines.");
		return TRUE;
	}

	ctext[nctexts++] = strdup(s);
	return FALSE;
}

/* dobtext(s)
	Sets btext, the beacon/CQ string.

	btext includes a carriage return.
*/
static int dobtext(char *s)
{
	btext = strdup(s);
	return FALSE;
}

/* dopttport(s)
	Sets the port and bit for the PTT output signal.
*/
static int dopttport(char *s)
{
	PTTPort = strtoul(s, &s, 0);
	s = sob(s);
	PTTBit = strtoul(s, &s, 0);
	s = sob(s);
	PTTLevel = strtoul(s, &s, 0);
	return FALSE;
}

/* dotxport(s)
	Sets the port and bit for the TX data line.
*/
static int dotxport(char *s)
{
	TXPort = strtoul(s, &s, 0);
	s = sob(s);
	TXBit = strtoul(s, &s, 0);
	return FALSE;
}

/* docdport(s)
	Sets the port and bit for the CD data line.
*/
static int docdport(char *s)
{
	CDPort = strtoul(s, &s, 0);
	s = sob(s);
	CDBit = strtoul(s, &s, 0);
	s = sob(s);
	CDLevel = strtoul(s, &s, 0);
	return FALSE;
}

/* dorxport(s)
	Sets the port and bit for the RX data line.
*/
static int dorxport(char *s)
{
	RXPort = strtoul(s, &s, 0);
	s = sob(s);
	RXBit = strtoul(s, &s, 0);
	return FALSE;
}

/* docapture(s)
	This buffer specifies the size of the capture buffer.  The larger this
	number, the less disk writes required.  This number should be a multiple
	of a sector size (512).
*/
static int docapture(char *s)
{
	CaptureBuffer = atol(s);
	if(CaptureBuffer < 512 || CaptureBuffer > 65536) {
		error("CAPTURE must be between 512 and 65536 bytes");
		return TRUE;
	}
	return FALSE;
}

/* dodefine(s)
	Defines function keys (F1 through F10).
*/
static int dodefine(char *s)
{
	char	keyname[80];
	int	keynumber;

/* extract function key */
	s = extract(s, keyname);
	keynumber = atoi(keyname + 1);
	if(*keyname != 'f' || keynumber < 0 || keynumber > 10) {
		error("Function key F1-F10 expected");
		return TRUE;
	}

/* extract key definition */
	if(convertkey(fkeys[keynumber-1], s)) {
		error("Error in definition for function key %s",keyname);
		return TRUE;
	} else
		return FALSE;
}

/* doinit(s)
	Send an initializing byte value to a port upon startup.
*/
static int doinit(char *s)
{
	word	port,value;

	port = strtoul(s, &s, 0);
	s = sob(s);
	value = strtoul(s, &s, 0);
	outportb(port, value);
	return FALSE;
}

/* dobeacon(s)
	Sets the interval for beaconing (in seconds).
*/
static int dobeacon(char *s)
{
	BeaconInt = atoi(s);
	return FALSE;
}

/* dobaddr(s)
	Sets the destination call and path for beacon packets.
*/
static int dobaddr(char *s)
{
	baddr = strdup(s);
	return FALSE;
}

/* doautowrap(s)
	Sets the column at which automatic word wrap is active.
*/
static int doautowrap(char *s)
{
	AutoWrap = atoi(s);
	if(AutoWrap && (AutoWrap < 10 || AutoWrap > 79)) {
		error("AUTOWRAP must be zero or between 10 and 79 bytes");
		return TRUE;
	}
	return FALSE;
}

/* dopaclen(s)
	Sets the maximum transmitted packet length.
*/
static int dopaclen(char *s)
{
	Paclen = atoi(s);
	if(Paclen < 1 || Paclen > 256) {
		error("PACLEN must be between 1 and 256 bytes");
		return TRUE;
	}
	return FALSE;
}

/* doclock(s)
	Enables or disables attempted clock skew adjust.
*/
static int doclock(char *s)
{
	ClockSkew = atoi(s);
	return FALSE;
}