/***************************************************************************
 *  COMMON.C								   *
 *	This file contains common functions				   *
 *									   *
 *	  ClearNcb () - Clear out a NCB					   *
 *	  NetBiosRequest() - Submit a NetBIOS call. OS dependent	   *
 *									   *
 *  History:	Alok Sinha  October, 1991    Created			   *
 *									   *
 ***************************************************************************/

// Include files
#include <ncb.h>
#include <common.h>

#include <stdio.h>
#include <memory.h>
#include <string.h>
#include <stdlib.h>
#include <process.h>

/*
 * For windows, we will call the NetBiosCall API from a assembly
 * routine. The prototype is defined below.
 */
#if defined (WIN)
extern unsigned int far pascal NetBios ( char far *NCB);
#endif

#if defined (DOS)
#include <dos.h>
#endif

/*
 * Global Handle for OS/2. Designed for simplicity.
 */
#if defined (OS2)
unsigned short far usNetHandle;
unsigned  short NetBiosExit(  );
#endif

/*
 * FUNCTION    :: ClearNcb()
 * PARAMETERS  ::
 *	 [in]  Ncb	  :- A NCB structure pointer which will be set to 0.
 * RETURN VALUE:
 *		NONE.
 */

void	       ClearNcb       ( NCB far *pNcb)
{
    /* Fill NCB with 0x00   */
    memset ( (void  *) pNcb, 0x00, sizeof (NCB) );
}

/*
 * FUNCTION    :: NetBiosRequest()
 * PARAMETERS  ::
 *	 [in]  Ncb	 :- A NCB structure pointer which will be submitted
 *
 * RETURN VALUE:
 *		Return value from submitting NCB.
 */

unsigned char NetBiosRequest (NCB far *pNcb)
{

#if defined( DOS )

    union  REGS InRegs, OutRegs;
    struct SREGS SegRegs;

    /* set ES:BX to point to the NCB */

    InRegs.x.bx = FP_OFF(pNcb);
    SegRegs.es	= FP_SEG(pNcb);

    /* Make the Call	*/
    return (unsigned char) int86x(NETBIOS_CALL, &InRegs, &OutRegs, &SegRegs);

#elif defined( OS2 )

    return (unsigned char ) NetBiosSubmit ( usNetHandle, // Global net handle
                                            0,           // Single NCB. Do not retry on error
                                            pNcb);


#elif defined( WIN )
    return NetBios ( (char far *)pNcb);
#endif /* OS dependent calls	*/

}

#if defined(OS2)

/*
 * FUNCTION    :: NetBiosInit()
 * PARAMETERS  ::
 *	 [in]  pchNetBiosName	:- A char far pointer to logical NetBIOS name
 *				   NULL means first netbios driver.
 *	 [in]  usOption		:- Mode of operation.
 *				   NB_EXCLUSIVE, NB_PRIVILEGE, NB_REGULAR
 *				   See "Microsoft Lan Manager Programmer's
 *				   Reference" for detail.
 *	 [out] usNetHandle (Global):- It is set to a correct handle, if
 *				   return value is NO_ERROR
 *
 * RETURN VALUE:
 *		NO_ERROR if call was successful. else a OS/2 error.
 *
 * COMMENT:
 *	    In this sample program, we use a simple global handle
 *	    for storing the handle returned by NetBiosOpen(). In fact,
 *	    the following code will not even guard against multiple
 *	    calls to NetBiosInit and over-riding "old" handle if one
 *	    exists already. One should carefully modify these routines
 *	    if these were to be used for business applications.
 */

unsigned  short NetBiosInit( char far	  *pchNetBiosName,
			   unsigned short usOption
			 )
{
    /* Clear the global handle */
    usNetHandle = 0;

    return NetBiosOpen ( pchNetBiosName,    // driver name e.g. "net1"
			 (char far *)0,	    // reserved
			 usOption,
			 &usNetHandle
			);

}

/*
 * FUNCTION    :: NetBiosExit()
 * PARAMETERS  ::
 *		   NONE
 * RETURN VALUE:
 *		NO_ERROR if call was successful. else a OS/2 error.
 */

unsigned  short NetBiosExit(  )
{

    return NetBiosClose ( usNetHandle,
			  (unsigned short) 0 // reserved
			);

}

#endif /* OS/2 */

/*
 * FUNCTION    :: CopyToBuffer()
 * PARAMETERS  ::
 *	 [in/out] dest	:- char pointer to a buffer containing a NetBIOS name.
 *	 [in	] src	:- char pointer to a buffer containing a name.
 *
 * RETURN VALUE:
 *		NONE.
 * Comment
 */

void CopyToBuffer ( char *pchDest , char *pchSrc)
{
    register count;

    /* Check for null pointer */
    if ((!pchDest) || ( ! pchSrc))
        return ;

    /* set the name field with nulls */
    memset ( pchDest, 0x20, NCBNAMSZ);

    /* copy from source to destination */
    count =  NCBNAMSZ;
    while ((*pchSrc) && ( count))
    {
        *pchDest++ = *pchSrc++;
        count--;
    }
    return;
}

/*
 * FUNCTION    :: NetCleanUp()
 * PARAMETERS  ::
 *		  [in]	Result: The result to be exited with to system.
 * RETURN VALUE:
 *		NO_ERROR if call was successful. else a OS/2 error.
 *
 * Comment:
 *		Primarily needed for closing NetBios Handle for OS/2.
 */

int NetCleanUp ( int Result)
{

#ifdef OS2
    unsigned short usRc;

    if ((usRc = NetBiosExit () )!= NO_ERROR)
        printf("NetBiosExit Error [%d]\n", usRc);
#endif

    return Result;
}

/*
 * FUNCTION    :: NetInit()
 * PARAMETERS  ::
 *		  [in]	ucLana: The LAN Adaptor number
 * RETURN VALUE:
 *		NO_ERROR if call was successful. Else exit to system
 *
 * Comment:
 *		Primarily needed for opening NetBios for OS/2.
 *		The algorithm for determining netname (e.g. "net1")
 *		works on Microsoft Lan Manager systems and not
 *		guaranteed to work on other systems. One
 *		should call NetBiosEnum() to determine the
 *		correct netname (See "Microsoft
 *		Lan Manager: Programmer's Reference" for detail)
 */

int NetInit ( unsigned char ucLana)
{
#ifdef OS2

    char chNetName [ 20 ];
    unsigned short usRc;

    if (ucLana==0)
	memcpy(chNetName, "net1", sizeof("net1"));
    else if (ucLana==1)
	memcpy(chNetName, "net2", sizeof("net2"));
    else
    {
	printf("Use NetBiosEnum to determine the correct Net Name\n");
	exit (1);
    }

    usRc = NetBiosInit( chNetName,  // Net Name
			1	    // regular open
		      );
    if (usRc)
    {
	printf("NetBiosInit Error:: RC [%d]\n", usRc);
	exit (1);
    }

#endif
    return 0;

}
