/*
 * WINDOWS ICON EXTRACTION UTILITY - MAIN APPLICATION
 *
 * LANGUAGE : Microsoft C 6.0
 * TOOLKIT  : Windows 3.0 SDK
 * MODEL    : Medium
 * STATUS   : Operational
 *
 * Copyright (C) 1991 - All Rights Reserved
 *
 * Eikon Systems, Inc.
 * 989 East Hillsdale Blvd, Suite 260
 * Foster City, California 94404
 *
 * 07/04/91 - Kevin P. Welch - initial creation.
 *
 */

#include <windows.h>
#include <string.h>
#include "iconex.h"

/*
 * WinMain( hInst, hPrevInst, lpszCmdLine, wCmdShow ) : int
 *
 *    hInst          handle of current instance
 *    hPrevInst      handle to previous instance (if any)
 *    lpszCmdLine    pointer to command line arguments
 *    wCmdShow       initial ShowWindow command
 *
 * This function is the system entry point for the application
 * and is responsible for defining the appropriate window
 * classes and for processing all the messages.  Note how
 * the dialog box manager is responsible for the operation of
 * the icon extraction window.
 *
 */

int PASCAL WinMain(
   HANDLE    hInst,
   HANDLE    hPrevInst,
   LPSTR     lpszCmdLine,
   WORD      wCmdShow )
{
   /* local variables */
   FARPROC lpProc;

   /* display main dialog box */
   lpProc = MakeProcInstance( (FARPROC)IconExDlgFn, hInst );
   DialogBox( hInst, "IconEx", NULL, lpProc );
   FreeProcInstance( lpProc );

   /* end program */
   return( FALSE );

}

/*
 * IconExDlgFn( hWnd, wMsg, wParam, lParam ) : BOOL
 *
 *    hWnd       handle to palette window
 *    wMsg       message number
 *    wParam     single word parameter
 *    lParam     double word parameter
 *
 * This function is responsible for processing all the messages
 * which relate to the icon extraction dialog box.  This mainly
 * involves the definition and retrieval of the various
 * events generated by the user.
 *
 */

BOOL FAR PASCAL IconExDlgFn(
   HWND    hDlg,
   WORD    wMsg,
   WORD    wParam,
   LONG    lParam )
{
   /* local variables */
   BOOL  bResult;

   /* initialization */
   bResult = TRUE;

   /* process messages */
   switch( wMsg )
      {
      case WM_MEASUREITEM : /* set listbox dimensions */

         /* check for listbox id */
         if ( ((LPMEASUREITEMSTRUCT)lParam)->CtlID == ID_ICONS )
            ((LPMEASUREITEMSTRUCT)lParam)->itemHeight =
               GetSystemMetrics(SM_CYICON) + 8;

         break;
      case WM_INITDIALOG : /* initialize dialog box */
         {
            /* initialize property lists */
            SetProp( hDlg, ICONCLIPFMT, NULL );
            SetProp( hDlg, ICONDATAHANDLE, NULL );

            /* update property list */
            SetProp(hDlg,
                    ICONCLIPFMT,
                    RegisterClipboardFormat("SDKPAINT")
            );

         /* center dialog box */
         CenterPopup( hDlg, NULL );

         /* define icon for dialog box - rather dirty */
         SetClassWord(hDlg,
                      GCW_HICON,
                      LoadIcon(INSTANCE(hDlg),"Icon")
         );

         /* update directory and path fields */
         DlgDirList( hDlg, "*.exe", ID_FILELIST, ID_PATH, 0x4010 );

         /* set and select initial search spec */
         SetDlgItemText( hDlg, ID_FILE, "*.exe" );
         SendDlgItemMessage(hDlg,
                            ID_FILE,
                            EM_SETSEL,
                            NULL,
                            MAKELONG(0,0x7FFF)
         );

         }
         break;
      case WM_COMMAND : /* window command */

      /* process sub-message */
      switch( wParam )
         {
         case ID_FILE :
            break;
         case ID_FILELIST : /* directory listbox */

            /* process notification codes */
            switch( HIWORD(lParam) )
               {
            case LBN_SELCHANGE : /* selection changed */
               {
                  CHAR  szDir[128];

                  /* change directories */
                  if ( DlgDirSelect(hDlg,szDir,ID_FILELIST) ) {
                     strcat( szDir, "*.exe" );
                     SetDlgItemText( hDlg, ID_FILE, szDir );
                  } else {
                     SetDlgItemText( hDlg, ID_FILE, szDir );
                     SendDlgItemMessage(hDlg,
                                        ID_FILE,
                                        EM_SETSEL,
                                        NULL,
                                        MAKELONG(0,0x7FFF)
                     );
                  }

               }
               break;
            case LBN_DBLCLK : /* double-click */
               {
                  CHAR  szDir[128];
                  CHAR  szFile[128];

                  /* change directories of open file */
                  if ( DlgDirSelect(hDlg,szDir,ID_FILELIST) ) {

                     /* change directories */
                     strcat( szDir, "*.exe" );
                     SetDlgItemText( hDlg, ID_FILE, "*.exe" );
                     DlgDirList( hDlg, szDir, ID_FILELIST, ID_PATH, 0x4010 );

                  } else {

                     /* define file name */
                     GetDlgItemText( hDlg, ID_PATH, szFile, sizeof(szFile) );
                     if ( szFile[strlen(szFile)-1] != '\\' )
                        strcat( szFile, "\\" );
                     strcat( szFile, szDir );

                     /* extract icons from EXE file */
                     OpenExeFile( hDlg, szFile );

                  }

               }
               break;
            default :
               break;
            }

            break;
         case ID_ICONS :
            {
               /* process notification codes */
               if ( HIWORD(lParam) == LBN_DBLCLK ) {
                  /* simulate "Copy" button being clicked */
                  SendMessage(hDlg,
                              WM_COMMAND,
                              ID_COPY,
                              MAKELONG( GetDlgItem(hDlg,ID_COPY), BN_CLICKED )
                  );
               }
            }
            break;
         case ID_OPEN :
            {
               CHAR  szDir[128];
               CHAR  szFile[128];

               /* retrieve file name */
               GetDlgItemText( hDlg, ID_FILE, szDir, sizeof(szDir) );

               /* change directory or open file */
               if ( strchr(szDir,'*') || strchr(szDir,'?') ) {

                  /* change to new directory */
                  DlgDirList( hDlg, szDir, ID_FILELIST, ID_PATH, 0x4010 );

               } else {

                  /* construct file name if path not specified */
                  if ( !strchr(szDir,':') && !strchr(szDir,'\\') ) {
                     GetDlgItemText( hDlg, ID_PATH, szFile, sizeof(szFile) );
                     if ( szFile[strlen(szFile)-1] != '\\' )
                        strcat( szFile, "\\" );
                     strcat( szFile, szDir );
                  } else
                     strcpy( szFile, szDir );

                  /* open exe file */
                  OpenExeFile( hDlg, szFile );
               }
            }
            break;
         case ID_COPY :
            {
               WORD       wResult;
               WORD       wIndex;

               wIndex = (WORD)SendDlgItemMessage(hDlg,
                                                 ID_ICONS,
                                                 LB_GETCURSEL,
                                                 0,
                                                 0L );

               wResult = IconCopyToClipboard(hDlg,
                                             GetProp( hDlg, ICONDATAHANDLE ),
                                             wIndex,
                                             GetProp( hDlg, ICONCLIPFMT ) );

               if (wResult)
                  WARNING( hDlg, "Unable to copy icon to clipboard!" );
            }
            break;
         case ID_EXIT : /* exit program */
         EndDialog( hDlg, TRUE );
         break;
         case ID_ABOUT : /* about program */
            Dialog( hDlg, "About", AboutDlgFn );
            break;
         default : /* something else */
         break;
      }

      break;
      case WM_SYSCOMMAND : /* system command */

      /* process sub-messages */
      switch( wParam )
         {
      case SC_CLOSE : /* destroy dialog box */
         EndDialog( hDlg, TRUE );
         break;
      default :
         bResult = FALSE;
         break;
      }

      break;
      case WM_DRAWITEM : /* draw icon list */
         {
            LPDRAWITEMSTRUCT    lp;
            HBRUSH              hbr;
            HICON               hIcon;
            RECT                rcSelect;

            /* check for listbox id */
            lp = (LPDRAWITEMSTRUCT)lParam;
            if ( lp->CtlID == ID_ICONS )
               if ( lp->itemID == -1 ) {

                  /* define selection rectangle */
                  CopyRect( &rcSelect, &lp->rcItem );
                  InflateRect( &rcSelect, -3, -3 );

                  /* draw focus rectangle */
                  DrawFocusRect( lp->hDC, &rcSelect );

               } else {

                  /* process action code */
                  switch( lp->itemAction )
                     {
                  case ODA_DRAWENTIRE : /* draw entire icon */
                     {
                        CHAR    lpszIconDescrip[MAXLENGTH];
                        WORD    wResult;

                        /* retrieve icon */
                        wResult = IconCreateIcon(hDlg,
                                                 GetProp(hDlg,ICONDATAHANDLE),
                                                 lp->itemID,&hIcon
                                  );

                        /* process return code */
                        switch( wResult )
                          {
                        case IDERR_SUCCESS : /* successful */

                          /* draw icon */
                          DrawIcon(lp->hDC,
                                   lp->rcItem.left+4,
                                   lp->rcItem.top+4,
                                   hIcon
                          );

                          /* delete icon */
                          DestroyIcon( hIcon );

                          break;
                        default :
                          WARNING( hDlg, "Unable to draw icon!" );
                          break;
                        }

                        /* draw icon label */
                        CopyRect( &rcSelect, &lp->rcItem );
                        rcSelect.top += 4;
                        rcSelect.left += GetSystemMetrics(SM_CXICON) + 8;

                        wResult=IconGetIconDescrip(GetProp(hDlg,ICONDATAHANDLE),
                                                   lp->itemID,
                                                   lpszIconDescrip
                        );

                        if (wResult == IDERR_SUCCESS)
                          DrawText(lp->hDC,
                                   lpszIconDescrip,
                                   -1,
                                   &rcSelect,
                                   DT_LEFT | DT_WORDBREAK
                          );

                     }

                     /* fall through to ODA_SELECT */

                  case ODA_SELECT : /* draw selection */

                     /* define selection rectangle */
                     CopyRect( &rcSelect, &lp->rcItem );
                     InflateRect( &rcSelect, -1, -1 );

                     /* create brush */
                     if ( lp->itemState & ODS_SELECTED ) {
                        hbr =   GetStockObject( BLACK_BRUSH );
                        FrameRect( lp->hDC, &rcSelect, hbr );
                     } else {
                        hbr = CreateSolidBrush( GetSysColor(COLOR_WINDOW) );
                        FrameRect( lp->hDC, &rcSelect, hbr );
                        DeleteObject( hbr );
                     }

                     break;
                  case ODA_FOCUS : /* draw focus */

                     /* define focus rectangle */
                     CopyRect( &rcSelect, &lp->rcItem );
                     InflateRect( &rcSelect, -3, -3 );

                     /* draw focus rectangle */
                     DrawFocusRect( lp->hDC, &rcSelect );

                     break;
                  }

               }

         }
         break;
      case WM_DESTROY : /* window being destroyed */
         {
          HANDLE   hIconData;

          /* remove property list(s) */
          RemoveProp( hDlg, ICONCLIPFMT );
          hIconData = RemoveProp( hDlg, ICONDATAHANDLE );
          if ( hIconData )
             IconFree( hIconData );
         }
         break;
      default :
      bResult = FALSE;
      break;
   }

   /* return result */
   return( bResult );

}

/*
 * AboutDlgFn( hDlg, wMsg, wParam, lParam ) : BOOL ;
 *
 *    hDlg       handle to dialog box
 *    wMsg       message or event
 *    wParam     word portion of message
 *    lParam     long portion of message
 *
 * This function is responsible for processing all the messages
 * that relate to the About dialog box.  About the only useful actions
 * this function performs is to center the dialog box and to wait for
 * the OK button to be pressed.
 *
 */

BOOL FAR PASCAL AboutDlgFn(HWND   hDlg,
                           WORD   wMsg,
                           WORD   wParam,
                           LONG   lParam )
{
   BOOL        bResult;

   /* process message */
   switch( wMsg )
      {
   case WM_INITDIALOG :
      bResult = TRUE;
      CenterPopup( hDlg, GetParent(hDlg) );
      break;
   case WM_COMMAND :

      /* process sub-message */
      if ( wParam == IDOK ) {
         bResult = TRUE;
         EndDialog( hDlg, TRUE );
      } else
         bResult = FALSE;

      break;
   default :
      bResult = FALSE;
      break;
   }

   /* return final result */
   return( bResult );

}

/*
 * OpenExeFile( hDlg, lpszFile ) : BOOL;
 *
 *       hDlg       handle to dialog box
 *       lpszFile   name of EXE file to open
 *
 * This utility function opens the named exe file, extracts the
 * icons, updates the icon list and the dialog box caption.  A
 * value of TRUE is returned if the function was successful, and
 * FALSE otherwise.
 *
 */

BOOL FAR PASCAL OpenExeFile( hDlg, lpszFile )
      HWND     hDlg;
      LPSTR    lpszFile;
{
      WORD     wCount;
      BOOL     bResult;
      HCURSOR  hOldCursor;
      CHAR     szTemp[128];

      /* initialization */
      bResult = FALSE;

      /* check for valid dialog handle */
      if ( IsWindow(hDlg) ) {
         HANDLE      hIconDataOld;
         HANDLE      hIconDataNew;

         /* initialize */
         hIconDataNew = NULL;

         /* activate hourglass cursor */
         hOldCursor = SetCursor( LoadCursor(NULL,IDC_WAIT) );

         /* open EXE file */
         switch( IconExtract(lpszFile, &hIconDataNew ,&wCount) )
            {
         case IDERR_SUCCESS : /* successful */

            /* get handle to old icon data */
            hIconDataOld = GetProp(hDlg,ICONDATAHANDLE);

            /* delete old icon data */
            if (hIconDataOld && IconFree(hIconDataOld) != IDERR_SUCCESS)
               WARNING( hDlg, "Unable to free allocated memory!" );

            /* save memory handle */
            SetProp( hDlg, ICONDATAHANDLE, hIconDataNew );

            /* enable copy button */
            EnableWindow( GetDlgItem(hDlg,ID_COPY), TRUE );

            /* update list of icons */
         SendMessage(GetDlgItem(hDlg,ID_ICONS),
                     WM_SETREDRAW,
                     FALSE,
                     0L
            );
            SendMessage(GetDlgItem(hDlg,ID_ICONS),
                        LB_RESETCONTENT,
                        0,
                        0L
            );
            while ( wCount-- )
               SendMessage(GetDlgItem(hDlg,ID_ICONS),
                           LB_ADDSTRING,
                           0,
                           0L
               );
         SendMessage(GetDlgItem(hDlg,ID_ICONS),
                     WM_SETREDRAW,
                     TRUE,
                     0L
            );

            /* redefine caption */
            wsprintf(szTemp,
                     "Icon Extractor - %s",
                     (LPSTR)(_fstrrchr(lpszFile,'\\')+1)
            );
            SetWindowText( hDlg, szTemp );

            /* select first icon */
         InvalidateRect( GetDlgItem(hDlg,ID_ICONS), NULL, TRUE );
            SendMessage(GetDlgItem(hDlg,ID_ICONS),
                        LB_SETCURSEL,
                        0,
                        0L
            );

            break;
         case IDERR_ALLOCFAIL :
            WARNING( hDlg, "Memory allocation failure!" );
            break;
         case IDERR_LOCKFAIL :
            WARNING( hDlg, "Memory access failure!" );
            break;
         case IDERR_OPENFAIL :
            WARNING( hDlg, "Unable to open file!" );
            break;
         case IDERR_READFAIL :
            WARNING( hDlg, "Unable to read file!" );
            break;
         case IDERR_INVALIDPARAM :
            WARNING( hDlg, "Invalid parameters!" );
            break;
         case IDERR_FILETYPEBAD :
            WARNING( hDlg, "Not an executable file!" );
            break;
         case IDERR_EXETYPEBAD :
            WARNING( hDlg, "Unsupported executable file type!" );
            break;
         case IDERR_WINVERSIONBAD :
            WARNING( hDlg, "Unsupported Windows executable version!" );
            break;
         case IDERR_RESTABLEBAD :
            WARNING( hDlg, "Invalid resource table in executable file!" );
            break;
         case IDERR_ICONBAD :
            WARNING( hDlg, "Unsupported icon(s) in executable file!" );
            break;
         case IDERR_NOICONS :
            WARNING( hDlg, "No Windows 3.0 icons found in file!" );
            break;
         default :
            WARNING( hDlg, "Internal error!" );
            break;
         }

         /* restor old cursor */
         SetCursor( hOldCursor );

      }

      /* return result */
      return( bResult );

}

/*
 * Dialog( hParentWnd, lpszTemplate, lpfnDlgProc ) : BOOL
 *
 *    hParentWnd        handle to parent window
 *    lpszTemplate      dialog box template
 *    lpfnDlgProc       dialog window function
 *
 * This utility function displays the specified dialog box, using the
 * template provided.  It automatically makes a new instance of the
 * dialog box function.  Note that this function will NOT work
 * correctly if an invalid or NULL parent window handle is provided.
 *
 */

BOOL FAR PASCAL Dialog( hParentWnd, lpszTemplate, lpfnDlgProc )
   HWND        hParentWnd;
   LPSTR       lpszTemplate;
   FARPROC     lpfnDlgProc;
{
   /* local variables */
   BOOL           bResult;
   FARPROC        lpProc;

   /* display palette dialog box */
   lpProc = MakeProcInstance( lpfnDlgProc, INSTANCE(hParentWnd) );
   bResult = DialogBox(INSTANCE(hParentWnd),
                       lpszTemplate,
                       hParentWnd,
                       lpProc
             );
   FreeProcInstance( lpProc );

   /* return result */
   return( bResult );

}

/*
 * CenterPopup( hWnd, hParentWnd ) : BOOL
 *
 *    hWnd              window handle
 *    hParentWnd        parent window handle
 *
 * This routine centers the popup window in the screen or display
 * using the window handles provided.  The window is centered over
 * the parent if the parent window is valid.  Special provision
 * is made for the case when the popup would be centered outside
 * the screen - in this case it is positioned at the appropriate
 * border.
 *
 */

BOOL FAR PASCAL CenterPopup(
      HWND     hWnd,
      HWND     hParentWnd
   )
{
   /* local variables */
   int      xPopup;
   int      yPopup;
   int      cxPopup;
   int      cyPopup;
   int      cxScreen;
   int      cyScreen;
   int      cxParent;
   int      cyParent;
   RECT     rcWindow;

   /* retrieve main display dimensions */
   cxScreen = GetSystemMetrics( SM_CXSCREEN );
   cyScreen = GetSystemMetrics( SM_CYSCREEN );

   /* retrieve popup rectangle  */
   GetWindowRect( hWnd, (LPRECT)&rcWindow );

   /* calculate popup extents */
   cxPopup = rcWindow.right - rcWindow.left;
   cyPopup = rcWindow.bottom - rcWindow.top;

   /* calculate bounding rectangle */
   if ( hParentWnd ) {

      /* retrieve parent rectangle */
      GetWindowRect( hParentWnd, (LPRECT)&rcWindow );

      /* calculate parent extents */
      cxParent = rcWindow.right - rcWindow.left;
      cyParent = rcWindow.bottom - rcWindow.top;

      /* center within parent window */
      xPopup = rcWindow.left + ((cxParent - cxPopup)/2);
      yPopup = rcWindow.top + ((cyParent - cyPopup)/2);

      /* adjust popup x-location for screen size */
      if ( xPopup+cxPopup > cxScreen )
         xPopup = cxScreen - cxPopup;

      /* adjust popup y-location for screen size */
      if ( yPopup+cyPopup > cyScreen )
         yPopup = cyScreen - cyPopup;

   } else {

      /* center within entire screen */
      xPopup = (cxScreen - cxPopup) / 2;
      yPopup = (cyScreen - cyPopup) / 2;

   }

   /* move window to new location & display */
   MoveWindow(
      hWnd,
      ( xPopup > 0 ) ? xPopup : 0,
      ( yPopup > 0 ) ? yPopup : 0,
      cxPopup,
      cyPopup,
      TRUE
   );

   /* normal return */
   return( TRUE );

}
