/*==================================================================*/
/*                                                                  */
/* File    : TICK.C                                                 */
/*                                                                  */
/* Purpose : Routines pertaining to the adding and modification of  */
/*           tickers.                                               */
/*                                                                  */
/* History :                                                        */
/*                                                                  */
/* Written by Marc Adler/Magma Systems for Microsoft Systems Journal*/
/*==================================================================*/

#include <windows.h>
#include "stock.h"


/*******************************************************************/
/*                                                                 */
/* Function : AddTickDlgProc()                                     */
/*                                                                 */
/* Purpose  : Dialog box proc for adding tickers.                  */
/*                                                                 */
/* Returns  :                                                      */
/*                                                                 */
/*******************************************************************/
BOOL FAR PASCAL AddTickDlgProc(hDlg, msg, wParam, lParam)
  HWND hDlg;
  WORD msg;
  WORD wParam;
  LONG lParam;
{
  RECT        r;
  char        buf[80];
  LPSTOCKINFO lpStockInfo;
  LPTICK      lpTick;
  TICK        tick;
  HWND        hWnd;

  static int idxTickEdited;

  switch (msg)
  {
    case WM_INITDIALOG:
      /*
         Make sure that there is a current stock information record
         which we can append the ticker onto.
      */
      if (!hCurrStockInfo)
        EndDialog(hDlg, FALSE);

      if ((idxTickEdited = (int) lParam) != -1)
      {
        /*
          Get a pointer to the stock info record
        */
        lpStockInfo = (LPSTOCKINFO) GlobalLock(hCurrStockInfo);
        if (lpStockInfo == NULL)
        {
          ErrorMessage(ERR_BADLOCK);
          goto byebye;
        }
        if (!(lpTick = (LPTICK) GlobalLock(lpStockInfo->hTicks)))
        {
          ErrorMessage(ERR_BADLOCK);
          GlobalUnlock(hCurrStockInfo);
          goto byebye;
        }

        SetDlgItemLong(hDlg, ID_TICK_PRICE, lpTick[idxTickEdited].price);
        SetDlgItemLong(hDlg, ID_TICK_VOLUME, lpTick[idxTickEdited].dwVolume);

        GlobalUnlock(lpStockInfo->hTicks);
        GlobalUnlock(hCurrStockInfo);
  
      }

      return TRUE;


    case WM_COMMAND:
      switch (wParam)
      {
        /*
          The user chose the OK button...
        */
        case IDOK:
          /*
            Get a pointer to the stock info record
          */
          lpStockInfo = (LPSTOCKINFO) GlobalLock(hCurrStockInfo);
          if (lpStockInfo == NULL)
          {
            ErrorMessage(ERR_BADLOCK);
            goto byebye;
          }

          /*
            Are we entering the first ticker? If so, then allocate a
            ticker array for the stock. We allocate 64 tickers
            initially.
          */
          if (lpStockInfo->hTicks == NULL)
          {
            lpStockInfo->StockFile.nTicks = 0;
            lpStockInfo->nTicksAllocated  = TICKER_CHUNK;
            lpStockInfo->hTicks = GlobalAlloc(GMEM_MOVEABLE | GMEM_ZEROINIT,
                                       (DWORD) sizeof(TICK) * TICKER_CHUNK);
            if (lpStockInfo->hTicks == NULL)
            {
tickerr:
              ErrorMessage(ERR_CANTALLOCTICKER);
              GlobalUnlock(hCurrStockInfo);
              goto byebye;
            }
          }

          /*
            Make sure that we do not overflow the ticker array. If
            there is a chance of this, then reallocate.
          */
          if (idxTickEdited == -1 && 
              lpStockInfo->StockFile.nTicks + 1 >= 
                                     lpStockInfo->nTicksAllocated)
          {
            HANDLE h;
            lpStockInfo->nTicksAllocated *= 2;
            h = GlobalReAlloc(lpStockInfo->hTicks,
                              (DWORD) sizeof(TICK) *
                                      lpStockInfo->nTicksAllocated,
                              GMEM_MOVEABLE);
            if (h == NULL)
              goto tickerr;
            lpStockInfo->hTicks = h;
          }

          /*
            Get a pointer to the ticker array
          */
          if (!(lpTick = (LPTICK) GlobalLock(lpStockInfo->hTicks)))
          {
            ErrorMessage(ERR_BADLOCK);
            GlobalUnlock(hCurrStockInfo);
            goto byebye;
          }

          /*
            Get the price and volume and out it in the last element
            of the ticker array.
            KLUDGE NOTES :
              a) We really should sort the records by date...
              b) We ignore the date here.
          */
          tick.price = GetDlgItemLong(hDlg, ID_TICK_PRICE, 
                                      NULL, FALSE);
          tick.dwVolume = GetDlgItemLong(hDlg, ID_TICK_VOLUME,
                                      NULL, FALSE);

          /*
            Copy the ticker structure
          */
          if (idxTickEdited == -1)
            lpTick[lpStockInfo->StockFile.nTicks++] = tick;
          else
            lpTick[idxTickEdited] = tick;
          lpStockInfo->dwFlags |= STATE_DIRTY;

          /*
            Unlock the memory and get outta here...
          */
          GlobalUnlock(lpStockInfo->hTicks);
          GlobalUnlock(hCurrStockInfo);
          EndDialog(hDlg, TRUE);
          break;


        /*
          The user chose the CANCEL button....
        */
        case IDCANCEL :
byebye:
          EndDialog(hDlg, FALSE);
          break;
      }
      return TRUE;

    default:
      return FALSE;
  }
}


/*******************************************************************/
/*                                                                 */
/* Function : ChangeTickDlgProc()                                  */
/*                                                                 */
/* Purpose  : Dialog box proc for editing tickers.                 */
/*                                                                 */
/* Returns  :                                                      */
/*                                                                 */
/*******************************************************************/
BOOL FAR PASCAL ChangeTickDlgProc(hDlg, msg, wParam, lParam)
  HWND hDlg;
  WORD msg;
  WORD wParam;
  LONG lParam;
{
  HWND        hLB;
  int         i, rc;
  char        buf[80];
  LPSTOCKINFO lpStockInfo;
  LPTICK      lpTick;
  TICK        tick;
  FARPROC     lpfn;

  switch (msg)
  {
    case WM_INITDIALOG:
      /*
         Make sure that there is a current stock information record
         which we can append the ticker onto.
      */
      if (!hCurrStockInfo)
        EndDialog(hDlg, FALSE);

      lpStockInfo = (LPSTOCKINFO) GlobalLock(hCurrStockInfo);
      if (lpStockInfo == NULL)
      {
        ErrorMessage(ERR_BADLOCK);
        goto byebye;
      }
      /*
        Get a pointer to the ticker array
      */
      if (!(lpTick = (LPTICK) GlobalLock(lpStockInfo->hTicks)))
      {
        ErrorMessage(ERR_BADLOCK);
        GlobalUnlock(hCurrStockInfo);
        goto byebye;
      }

      hLB = GetDlgItem(hDlg, ID_LISTBOX);
      SendMessage(hLB, WM_SETREDRAW, FALSE, 0L);

      for (i = 0;  i < lpStockInfo->StockFile.nTicks;  i++)
      {
        sprintf(buf, "%d\t%ld\t%ld", i, lpTick[i].price, lpTick[i].dwVolume);
        SendMessage(hLB, LB_ADDSTRING, -1, (DWORD) (LPSTR) buf);
      }

      SendMessage(hLB, WM_SETREDRAW, TRUE,  0L);

      GlobalUnlock(lpStockInfo->hTicks);
      GlobalUnlock(hCurrStockInfo);
      return TRUE;


    case WM_COMMAND:
      switch (wParam)
      {
        /*
          The user chose the OK button...
        */
        case ID_LISTBOX :
          if (HIWORD(lParam) != LBN_DBLCLK)
            return TRUE;
          /*
            else fall through ...
          */

        case IDOK:
          lpStockInfo = (LPSTOCKINFO) GlobalLock(hCurrStockInfo);
          if (lpStockInfo == NULL)
          {
            ErrorMessage(ERR_BADLOCK);
            goto byebye;
          }
          /*
            Get a pointer to the ticker array
          */
          if (!(lpTick = (LPTICK) GlobalLock(lpStockInfo->hTicks)))
          {
            ErrorMessage(ERR_BADLOCK);
            GlobalUnlock(hCurrStockInfo);
            goto byebye;
          }

          hLB = GetDlgItem(hDlg, ID_LISTBOX);
          i = (int) SendMessage(hLB, LB_GETCURSEL, 0, 0L);
          if (i == LB_ERR)
            break;

          /*
            'i' is the index of the entry we want to edit...
          */
          lpfn = MakeProcInstance(AddTickDlgProc, hThisInstance);
          rc = DialogBoxParam(hThisInstance, "AddTick", hDlg, lpfn, (LONG) i);
          FreeProcInstance(lpfn);

          if (rc == TRUE)
          {
            SendMessage(hLB, LB_DELETESTRING, i, 0L);
            sprintf(buf, "%d\t%ld\t%ld", i, lpTick[i].price, lpTick[i].dwVolume);
            SendMessage(hLB, LB_INSERTSTRING, i, (DWORD) (LPSTR) buf);
            SendMessage(hLB, LB_SETCURSEL, i, 0L);
          }

          GlobalUnlock(lpStockInfo->hTicks);
          GlobalUnlock(hCurrStockInfo);
          break;

        /*
          The user chose the CANCEL button....
        */
        case IDCANCEL :
byebye:
          EndDialog(hDlg, FALSE);
          break;
      }
      return TRUE;

    default:
      return FALSE;
  }
}


LONG FAR PASCAL GetDlgItemLong(HWND hDlg, WORD id, BOOL FAR *lpTranslated,
                    BOOL bSigned)
{
  extern long atol();
  char szBuf[64];

  GetDlgItemText(hDlg, id, (LPSTR) szBuf, sizeof(szBuf));
  return atol(szBuf);
}

VOID FAR PASCAL SetDlgItemLong(HWND hDlg, WORD id, LONG lNum)
{
  char szBuf[64];

  sprintf(szBuf, "%ld", lNum);
  SetDlgItemText(hDlg, id, (LPSTR) szBuf);
}

