/* POPPADP.C -- Popup Editor Printing Functions */

#include <windows.h>
#include <drivinit.h>
#include <string.h>
#include <stdio.h>
#include "stock.h"

int PASCAL GraphWndPrint(HWND hWnd, HDC hDC, LPSTOCKINFO lpStockInfo);

BOOL bUserAbort = FALSE;
HWND hDlgPrint = NULL;

char szDevice[160];	/* Contains the device, the driver, and the port */
PSTR szDriver;		/* Pointer to the driver name			 */
PSTR szPort;		/* Port, ie, LPT1				 */
PSTR szTitle;		/* Global pointer to job title			 */
int iPrinter = 0;	/* level of available printer support.		 */
			/* 0 - no printer available			 */
			/* 1 - printer available			 */
			/* 2 - driver supports 3.0 device initialization */
HANDLE hInitData=NULL;	/* handle to initialization data		 */

char szExtDeviceMode[] = "EXTDEVICEMODE";


/****************************************************************************
 *									    *
 *  FUNCTION   : GetPrinterDC ()					    *
 *									    *
 *  PURPOSE    : Creates a printer display context for the default device.  *
 *		 As a side effect, it sets the szDevice and szPort variables*
 *		 It also sets iPrinter to the supported level of printing.  *
 *									    *
 *  RETURNS    : HDC   - A handle to printer DC.			    *
 *									    *
 ****************************************************************************/
HDC FAR PASCAL GetPrinterDC(void)
{
    char     szT[32];
    HDC      hdc;
    LPSTR    lpdevmode = NULL;

    iPrinter = 0;

    /* Get the printer information from win.ini into a buffer and
     * null terminate it.
     */
    GetProfileString ( "windows", "device", "" ,szDevice, sizeof(szDevice));
    for (szDriver = szDevice; *szDriver && *szDriver != ','; szDriver++)
	;
    if (*szDriver)
	*szDriver++ = 0;

    /* From the current position in the buffer, null teminate the
     * list of ports
     */
    for (szPort = szDriver; *szPort && *szPort != ','; szPort++)
	;
    if (*szPort)
	*szPort++ = 0;

    /* if the device, driver and port buffers all contain meaningful data,
     * proceed.
     */
    if (!*szDevice || !*szDriver || !*szPort){
	*szDevice = 0;
	return NULL;
    }

    /* Create the printer display context */
    if (hInitData){
	/* Get a pointer to the initialization data */
	lpdevmode = (LPSTR) LocalLock (hInitData);

	if (lstrcmp (szDevice, lpdevmode)){
	    /* User has changed the device... cancel this setup, as it is
	     * invalid (although if we worked harder we could retain some
	     * of it).
	     */
	    lpdevmode = NULL;
	    LocalUnlock (hInitData);
	    LocalFree (hInitData);
	    hInitData = NULL;
	}
    }
    hdc = CreateDC (szDriver, szDevice, szPort, lpdevmode);

    /* Unlock initialization data */
    if (hInitData)
	LocalUnlock (hInitData);

    if (!hdc)
	return NULL;


    iPrinter = 1;

    /* Find out if ExtDeviceMode() is supported and set flag appropriately */
    if (GetProcAddress (GetModuleHandle (szDriver), szExtDeviceMode))
	iPrinter = 2;

    return hdc;

}

/****************************************************************************
 *									    *
 *  FUNCTION   : AbortProc()						    *
 *									    *
 *  PURPOSE    : To be called by GDI print code to check for user abort.    *
 *									    *
 ****************************************************************************/
BOOL FAR PASCAL AbortProc(hPrinterDC, nCode)
  HDC hPrinterDC;
  short nCode;
{
  MSG msg;

  while (!bUserAbort && PeekMessage(&msg, NULL, 0, 0, PM_REMOVE))
    if (!hDlgPrint || !IsDialogMessage(hDlgPrint, &msg))
    {
      TranslateMessage(&msg);
      DispatchMessage(&msg);
    }
  return !bUserAbort;
}


/****************************************************************************
 *									    *
 *  FUNCTION   : PrintDlgProc ()					    *
 *									    *
 *  PURPOSE    : Dialog function for the print cancel dialog box.	    *
 *									    *
 *  RETURNS    : TRUE  - OK to abort/ not OK to abort			    *
 *		 FALSE - otherwise.					    *
 *									    *
 ****************************************************************************/
BOOL FAR PASCAL PrintDlgProc(hDlg, iMessage, wParam, lParam)
  HWND hDlg;
  unsigned iMessage;
  WORD wParam;
  LONG lParam;
{
  switch (iMessage)
  {
  case WM_INITDIALOG:
    /* Set up information in dialog box */
    SetDlgItemText (hDlg, IDD_PRINTDEVICE, (LPSTR) szDevice);
    SetDlgItemText (hDlg, IDD_PRINTPORT, (LPSTR) szPort);
    SetDlgItemText (hDlg, IDD_PRINTTITLE, (LPSTR) szTitle);
    EnableMenuItem(GetSystemMenu(hDlg, FALSE), SC_CLOSE,
		   MF_GRAYED);
    break;

  case WM_COMMAND:
    bUserAbort = TRUE;
    EnableWindow(GetParent(hDlg), TRUE);
    DestroyWindow(hDlg);
    hDlgPrint = 0;
    break;

  default:
    return FALSE;
  }
  return TRUE;
}


/****************************************************************************
 *									    *
 *  FUNCTION   : PrintFile ()						    *
 *									    *
 *  PURPOSE    : Prints the contents of the edit control.		    *
 *									    *
 ****************************************************************************/

BOOL PASCAL PrintFile(void)
{
  BOOL  bError = FALSE;
  char  szMsg[40];
  FARPROC lpfnAbortProc, lpfnPrintDlgProc;
  HDC   hPrnDC;
  RECT  rect;
  short yChar, nCharsPerLine, nLinesPerPage, 
        nTotalLines, nTotalPages, nPage, nLine, nLineNum = 0;
  TEXTMETRIC tm;

  if (!hWndActive)
    return FALSE;

  if (NULL == (hPrnDC = GetPrinterDC()))
    return TRUE;

  GetTextMetrics(hPrnDC, &tm);
  yChar = tm.tmHeight + tm.tmExternalLeading;

  nCharsPerLine = GetDeviceCaps(hPrnDC, HORZRES) / tm.tmAveCharWidth;
  nLinesPerPage = GetDeviceCaps(hPrnDC, VERTRES) / yChar;

  EnableWindow(hWndMain, FALSE);

  bUserAbort = FALSE;
  lpfnPrintDlgProc = MakeProcInstance(PrintDlgProc, hThisInstance);
  hDlgPrint = CreateDialog(hThisInstance, "PrintDlgBox", hWndMain, lpfnPrintDlgProc);
  if (hDlgPrint)
  {
    ShowWindow(hDlgPrint, SW_SHOW);
    UpdateWindow(hDlgPrint);
  }

  lpfnAbortProc = MakeProcInstance(AbortProc, hThisInstance);
  Escape(hPrnDC, SETABORTPROC, 0, (LPSTR) lpfnAbortProc, NULL);

  /*
    Start printing
  */
  strcpy(szMsg, "Stock");
  if (Escape(hPrnDC, STARTDOC, strlen(szMsg), szMsg, NULL) > 0)
  {
      HANDLE      hStockInfo;
      LPSTOCKINFO lpStockInfo;

      if ((hStockInfo = GetWindowWord(hWndActive, 0)) != NULL && 
          (lpStockInfo = (LPSTOCKINFO) GlobalLock(hStockInfo)))
      {
        GraphWndPaint(hWndActive, hPrnDC, lpStockInfo, TRUE);
        GlobalUnlock(hStockInfo);
      }
  }
  else
    bError = TRUE;

  /*
    End the printing operation
  */
end:
  if (!bError)
  {
    Escape(hPrnDC, NEWFRAME, 0, NULL, NULL);
    Escape(hPrnDC, ENDDOC, 0, NULL, NULL);
  }

  if (!bUserAbort)
  {
    EnableWindow(hWndMain, TRUE);
    if (hDlgPrint)
      DestroyWindow(hDlgPrint);
    hDlgPrint = NULL;
  }

  if (bError)
    MessageBox(hWndMain, "Could not print", "Stock", MB_OK | MB_ICONEXCLAMATION);

  FreeProcInstance(lpfnPrintDlgProc);
  FreeProcInstance(lpfnAbortProc);
  DeleteDC(hPrnDC);

  return bError || bUserAbort;
}


/****************************************************************************
 *									    *
 *  FUNCTION   : GetInitializationData()				    *
 *									    *
 *  PURPOSE    : Gets DC initialization data from a printer driver	    *
 *		 supporting ExtDeviceMode(). Called in response to the	    *
 *		 File/Printer setup menu selection.			    *
 *									    *
 *		 This function allows the user to change the printer	    *
 *		 settings FOR MULTIPAD ONLY.  This allows Multipad to print *
 *		 in a variety of settings without messing up any other	    *
 *		 applications. In a more sophisticated application, this    *
 *		 setup could even be saved on a document-by-document basis. *
 *									    *
 ****************************************************************************/
BOOL FAR PASCAL GetInitializationData( hwnd )
HWND hwnd ;
{
    LPSTR     lpOld;
    LPSTR     lpNew;
    FARPROC   lpfn;
    HANDLE    hT,hDrv;
    char      sz[32];
    WORD      cb;
    int       flag;

    /* Pop up dialog for user and retain data in app buffer */
    flag = DM_PROMPT | DM_COPY;

    /* Load the device driver and find the ExtDeviceMode() function */
    wsprintf (sz, "%s.drv", (LPSTR)szDriver);
    if ((hDrv = LoadLibrary (sz)) < 32)
	return FALSE;
    if (!(lpfn = GetProcAddress (hDrv, szExtDeviceMode)))
	return FALSE;

    if (hInitData){
	/* We have some old data... we want to modify the previously specified
	 * setup rather than starting with the default setup.
	 */
	lpOld = (LPSTR)LocalLock(hInitData);
	flag |= DM_MODIFY;
    }
    else
	lpOld = NULL;

    /* Get the number of bytes needed for the init data */
    cb = (*lpfn) (hwnd,
		  hDrv,
		  NULL,
		  (LPSTR)szDevice,
		  (LPSTR)szPort,
		  (LPDEVMODE)NULL,
		  (LPSTR)NULL,
		  0);

    /* Grab some memory for the new data and lock it. */
    hT	  = LocalAlloc (LHND,cb);
    lpNew = (LPSTR)LocalLock (hT);

    /* Post the device mode dialog. 0 flag iff user hits OK button */
    if ((*lpfn) (hwnd,
		 hDrv,
		 (LPDEVMODE)lpNew,
		 (LPSTR)szDevice,
		 (LPSTR)szPort,
		 (LPDEVMODE)lpOld,
		 (LPSTR)NULL,
		 flag)==IDOK)
	flag = 0;

    /* Unlock the input structures */
    LocalUnlock (hT);
    if (hInitData)
	LocalUnlock (hInitData);

    /* If the user hit OK and everything worked, free the original init.
     * data and retain the new one.  Otherwise, toss the new buffer.
     */
    if (flag)
	LocalFree (hT);
    else{
	if (hInitData)
	    LocalFree (hInitData);
	hInitData = hT;
    }

    FreeLibrary(hDrv);
    return (!flag);
}

