/*=================================================================*/
/*                                                                 */
/* File    : DB.C                                                  */
/*                                                                 */
/* Purpose : Simple database routines for the stock charting app   */
/*                                                                 */
/* History :                                                       */
/*                                                                 */
/* (C) Copyright 1990,1991 Marc Adler/Magma Systems                */
/*=================================================================*/

#include <string.h>
#include <fcntl.h>
#include <io.h>
#include <sys\types.h>
#include <sys\stat.h>

#include "windows.h"
#include "stock.h"


HANDLE PASCAL StockInfoAlloc(void)
{
  HANDLE hStockInfo;
  LPSTOCKINFO lpStockInfo;
  LPGRAPHINFO lpGraphInfo;

  if ((hStockInfo = GlobalAlloc(GMEM_MOVEABLE | GMEM_ZEROINIT, 
                                (DWORD) sizeof(STOCKINFO))) == NULL)
    return NULL;
  if ((lpStockInfo = (LPSTOCKINFO) GlobalLock(hStockInfo)) == NULL)
  {
    ErrorMessage(ERR_CANTALLOCINFO);
    GlobalFree(hStockInfo);
    return NULL;
  }

  lstrcpy(lpStockInfo->szFileName, (LPSTR) "(Untitled)");

  /*
   * Set the initial attributes of the graph 
   */
  lpGraphInfo = &lpStockInfo->StockFile.graphinfo;
  lpGraphInfo->dwMinPrice     = 1;
  lpGraphInfo->dwMaxPrice     = 100;
  lpGraphInfo->dwScaleFactor  = 1;
  lpGraphInfo->dwTickInterval = 1;
  lpGraphInfo->iDenominator   = 1;

  lpStockInfo->StockFile.dwMagic = MAGIC_COOKIE;
  lpStockInfo->dwFlags |= (STATE_HAS_VGRID | STATE_HAS_HGRID);
  lpStockInfo->graphAttrs.iMappingMode = MM_ISOTROPIC;
  lpStockInfo->graphAttrs.idxPen = 0;
  lpStockInfo->graphAttrs.clrPen = RGB(0x00, 0x00, 0x00);
  lpStockInfo->graphAttrs.idxBackground = 7;
  lpStockInfo->graphAttrs.clrBackground = RGB(0xFF, 0xFF, 0xFF);
  CheckMenuItem(GetMenu(hWndMain), ID_GRAPH_ISOTROPIC, 
                                   MF_BYCOMMAND | MF_CHECKED);
  CheckMenuItem(GetMenu(hWndMain), ID_GRAPH_ANISOTROPIC, 
                                   MF_BYCOMMAND | MF_UNCHECKED);

  GlobalUnlock(hStockInfo);
  return hStockInfo;
}



/*******************************************************************/
/*                                                                 */
/* Function : StockFIleRead()                                      */
/*                                                                 */
/* Purpose  : Reads a stock file in from the disk and allocates a  */
/*            stock information structure for it.                  */
/*                                                                 */
/* Returns  : TRUE if the stock file was read, FALSE if not.       */
/*                                                                 */
/*******************************************************************/
int PASCAL
StockFileRead(LPSTR szFileName)
{
  char sz[128];
  int fd;
  int rc = FALSE;
  HANDLE hStockInfo;
  LPSTOCKINFO lpStockInfo;
  LPTICK lpTick;
  HWND hWnd;

  /*
   * Make sure it has the .STO extension if the user didn't
   * specify one 
   */
  lstrcpy((LPSTR) sz, szFileName);
  if (!strchr(sz, '.'))
    strcat(sz, ".sto");

  /*
   * Open the file for reading 
   */
  if ((fd = _lopen(sz, O_RDONLY | O_BINARY)) < 0)
  {
    return ErrorMessage(ERR_CANTOPEN);
  }

  /*
    Allocate a stock information structure
  */
  if ((hStockInfo = StockInfoAlloc()) == NULL)
    return FALSE;
  if ((lpStockInfo = (LPSTOCKINFO) GlobalLock(hStockInfo)) == NULL)
  {
    ErrorMessage(ERR_CANTALLOCINFO);
    GlobalFree(hStockInfo);
    return NULL;
  }

  /*
   * Save the absolute path name of the file 
   */
  lstrcpy(lpStockInfo->szFileName, (LPSTR) sz);

  /*
   * Read the header 
   */
  if (_lread(fd, (LPSTR) & lpStockInfo->StockFile, sizeof(STOCKFILE))
                  != sizeof(STOCKFILE))
  {
    ErrorMessage(ERR_BADHEADER);
    GlobalUnlock(hStockInfo);
    GlobalFree(hStockInfo);
    goto bye;
  }

  /*
   * Check the magic number to see if it's really a stock db
   * file 
   */
  if (lpStockInfo->StockFile.dwMagic != MAGIC_COOKIE)
  {
    ErrorMessage(ERR_BADSIG);
    GlobalUnlock(hStockInfo);
    GlobalFree(hStockInfo);
    goto bye;
  }

  /*
   * Allocate global memory to hold the ticks 
   */
  if ((lpStockInfo->hTicks = GlobalAlloc(GMEM_MOVEABLE,
         (DWORD)sizeof(TICK)*lpStockInfo->StockFile.nTicks)) == NULL)
  {
    ErrorMessage(ERR_CANTALLOCTICKER);
    goto bye;
  }
  lpStockInfo->nTicksAllocated = lpStockInfo->StockFile.nTicks;
  if ((lpTick = (LPTICK) GlobalLock(lpStockInfo->hTicks)) == NULL)
  {
    ErrorMessage(ERR_CANTALLOCTICKER);
    GlobalFree(lpStockInfo->hTicks);
    goto bye;
  }

  /*
   * Read the ticks into the global memory buffer 
   */
  if (_lread(fd, (LPSTR) lpTick, 
                 sizeof(TICK) * lpStockInfo->StockFile.nTicks) !=
                 sizeof(TICK) * lpStockInfo->StockFile.nTicks)
  {
    ErrorMessage(ERR_CANTREADTICKER);
    GlobalUnlock(lpStockInfo->hTicks);
    GlobalFree(lpStockInfo->hTicks);
    goto bye;
  }

  /*
   * Create a window for this guy 
   */
  hWnd = lpStockInfo->hWnd = GraphCreateWindow((LPSTR) sz);
  SetWindowWord(hWnd, 0, hStockInfo);

  /*
   * Link this entry onto the stock info list 
   */
  hCurrStockInfo = hStockInfo;

  GlobalUnlock(lpStockInfo->hTicks);
  rc = TRUE;

  /*
   * Done reading .... close the file 
   */
bye:
  close(fd);

  GlobalUnlock(hStockInfo);
  if (hWnd)
  {
    InvalidateRect(hWnd, (LPRECT) NULL, FALSE);
    UpdateWindow(hWnd);
  }
  return rc;
}


int PASCAL StockFileSave(HANDLE hStockInfo)
{
  int         fd;
  LPTICK      lpTicks;
  LPSTOCKINFO lpStockInfo;

  lpStockInfo = (LPSTOCKINFO) GlobalLock(hStockInfo);
  lpTicks = (LPTICK) GlobalLock(lpStockInfo->hTicks);

MessageBox(hWndMain, lpStockInfo->szFileName, "StockName", MB_OK);

  if ((fd = _lopen(lpStockInfo->szFileName, OF_WRITE)) < 0)
  {
    if ((fd = _lcreat(lpStockInfo->szFileName, 0)) < 0)
      return ErrorMessage(ERR_CANTOPEN);
  }

  _lwrite(fd, (LPSTR) & lpStockInfo->StockFile, sizeof(STOCKFILE));
  _lwrite(fd, (LPSTR) lpTicks, 
              sizeof(TICK) * lpStockInfo->StockFile.nTicks);
  close(fd);

  lpStockInfo->dwFlags &= ~STATE_DIRTY;
  GlobalUnlock(lpStockInfo->hTicks);
  GlobalUnlock(hStockInfo);

  MessageBox(hWndMain, "The file has been saved", "Stock", MB_OK);
  return TRUE;
}


int PASCAL ErrorMessage(WORD idMsg)
{
  char szMsg[80];

  if (LoadString(hThisInstance, idMsg, (LPSTR) szMsg, sizeof(szMsg)))
  {
    MessageBox(hWndMain, szMsg, NULL, MB_OK | MB_ICONEXCLAMATION);
  }
  return FALSE;
}

