//=================================================================
//  INTERFACE.C
//=================================================================

#include <windows.h>
#include <netcons.h>
#include <neterr.h>
#include <wksta.h>
#include <nmpipe.h>
#include <string.h>
#include <mailslot.h>

#include "pbxpkt.h"
#include "runtime.h"

#define MAX_MAILSLOT_NAME   255
#define TIMEOUT 	    1000L
#define OF_PIPE 	    (OF_READWRITE | OF_SHARE_DENY_NONE)
#define ERROR_SEM_TIMEOUT   121
#define MAXTRIES	    5

static WORD GetMachineName(LPSTR szWkstaName);

//=================================================================
//  PbxRegister() -- Registers a user with PBX.
//=================================================================

WORD PbxRegister(int hPipe, LPSTR szName, WORD wClientId)
{
    PBXPKT packet;
    WORD   err;

    packet.usPktID = REGISTER;
    packet.usPktSize = sizeof(PBXPKT);
    packet.usNameCnt = 1;

    packet.aPBXName[0].usClientType = wClientId;
    _fstrcpy(packet.aPBXName[0].pszName, szName);

    err = _lwrite(hPipe, (LPSTR) &packet, sizeof(PBXPKT));

    return ( err <= 0 ? -1 : 0 );
}

//=================================================================
//  PbxConnent() -- Connects two users together.
//=================================================================

WORD PbxConnect(int hPipe, LPSTR szName, WORD wClientId)
{
    PBXPKT packet;
    WORD   err;

    packet.usPktID = CONNECT;
    packet.usPktSize = sizeof(PBXPKT);
    packet.usNameCnt = 1;

    packet.aPBXName[0].usClientType = wClientId;
    _fstrcpy(packet.aPBXName[0].pszName, szName);

    err = _lwrite(hPipe, (LPSTR) &packet, sizeof(PBXPKT));

    return ( err <= 0 ? -1 : 0 );
}

//=================================================================
//  PbxSearch() -- Searches for a PBX and returns its name.
//=================================================================

WORD PbxSearch(LPSTR szPbxName)
{
    WORD hMailSlot, wBytes = 0, err;
    WORD wNextSize = 0, wNextPriority = 0;
    char szMailSlot[MAX_MAILSLOT_NAME];
    char szWkstaName[UNCLEN+1] = "\\\\";
    WORD nMaxTries = MAXTRIES;

    //=============================================
    //	First make the mailslot
    //=============================================

    if ( err = DosMakeMailslot(ANNOUNCELINE, UNCLEN+1,
				UNCLEN+1, &hMailSlot) )
    {
	return err;
    }

    strcpy(szMailSlot, "\\\\*");
    strcat(szMailSlot, ANNOUNCELINE);
    GetMachineName(&szWkstaName[2]);

    //=============================================
    //	Write to the mailslot the machine name
    //=============================================

    if ( err = DosWriteMailslot(szMailSlot, szWkstaName,
				 strlen(szWkstaName),
				 ANNOUNCEPRIORITY, 2, 0) )

    {
	DosDeleteMailslot(hMailSlot);
	return err;
    }

    //=============================================
    //	Read the mailslot for the reply.
    //	We have to read more than once because
    //	we can receive our own broadcast. We will
    //	read upto MAXTRIES times before reporting
    //	timeout.
    //=============================================

    do {
	err = DosReadMailslot(hMailSlot, szPbxName,
			      &wBytes, &wNextSize,
			      &wNextPriority, TIMEOUT);

	//============================================
	//  We have received the reply but we do
	//  not know if it is from PBX or if it is
	//  our broadcast.
	//============================================

	if ( !err && _fstrcmp(szPbxName, szWkstaName) == 0 )
	{
	    err = -1;	    //... it was our broadcast
	}
    }
    while(err && --nMaxTries);

    DosDeleteMailslot(hMailSlot);

    if ( err && nMaxTries == 0 )    //... ignore err and report
    {				    //... timeout condition
	return ERROR_SEM_TIMEOUT;
    }

    return err;
}

//=================================================================
//  OpenPIpe() -- Opens the pipe
//=================================================================

WORD OpenPipe(int *phPipe, LPSTR szPipeName)
{
    OFSTRUCT of;
    WORD     hPipe;

    *phPipe = OpenFile(szPipeName, &of, OF_PIPE);

    return (*phPipe == -1 ? of.nErrCode : 0);
}

//=================================================================
//  PeekPIpe() -- Retrieves the contents of a pipe without removing
//  it.
//=================================================================

WORD PeekPipe(int hPipe, LPVOID buf, WORD nBytes,
	      LPWORD nBytesRead, LPWORD nAvail)
{
    AVAILDATA  avail;
    WORD       wPipeState, err;

    err = DosPeekNmPipe(hPipe, buf, nBytes, nBytesRead,
			(PAVAILDATA) &avail, (LPWORD) &wPipeState);

    *nAvail = avail.cbpipe;

    return err;
}

//=================================================================
//  GetMachineName -- Retrieves the local computer name.
//=================================================================

static WORD GetMachineName(LPSTR szWkstaName)
{
    HANDLE hWksta;
    struct wksta_info_0 far *wksta = NULL;
    unsigned rc, total;

    //... This first call will fail, but it will return the
    //... actual number of bytes needed to complete the call.

    rc = NetWkstaGetInfo(NULL, 0, (LPVOID) wksta, 0, &total);

    if ( rc == ERROR_MORE_DATA || rc == NERR_BufTooSmall)
    {
	hWksta = GlobalAlloc(GMEM_MOVEABLE, total);
	wksta = (LPVOID) GlobalLock(hWksta);

	rc = NetWkstaGetInfo(NULL, 0, (LPVOID) wksta, total, &total);

	if ( rc == NERR_Success )
	{
	    _fstrcpy((LPSTR) szWkstaName, wksta->wki0_computername);
	}

	GlobalFree(hWksta);
    }

    return rc;
}
