/********************************************************************
ENGINE.C:   Routines to load, initialize and call a 32 bit engine.

The following are code examples for performing mixed 16/32 bit
programming in the Windows environment.  The 32 bit code is a
standard format EXP file generated by Phar Lap's 386|LINK.

This code is all in the 16 bit part of the program, and was
compiled using Microsoft C Version 6.0 and MASM 5.1.
********************************************************************/

#include    <windows.h>     /* standard Windows header file.       */
#include    "winmem32.h"    /* declarations of Winmem32 functions. */
#include    "front.h"       /* front header file.                  */
#include    "engine.h"      /* engine header file.                 */

/*  Important addresses and segment selectors */

static  HANDLE  hDataSeg;           /* Selector of Data Segment    */
static  HANDLE  hCodeSeg;           /* Selector of Code Segment    */
static  DWORD   InitAddr;           /* Entry point address         */
static  DWORD   EvalAddr;           /* Address of Action Routine   */
static  DWORD   CommAddr;           /* Address of Comm Buffer      */
static  DWORD   StackAddr;          /* Initial Stack Pointer       */
static  DWORD   ParamBlock;         /* Parameter Block Offset      */

static  BOOL    bEnginePresent = FALSE; /* Is engine present flag. */

/********************************************************************
    This routine allocates a 32 bit segment, loads the kernel
    and sets up the addresses to call into the kernel.
********************************************************************/
int     LoadEngine(HWND hWnd,LPSTR name)    {
struct      EXPHeader exphead;
int         hFile,extra;
DWORD       segsize,filesize,chunksize;
char huge   *pUse16CodeSegment;
char huge   *pCodeSegment;
WORD        ret;
OFSTRUCT    of;

    if (bEnginePresent) {

        KillEngine(hWnd);

    }

    hFile = OpenFile(name,&of,OF_READ);

    if (hFile == -1)
        return 0;

    _lread(hFile,(LPSTR)&exphead,sizeof(struct EXPHeader));

    /* test for magic number */
    if (exphead.signature != 0x504D)    {

        _lclose(hFile);
        return 0;

    }

    filesize = (DWORD)(exphead.sizeinblocks) * 512L +
           (DWORD)(exphead.sizemod512) - 1024L;

    /* Allocate a large kernel */
    extra = 1000;   /* add 4 extra megabytes to kernel */

    segsize = filesize +
        (DWORD)(exphead.mindatapages + extra) * 4096L;

    ret = Global32Alloc(segsize,&hDataSeg,segsize,0);

    if (ret != 0)
        return 0;

    /* Lock the data segment */
    pCodeSegment = (char huge *)GlobalLock(hDataSeg);

    /* bypass header */

    _llseek(hFile,0x200L,0);

    /* load in the 32-Bit engine */

    while (filesize > 0)    {

        chunksize = (filesize >= MAXCHUNK) ?
                MAXCHUNK : filesize;

        _lread(hFile,pCodeSegment,chunksize);

        filesize     -= chunksize;

        pCodeSegment += chunksize;
    }

    /* Close the file */
    _lclose(hFile);

    /* Unlock the data segment */
    GlobalUnlock(hDataSeg);

    /* we have now read the EXP file; create a code alias */

    ret = Global32CodeAlias(hDataSeg,&hCodeSeg,0);

    if (ret != 0)
        return 0;

    /*
       Initialize addresses from parameters stored in the EXP
       header.
    */

    StackAddr = exphead.initESP;
    InitAddr  = exphead.initEIP;

    /* Engine has now been loaded */

    bEnginePresent = TRUE;

    /*
       Do the initial call to the engine, which will set up
       the call buffer, as well as get the address of the
       Parameter Block.

     */

    InitEngine(hWnd);

    return 1;
}

/********************************************************************
    This routines cleans up the kernel.
********************************************************************/
void    KillEngine(HWND hWnd)   {

    if (!bEnginePresent)    {

        MessageBox(hWnd,"Engine Is Not Present!","Front",
            MB_OK | MB_ICONHAND);
        return;

    }

    Global32CodeAliasFree(hDataSeg,hCodeSeg,0);
    Global32Free(hDataSeg,0);

    bEnginePresent = FALSE;
}

/********************************************************************
    This routine calls through to the 32 bit kernel.
********************************************************************/
void    CallEngine(HWND hWnd)   {

    if (!bEnginePresent)    {

        MessageBox(hWnd,"Engine Is Not Present!","Front",
            MB_OK | MB_ICONHAND);
        return;
    }

    Call32(hCodeSeg,hDataSeg,EvalAddr,StackAddr,(LONG)0,0L);
}

/********************************************************************
    Initialize the engine and set up helper segments for 16 / 32
    communications.
********************************************************************/
void    InitEngine(HWND hWnd)   {

    if (!bEnginePresent)    {

        MessageBox(hWnd,"Engine Is Not Present!","Front",
            MB_OK | MB_ICONHAND);
        return;
    }

    ParamBlock = Call32(hCodeSeg,
                hDataSeg,
                InitAddr,
                StackAddr,
                (LONG)0,0L);

    /* Get offset of Action Routine */
    EvalAddr = GetAction();

    /* Offset of Comm Buffer    */
    CommAddr = GetBuffer();
}
