#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <errno.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/time.h>
#include <fcntl.h>

#include "pty.h"
#include "tty.h"
#include "mesg.h"
#include "misc.h"
#include "mutex.h"


/*
 * --------------------------------------------------------------------------
 *  Utility functions
 * --------------------------------------------------------------------------
 */

static int empty = 1;
static uchar unChar;

int GetChar()
{
    uchar ch; int try = 3;

    if (!empty) return unChar;

    do {
	if (read(FROM_PC, &ch, 1) == 1) {
	    return ch & 0xFF;
	}
    } while (--try);
    Warning("Fail to get a char\r\n");
    return -1;
}

int UnGetChar(ch)
uchar ch;
{
    if (empty) {
	empty = 0;
	unChar = ch;
	return 0;
    }
    return -1;
}

void SendInt(num)
int num;
{
    uchar buf[2];

    num += MAX_SIGNED_INT;
    buf[0] = HighByte(num);
    buf[1] = LowByte(num);
    Match(write(TO_PC, buf, 2), 2);
}

int GetInt()
{
    int num;

    num = GetChar() << 8;
    return num + GetChar() - MAX_SIGNED_INT;
}




/*
 *  Global variables
 */
int	done = 0;			/* to be modified by ProblemChild()  */
int	numWindows = 0;			/* # of windows so far               */
int	maxFd = 0;			/* max fd in readSet                 */
fd_set	readSet;			/* for detecting input from PC/wins  */
fd_set	PCReadSet;
Window	*focus = NULL;
Window	windows[MAX_NUM_WINDOWS +1];
uchar	inBuf[INBUF_SIZE];		/* input/output buffer               */
uchar	outBuf[OUTBUF_SIZE +2];
uchar	comRecId = CONSOLE;		/* comm data receiver                */

/* default value even if PC doesn't initialize them */
int scrRow = 1, scrCol = 30;



int EvalReadSet()
{
    int i;

    FD_ZERO(&readSet);
    /* FD_SET(FROM_PC, &readSet); */
    for (i = 0; i <= MAX_WIN_ID; i++)
	if (windows[i].pid) FD_SET(windows[i].pty.master, &readSet);
    return 0;
}


int PutToFocus(ch)
uchar ch;
{
    if (!focus) {
	sprintf(echoBuf, "Try to put %d to focus(NULL)\r\n", ch);
	Inform(echoBuf);
	return -1;
    }
    Match(write(focus->pty.master, &ch, 1), 1);
    return 0;
}


int Sync()
{
    int nBytes = 0, id;

    SendVirDpy();
    for (id = MIN_WIN_ID; id <= MAX_WIN_ID; id++) {
	if (!windows[id].pid) continue;
	outBuf[nBytes++] = MX_AT;
	outBuf[nBytes++] = WX_AT;
	outBuf[nBytes++] = MX_OPEN;
	outBuf[nBytes++] = WX_OPEN;
	outBuf[nBytes++] = id;
	outBuf[nBytes++] = windows[id].row0;
	outBuf[nBytes++] = windows[id].col0;
	outBuf[nBytes++] = windows[id].nRows;
	outBuf[nBytes++] = windows[id].nCols;
    }
    Check(nBytes, 0);
    Match(write(TO_PC, outBuf, nBytes), nBytes);
    comRecId = CONSOLE;
    return 0;
}


int GetVirDpy()
{
    scrRow = GetChar();
    scrCol = GetChar();
    return 0;
}

int SendVirDpy()
{
    int nBytes = 0;

    outBuf[nBytes++] = MX_AT;
    outBuf[nBytes++] = WX_AT;
    outBuf[nBytes++] = MX_SCR_POS;
    outBuf[nBytes++] = WX_SCR_POS;
    outBuf[nBytes++] = scrRow;
    outBuf[nBytes++] = scrCol;
    Match(write(TO_PC, outBuf, nBytes), nBytes);
    return 0;
}



int TermOpen()
{
#define	MAX_ARG		30
    char *file, *argv[MAX_ARG], *getenv();	/* *rindex(); */
    struct winsize win;
    int id;


    /* find a id with empty slot */
    for (id = 0; windows[id].pid; id++) ;


	/* get the geometry */
    windows[id].row0 = GetChar();
    windows[id].col0 = GetChar();
    windows[id].nRows = GetChar();
    windows[id].nCols = GetChar();

    if ((file = getenv("SHELL")) == NULL)
	file = "/bin/csh";
    if ((argv[0] = rindex(file, '/')) == NULL)
	argv[0] = "-sh";
    argv[1] = NULL;


    /* why this line? */
    if (pty_master(&windows[id].pty) < 0) return -1;

    if ((windows[id].pid = fork()) < 0) {
	perror("fork");
	exit(-1);
    }

	/* --- CHILD --- */
    else if (windows[id].pid == CHILD) {
	int i;		/* loop counter */

	    /* Set up the other half of pseudo-terminal */
	if (pty_slave(&windows[id].pty) < 0) {
	    perror("pty_slave():");
	    exit(-1);
	}

	    /* Give the slave the same term cap as ordinary stdin */
	if (tty_copy(STDIN, windows[id].pty.slave) < 0) {
	    perror("tty_copy():");
	    exit(-1);
	}

	ioctl(STDIN, TIOCGWINSZ, &win);
	win.ws_row = windows[id].nRows; win.ws_col = windows[id].nCols;
	ioctl(windows[id].pty.slave, TIOCSWINSZ, &win);
	if (tty_noraw(windows[id].pty.slave) < 0 ||
		tty_echo(windows[id].pty.slave) < 0) {
	    perror("TermOpen():");
	    exit(-1);
	}

	/*
	 *  Close all unnecessary fd's.  Close master but not slave since it
	 *  has just been open in this child block (this is child!).
	 */
	for (i = 0; i <= MAX_WIN_ID; i++)
	    if (windows[i].pid) close(windows[i].pty.master);
	/* close its master now, since we doesn't need it any more */
	close(windows[id].pty.master);
	close(STDIN);
	close(STDOUT);
	close(STDERR);

	    /* reopen stdin, stdout & stderr for this process (child) */
	if (dup2(windows[id].pty.slave, STDIN) < 0 ||
		dup2(windows[id].pty.slave, STDOUT) < 0 ||
		dup2(windows[id].pty.slave, STDERR) < 0) {
	    perror("dup2()");
	    exit(-1);
	}

	    /* now even the slave is no longer needed, so close it as well */
	close(windows[id].pty.slave);

	if (execvp(file, argv) < 0) {
	    Perr("execvp()");
	    exit(-1);
	}
    }		/* --- END OF CHILD BLOCK --- */

	/* --- PARENT --- */
    maxFd = max(windows[id].pty.master, maxFd);
    numWindows++;
    EvalReadSet();

    /* the new window become the new focus */
    focus = &windows[id];

    return 0;
}



int TermClose()
{
    uchar ch;


    if (read(FROM_PC, &ch, 1) < 0) {
	perror("read():");
	return -1;
    }
    if (windows[ch].pid == 0 || kill(windows[ch].pid, SIGKILL) < 0)
	return -1;
    return 0;
}


int TermGeom()
{
    int id;
    struct winsize win;

    id = GetChar();
    if (id >= MIN_WIN_ID && id <= MAX_WIN_ID && windows[id].pid) {
	windows[id].row0 = GetChar();
	windows[id].col0 = GetChar();
	windows[id].nRows = win.ws_row = GetChar();
	windows[id].nCols = win.ws_col = GetChar();
	return ioctl(windows[id].pty.master, TIOCSWINSZ, &win) == -1 ? -1 : 0;
    }
    return -1;
}



int TermChange()
{
    int id;

    id = GetChar();
    if (id >= MIN_WIN_ID && id <= MAX_WIN_ID && windows[id].pid) {
	focus = &windows[id];
	return 0;
    }
    sprintf(echoBuf, "Can't change term to window #%d\r\n", id);
    Inform(echoBuf);
    return -1;
}


/*
 *  Solely for debug use right now.
 */
int Attention()
{
    sprintf(echoBuf,
	    "numWin = %d, comRecId = %d, focus = %08X\r\n",
	    numWindows, comRecId, focus);
    Inform(echoBuf);
    return 0;
}


/*
 *  Quit mutex, very dangerous.
 */
Quit()
{
    MutexEnd();
    exit(0);
    Inform("Impossible\r\n");
}



int ProblemChild()
{
    static uchar cmd[6] = {MX_AT, WX_AT, MX_CLOSE, WX_CLOSE};
    int pid, state, id;


    pid = wait(&state);

    /* find out problem child's id */
    for (id = MIN_WIN_ID; id <= MAX_WIN_ID && pid != windows[id].pid; id++)
	;

#ifdef	DEBUG
    sprintf(echoBuf, "--- window #%d die ---\r\n", id);
    Inform(echoBuf);
#endif

    close(windows[id].pty.master);
    if (id < MIN_WIN_ID || id > MAX_WIN_ID) {
	Warning("id out of range\n");
	exit(-1);
    }

    if (focus == &windows[id]) {
	focus = NULL;
#ifdef	DEBUG
	Inform("Change focus --> NULL\r\n");
#endif
    }

    cmd[4] = id;
    windows[id].pid = 0;
    Match(write(TO_PC, cmd, 5), 5);
    fsync(TO_PC);
    if (--numWindows == 0)
	done = 1;

    /* EvalReadSet(); */
}

/* ------------------ ProblemChild() --------------------------------------- */



int MutexIn()
{
    static int state = MX_NIL, ch;
    static uchar mesg[5] = {MX_AT, WX_AT};	/* reply mesg */
    int rval;


    while (1) {
	switch (state) {
	case MX_NIL:
	    ch = GetChar();
	    if (ch != MX_AT) {
		PutToFocus(ch);
		return 0;
	    }
	    state = MX_AT;
	    return 0;

	case MX_1:	/* already get [MX_AT, WX_AT, 0] */
	    ch = GetChar();
	    if (ch == ' ') {
		state = MX_2;
		return 0;
	    }
	    PutToFocus(MX_AT); PutToFocus(WX_AT); PutToFocus(0);
	    UnGetChar(ch);
	    return 0;

	case MX_2:	/* already get [MX_AT, WX_AT, 0, ' '] */
	    ch = GetChar();
	    if (!ch) {
		rval = Sync();
		break;
	    }
	    PutToFocus(MX_AT); PutToFocus(WX_AT);
	    PutToFocus(0); PutToFocus(' ');
	    UnGetChar();
	    return 0;

	case MX_AT:	/* expect WX_AT */
	    ch = GetChar();
	    if (ch == WX_AT)
		state = WX_AT;
	    else
		PutToFocus(MX_AT), PutToFocus(ch);
	    return;

	case WX_AT:	/* get first control code */
	    ch = GetChar();
	    state = ch ? MX_AT2 : MX_1;
	    return 0;

	case MX_AT2:	/* get the complement of the control code */
	    state = ch;
	    ch = GetChar();
	    if ((state ^ ch) != 0xFF) {
		state = MX_NIL;
		PutToFocus(MX_AT); PutToFocus(WX_AT);
		PutToFocus(state); PutToFocus(ch);
		return 0;
	    }
	    continue;

	case MX_CLOSE:	rval = TermClose();	break;
	case MX_CHANGE:	rval = TermChange();	break;
	case MX_END:	rval = Quit();		break;
	case MX_GEOM:	rval = TermGeom();	break;
	case MX_OPEN:	rval = TermOpen();	break;
	case MX_SCR_POS:rval = GetVirDpy();	break;
	case MX_SYNC:	rval = Sync();		break;
	case MX_USER:	rval = Attention();	break;

	default:
	    sprintf("Unknown state in MutexIn(): %d\r\n", state);
	    Inform(echoBuf);
	    state = MX_NIL;
	    continue;
	}
	if (rval < 0) {
	    sprintf(echoBuf, "MutexIn(): FAIL to do the request #%d\r\n",state);
	    Inform(echoBuf);
	}
	state = MX_NIL;

	mesg[2] = rval < 0 ? MX_NAK : MX_ACK;
	mesg[3] = ~mesg[2];
	Match(write(TO_PC, mesg, 4), 4);
	fsync(TO_PC);

	return rval == MX_NAK ? -1 : 0;
    }
}



int MutexOut(tid)
int tid;
{
    /* the command string to switch window if differ from the last one */
    static uchar swCmd[6] = {MX_AT, WX_AT, MX_CHANGE, WX_CHANGE};
    int nBytes;

    if ((nBytes = read(windows[tid].pty.master, outBuf, OUTBUF_SIZE)) < 0)
	return -1;

    if (tid != comRecId) {
	swCmd[4] = comRecId = tid;
	Match(write(TO_PC, swCmd, 5), 5);
    }

    Match(write(TO_PC, outBuf, nBytes), nBytes);
    fsync(TO_PC);	/* this is not neccessary */
    return 0;
}


int MutexLoop()
{
    int rval, i, n;
    struct timeval timeOut = {0L, 0L};
    struct timeval timeOut2 = {0L, 50000L};	/* 5/1000 sec */


    EvalReadSet();

    while (!done) {
	FD_SET(FROM_PC, &PCReadSet);
	if (select(MAX_NUM_WINDOWS, &PCReadSet, 0, 0, &timeOut2) > 0) {
	    rval = MutexIn();
	}

	if ((n = select(MAX_NUM_WINDOWS, &readSet, 0, 0, &timeOut)) > 0) {
	    if (FD_ISSET(FROM_PC, &readSet)) {
		rval = MutexIn();
		n--;
#ifdef	DEBUG	/* ----------------------- */
		if (rval < 0) Warning("MutexIn() return -1\r\n");
#endif	/* ----------------------- */
	    }
	    for (i = 0; i <= MAX_WIN_ID && n; i++)
		if (windows[i].pid &&
			    FD_ISSET(windows[i].pty.master, &readSet)) {
		    rval = MutexOut(i);
		    n--;
#ifdef	DEBUG	/* ----------------------- */
		    if (rval < 0) {
			sprintf(echoBuf, "MutexOut(%d) return -1\r\n", i);
			Warning(echoBuf);
		    }
#endif	/* ----------------------- */
		}
	}

	EvalReadSet();
    }
    return 0;
}


int MutexBegin()
{
    uchar mesg[5] = {MX_AT, WX_AT, MX_START, WX_START};

    Check(isatty(STDIN), 0);
    Check(isatty(STDOUT), 0);
    if (tty_raw(STDIN) < 0 || tty_noecho(STDIN) < 0) {
	tty_noraw(STDIN);
	tty_echo(STDIN);
	Warning("tty_raw()/tty_noecho() fail\n");
	return -1;
    }
    Match(write(TO_PC, mesg, 4), 4);
    fsync(TO_PC);
    return 0;
}


int MutexEnd()
{
    uchar mesg[5] = {MX_AT, WX_AT, MX_END, WX_END};

    Match(write(TO_PC, mesg, 4), 4);
    fsync(TO_PC);

    if (tty_noraw(STDIN) < 0 || tty_echo(STDIN) < 0) {
	Warning("Cannot switch back to cook/echo mode\r\n");
	return -1;
    }
    return 0;
}


main()
{
    /*
     *  keep track of children, set done = 1 after all children have died
     */
    signal(SIGCHLD, ProblemChild);

    Check(MutexBegin(), -1);

    FD_ZERO(&PCReadSet);

    MutexLoop();
    MutexEnd();
    return 0;
}

uchar _echoBuf[ECHO_BUF_SIZE];
uchar *echoBuf = _echoBuf;

