/* CAPTURE.C by Bruce Eckel, Revolution2 Real-time Consulting.
Uses the RTD AD1000 board to capture a set of samples. */
#include "capture.h"
#include <stdio.h>

/* Uncomment the following or define it on the compiler
command line (see the makefile) with -DTEST_BOARD to 
generate test code for a newly-installed AD1000 board: */
/* #define TEST_BOARD */

/* Print a value as 1s and 0s so you can see the bit
patterns from the A/D converter: */
void print_binary(unsigned int binary) {
  int i;
  for(i = 15; i >= 0; i--)
    putchar((binary & (1 << i)) ? '1' : '0');
  /* The above statement uses C's "ternary operator",
  which is an if-else statement which returns a value.
  In words, if the bitwise AND of "binary" and 1 shifted
  left by "i" places produces a non-zero value, then the
  character '1' is produced, else the character '0' is
  produced. */
}

/* Print the AD1000 value as a voltage, so you can compare
it with a voltmeter reading: */
void print_value(unsigned int reading) {
  if(reading & 0x0800) {
    putchar('+');
    reading &= 0x07ff;  /* strip off "plus sign" bit */
  } else {
    putchar('-');
    /* Flip the bits and mask them so a negative number near
    zero becomes a small negative value: */
    reading = (~reading) & 0x07ff;
  }
  /* now scale and print the value */
  printf("%1.3f", 5.0 * ((float)reading) / ((float)0x07ff) );
}

/* Capture a block of samples: */
void capture(unsigned * buf, int bufsize, unsigned threshold, int rate) {
  unsigned int sample;
  int i, j;
  /* First, wait until the threshold is exceeded: */
  while(1) {
    GET_VALUE(sample);
    /* The high bit is added to threshold to make it a positive
       A/D value: */
    if(sample  > (threshold | 0x800))
      break;  /* out of while(1) loop */
  }
  /* Then capture a block of samples: */
  for(i = 0; i < bufsize; i++) {
    GET_VALUE(buf[i]);
    for(j = 0; j < rate; j++)
      ;  /* delay by a factor of "rate" between samples */
  }
}

#ifdef TEST_BOARD
/* The following program creates a digital voltmeter out of
a selected channel of the AD1000.  You should run this program
and test a 1.5 volt battery, or a potentiometer connected to a
DC power source, to insure that your AD1000 is installed and
working properly. */
main() {
  const av = 10000;  /* Averaging factor to remove noise */
  /* An av of 10000 gives incredibly accurate results (in
  complete agreement with my voltmeter).  Try smaller numbers
  to see the rattle in your system. */
  unsigned int sample;
  long sum;
  int i;
  MUX(0);  /* select channel */
  while(1) {
    sum = 0;
    for(i = 1; i < av; i++) {
      GET_VALUE(sample);
      sum += sample;  /* Add up a bunch of samples */
    }
    sum /= av;  /* calculate the average */
    putchar('\r');
    print_binary(sum);
    putchar(' ');
    print_value(sum);
    if(kbhit()) exit(1);
  }
}
#endif /* TEST_BOARD */
