/*	ISR.h -  support for writing interrupt handlers

        Supports ISR article in Micro Cornucopia Magazine Issue #46
*/

#ifndef uchar
#include <defs.h>
#endif

#ifndef OP_IRET
#include <cpu.h>
#endif

#ifndef AX
#include <call.h>
#endif

#define ISR_HDR_LOADED

/*	REGS  - Structure of a register file.

	A far pointer to this structure is passed to the 
	interrupt handlers by the traps. The CPU registers are
	also passed in the order shown below.

	It's therefore possible to declare a handler like this:

	   void far int_24_handler( isr, r1, r2 )
	   isrh *isr;
	   __pregs r1, r2;
	   {}

	Which is much easier to type than the alternative...
	This only works if your compiler supports passing structures
	by value (By now, there probably aren't many that don't...)
*/

typedef struct ISR_REGS {
  __regdp esbx;
  __reg c;
  __reg3r dsdxax;
  unsigned int di;
  __regdp bpsi;
  __pcreg pc;
  __regpsw psw;
} isr_regs;


/*	The code to intercept an interrupt and call a 'C function
	is copied into a structure of the following type for each
	handler that's needed. The required patches to the code are
	then made. isr_install() sets up the interrupt vectors,
	and returns. isr_restore() restores the previous handler, and
	isr_pass() calls the previous handler. The code for this trap
	is in ISRHT.C.
*/

typedef struct ISRH {
   uchar code1[17];
   uint isrhs;
   uchar code2[2];
   uint isrho;
   uchar code3[2];
   void far (*hndlr)();
   uchar code4[11];
   uchar code_jmpf;
   void far (*prev_hndlr)();
   int isr_num;
   int count;
   void far *stack;
} isrh;

#define isr_pass(i) ( (i)->code_jmpf= i->prev_hndlr ? OP_JMPF : OP_IRET )
#define isr_iret(i) ( (i)->code_jmpf=                           OP_IRET )
#define isr_retf(i) ( (i)->code_jmpf=                           OP_RETF )

void far *isr_get_vector(int inum);
void isr_set_vector(int inum,void far (*farfuncp)());
void isr_install(isrh *isr,int isr_num, void far (*handler)());
void isr_restore(isrh *isr);
