/*	CALL.h   -  header to allow low level routines to
	            look like they work directly with CPU
	            registers and do callf/ints "naturally!"
	            Works with the functions in CALL.C/ISR.*

        Supports ISR article in Micro Cornucopia Magazine Issue #46
*/


#ifndef OP_RETF
#include <cpu.h>
#endif


/*	These typedefs describe the types of structures that are found
	in the final register structure at the end.

	These aren't intended to be used by humans...
*/

typedef union __REG {
   struct { 
      unsigned char l, h; 
   } b;
   unsigned int  x;
} __reg;

typedef union __PCREG {
   void far (*x)();
   struct { 
      unsigned int l, h; 
   } w;
} __pcreg;

typedef union __REGDP {
   struct { 
      __reg r; 
      unsigned int s; 
   } i;
   unsigned char far *p;
   unsigned long l;
} __regdp;

typedef union __REGPSW {
   __flags bit;
   unsigned int w;
} __regpsw;

typedef union __REG3R {
   struct {
      __reg ax, dx;
      unsigned int ds;
   } r1;
   struct {
      __reg ax;
      __regdp dsdx;
   } r2;
   struct {
      __regdp dxax;
      unsigned int ds;
   } r3;
} __reg3r;


/*	This is the main register file structure. The interrupt handlers
	all push and pop registers in this order, and the functions
	in CALL.C work with this "register file" organization. Note that 
	the stack pointer isn't actually used. That object was included 
	in the structure for "completeness" and/or "future expansion,"
	etc. 
*/

typedef struct __REGS {
  __regdp esbx;
  __reg c;
  __reg3r dsdxax;
  unsigned int di;
  __regdp bpsi;
  __pcreg pc;
  __regpsw psw;
  __regdp stk;
} __regs;

/*	These macros are provided to make it easier to access
	objects in the above structure. Use either p->AX or s.AX
	as appropriate to keep the code from looking too silly...
*/

#define AX       dsdxax.r1.ax.x
#define AH       dsdxax.r1.ax.b.h
#define AL       dsdxax.r1.ax.b.l

#define BX       esbx.i.r.x
#define BH       esbx.i.r.b.h
#define BL       esbx.i.r.b.l

#define CX       c.x
#define CH       c.b.h
#define CL       c.b.l

#define DX       dsdxax.r1.dx.x
#define DH       dsdxax.r1.dx.b.h
#define DL       dsdxax.r1.dx.b.l

#define DI       di

#define SI       bpsi.i.r.x
#define BP       bpsi.i.s

#define DS       dsdxax.r1.ds
#define ES       esbx.i.s

#define BPSI     bpsi.p
#define DXAX     dsdxax.r3.dxax.l
#define DSDX     dsdxax.r2.dsdx.p
#define ESBX     esbx.p

#define PSW      psw.w

#define PC       pc.x
#define CS       pc.w.h
#define IP       pc.w.l

#define STKPTR   stk.p
#define SS       stk.i.s
#define SP       stk.i.r.x

#define OF	 psw.bit.of	 
#define DIR	 psw.bit.dir	 
#define INTE     psw.bit.inte 
#define TRAP     psw.bit.trap 
#define SIGN     psw.bit.sign 
#define ZERO     psw.bit.zero 
#define AUXC     psw.bit.auxc 
#define PE	 psw.bit.pe	 
#define CARRY    psw.bit.carry


/*	The macros here all refer to something called _REGS,
	which must be defined in your module. It can either 
	be a structure, or another #defined macro which 
	refers to a structure, etc.

	The idea here is that modules that contain many DOS or 
	BIOS calls can put a __regs _REGS; declaration at the 
	start of every function. As such, _AX allows access to 
	the AX object in the structure, etc.

	This makes the source code look MUCH better!
*/


#define _REGSP   ((__regs far *)&(_REGS))

#define _AX      (_REGS.AX)
#define _AH      (_REGS.AH)
#define _AL      (_REGS.AL)

#define _BX      (_REGS.BX)
#define _BH      (_REGS.BH)
#define _BL      (_REGS.BL)

#define _CX      (_REGS.CX)
#define _CH      (_REGS.CH)
#define _CL      (_REGS.CL)

#define _DX      (_REGS.DX)
#define _DH      (_REGS.DH)
#define _DL      (_REGS.DL)

#define _SI      (_REGS.SI)
#define _DI      (_REGS.DI)
#define _BP      (_REGS.BP)

#define _DS      (_REGS.DS)
#define _ES      (_REGS.ES)

#define _BPSI    (_REGS.BPSI)
#define	_DXAX    (_REGS.DXAX)
#define _DSDX    (_REGS.DSDX)
#define _ESBX    (_REGS.ESBX)

#define _PSW     (_REGS.PSW)

#define _PC      (_REGS.PC)
#define _IP      (_REGS.IP)
#define _CS      (_REGS.CS)

#define _STKPTR  (_REGS.STKPTR)
#define _SS      (_REGS.SS)
#define _SP      (_REGS.SP)

#define _OF      (_REGS.OF)
#define _DIR     (_REGS.DIR)
#define _INTE    (_REGS.INTE)
#define _TRAP    (_REGS.TRAP)
#define _SIGN    (_REGS.SIGN)
#define _ZERO    (_REGS.ZERO)
#define _AUXC    (_REGS.AUXC)
#define _PE      (_REGS.PE)
#define _CARRY   (_REGS.CARRY)


/*	The following functions are in the module CALL.C.
	Call_8086() calls the far function at _PC, while
	intn_8086() expects an interrupt number to be in
	_IP. Seg86 simply fills the __r structures' segment
	registers with the values found in the CPU registers
	at the time of the call.
*/

unsigned int far seg86( __regs far *ri );
unsigned int far call_8086( __regs far *ri, __regs far *ro );
unsigned int far intn_8086( __regs far *ri, __regs far *ro );
void __regs_dump(__regs far *rp);


/*	These macros do what they seem to do...

	NOTE: the cast to (void far (*)()) for _PC in the
	call86() macro doesn't work unless x appears before
	l,h in the typedef for PCREG (Manx 'C 86, V. 4.10a.)
*/

#define call86(pc)  ( _PC = (pc), call_8086( (_REGSP), (_REGSP) ) )
#define int86(i)    ( _IP = ( i), intn_8086( (_REGSP), (_REGSP) ) )
#define pcdos(f)    ( _AH = ( f), int86( 0x21 )                   )


#asm
	;If a 'C function is declared as being far, and having two
	;far pointers (_RI and _RO) to __regs structures as its 
	;first two parameters, the instruction "les bx, _RI" would 
	;set es:bx to point to the _RI __regs structure, and
	;"les bx, _RO" would set es:bx to point to the _RO __regs
	;structure.
	;
_RI	equ   dword ptr  6[bp]
_RO	equ   dword ptr 10[bp]

	;The following are offsets from ES:BX to the contents
	;of the __regs structure.
	;
_BX	equ   es:word ptr 00[bx]
_BL	equ   es:byte ptr 00[bx]
_BH	equ   es:byte ptr 01[bx]

_ES	equ   es:word ptr 02[bx]

_CX	equ   es:word ptr 04[bx]
_CL	equ   es:byte ptr 04[bx]
_CH	equ   es:byte ptr 05[bx]

_AX	equ   es:word ptr 06[bx]
_AL	equ   es:byte ptr 06[bx]
_AH	equ   es:byte ptr 07[bx]

_DX	equ   es:word ptr 08[bx]
_DL	equ   es:byte ptr 08[bx]
_DH	equ   es:byte ptr 09[bx]

_DS	equ   es:word ptr 10[bx]

_DI	equ   es:word ptr 12[bx]
_SI	equ   es:word ptr 14[bx]
_BP	equ   es:word ptr 16[bx]

_IP	equ   es:word ptr 18[bx]
_CS	equ   es:word ptr 20[bx]

_PSW	equ   es:word ptr 22[bx]

_SP	equ   es:word ptr 24[bx]
_SS	equ   es:word ptr 26[bx]
#endasm


