/******************************************************************************
* EMS - expand memory support routines.					      *
*									      *
*					Written by Gershon Elber,  Dec. 1990  *
*******************************************************************************
* History:								      *
*  9 Dec 90 - Version 1.0 by Gershon Elber.				      *
******************************************************************************/

#include <stdio.h>
#include <string.h>
#include <dos.h>
#include "intr_loc.h"
#include "intr_ems.h"

/* #define DEBUG_EMS */

#define EMS_DEVICE_NAME "EMMXXXX0"

#define EMS_INT_NUM	0x67

#define EMS_SEGMENT	0x41
#define EMS_ALLOC	0x43
#define EMS_MAP		0x44
#define EMS_FREE	0x45

static unsigned int Segment = 0;

/******************************************************************************
* Returns TRUE iff EMS has been detected.				      *
******************************************************************************/
IntrBType EMSDetected(void)
{
    char *p = (char *) getvect(EMS_INT_NUM);

    if (p == NULL) return FALSE;

    p = MK_FP(FP_SEG(p), 10);		      /* Get pointer to device name. */

    if (strncmp(p, EMS_DEVICE_NAME, 8) != 0) return FALSE;

    Segment = EMSSegment();

    return TRUE;
}

/******************************************************************************
* Returns the segment of the base EMS address in the system, 0 if error.      *
******************************************************************************/
unsigned int EMSSegment(void)
{
    union REGS regs;

    regs.h.ah = EMS_SEGMENT;

    int86(EMS_INT_NUM, &regs, &regs);

    if (regs.h.ah != 0) return 0;

    return regs.x.bx;
}

/******************************************************************************
* EMS alloc routine. Returns EMS handle if succesfull, 0 if failed.	      *
******************************************************************************/
int EMSAlloc(int NumPages)
{
    union REGS regs;

    regs.x.bx = NumPages;
    regs.h.ah = EMS_ALLOC;

    int86(EMS_INT_NUM, &regs, &regs);

    if (regs.h.ah != 0) return 0;

    return regs.x.dx;
}

/******************************************************************************
* EMS mapping routine. The specified page number in the given EMS handle is   *
* mapped to the specified physical page.				      *
* Returns a pointer to physical address.				      *
******************************************************************************/
void *EMSMap(int EMSHandle, int LogicPageNum, int PhysicPageNum)
{
    union REGS regs;

    regs.x.bx = LogicPageNum;
    regs.x.dx = EMSHandle;
    regs.h.al = PhysicPageNum;
    regs.h.ah = EMS_MAP;

    int86(EMS_INT_NUM, &regs, &regs);

    if (regs.h.ah != 0) return NULL;

    return MK_FP(Segment + 0x0400 * PhysicPageNum, 0x0000);
}

/******************************************************************************
* EMS free routine. Frees the EMS handle.				      *
******************************************************************************/
void EMSFree(int EMSHandle)
{
    union REGS regs;

    regs.x.dx = EMSHandle;
    regs.h.ah = EMS_FREE;

    int86(EMS_INT_NUM, &regs, &regs);
}

#ifdef DEBUG_EMS

/******************************************************************************
* Simple tests for the above routines.					      *
******************************************************************************/
void main(void)
{
    int Handle1, Handle2;
    void *p1, *p2;

    if (EMSDetected())
	fprintf(stderr, "EMS driver detected.\n");
    else {
	fprintf(stderr, "EMS driver not found.\n");
        exit(1);
    }

    if ((Handle1 = EMSAlloc(1)) > 0)
	fprintf(stderr, "EMS alloc one page - got handle \#%d.\n", Handle1);
    else {
	fprintf(stderr, "EMS alloc failed.\n");
        exit(2);
    }

    if ((p1 = EMSMap(Handle1, 0, 0)) != NULL)
	fprintf(stderr, "EMS map one page - address %p.\n", p1);
    else {
	fprintf(stderr, "EMS map failed.\n");
        exit(2);
    }

    if ((Handle2 = EMSAlloc(5)) > 0)
	fprintf(stderr, "EMS alloc five page - got handle \#%d.\n", Handle2);
    else {
	fprintf(stderr, "EMS alloc failed.\n");
        exit(2);
    }

    if ((p2 = EMSMap(Handle2, 0, 1)) != NULL)
	fprintf(stderr, "EMS map one page - address %p.\n", p2);
    else {
	fprintf(stderr, "EMS map failed.\n");
        exit(2);
    }

    EMSFree(Handle1);
    EMSFree(Handle2);
}

#endif /* DEBUG_EMS */
