/*
 * trans.c - main control of the translation process.
 */

#include "../h/gsupport.h"
#include "globals.h"
#include "trans.h"
#include "tsym.h"
#include "tree.h"
#include "token.h"
#include "tcode.h"
#include "tproto.h"

/*
 * Prototypes.
 */

hidden FILE	*preprocess	Params((char *filename));
hidden novalue	trans1		Params((char *filename));

int tfatals = 0;		/* total number of fatal errors */
int twarns = 0;			/* total number of warnings */
int in_line;			/* current input line number */
int incol;			/* current input column number */
int peekc;			/* one-character look ahead */

FILE *srcfile;			/* current input file */

/*
 * translate a number of files, returning an error count
 */
int trans()
   {
   register struct pentry *proc;
   struct srcfile *sf;

   for (sf = srclst; sf != NULL; sf = sf->next)
      trans1(sf->name);	/* translate each file in turn */

   /*
    * Resolve undeclared references.
    */
   for (proc = proc_lst; proc != NULL; proc = proc->next)
      resolve(proc);

#ifdef DeBug
   symdump();
#endif					/* DeBug */

   if (tfatals == 0) {
      chkstrinv();  /* see what needs to be available for string invocation */
      chkinv();     /* perform "naive" optimizations */
      }

   if (tfatals == 0)
      typeinfer();        /* perform type inference */

   if (just_type_trace)
      return tfatals;     /* stop without generating code */

   if (tfatals == 0) {
      var_dcls();         /* output declarations for globals and statics */
      const_blks();       /* output blocks for cset and real literals */
      for (proc = proc_lst; proc != NULL; proc = proc->next)
         proccode(proc);  /* output code for a procedure */
      recconstr(rec_lst); /* output code for record constructors */
      }

   /*
    * Report information about errors and warnings and be correct about it.
    */
   if (tfatals == 1)
      fprintf(stderr, "1 error; ");
   else if (tfatals > 1)
      fprintf(stderr, "%d errors; ", tfatals);
   else if (verbose > 0)
      fprintf(stderr, "No errors; ");

   if (twarns == 1)
      fprintf(stderr, "1 warning\n");
   else if (twarns > 1)
      fprintf(stderr, "%d warnings\n", twarns);
   else if (verbose > 0)
      fprintf(stderr, "no warnings\n");
   else if (tfatals > 0)
      fprintf(stderr, "\n");

#ifdef TranStats
   tokdump();
#endif					/* TranStats */

   return tfatals;
   }

/*
 * translate one file.
 */
static novalue trans1(filename)
char *filename;
   {
   in_line = 1;			/* start with line 1, column 0 */
   incol = 0;
   peekc = 0;			/* clear character lookahead */

   if (m4pre)
      srcfile = preprocess(filename);
   else if (strcmp(filename,"-") == 0) {
      srcfile = stdin;
      filename = "stdin";
      }
   else
      srcfile = fopen(filename,"r");
   if (srcfile == NULL) {
      tfatal(filename, "cannot open source file");
      return;
      }
   if (verbose > 0)
      fprintf(stderr, "%s:\n",filename);

   tok_loc.n_file = filename;
   in_line = 1;

   yyparse();				/* Parse the input */

   /*
    * Close the the input file.
    */

   if (!m4pre) {
      if (srcfile != stdin) fclose(srcfile);
      }
   /* "else" is below in conditional */

/*
 * The following code is operating-system dependent [@trans.01].  This code
 *  closes the pipe for the preprocessor if processing was done.
 */

#if PORT
/* nothing to do */
Deliberate Syntax Error
#endif					/* PORT */

#if AMIGA || ATARI_ST || MACINTOSH || MSDOS || MVS || VM || VMS
/* nothing to do */
#endif					/* AMIGA || ATARI_ST || ... */

#if UNIX
   else if (pclose(srcfile) != 0)
      quit("m4 terminated abnormally");
#endif					/* UNIX */

/*
 * End of operating-system specific code.
 */
   }

/*
 * writecheck - check the return code from a stdio output operation
 */
novalue writecheck(rc)
   int rc;

   {
   if (rc < 0)
      quit("unable to write to icode file");
   }

/*
 * preprocess - open a pipe to the m4 preprocessor.
 */
static FILE *preprocess(filename)
char *filename;
{
/*
 * The following code is operating-system dependent [@trans.02]. This code opens
 *  a pipe to m4(1) on systems where the "-m" flag is accepted.
 */

#if PORT
   /* can't get here */
Deliberate Syntax Error
#endif					/* PORT */

#if AMIGA || ATARI_ST || MACINTOSH || MSDOS || MVS || VM || VMS
   /* can't get here */
#endif					/* AMIGA || ATARI_ST || ... */

#if UNIX
      {
      FILE *f, *popen();
      char *s = alloc((unsigned int)(4+strlen(filename)));
      sprintf(s,"m4 %s",filename);
      f = popen(s,"r");
      free(s);
      return f;
      }
#endif					/* UNIX */

/*
 * End of operating-system specific code.
 */
}
