/***********************************************************************/
/* COMM1.C - Commands A-E                                              */
/* This file contains all commands that can be assigned to function    */
/* keys or typed on the command line.                                  */
/***********************************************************************/
/*
 * THE - The Hessling Editor. A text editor similar to VM/CMS xedit.
 * Copyright (C) 1991,1992 Mark Hessling
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to:
 *
 *    The Free Software Foundation, Inc.
 *    675 Mass Ave,
 *    Cambridge, MA 02139 USA.
 *
 *
 * If you make modifications to this software that you feel increases
 * it usefulness for the rest of the community, please email the
 * changes, enhancements, bug fixes as well as any and all ideas to me.
 * This software is going to be maintained and enhanced as deemed
 * necessary by the community.
 *
 * Mark Hessling                     email: M.Hessling@itc.gu.edu.au
 * 36 David Road                     Phone: +61 7 849 7731
 * Holland Park                      Fax:   +61 7 875 7877
 * QLD 4121
 * Australia
 */
#include <stdio.h>
#include <stdlib.h>

#include "the.h"

/*#define DEBUG 1*/

/*-------------------------- external data ----------------------------*/
extern LINE *next_line,*curr_line;
extern VIEW_DETAILS *vd_current,*vd_first,*vd_mark;
extern char current_screen;
extern SCREEN_DETAILS screen[MAX_SCREENS];        /* screen structures */
extern WINDOW *foot,*error_window,*divider;
extern char error_on_screen;
extern unsigned char *rec;
extern unsigned short rec_len;
extern unsigned char *cmd_rec;
extern unsigned short cmd_rec_len;
extern unsigned char mode_insert;        /* defines insert mode toggle */
extern unsigned char in_profile;    /* indicates if processing profile */

extern unsigned char temp_cmd[150];
extern unsigned char dir_filename[MAX_FILE_NAME+1];
extern unsigned char dir_pathname[MAX_FILE_NAME+1];
extern unsigned char sp_path[MAX_FILE_NAME+1] ;
extern unsigned char sp_fname[MAX_FILE_NAME+1] ;
extern unsigned char dir_path[MAX_FILE_NAME+1] ;    /* for dir and ls commands */
/*---------------------- function definitions -------------------------*/
#ifdef PROTO
void split_command(unsigned char *,unsigned char *,unsigned char *);
int param_split(unsigned char *,unsigned char *[],int );
long valid_target(unsigned char *);
int get_row_for_focus_line(int,long,long);
void redraw_window(WINDOW *);
void add_command(unsigned char *);
#else
void split_command();
int param_split();
long valid_target();
int get_row_for_focus_line();
void redraw_window();
void add_command();
#endif
/*man-start*********************************************************************
COMMAND
     add - add blank line

SYNTAX
     ADD [n]

DESCRIPTION
     The ADD command inserts the specified number of blank lines after
     the current_line (if issued from the command line) or after the
     focus_line (if issued in the main of prefix windows).
     The cursor is positioned in the column corresponding to the first
     column not containing a space in the line above.

COMPATIBILITY
     Compatible.

DEFAULT
     With no parameters, 1 line is added.

SEE ALSO
     Sos_addline

STATUS
     Complete
**man-end**********************************************************************/
#ifdef PROTO
int Add(unsigned char *params)
#else
int Add(params)
unsigned char *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
#define ADD_PARAMS  1
 unsigned char *word[ADD_PARAMS+1];
 char parm[ADD_PARAMS];
 unsigned short num_params;
 long num_lines;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm1.c:   Add");
#endif
/*---------------------------------------------------------------------*/
/* Validate the parameters that have been supplied. The one and only   */
/* parameter should be a positive integer greater than zero.           */
/* If no parameter is supplied, 1 is assumed.                          */
/*---------------------------------------------------------------------*/
 num_params = param_split(params,word,ADD_PARAMS);
 if (num_params == 0)
    {
     num_params = 1;
     word[0] = (unsigned char *)"1";
    }
 if (num_params != 1)
    {
     display_error(1,word[1]);
#ifdef TRACE
     trace_return();
#endif
     return(OK);
    }
 if (!valid_positive_integer(word[0]))
    {
     display_error(4,word[0]);
#ifdef TRACE
     trace_return();
#endif
     return(OK);
    }
 num_lines = atol(word[0]);
 insert_new_line((unsigned char *)"",0,num_lines,FALSE);
#ifdef TRACE
 trace_return();
#endif
 return(OK);
}
/*man-start*********************************************************************
COMMAND
     all - select and display restricted set of lines

SYNTAX
     ALL [string target]

DESCRIPTION

COMPATIBILITY
     Compatible.

DEFAULT

SEE ALSO

STATUS
     Not Started
**man-end**********************************************************************/
#ifdef PROTO
int All(unsigned char *params)
#else
int All(params)
unsigned char *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm1.c:   All");
#endif
 display_error(0,(unsigned char *)"This function has not yet been implemented");
#ifdef TRACE
 trace_return();
#endif
 return(OK);
}
/*man-start*********************************************************************
COMMAND
     autosave - set autosave period

SYNTAX
     AUtosave n|OFF

DESCRIPTION
     The AUTOSAVE command sets the interval between automatic saves
     of the file, or turns it off altogether.

COMPATIBILITY
     Compatible.

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
int Autosave(unsigned char *params)
#else
int Autosave(params)
unsigned char *params;
#endif
/***********************************************************************/
{
/*--------------------------- extern data -----------------------------*/
extern unsigned char AUSx;
/*--------------------------- local data ------------------------------*/
#define AUS_PARAMS  1
 unsigned char *word[AUS_PARAMS+1];
 register int i;
 unsigned short num_params;
 long num_pages,num_lines;
 unsigned short x,y;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm1.c:   Autosave");
#endif
 num_params = param_split(params,word,AUS_PARAMS);
 if (num_params == 0)
   {
    display_error(3,(unsigned char *)"");
#ifdef TRACE
    trace_return();
#endif
    return(OK);
   }
 if (num_params != 1)
   {
    display_error(2,(unsigned char *)"");
#ifdef TRACE
    trace_return();
#endif
    return(OK);
   }
 if (strcmp(word[0],"off") == 0
 ||  strcmp(word[0],"OFF") == 0)
   {
    CURRENT_FILE->autosave = 0;
#ifdef TRACE
    trace_return();
#endif
    return(OK);
   }
 if (!valid_positive_integer(word[0]))
   {
    display_error(4,(unsigned char *)word[0]);
#ifdef TRACE
    trace_return();
#endif
    return(OK);
   }
 if (in_profile)
    AUSx = (unsigned char)atoi(word[0]);
 else
    CURRENT_FILE->autosave = (unsigned char)atoi(word[0]);
 return(OK);
}
/*man-start*********************************************************************
COMMAND
     backward - scroll backwards one screen

SYNTAX
     BACkward [n]

DESCRIPTION
     The BACKWARD command scrolls the file contents backwards one full
     screen.

COMPATIBILITY
     Compatible.

DEFAULT
     With no parameters, 1 screen is scrolled.

SEE ALSO
     forward,top

STATUS
     Complete
**man-end**********************************************************************/
#ifdef PROTO
int Backward(unsigned char *params)
#else
int Backward(params)
unsigned char *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
#define BAC_PARAMS  1
 unsigned char *word[BAC_PARAMS+1];
 char parm[BAC_PARAMS];
 register int i;
 unsigned short num_params;
 long num_pages,num_lines;
 unsigned short x,y;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm1.c:   Backward");
#endif
 num_params = param_split(params,word,BAC_PARAMS);
 if (num_params == 0)
    {
     num_params = 1;
     word[0] = (unsigned char *)"1";
    }
 if (num_params != 1)
    {
     display_error(1,(unsigned char *)word[1]);
#ifdef TRACE
    trace_return();
#endif
     return(OK);
    }
 if (strcmp(word[0],"*") == 0)
   {
    CURRENT_VIEW->current_line = 0L;
    post_process_line(CURRENT_VIEW->focus_line);
    CURRENT_VIEW->focus_line = 0L;
    pre_process_line(CURRENT_VIEW->focus_line);
    if (!in_profile)
      {
       getyx(CURRENT_WINDOW_MAIN,y,x);
       show_page();
       y = get_row_for_focus_line(CURRENT_VIEW->current_row,
                               CURRENT_VIEW->focus_line,
                               CURRENT_VIEW->current_line);
       wmove(CURRENT_WINDOW_MAIN,y,x);
      }
#ifdef TRACE
    trace_return();
#endif
    return(OK);
   }
 if ((num_pages = atol(word[0])) == 0L)
   {
    display_error(4,(unsigned char *)word[0]);
#ifdef TRACE
    trace_return();
#endif
    return(OK);
   }
 num_lines = CURRENT_VIEW->current_line - ((CURRENT_SCREEN.rows-1)*num_pages);
 CURRENT_VIEW->current_line = max(num_lines,0);

 post_process_line(CURRENT_VIEW->focus_line);
 CURRENT_VIEW->focus_line = calculate_focus_line(CURRENT_VIEW->current_row,
                                                CURRENT_VIEW->focus_line,
                                                CURRENT_VIEW->current_line);
 pre_process_line(CURRENT_VIEW->focus_line);
 if (in_profile)
   {
#ifdef TRACE
    trace_return();
#endif
    return(OK);
   }
 getyx(CURRENT_WINDOW_MAIN,y,x);
 show_page();
 if (CURRENT_VIEW->current_window != WINDOW_COMMAND)
   {
    y = get_row_for_focus_line(CURRENT_VIEW->current_row,
                               CURRENT_VIEW->focus_line,
                               CURRENT_VIEW->current_line);
    wmove(CURRENT_WINDOW,y,x);
   }
#ifdef TRACE
 trace_return();
#endif
 return(OK);
}
/*man-start*********************************************************************
COMMAND
     bottom - move to the bottom of the file

SYNTAX
     BOTtom

DESCRIPTION
     The BOTTOM command moves to the very end of the current file.
     The "Bottom-of-file" line is set to the current_line.

     "BOTTOM" is equivalent to "FORWARD *".

COMPATIBILITY
     Compatible.

SEE ALSO
     forward,top

STATUS
     Complete
**man-end**********************************************************************/
#ifdef PROTO
int Bottom(unsigned char *params)
#else
int Bottom(params)
unsigned char *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 short rc;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm1.c:   Bottom");
#endif
 rc = Forward("*");
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     cancel - quickly exit from THE

SYNTAX
     CANcel

DESCRIPTION
     The CANCEL command exits from THE quickly by QQUITting out of all
     files currently in the ring that do not have any outstanding
     alterations.

COMPATIBILITY
     Compatible.

SEE ALSO
     ccancel

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
int Cancel(unsigned char *params)
#else
int Cancel(params)
unsigned char *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 VIEW_DETAILS *save_current_view=(VIEW_DETAILS *)NULL;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm1.c:   Cancel");
#endif
 CURRENT_VIEW = vd_first;
 while (CURRENT_VIEW != (VIEW_DETAILS *)NULL)
   {
    if (CURRENT_FILE->save_alt == 0)
       Qquit((unsigned char *)"");
    else
      {
       save_current_view = CURRENT_VIEW;
       CURRENT_VIEW = CURRENT_VIEW->next;
      }
   }
 if (save_current_view != (VIEW_DETAILS *)NULL)
   {
    CURRENT_VIEW = save_current_view;
    pre_process_line(CURRENT_VIEW->focus_line);
    show_page();
    if (CURRENT_VIEW->prefix_on)
       touchwin(CURRENT_WINDOW_PREFIX);
    touchwin(CURRENT_WINDOW_COMMAND);
    touchwin(CURRENT_WINDOW_MAIN);
    touchwin(CURRENT_WINDOW);
   }
#ifdef TRACE
 trace_return();
#endif
 return(QUIT);
}
/*man-start*********************************************************************
COMMAND
     case - set case sensitivity parameters

SYNTAX
     CASE Mixed|Lower|Upper [Respect|Ignore] [Respect|Ignore]

DESCRIPTION
     The CASE command sets the editor's handling of the case of text.

     The first option (which is mandatory) controls how text is entered
     by the user. When LOWER or UPPER are in effect, the shift or caps
     lock keys have no effect on the text being entered. When MIXED is
     in effect, text is entered in the case set by the use of the shift
     and caps lock keys.

     The second option determines how the editor determines if a string
     target matches text in the file when the target is used in a LOCATE
     command.  With IGNORE in effect, a match is
     found irrespective of the case of the target or the found text.
     The following strings are treated as equivalent: the THE The ThE...
     With RESPECT in effect, the target and text must be the same case.
     Therefore a target of 'The' only matches text containing 'The', not
     'THE' or 'ThE' etc.

     The third option determines how the editor determines if a string
     target matches text in the file when the target is used in a CHANGE
     command.  With IGNORE in effect, a match is
     found irrespective of the case of the target or the found text.
     The following strings are treated as equivalent: the THE The ThE...
     With RESPECT in effect, the target and text must be the same case.
     Therefore a target of 'The' only matches text containing 'The', not
     'THE' or 'ThE' etc.

COMPATIBILITY
     Compatible.

DEFAULT
     MIXED IGNORE RESPECT

STATUS
     Complete
**man-end**********************************************************************/
#ifdef PROTO
int Case(unsigned char *params)
#else
int Case(params)
unsigned char *params;
#endif
/***********************************************************************/
{
/*--------------------------- extern data -----------------------------*/
extern unsigned char CASE_Ex;
extern unsigned char CASE_Lx;
extern unsigned char CASE_Cx;
/*--------------------------- local data ------------------------------*/
#define CAS_PARAMS  3
 char parm[CAS_PARAMS];
 unsigned char *word[CAS_PARAMS+1];
 register int i;
 unsigned short num_params;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm1.c:   Case");
#endif
 num_params = param_split(params,word,CAS_PARAMS);
 for (i=0;i<CAS_PARAMS;i++)
     parm[i] = UNDEFINED_OPERAND;
 if (equal("mixed",word[0],1))
    parm[0] = CASE_MIXED;
 if (equal("upper",word[0],1))
    parm[0] = CASE_UPPER;
 if (equal("lower",word[0],1))
    parm[0] = CASE_LOWER;
 if (parm[0] == UNDEFINED_OPERAND)
    {
     display_error(1,(unsigned char *)word[0]);
#ifdef TRACE
     trace_return();
#endif
     return(OK);
    }
 if (strcmp(word[1],"") == 0)
    parm[1] = CASE_IGNORE;
 else
   {
    if (equal("respect",word[1],1))
       parm[1] = CASE_RESPECT;
    if (equal("ignore",word[1],1))
       parm[1] = CASE_IGNORE;
   }
 if (parm[1] == UNDEFINED_OPERAND)
    {
     display_error(1,(unsigned char *)word[1]);
#ifdef TRACE
     trace_return();
#endif
     return(OK);
    }
 if (strcmp(word[2],"") == 0)
    parm[2] = CASE_RESPECT;
 else
   {
    if (equal("respect",word[2],1))
       parm[2] = CASE_RESPECT;
    if (equal("ignore",word[2],1))
       parm[2] = CASE_IGNORE;
   }
 if (parm[2] == UNDEFINED_OPERAND)
    {
     display_error(1,(unsigned char *)word[2]);
#ifdef TRACE
     trace_return();
#endif
     return(OK);
    }
/*---------------------------------------------------------------------*/
/* When the CASE command is found in the profile file, the default     */
/* CASE settings for ALL files to be editted is set.                   */
/*---------------------------------------------------------------------*/
 if (in_profile)
   {
    CASE_Ex = parm[0];
    CASE_Lx = parm[1];
    CASE_Cx = parm[2];
   }
 else
   {
    CURRENT_VIEW->case_enter  = parm[0];
    CURRENT_VIEW->case_locate = parm[1];
    CURRENT_VIEW->case_change = parm[2];
   }
#ifdef TRACE
 trace_return();
#endif
 return(OK);
}
/*man-start*********************************************************************
COMMAND
     ccancel - quickly exit from THE

SYNTAX
     CCancel

DESCRIPTION
     The CCANCEL command exits from THE quickly by QQUITting out of all
     files currently in the ring. Any changes made to any of the files
     will be lost.

COMPATIBILITY
     Compatible.

SEE ALSO
     cancel

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
int Ccancel(unsigned char *params)
#else
int Ccancel(params)
unsigned char *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm1.c:   Ccancel");
#endif
 CURRENT_VIEW = vd_first;
 while (CURRENT_VIEW != (VIEW_DETAILS *)NULL)
   {
    Qquit((unsigned char *)"");
   }
#ifdef TRACE
 trace_return();
#endif
 return(QUIT);
}
/*man-start*********************************************************************
COMMAND
     change - change file text

SYNTAX
     Change /string1/string2/ [target] [n] [m]

DESCRIPTION
     The CHANGE command changes one string of text to another.

     The first parameter to the change command is the old and new
     string values, seperated by delimiters.
     The allowable delimiters are '/' '\' and '@'.

     The second parameter is the target; how many lines are to be
     searched for occurrences of the first string to be changed.

     The third parameter determines how many occurrences of 'string1'
     are to be changed on each line.

     The fourth parameter determines at which occurrences of 'string1'
     on the line are changes to commence.

COMPATIBILITY
     Compatible.

DEFAULT
     1 1 1

SEE ALSO
     schange

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
int Change(unsigned char *params)
#else
int Change(params)
unsigned char *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 short rc;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm1.c:   Change");
#endif

 rc = execute_change_command(params,FALSE);

#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     cmatch - find matching bracket character

SYNTAX
     ** effective only if bound to a key **

DESCRIPTION
     The CMATCH command searches for the matching bracket character to
     the character under the cursor.

     It handles nested sets of matching pairs.
     The matching character pairs are '[]{}<>()'.

COMPATIBILITY
     Compatible.

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
int Cmatch(unsigned char *params)
#else
int Cmatch(params)
unsigned char *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 static unsigned char *match = (unsigned char *)"[]{}<>()";
 unsigned short x,y;
 unsigned char ch,match_ch;
 register int i;
 char direction_backward;
 short matches=1,match_col=(-1),start_col;
 long offset=0;
 LINE *curr;
 WINDOW *w;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm1.c:   Cmatch");
#endif
/*---------------------------------------------------------------------*/
/* This command only allowed to be issued from with the MAIN window.   */
/*---------------------------------------------------------------------*/
 if (CURRENT_VIEW->current_window != WINDOW_MAIN
 || in_profile)
   {
    display_error(66,(unsigned char *)"");
#ifdef TRACE
    trace_return();
#endif
    return(OK);
   }
 if (CURRENT_VIEW->focus_line == 0
 ||  CURRENT_VIEW->focus_line == CURRENT_FILE->number_lines+1)
   {
    display_error(66,(unsigned char *)"");
#ifdef TRACE
    trace_return();
#endif
    return(OK);
   }
 getyx(CURRENT_WINDOW,y,x);
/*---------------------------------------------------------------------*/
/* Check if the character under the cursor is a valid match character. */
/*---------------------------------------------------------------------*/
 w = CURRENT_WINDOW;
 ch = (unsigned char)winch(w) & A_CHARTEXT;
 match_ch = 0;
 for (i=0;i<strlen(match);i++)
    if (ch == *(match+i))
      {
       direction_backward = (i % 2);
       match_ch = (direction_backward) ? *(match+i-1) : *(match+i+1);
       break;
      }
 if (match_ch == 0)
   {
    display_error(67,(unsigned char *)"");
#ifdef TRACE
    trace_return();
#endif
    return(OK);
   }
/*---------------------------------------------------------------------*/
/* Calculate the actual position of the character in the LINE.         */
/*---------------------------------------------------------------------*/
 start_col = CURRENT_VIEW->verify_col + x - 1;
 start_col += (direction_backward) ? (-1) : 1;
/*---------------------------------------------------------------------*/
/* Find the focus line linked list entry.                              */
/*---------------------------------------------------------------------*/
 post_process_line(CURRENT_VIEW->focus_line);
 curr = ll_find(CURRENT_FILE->first_line,CURRENT_VIEW->focus_line);
 while (curr->next != NULL && curr->prev != NULL)
   {
    if (direction_backward)
      {
       for (i=start_col;i>(-1);i--)
          {
           if (*(curr->line+i) == ch)
             matches++;
           else
              if (*(curr->line+i) == match_ch)
                matches--;
           if (matches == 0)       /* found matching one */
             {
              match_col = i;
              break;
             }
          }
       if (match_col != (-1))
         break;
       curr = curr->prev;
       offset--;
       start_col = curr->length;
      }
    else
      {
       for (i=start_col;i<curr->length;i++)
          {
           if (*(curr->line+i) == ch)
             matches++;
           else
              if (*(curr->line+i) == match_ch)
                matches--;
           if (matches == 0)       /* found matching one */
             {
              match_col = i;
              break;
             }
          }
       if (match_col != (-1))
         break;
       curr = curr->next;
       offset++;
       start_col = 0;
      }
   }
 if (match_col == (-1))  /* no match found */
   {
    display_error(68,(unsigned char *)"");
#ifdef TRACE
    trace_return();
#endif
    return(OK);
   }
/*---------------------------------------------------------------------*/
/* If we get here, we have found the matching character, so we have to */
/*  move the cursor to the new column and/or line.                     */
/*---------------------------------------------------------------------*/
 if (offset == 0L)
   {
    if (match_col >= CURRENT_VIEW->verify_col-1
    &&  match_col <= (CURRENT_SCREEN.cols+(CURRENT_VIEW->verify_col-1))-1)
/*---------------------------------------------------------------------*/
/* If the new cursor position is in the same panel and on the same line*/
/* just move the cursor there and get out.                             */
/*---------------------------------------------------------------------*/
      {
       wmove(CURRENT_WINDOW,y,match_col-(CURRENT_VIEW->verify_col-1));
#ifdef TRACE
       trace_return();
#endif
       return(OK);
      }
   else
      {
       x = CURRENT_SCREEN.cols / 2;
       CURRENT_VIEW->verify_col = max(1,match_col-(short)x);
       show_page();
       wmove(CURRENT_WINDOW,y,(match_col-(CURRENT_VIEW->verify_col-1)));
#ifdef TRACE
       trace_return();
#endif
       return(OK);
      }
   }

 CURRENT_VIEW->focus_line += offset;
 pre_process_line(CURRENT_VIEW->focus_line);
 if (offset + y <= 0
 ||  offset + y >= CURRENT_SCREEN.rows)
   {
    CURRENT_VIEW->current_line = CURRENT_VIEW->focus_line;
    y = CURRENT_VIEW->current_row;
   }
 else
    y = get_row_for_focus_line(CURRENT_VIEW->current_row,
                               CURRENT_VIEW->focus_line,
                               CURRENT_VIEW->current_line);
 if (match_col >= CURRENT_VIEW->verify_col-1
 &&  match_col <= (CURRENT_SCREEN.cols+(CURRENT_VIEW->verify_col-1))-1)
    x = match_col-(CURRENT_VIEW->verify_col-1);
 else
   {
    x = CURRENT_SCREEN.cols / 2;
    CURRENT_VIEW->verify_col = max(1,match_col-(short)x);
    x = (match_col-(CURRENT_VIEW->verify_col-1));
   }

 show_page();
 wmove(CURRENT_WINDOW,y,x);
#ifdef TRACE
 trace_return();
#endif
 return(OK);
}
/*man-start*********************************************************************
COMMAND
     cmdarrows - sets the functionality of the up and down arrows on the
                  command line.

SYNTAX
     CMDArrows RETRIEVE|TAB

DESCRIPTION
     The CMDARROWS command determines the action that occurs when the up
     and down arrows are hit while the cursor is on the command line.
     CMDARROWS RETRIEVE (the default) will set the up and down arrows
     to retrieve the last or next command entered on the command line.
     CMDARROWS TAB will set the up and down arrows to move to the last
     or first line respectively of the main window.

COMPATIBILITY
     New function.

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
int Cmdarrows(unsigned char *params)
#else
int Cmdarrows(params)
unsigned char *params;
#endif
/***********************************************************************/
{
/*------------------------- external data -----------------------------*/
extern unsigned char CMDARROWSTABx;
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm1.c:   Cmdarrows");
#endif
 if (strcmp(params,"tab") == 0)
   {
    CMDARROWSTABx = TRUE;
#ifdef TRACE
    trace_return();
#endif
    return(OK);
   }
 if (strcmp(params,"retrieve") == 0)
   {
    CMDARROWSTABx = FALSE;
#ifdef TRACE
    trace_return();
#endif
    return(OK);
   }

 display_error(1,(unsigned char *)params);
#ifdef TRACE
 trace_return();
#endif
 return(OK);
}
/*man-start*********************************************************************
COMMAND
     colour - set colours for display

SYNTAX
     [SET] colour | color area [modifier] [foreground background]

DESCRIPTION
     The COLOUR command changes the colours or display attributes of
     various display areas in THE.

     Valid values for 'area':
       filearea - area containing file lines
       curline  - the current line
       block    - marked block
       cblock   - current line if in marked block
       cmdline  - command line
       idline   - line containing file specific info
       msgline  - error messages
       arrow    - command line prompt
       prefix   - prefix area
       pending  - pending commands in prefix
       scale    - line showing scale line (N/A)
       tofeof   - *** Top of File *** and *** Bottom of File *** lines
       ctofeof  - as above if the same as current line
       tabline  - line showing tab positions (N/A)
       shadow   - hidden line marker lines (N/A)
       statarea - line showing status of editting session
       divider  - dividing line between vertical split screens

     Valid values for 'foreground' and 'background':
       black,blue,green,cyan,red,magenta,yellow,white

     Valid values for 'modifier':
       normal,blink,bold,reverse,underline

COMPATIBILITY
     Does not implement all modifiers.

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
int Colour(unsigned char *params)
#else
int Colour(params)
unsigned char *params;
#endif

/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 struct areas
 {
  unsigned char *area;
  short area_min_len;
  short area_window;
 };
 typedef struct areas AREAS;
 static AREAS valid_areas[ATTR_MAX]=
 {
  {(unsigned char *)"filearea",1,WINDOW_MAIN},
  {(unsigned char *)"curline",2,WINDOW_MAIN},
  {(unsigned char *)"block",1,WINDOW_MAIN},
  {(unsigned char *)"cblock",2,WINDOW_MAIN},
  {(unsigned char *)"cmdline",1,WINDOW_COMMAND},
  {(unsigned char *)"idline",1,WINDOW_IDLINE},
  {(unsigned char *)"msgline",1,WINDOW_ERROR},
  {(unsigned char *)"arrow",1,WINDOW_ARROW},
  {(unsigned char *)"prefix",2,WINDOW_PREFIX},
  {(unsigned char *)"pending",1,WINDOW_PREFIX},
  {(unsigned char *)"scale",1,WINDOW_MAIN},
  {(unsigned char *)"tofeof",2,WINDOW_MAIN},
  {(unsigned char *)"ctofeof",2,WINDOW_MAIN},
  {(unsigned char *)"tabline",1,WINDOW_MAIN},
  {(unsigned char *)"shadow",2,WINDOW_MAIN},
  {(unsigned char *)"statarea",2,WINDOW_FOOTING},
  {(unsigned char *)"divider",1,WINDOW_DIVIDER}
 };
#define COL_PARAMS 2
 unsigned char *word[COL_PARAMS+1];
 char parm[COL_PARAMS];
 register int i;
 unsigned short num_params;
 int area;
 chtype fg,bg,mod;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm1.c:   Colour");
#endif
 num_params = param_split(params,word,COL_PARAMS);
 if (num_params < 2 )
    {
     display_error(3,(unsigned char *)"");
#ifdef TRACE
     trace_return();
#endif
     return(OK);
    }
/*---------------------------------------------------------------------*/
/* Check that the supplied area matches one of the values in the area  */
/* array and that the length is at least as long as the minimum.       */
/*---------------------------------------------------------------------*/
 parm[0] = FALSE;
 for (i=0;i<ATTR_MAX;i++)
    {
     if (equal(valid_areas[i].area,word[0],valid_areas[i].area_min_len))
       {
        parm[0] = TRUE;
        area = i;
        break;
       }
    }
 if (parm[0] == FALSE)
    {
     display_error(1,(unsigned char *)word[0]);
#ifdef TRACE
     trace_return();
#endif
     return(OK);
    }
/*---------------------------------------------------------------------*/
/* Determine colours and modifiers.                                    */
/*---------------------------------------------------------------------*/

 if (parse_colours(word[1],&fg,&bg,&mod) == ERROR)
   {
#ifdef TRACE
    trace_return();
#endif
    return(ERROR);
   }
/*---------------------------------------------------------------------*/
/* If we are still in the profile, save the colour setup for later...  */
/*---------------------------------------------------------------------*/

 if (in_profile)
   {
    colour[area] = (chtype)1;
    save_fg[area] = fg;
    save_bg[area] = bg;
    save_mod[area] = mod;
#ifdef TRACE
    trace_return();
#endif
    return(OK);
   }
/*---------------------------------------------------------------------*/
/* Set the colours.                                                    */
/*---------------------------------------------------------------------*/

 set_colour(area,fg,bg,mod);

/*---------------------------------------------------------------------*/
/* Update the appropriate window with the new colour combination.      */
/*---------------------------------------------------------------------*/

 switch (valid_areas[area].area_window)
   {
    case WINDOW_MAIN:
                        if (area == ATTR_FILEAREA)
                           wattrset(CURRENT_WINDOW_MAIN,colour[area]);
                        redraw_window(CURRENT_WINDOW_MAIN);
                        show_page();
                        touchwin(CURRENT_WINDOW_MAIN);
                        wnoutrefresh(CURRENT_WINDOW_MAIN);
                        break;
    case WINDOW_PREFIX:
                        if (CURRENT_VIEW->prefix_on)
                          {
                           wattrset(CURRENT_WINDOW_PREFIX,colour[area]);
                           redraw_window(CURRENT_WINDOW_PREFIX);
                           touchwin(CURRENT_WINDOW_PREFIX);
                           wnoutrefresh(CURRENT_WINDOW_PREFIX);
                          }
                        break;
    case WINDOW_COMMAND:
                        wattrset(CURRENT_WINDOW_COMMAND,colour[area]);
                        redraw_window(CURRENT_WINDOW_COMMAND);
                        touchwin(CURRENT_WINDOW_COMMAND);
                        wnoutrefresh(CURRENT_WINDOW_COMMAND);
                        break;
    case WINDOW_ARROW:
                        wattrset(CURRENT_WINDOW_ARROW,colour[area]);
                        redraw_window(CURRENT_WINDOW_ARROW);
                        touchwin(CURRENT_WINDOW_ARROW);
                        wnoutrefresh(CURRENT_WINDOW_ARROW);
                        break;
    case WINDOW_IDLINE:
                        wattrset(CURRENT_WINDOW_IDLINE,colour[area]);
                        redraw_window(CURRENT_WINDOW_IDLINE);
                        touchwin(CURRENT_WINDOW_IDLINE);
                        wnoutrefresh(CURRENT_WINDOW_IDLINE);
                        break;
    case WINDOW_FOOTING:
                        wattrset(foot,colour[area]);
                        redraw_window(foot);
                        touchwin(foot);
                        wnoutrefresh(foot);
                        break;
    case WINDOW_ERROR:
                        wattrset(error_window,colour[area]);
                        break;
    case WINDOW_DIVIDER:
                        wattrset(divider,colour[area]);
                        break;
    default:
                        break;
   }
#ifdef TRACE
 trace_return();
#endif
 return(OK);
}

/*man-start*********************************************************************
COMMAND
     control_char - allow control characters to be entered

SYNTAX
     ** effective only if bound to a key **

DESCRIPTION
     The CONTROL_CHAR command prompts the user to enter a control character.

COMPATIBILITY
     New feature.

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
int Control_char(unsigned char *params)
#else
int Control_char(params)
unsigned char *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 unsigned short key;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm1.c:   Control_char");
#endif
 display_error(0,(unsigned char *)"Press the character you require.");
 touchwin(error_window);
 doupdate();
 key = my_getch(CURRENT_WINDOW);
 if (islower(key))
    key = toupper(key);
 if (key >= '@'
 &&  key <= '_')
   {
    error_on_screen = NO;
    touchwin(foot);
#ifdef TRACE
    trace_return();
#endif
    return((RAW_KEY*2)+key-'@');
   }
 display_error(69,(unsigned char *)"- must be between '@' and '_'");
#ifdef TRACE
 trace_return();
#endif
 return(ERROR);
}
/*man-start*********************************************************************
COMMAND
     copy - copies text from one position to another

SYNTAX
     COPY BLOCK

DESCRIPTION
     The COPY command copies text from one part of a file to another.
     The text can be in the same file or a different file.

COMPATIBILITY
     Does not implement target option.

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
int Copy(unsigned char *params)
#else
int Copy(params)
unsigned char *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
#define COPY_PARAMS  1
 unsigned char *word[COPY_PARAMS+1];
 unsigned short num_params;
 long num_lines;
 LINE *curr,*old_curr;
 register int i;
 unsigned short y,x;
 unsigned long li;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm1.c:   Copy");
#endif
 num_params = param_split(params,word,COPY_PARAMS);
 if (strcmp(word[0],"block") != 0)
   {
    display_error(1,(unsigned char *)word[0]);
#ifdef TRACE
    trace_return();
#endif
    return(OK);
   }
/*---------------------------------------------------------------------*/
/* If on the command line, error.                                      */
/*---------------------------------------------------------------------*/
 if (CURRENT_VIEW->current_window == WINDOW_COMMAND)
   {
    display_error(38,(unsigned char *)"");
#ifdef TRACE
    trace_return();
#endif
    return(OK);
   }
/*---------------------------------------------------------------------*/
/* If no marked block in any view, return error.                       */
/*---------------------------------------------------------------------*/
 if (MARK_VIEW == (VIEW_DETAILS *)NULL)
   {
    display_error(44,(unsigned char *)"");
#ifdef TRACE
    trace_return();
#endif
    return(OK);
   }
 post_process_line(CURRENT_VIEW->focus_line);
/*---------------------------------------------------------------------*/
/* Determine the number of lines in the marked block (any view)        */
/*---------------------------------------------------------------------*/
 num_lines = MARK_VIEW->mark_end_line-MARK_VIEW->mark_start_line+1L;
/*---------------------------------------------------------------------*/
/* Find the LINE pointer for start of marked block.                    */
/*---------------------------------------------------------------------*/
 old_curr = ll_find(MARK_FILE->first_line,MARK_VIEW->mark_start_line);
/*---------------------------------------------------------------------*/
/* Find the LINE pointer for the current focus line.                   */
/* If the focus line is the bottom of file line, subtract 1 from it.   */
/*---------------------------------------------------------------------*/
 if (CURRENT_VIEW->focus_line == CURRENT_FILE->number_lines+1)
    li = 1L;
 else
    li = 0L;
 curr = ll_find(CURRENT_FILE->first_line,CURRENT_VIEW->focus_line-li);
/*---------------------------------------------------------------------*/
/* Add each line from the marked view after the current focus line.    */
/*---------------------------------------------------------------------*/
 for (i=0;i<num_lines;i++)
    {
     if ((curr = add_line(CURRENT_FILE->first_line,curr,
                          old_curr->line,old_curr->length)) == NULL)
       {
        display_error(30,(unsigned char *)"");
#ifdef TRACE
        trace_return();
#endif
        return(ERROR);
       }
     old_curr = old_curr->next;
    }
/*---------------------------------------------------------------------*/
/* Increment the number of lines counter for the current file and the  */
/* number of alterations.                                              */
/*---------------------------------------------------------------------*/
 increment_alt();
 CURRENT_FILE->number_lines += num_lines;
/*---------------------------------------------------------------------*/
/* Set the highlighting of lines up. The newly copied block remains    */
/* marked in the current view and the previous marked block is reset.  */
/*---------------------------------------------------------------------*/
 MARK_VIEW->mark_start_line = MARK_VIEW->mark_end_line = (-1L);
 CURRENT_VIEW->mark_start_line = CURRENT_VIEW->focus_line + 1L;
 CURRENT_VIEW->mark_end_line = CURRENT_VIEW->focus_line + num_lines;

 CURRENT_VIEW->focus_line = CURRENT_VIEW->mark_start_line;

 MARK_VIEW = CURRENT_VIEW;

/*---------------------------------------------------------------------*/
/* The following does a 'reset block' in the current view.             */
/*---------------------------------------------------------------------*/
 CURRENT_VIEW->mark_start_line = CURRENT_VIEW->mark_end_line = (-1L);
 MARK_VIEW = (VIEW_DETAILS *)NULL;

 pre_process_line(CURRENT_VIEW->focus_line);

 getyx(CURRENT_WINDOW,y,x);
 if (y == CURRENT_SCREEN.rows-1)      /* on bottom line of window */
   {
    CURRENT_VIEW->current_line = CURRENT_VIEW->focus_line;
    y = CURRENT_VIEW->current_row;
    i = 0;
   }
 else
   i = 1;
 show_page();
 wmove(CURRENT_WINDOW,y+i,x);

#ifdef TRACE
 trace_return();
#endif
 return(OK);
}
/*man-start*********************************************************************
COMMAND
     define - assign a command to a key

SYNTAX
     DEFine key-name command [parameters]

DESCRIPTION
     The DEFINE command allows the user assign a command and an optional
     parameter to a key.

     key-names correspond to the key values specified in the 'curses.h'
     file.

COMPATIBILITY
     Minimal. No support for in-memory macro commands.

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
int Define(unsigned char *params)
#else
int Define(params)
unsigned char *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
#define DEF_PARAMS  3
 unsigned char *word[DEF_PARAMS+1];
 char parm[DEF_PARAMS];
 register int i;
 unsigned short num_params;
 long num_lines,true_line;
 unsigned short x,y;
 LINE *curr;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm1.c:   Define");
#endif
 num_params = param_split(params,word,DEF_PARAMS);
 if (num_params < 2)
   {
    display_error(3,(unsigned char *)"");
#ifdef TRACE
    trace_return();
#endif
    return(OK);
   }
 if (num_params > 3)
   {
    display_error(2,(unsigned char *)"");
#ifdef TRACE
    trace_return();
#endif
    return(OK);
   }
 add_define(word[0],word[1],word[2]);
#ifdef TRACE
 trace_return();
#endif
 return(OK);
}
/*man-start*********************************************************************
COMMAND
     delete - delete lines from a file

SYNTAX
     DELete [target]

DESCRIPTION
     The DELETE command allows the user remove lines from the current
     file. The number of lines removed depends on the target specified.
     Lines are removed starting with the current_line.

COMPATIBILITY
     Compatible.

DEFAULT
     1 (the current line)

SEE ALSO
     Sos_delete

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
int Delete_line(unsigned char *params)
#else
int Delete_line(params)
unsigned char *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
#define DEL_PARAMS  1
 unsigned char *word[DEL_PARAMS+1];
 char parm[DEL_PARAMS];
 register int i;
 unsigned short num_params;
 long num_lines,true_line;
 unsigned short x,y;
 char direction;
 LINE *curr;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm1.c:   Delete_line");
#endif
/*---------------------------------------------------------------------*/
/* Validate the parameters that have been supplied. The one and only   */
/* parameter should be a positive integer greater than zero.           */
/* If no parameter is supplied, 1 is assumed.                          */
/*---------------------------------------------------------------------*/
 num_params = param_split(params,word,DEL_PARAMS);
 if (num_params == 0)
    {
     num_params = 1;
     word[0] = (unsigned char *)"1";
    }
 if (num_params != 1)
   {
    display_error(1,(unsigned char *)word[1]);
#ifdef TRACE
    trace_return();
#endif
    return(OK);
   }
/*---------------------------------------------------------------------*/
/* Get the correct number of lines to delete. If too many are specified*/
/* the following call will reduce the number of lines to 1 less than   */
/* the last line.                                                      */
/*---------------------------------------------------------------------*/
 if (strcmp(word[0],"block") == 0)
   {
    if (CURRENT_VIEW->mark_end_line == (-1L)
    &&  CURRENT_VIEW->mark_start_line == (-1L))
      {
       display_error(44,(unsigned char *)"");
#ifdef TRACE
       trace_return();
#endif
       return(OK);
      }
    num_lines = CURRENT_VIEW->mark_end_line-CURRENT_VIEW->mark_start_line+1L;
   }
 else
    if ((num_lines = valid_target(word[0])) == TARGET_ERROR)
      {
       display_error(4,(unsigned char *)word[0]);
#ifdef TRACE
       trace_return();
#endif
       return(OK);
      }
 if (num_lines == TARGET_NOT_FOUND)
   {
    display_error(17,(unsigned char *)"");
#ifdef TRACE
    trace_return();
#endif
    return(OK);
   }
/*---------------------------------------------------------------------*/
/* Determine in which direction we are deleting.                       */
/*---------------------------------------------------------------------*/
 if (num_lines < 0)
   {
    direction = DIRECTION_BACKWARD;
    num_lines = num_lines * (-1L);
   }
 else
   direction = DIRECTION_FORWARD;
/*---------------------------------------------------------------------*/
/* Check from which window the command was issued and make adjustments */
/* as required.                                                        */
/* Commands issued from the command window relate to the current line, */
/* commands issued from either the prefix or main window relate to the */
/* focus line.                                                         */
/*---------------------------------------------------------------------*/
 if (CURRENT_VIEW->current_window == WINDOW_COMMAND
 || in_profile)
   {
    true_line = CURRENT_VIEW->current_line;
    if (true_line == 0L)
      {
       true_line++;
       num_lines--;
      }
    if (true_line == CURRENT_FILE->number_lines+1)
      {
       true_line--;
       num_lines--;
      }
   }
 else
   {
    true_line = CURRENT_VIEW->focus_line;
    if (true_line == CURRENT_FILE->number_lines+1L || true_line == 0L)
      {
       display_error(38,(unsigned char *)"");
#ifdef TRACE
       trace_return();
#endif
       return(OK);
      }
   }

 post_process_line(true_line);

 if (strcmp(word[0],"block") == 0)
    if (in_profile)
      {
       display_error(25,(unsigned char *)word[0]);
#ifdef TRACE
       trace_return();
#endif
       return(ERROR);
      }
    else
       true_line = CURRENT_VIEW->mark_start_line;
/*---------------------------------------------------------------------*/
/* Find the current LINE pointer for the true_line.                    */
/* This is the line to be deleted.                                     */
/*---------------------------------------------------------------------*/
 curr = ll_find(CURRENT_FILE->first_line,true_line);
/*---------------------------------------------------------------------*/
/* Delete from the linked list the number of lines specified in the    */
/* direction specified.                                                */
/*---------------------------------------------------------------------*/
/* for (i=0;i<num_lines,curr->next != NULL;i++) */
 for (i=0;i<num_lines;i++)
    {
     add_to_recovery_list(curr->line,curr->length);
     curr = ll_del(CURRENT_FILE->first_line,curr,direction);
    }
/*---------------------------------------------------------------------*/
/* Special problem for sos_delline. As the current_line is left where  */
/* it is, when deleting a line using sos_delline and the current_line  */
/* is on the bottom-of-file the current_line MUST be reduced by the    */
/* number of lines deleted. This test can be done here as it is not    */
/* possible to have this situation when the Delete command is issued   */
/* from the command window.                                            */
/*---------------------------------------------------------------------*/
 if (CURRENT_VIEW->current_line == CURRENT_FILE->number_lines+1)
    CURRENT_VIEW->current_line -= num_lines;
/*---------------------------------------------------------------------*/
/* Decrement the number of lines counter for the current file and the  */
/* number of alterations.                                              */
/*---------------------------------------------------------------------*/
 increment_alt();
 CURRENT_FILE->number_lines -= num_lines;

 if (direction == DIRECTION_BACKWARD)
   {
    CURRENT_VIEW->current_line -= num_lines;
    CURRENT_VIEW->focus_line -= num_lines;
   }
/*---------------------------------------------------------------------*/
/* If we are in the profile, then don't do any more processing.        */
/*---------------------------------------------------------------------*/
 if (in_profile)
   {
    pre_process_line(CURRENT_VIEW->focus_line);
#ifdef TRACE
    trace_return();
#endif
    return(OK);
   }
 if (strcmp(word[0],"block") == 0)
   {
    CURRENT_VIEW->mark_start_line = CURRENT_VIEW->mark_end_line = (-1L);
    MARK_VIEW = (VIEW_DETAILS *)NULL;
   }
 else
/*---------------------------------------------------------------------*/
/* Fix the positioning of the marked block (if there is one and it is  */
/* in the current view and we are not deleting a block).               */
/*---------------------------------------------------------------------*/
  {
   if (direction == DIRECTION_FORWARD)
      adjust_marked_block(NO,true_line,num_lines);
   else
      adjust_marked_block(NO,true_line-num_lines+1,num_lines);
  }
 if (strcmp(word[0],"block") == 0)
   {
    if (CURRENT_VIEW->current_line > CURRENT_FILE->number_lines)
       CURRENT_VIEW->current_line = CURRENT_FILE->number_lines;
    if (CURRENT_VIEW->focus_line > CURRENT_FILE->number_lines)
       CURRENT_VIEW->focus_line = CURRENT_VIEW->current_line;
   }
 pre_process_line(CURRENT_VIEW->focus_line);
 show_page();
 if (CURRENT_VIEW->current_window != WINDOW_COMMAND)
   {
    getyx(CURRENT_WINDOW,y,x);
    y = get_row_for_focus_line(CURRENT_VIEW->current_row,
                               CURRENT_VIEW->focus_line,
                               CURRENT_VIEW->current_line);
    wmove(CURRENT_WINDOW,y,x);
   }
#ifdef TRACE
 trace_return();
#endif
 return(OK);
}
/*man-start*********************************************************************
COMMAND
     directory - list the specified directory

SYNTAX
     DIRectory [directory]

DESCRIPTION
     The DIRECTORY command displays all files in the specified directory.
     When no parameter is supplied, the current directory is displayed.

COMPATIBILITY
     Compatible.

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
int Directory(unsigned char *params)
#else
int Directory(params)
unsigned char *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
#define DIR_PARAMS  1
 unsigned char *word[DIR_PARAMS+1];
 unsigned short num_params;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm1.c:   Directory");
#endif
/*---------------------------------------------------------------------*/
/* Validate the parameters that have been supplied. The one and only   */
/* parameter should be the directory to display.                       */
/*---------------------------------------------------------------------*/
 num_params = param_split(params,word,DIR_PARAMS);
 if (num_params > 1)
   {
    display_error(1,(unsigned char *)word[1]);
#ifdef TRACE
    trace_return();
#endif
    return(OK);
   }
/*---------------------------------------------------------------------*/
/* Validate that the supplied directory is valid.                      */
/*---------------------------------------------------------------------*/
 if (splitpath(word[0]) == ERROR)
   {
    display_error(10,(unsigned char *)word[0]);
#ifdef TRACE
    trace_return();
#endif
    return(ERROR);
   }
 if (read_directory() == ERROR)
   {
    display_error(10,(unsigned char *)word[0]);
#ifdef TRACE
    trace_return();
#endif
    return(ERROR);
   }
 strcpy(temp_cmd,dir_pathname);
 strcat(temp_cmd,dir_filename);
 Edit(temp_cmd);
#ifdef TRACE
 trace_return();
#endif
 return(OK);
}
/*man-start*********************************************************************
COMMAND
     down_arrow - move the cursor down one line

SYNTAX
     ** effective only if bound to a key **

DESCRIPTION
     The down_arrow command moves the cursor down one line in the main
     window. Scrolling of the window occurs if the cursor is on the last
     line of the window.

     When on the command line, this command moves forward through the
     list of previous command line commands or tabs to the first line of
     the main window depending on the value set by the function
     cmdarrows. (default is the former)

COMPATIBILITY
     Compatible.

SEE ALSO
     Up_arrow

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
int Down_arrow(unsigned char *params)
#else
int Down_arrow(params)
unsigned char *params;
#endif
/***********************************************************************/
{
/*------------------------- external data -----------------------------*/
extern unsigned char CMDARROWSTABx;
/*--------------------------- local data ------------------------------*/
 unsigned short x,y;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm1.c:   Down_arrow");
#endif
 switch(CURRENT_VIEW->current_window)
  {
   case WINDOW_PREFIX:
   case WINDOW_MAIN:
        getyx(CURRENT_WINDOW,y,x);
/*---------------------------------------------------------------------*/
/* If the cursor is on the last line of the file...                    */
/*---------------------------------------------------------------------*/
        if (CURRENT_VIEW->current_line+y-
           CURRENT_VIEW->current_row == CURRENT_FILE->number_lines+1)
           {
/*---------------------------------------------------------------------*/
/* ... and the last line of the file is on the current row, stay there.*/
/*---------------------------------------------------------------------*/
            if (CURRENT_VIEW->current_line ==
                CURRENT_FILE->number_lines+1)
                break;
/*---------------------------------------------------------------------*/
/* ... and the last line of the file is below the current row,         */
/* scroll the window up one line.                                      */
/*---------------------------------------------------------------------*/
            CURRENT_VIEW->current_line++;
            show_page();
            wmove(CURRENT_WINDOW,y-1,x);
            break;
           }
/*---------------------------------------------------------------------*/
/* If on the bottom of the window, scroll the window up 1 line.        */
/*---------------------------------------------------------------------*/
        if (y == CURRENT_SCREEN.rows-1) /* on bottom of window */
           {
            CURRENT_VIEW->current_line++;
            post_process_line(CURRENT_VIEW->focus_line);
            CURRENT_VIEW->focus_line++;
            pre_process_line(CURRENT_VIEW->focus_line);
            show_page();
            wmove(CURRENT_WINDOW,y,x);
            break;
           }
/*---------------------------------------------------------------------*/
/* We are in the middle of the window, so just move the cursor down    */
/* 1 line.                                                             */
/*---------------------------------------------------------------------*/
        wmove(CURRENT_WINDOW,y+1,x);
        post_process_line(CURRENT_VIEW->focus_line);
        CURRENT_VIEW->focus_line++;
        pre_process_line(CURRENT_VIEW->focus_line);
        break;
   case WINDOW_COMMAND:
/*---------------------------------------------------------------------*/
/* Cycle forward  through the command list or tab to first line.       */
/*---------------------------------------------------------------------*/
        if (CMDARROWSTABx)
          {
           getyx(CURRENT_WINDOW,y,x);
           if (CURRENT_VIEW->prefix_on != TRUE
           ||  CURRENT_VIEW->prefix_left != TRUE)
              x += 6;
           if (CURRENT_VIEW->current_line > CURRENT_VIEW->current_row)
             {
              CURRENT_VIEW->focus_line = CURRENT_VIEW->current_line -
                                        CURRENT_VIEW->current_row ;
              y = 0;
             }
           else
             {
              CURRENT_VIEW->focus_line = 0;
              y = CURRENT_VIEW->current_row -
                                        CURRENT_VIEW->current_line ;
             }
           pre_process_line(CURRENT_VIEW->focus_line);
           CURRENT_VIEW->current_window = WINDOW_MAIN;
           wmove(CURRENT_WINDOW,y,x);
          }
        else
           Retrieve("+");
        break;
   default:
        display_error(2,(unsigned char *)"");
        break;
  }
#ifdef TRACE
 trace_return();
#endif
 return(OK);
}
/*man-start*********************************************************************
COMMAND
     edit - edit another file

SYNTAX
     Edit [filename]

DESCRIPTION
     The EDIT command allows the user to edit another file. The new file
     is placed in the file ring. The previous file being editted remains
     in memory and can be returned to by issuing an EDIT command without
     any parameters. Several files can be editted at once, and all files
     are arranged in a ring, with subsequent EDIT commands moving through
     the ring, one file at a time.

COMPATIBILITY
     Compatible.

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
int Edit(unsigned char *params)
#else
int Edit(params)
unsigned char *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 char old_current_file;
 unsigned short y,x;
#ifndef VMS
 SCREEN_DETAILS *screen0=&screen[0];
 SCREEN_DETAILS *screen1=&screen[1];
#endif
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm1.c:   Edit");
#endif
 if (strcmp(params,"") == 0)
   {
    if (CURRENT_VIEW->next == (VIEW_DETAILS *)NULL
    &&  CURRENT_VIEW->prev == (VIEW_DETAILS *)NULL)
       return(OK);
    wmove(CURRENT_WINDOW_COMMAND,0,0);
    wclrtoeol(CURRENT_WINDOW_COMMAND);
    memset(cmd_rec,' ',COLS);
    cmd_rec_len = 0;
    post_process_line(CURRENT_VIEW->focus_line);
    if (CURRENT_VIEW->next == (VIEW_DETAILS *)NULL)
       CURRENT_VIEW = vd_first;
    else
       CURRENT_VIEW = CURRENT_VIEW->next;
    pre_process_line(CURRENT_VIEW->focus_line);
    show_page();
    if (CURRENT_VIEW->prefix_on)
       touchwin(CURRENT_WINDOW_PREFIX);
    touchwin(CURRENT_WINDOW_COMMAND);
    touchwin(CURRENT_WINDOW_MAIN);
    CURRENT_SCREEN.screen_view = CURRENT_VIEW;
#ifdef TRACE
    trace_return();
#endif
    return(OK);
   }
 wmove(CURRENT_WINDOW_COMMAND,0,0);
 wclrtoeol(CURRENT_WINDOW_COMMAND);
 memset(cmd_rec,' ',COLS);
 cmd_rec_len = 0;
 post_process_line(CURRENT_VIEW->focus_line);
 if (get_file(params) == ERROR)
   {
#ifdef TRACE
    trace_return();
#endif
    return(ERROR);
   }
 set_up_windows();
 pre_process_line(CURRENT_VIEW->focus_line);
 show_page();
/* touchwin(CURRENT_WINDOW_PREFIX);*/
/* touchwin(CURRENT_WINDOW_COMMAND);*/
/* touchwin(CURRENT_WINDOW_MAIN);*/
/* touchwin(CURRENT_WINDOW);*/
#ifdef TRACE
 trace_return();
#endif
 return(OK);
}
/*man-start*********************************************************************
COMMAND
     enter - execute a command

SYNTAX
     ** effective only if bound to a key **

DESCRIPTION
     The ENTER command executes the command currently displayed on the
     command line, if the cursor is currently displayed there.
     If the key associated with ENTER is pressed while in the main or
     prefix window, then the cursor will move to the first column of the
     next line. If the mode is currently in 'insert', then a new line
     is added and the cursor placed at the first column of the new line.

COMPATIBILITY
     Compatible.

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
int Enter(unsigned char *params)
#else
int Enter(params)
unsigned char *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 unsigned short x,y;
 short rc;
 register int i;
 static char first=TRUE;
 static char number_of_commands=0;
 WINDOW *w;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm1.c:   Enter");
#endif
 if (CURRENT_VIEW->current_window == WINDOW_COMMAND)
   {
    for (i=0;i<cmd_rec_len;i++)
        temp_cmd[i] = cmd_rec[i];
    temp_cmd[cmd_rec_len] = '\0';
    strtrunc(temp_cmd);
    add_command(temp_cmd);
    rc = command_line(temp_cmd);
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
/* check for insert mode or not */
 if (mode_insert)
    Sos_addline("");
 else
   {
    Down_arrow((unsigned char *)"");
    getyx(CURRENT_WINDOW,y,x);
    wmove(CURRENT_WINDOW,y,0);
   }
#ifdef TRACE
 trace_return();
#endif
 return(OK);
}
/*man-start*********************************************************************
COMMAND
     eolout - set end of line terminator

SYNTAX
     EOLout CRLF|LF

DESCRIPTION
     The EOLOUT command allows the user to specify the combination of
     characters that terminate a line. Lines of text in U*ix files are
     usually terminated with a LF, whereas in DOS they usually end with
     a CR and LF combination.

COMPATIBILITY
     Does not implement CR option.

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
int Eolout(unsigned char *params)
#else
int Eolout(params)
unsigned char *params;
#endif
/***********************************************************************/
{
/*--------------------------- extern data -----------------------------*/
extern unsigned char EOLx;
/*--------------------------- local data ------------------------------*/
#define EOLOUT_PARAMS  1
 unsigned char *word[EOLOUT_PARAMS+1];
 unsigned short num_params;
 unsigned char eolchar;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("comm1.c:   Eolout");
#endif
 num_params = param_split(params,word,EOLOUT_PARAMS);
 if (strcmp(word[0],"lf") == 0)
    eolchar = EOLOUT_LF;
 else
    if (strcmp(word[0],"crlf") == 0)
       eolchar = EOLOUT_CRLF;
    else
      {
       display_error(1,(unsigned char *)word[0]);
#ifdef TRACE
       trace_return();
#endif
       return(ERROR);
      }
 if (in_profile)
    EOLx = eolchar;
 else
    CURRENT_FILE->eolout = eolchar;
#ifdef TRACE
 trace_return();
#endif
 return(OK);
}
