/*
 *  Copyright (c) 1992 John E. Davis  (davis@amy.tch.harvard.edu)
 *  All Rights Reserved.
 */
#include <stdio.h>
#include "sysdep.h"

#include <ssdef.h>
#include <rmsdef.h>
#include <dvidef.h>
#include <jpidef.h>
#include <libdef.h>
#include <descrip.h>
#include <iodef.h>
#include <ttdef.h>
#include <unixlib.h>
#include <rms.h>
#include <errno.h>

typedef struct {                /* I/O status block     */
        short i_cond;           /* Condition value      */
        short i_xfer;           /* Transfer count     */
        long  i_info;           /* Device information     */
} iosb;

typedef struct {                /* Terminal characteristics   */
        char  t_class;          /* Terminal class     */
        char  t_type;           /* Terminal type      */
        short t_width;          /* Terminal width in characters   */
        long  t_mandl;          /* Terminal's mode and length   */
        long  t_extend;         /* Extended terminal characteristics  */
}  termchar;

short TTY_CHANNEL_GLOBAL;

/*
 *      Exit Handler Control Block
 */
static struct argument_block
  {
      int forward_link;
      int (*exit_routine)();
      int arg_count;
      int *status_address;
      int exit_status;
  }
exit_block =
  {
      0,
      NULL,
      1,
      &exit_block.exit_status,
      0
  };

int vms_input_buffer;

static struct vms_ast_iosb
{
    short status;
    short offset;
    short termlen;
    short term;
} vms_ast_iosb;

extern void vms_que_key_ast();
int vms_event_flag;
int event_flag_mask;
int ast_stop_input = 0;
int waiting_for_ast;

int getkey_ast(int x)
{
   unsigned int c = 1000;

   if (waiting_for_ast)  SYS$SETEF (vms_event_flag);
   waiting_for_ast = 0;

   if (vms_ast_iosb.offset)
     {
	c = (unsigned int) vms_input_buffer;
     }

   if (c <= 255)
     {
	if (c == 7)
	  {
	     Lang_Error = -1;
	     KeyBoard_Quit = 1;
	  }

	if (INPUT_BUFFER_LEN < 253) INPUT_BUFFER[INPUT_BUFFER_LEN++] = c;
     }
   vms_que_key_ast();
}

void vms_que_key_ast()
{
   static int trmmsk [2] = { 0, 0 };
   int status;

   if (ast_stop_input) return;
   status = SYS$QIO (0, TTY_CHANNEL_GLOBAL,
		     IO$_READVBLK + IO$M_NOECHO | IO$_TTYREADALL,
		     &vms_ast_iosb, getkey_ast, 1,
		     &vms_input_buffer, 1, 0, trmmsk, 0, 0);
}

void init_tty()
{

   $DESCRIPTOR ( Term, "SYS$ERROR");
   /* Enable the Application Keypad */
   fputs("\033=\033[?1l", stdout);

   if (sys$assign ( &Term, &TTY_CHANNEL_GLOBAL, 0, 0 )
       != SS$_NORMAL)
     {
	fprintf(stderr,"Unable to assign input channel\n");
	exit(0);
     }

   if (NULL == exit_block.exit_routine)
     {
	void reset_tty();

	exit_block.exit_routine = reset_tty;
	SYS$DCLEXH(&exit_block);
     }

   /* allocate an event flag and clear it--- used by ast routines */
   if (!vms_event_flag) LIB$GET_EF (&vms_event_flag);
   SYS$CLREF (vms_event_flag);
   event_flag_mask = ((unsigned) 1 << (vms_event_flag % 32));

   waiting_for_ast = 0;
   ast_stop_input = 0;
   vms_que_key_ast();   /* set up the key ast */
}

void sys_flush(int fd)
{
}

void reset_tty()
{
   fputs("\033>", stdout);  /* Enable the Numeric Keypad */
}

unsigned char sys_getkey()
{
   unsigned char c;

   /* On VMS, the keyboard ast routine should be stuffing the buffer, so
   do nothing except sleep */

   /* clear the flag which ast will set */
   waiting_for_ast = 0;
   if (INPUT_BUFFER_LEN) return(getkey());
   SYS$CLREF (vms_event_flag);

   /* set up a flag for the ast so it knows to set the event flag */
   waiting_for_ast = 1;

   while (!INPUT_BUFFER_LEN)
     {
	/* this will return when ast does its job */
	SYS$WFLOR (vms_event_flag, event_flag_mask);
     }
   waiting_for_ast = 0;
   c = getkey();
   return(c);
}

int input_pending(int fd, int secs)
{
   if (secs) sleep(1);
   return(INPUT_BUFFER_LEN);
}

/*  This is to get the size of the terminal  */
void get_term_dimensions(int *cols, int *rows)
{
   int status;
   iosb iostatus;
   $DESCRIPTOR(devnam, "SYS$ERROR");
   struct
     {
	short row_buflen;
	short row_itmcod;
	int *row_bufadr;
	short *row_retlen;
	short col_buflen;
	short col_itmcod;
	int *col_bufadr;
	short *col_retlen;
	int listend;
     }
   itmlst =
     {
	sizeof(*rows), DVI$_TT_PAGE, rows, 0,
	sizeof(*cols), DVI$_DEVBUFSIZ, cols, 0,
	0
     };

   /* Get current terminal characteristics */
   status = sys$getdviw(0,           /* Wait on event flag zero  */
			0,           /* Channel to input terminal  */
			&devnam,     /* device name */
			&itmlst,	  /* Item descriptor List */
			&iostatus,   /* Status after operation */
			0, 0,        /* No AST service   */
			0);          /* nullarg */

   if (status&1) status = iostatus.i_cond;
   /* Jump out if bad status */
   if ((status & 1) == 0) exit(status);
}

/* returns 0 on failure, 1 on sucess */
int sys_delete_file(char *filename)
{
    return (1 + delete(filename));   /* 0: sucess; -1 failure */
}

int sys_rename(char *from, char *to)
{
    return(-1);
}

/* This routine converts unix type names to vms names */
int locate(char ch, char *string)
{
    int i;
    char c;

    i = 0;
    while (c = string[i++], (c != ch) && (c != '\0'));
    if (c == ch) return(i); else return (0);
}

char *unix2vms(char *file)
{
    int i,device,j,first,last;
    static char vms_name[80];
    char ch;

    if (locate('[',file)) return(file); /* vms_name syntax */
    if (!locate('/',file)) return(file); /* vms_name syntax */

    /* search for the ':' which means a device is present */
    device = locate(':',file);

    i = 0;
    if (device)
      {
          while (ch = file[i], i < device) vms_name[i++] = ch;
      }
    j = i;

    /* go from the  end looking for a '/' and mark it */
    i = strlen(file) - 1;
    while(ch = file[i], ch != '/' && i-- >= 0);
    if (ch == '/')
      {
          file[i] = ']';
          last = 0;
      }
    else last = 1;

    i = j;
    vms_name[j++] = '[';
    vms_name[j++] = '.';
    first = 0;
    while(ch = file[i++], ch != '\0')
      {
          switch (ch)
            {
              case '.':
                if (last) vms_name[j++] = '.';
                if (last) break;
                ch = file[i++];
                if (ch == '.')
                  {
                      if (!first) j--;  /* overwrite the dot */
                      vms_name[j++] = '-';
                  }
                else if (ch == '/'); /*  './' combinations-- do nothing */
                else if (ch == ']')
                  {
                      last = 1;
                      if (vms_name[j-1] == '.') j--;
                      vms_name[j++] = ']';
                  }

                else vms_name[j++] = '.';
                break;
              case '/':
                if (first)
                  {
                      vms_name[j++] = '.';
                  }
                else
                  {
                      first = 1;
                      /* if '/' is first char or follows a colon do nothing */
                      if ((i!=1) && (file[i-2] != ':'))
                        {
                            vms_name[j++] = '.';
                        }
                      else j--; /* overwrite the '.' following '[' */
                  }
                break;
              case ']':
                last = 1;
                if (vms_name[j-1] == '.') j--;
                vms_name[j++] = ']';
                break;
              default:
                vms_name[j++] = ch;
            }
      }
    return (vms_name);
}

/* these two from emacs source */
void define_logical_name (char *varname, char *string)
{
    struct dsc$descriptor_s strdsc =
      {strlen (string), DSC$K_DTYPE_T, DSC$K_CLASS_S, string};
    struct dsc$descriptor_s envdsc =
      {strlen (varname), DSC$K_DTYPE_T, DSC$K_CLASS_S, varname};
    struct dsc$descriptor_s lnmdsc =
      {7, DSC$K_DTYPE_T, DSC$K_CLASS_S, "LNM$JOB"};

    LIB$SET_LOGICAL (&envdsc, &strdsc, &lnmdsc, 0, 0);
}

void delete_logical_name (char *varname)
{
    struct dsc$descriptor_s envdsc =
      {strlen (varname), DSC$K_DTYPE_T, DSC$K_CLASS_S, varname};
    struct dsc$descriptor_s lnmdsc =
      {7, DSC$K_DTYPE_T, DSC$K_CLASS_S, "LNM$JOB"};

    LIB$DELETE_LOGICAL (&envdsc, &lnmdsc);
}

int do_attach_cmd()
{
   unsigned long pid;
   char *pidstr;

   if((pidstr = getenv("JED_ATTACH_TO")) != NULL)
     {
	delete_logical_name("JED_ATTACH_TO");
	(void) sscanf(pidstr,"%X",&pid);
	if (lib$attach(&pid) == SS$_NORMAL)
	  return(1);
        else
          return(0);
     }
   else return(0);
}

unsigned long SHELL_PID = 0;

/* here we try to attach to the parent otherwise just spawn a new one */
void sys_suspend()
{
   unsigned long parent_pid;
   unsigned long status = 0;
   char str[80], *file;

   /* stop the keyboard ast */
   SYS$SETAST (0);
   SYS$CLREF (vms_event_flag);
   waiting_for_ast = 1;
   ast_stop_input = 1;

   /* cancel all i/o on this channel */
   SYS$CANCEL (TTY_CHANNEL_GLOBAL);
   SYS$SETAST (1);
   SYS$WAITFR (vms_event_flag);
   waiting_for_ast = 0;

   parent_pid = getppid();

   /* try to attach to different process */
   if (do_attach_cmd()) status = SS$_NORMAL;

   else if (parent_pid && parent_pid != 0xffffffff)
   /* we attach to parent */
      status = lib$attach(&parent_pid);

   else if (SHELL_PID && SHELL_PID != 0xffffffff)
   /* try to attach to previous shell */
      status = lib$attach (&SHELL_PID);

   if (status != SS$_NORMAL)		/* others fail so spawn a new shell */
     {
	status = 0;
	fputs("Spawning Subprocess...\n",stdout);
	fflush(stdout);
	lib$spawn(0,0,0,0,0,&SHELL_PID,&status);
	/* if we attach back, status may come back unchanged */
	if ((status != 0) && (status != SS$_NORMAL))
	  {
	     sprintf(str,"Unable to spawn subprocess. Error = %X", status);
	     msg_error(str);
	     return;
	  }
     }

   if((file = getenv("JED_FILE_NAME")) != NULL)
     {
	find_file_cmd(file);
	delete_logical_name ("JED_FILE_NAME");
     }
}

/* returns 0 on success, -1 on syntax error
   -2 on bad dir, -3 on bad device, 1 if something else */
int vms_parse_file(char *old)
{
   struct FAB fab = cc$rms_fab;
   struct NAM nam = cc$rms_nam;
   char new[255];
   int status;

   fab.fab$l_fna = old;
   fab.fab$b_fns = strlen(old);
   fab.fab$b_dns = 0;
   fab.fab$w_ifi = 0;
   fab.fab$l_nam = &nam;
   nam.nam$l_esa = new;
   nam.nam$b_ess = 255;
   nam.nam$b_nop = NAM$V_SYNCHK;   /* syntax only */
   nam.nam$l_rlf = 0;

   status = sys$parse(&fab);
   new[nam.nam$b_esl] = 0;
   strcpy(old, new);
   while(*old != 0)
     {
	if ((*old == ';') && (*(old + 1) == 0)) *old = 0;
	else
	  {
	     if ((*old >= 'A') && (*old <= 'Z')) *old |= 0x20;
	     old++;
	  }
     }

   switch(status)
     {
	case RMS$_NORMAL: return 0;
	case RMS$_SYN: return -1;
	case RMS$_DNF: return -2;
	case RMS$_DEV: return -3;
	return 1;
     }
}

char *expand_filename(char *file)
{
   char *p, *p1, tmp[255];
   static char work[300];
   int len;

   strcpy(work, get_cwd()); strcat(work, file); file = work;

    /*  start at end and look for ']' then look for ':' */
    if (0 == (len = strlen(file))) return(file);
    p = file + (len - 1);
    while (p >= file) if (*p-- == ':')
      {
	 while((p >= file) && (*p != ':') && (*p != ']')) p--;
	 p++;
	 p1 = file;
         while(*p) *p1++ = *p++;
	 *p1 = 0;
	 break;
      }

    /* now remove "][" combinations */
    p = p1 = file;
    while (*p != 0)
      {
	 *p1 = *p;

	 if ((*p == ']') && (*(p + 1) == '['))
	   {
	      *p1 = '.';
	      p++;
	   }
	 p1++;
	 p++;
      }
    *p1 = 0;
    switch(vms_parse_file(file))
      {
	 case -1: msg_error("Filename syntax error."); break;
	 case -2: msg_error("Directory does not exist!"); break;
	 case -3: msg_error("Bad device name."); break;
	 case 0: break;
	 default: msg_error("Unknown filename error!");
      }
   p = file;
   while(*p != 0) if (*p++ == ']')
     {
	 if ((*p == '.') && (*(p + 1) == 0)) *p = 0;
         break;
      }
   return(file);

}

int vms_expand_filename(char *file,char *expanded_file)
{
    unsigned long status;
    static int context = 0;
    static char inputname[255] = "";
    $DESCRIPTOR(file_desc,inputname);
    $DESCRIPTOR(default_dsc,"SYS$DISK:[]*.*;");
    static struct dsc$descriptor_s  result =
	    {0, DSC$K_DTYPE_T, DSC$K_CLASS_D, NULL};

    if (strcmp(inputname, file))
      {
	  if (context)
	    {
		lib$find_file_end(&context);
	    }
	  context = 0;
	  strcpy(inputname, file);
	  file_desc.dsc$w_length = strlen(inputname);
      }

    if (RMS$_NORMAL == lib$find_file(&file_desc,&result,&context,
	           		     &default_dsc,0,0,&0))
      {
	  memcpy(expanded_file, result.dsc$a_pointer, result.dsc$w_length);
	  expanded_file[result.dsc$w_length] = '\0';
          return (1);
      }
    else
      {
          /* expanded_file[0] = '\0'; */      /* so file comes back as zero width */
          return(0);
      }
}

static int context = 0;

static char inputname[255] = "";
$DESCRIPTOR(file_desc,inputname);
$DESCRIPTOR(default_dsc,"SYS$DISK:[]*.*;");

static char *VMS_Star;

int sys_findnext(char *file)
{
   unsigned long status;
   char *f;
   static struct dsc$descriptor_s  result = {0, DSC$K_DTYPE_T, DSC$K_CLASS_D, NULL};

   if (RMS$_NORMAL == lib$find_file(&file_desc,&result,&context,
				    &default_dsc,0,0,&0))
     {
	memcpy(file, result.dsc$a_pointer, result.dsc$w_length);
	file[result.dsc$w_length] = 0;
	f = &file[result.dsc$w_length];
	if (*VMS_Star)
	  {
	     while ((f > file) && (*f != ';')) f--;
	     if (*f == ';') *f = 0;
	  }

	return (1);
     }
   else return(0);
}

int sys_findfirst(char *thefile)
{
   char *f, *f1, *s2 = "*.*";
   char *file;

   file = thefile;
   f = extract_file(file);

   while (*f && (*f != '*')) f++;

   VMS_Star = "";

   /* let user choose what to match with or take mine */
   if (! *f)
     {
	f = extract_file(file);
	while (*f && (*f != '.')) f++;
	if (*f) VMS_Star = "*"; else VMS_Star = s2;

	file = expand_filename(thefile);
	f = f1 = extract_file(file);
	if (VMS_Star == s2)
	  {
	     while (*f && (*f != '.')) f++;
	     *f = 0;
	  }
	else
	  {
	     while (*f1 && (*f1 != ';')) f1++;
	     *f1 = 0;
	  }
	strcpy(thefile, file);
     }

   strcpy(inputname, file);
   if (*VMS_Star) strcat(inputname, VMS_Star);
   file_desc.dsc$w_length = strlen(inputname);

   if (context) lib$find_file_end(&context);
   context = 0;
   return sys_findnext(thefile);
}

#include stat.h

/* returns 0 if file does not exist, 1 if it is not a dir, 2 if it is */
int sys_chmod(char *file, int what, int *mode)
{
   struct stat buf;
   int m, err;

   if (stat(file, &buf) < 0) switch (errno)
     {
	case EACCES: return(-1); /* es = "Access denied."; break; */
	case ENOENT: return(0);  /* ms = "File does not exist."; */
	case ENOTDIR: return(-2); /* es = "Invalid Path."; */
	default: return(-3); /* "stat: unknown error."; break;*/
     }

   m = buf.st_mode;

   *mode = m & 0777;

   if (m & S_IFDIR) return (2);
   return(1);
}
#if 0
   switch(vms_parse_file(file))
     {
	case 0: break;
	case -1:		       /* Filename syntax error. */
	case -2:   return(-2);	       /* Directory does not exist! */
	case 1:
	case -3:   return(-3);	       /* Bad device name. */
      default: return(-3);
     }
   return(1);
}

#endif

unsigned long sys_file_mod_time(char *file)
{
   struct stat buf;

   if (stat(file, &buf) < 0) return(0);
   return((unsigned long) buf.st_mtime);
}

