/*
 *  Copyright (c) 1992 John E. Davis  (davis@amy.tch.harvard.edu)
 *  All Rights Reserved.
 */
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include "buffer.h"
#include "ins.h"
#include "ledit.h"
#include "text.h"
#include "screen.h"
#include "cmds.h"
#include "paste.h"
#include "misc.h"
#include "slang.h"

/* This routine deletes multiple spaces except those following a period, '?'
   or a '!'.

   Returns the address of beginning of non whitespace */
unsigned char *text_format_line(void)
{
    unsigned char *p, *p1;
    int min;

    p = CLine->data;
    Point = 0;

    while(((*p == '\t') || (*p == ' ')) && (Point < CLine->len)) p++, Point++;
    min = Point;

    Point = CLine->len - 1;
    if (Point < 0) Point = 0;
    p = CLine->data + Point;

    while (Point > min)
      {
          if ((*p == ' ') || (*p == '\t'))
            {
	       Point--; p--;
	       p1 = p - 1;
	       if (((*p == ' ') || (*p == '\t')) && (Point > min)
		   && (*p1 != '.') && (*p1 != '?') && (*p1 != '!'))
		 {
		    del();
		    if (*p == '\t') *p = ' ';
		 }
	    }
          else
            {
                Point--; p--;
            }
      }
   return(CLine->data + min);
}

int wrap_line1(int format)
{
   unsigned char *p, *pmin;
   int col;

   if (format) pmin = text_format_line(); else pmin = CLine->data;
   eol();
   col = calculate_column();
   if (col < User_Vars.wrap_column)
     {
	return(0);
     }

   point_column(User_Vars.wrap_column - 1);
   p = CLine->data + Point;

   while(p > pmin)
     {
	if ((*p == ' ') || (*p == '\t')) break;
	p--;
     }

   if (p == pmin)
     {
	/* that failed, so go the other way */
	p = CLine->data + CLine->len;
	while(pmin < p)
	  {
	     if ((*pmin == ' ') || (*pmin == '\t')) break;
	     pmin++;
	  }
	if (p == pmin) return(0);
	p = pmin;
     }

   Point = (int) (p - CLine->data);
   trim_whitespace();
   newline();
   CLine = CLine->prev;
   return(1);
}

void wrap_line(int format)
{
    push_spot();
    wrap_line1(format);
    pop_spot();
}

/* Here a paragraph follows either an indentation, a '\\' char or two
   '\n' chars.  or a '%' char since tex uses this */

static int paragraph_sep_hook;

int is_paragraph_sep(void)
{
   unsigned char *p;
   int ret;
   p = CLine->data;
   
   if (paragraph_sep_hook && 
       (0 != (paragraph_sep_hook = lang_run_hooks("is_paragraph_separator", NULL))))
     {
	if (!lang_pop_integer(&ret)) ret = 1;
	return ret;
     }

   if ((*p == '\n') || (*p == '\\') || (*p == '%')) return(1);
   return(0);
}

int backward_paragraph(void)
{
   paragraph_sep_hook = 1;
   while(1)
     {
	eol();
	trim_whitespace();

	if (is_paragraph_sep()) return(1);
	if (CLine->prev == NULL) break;
	CLine = CLine->prev;
     }
   return(1);
}

int forward_paragraph(void)
{
   paragraph_sep_hook = 1;
   while(1)
     {
	eol();
	trim_whitespace();
	if (is_paragraph_sep()) return(1);
	if (CLine->next == NULL) break;
	CLine = CLine->next;
     }
   return(1);
}

int text_format_paragraph()
{
   unsigned char *p;
   int n, col;
   Line *end, *beg, *next;

   if (CBuf->modes != WRAP_MODE) return(0);
   push_spot();

   get_current_indent(&n);

   /* find paragraph start */
   backward_paragraph();
   if (is_paragraph_sep() && (CLine->next != NULL)) CLine = CLine->next;
   beg = CLine;

   forward_paragraph();
   if (CLine->next == NULL) end = NULL; else end = CLine;

   CLine = beg;
   Point = 0;
   /* Now loop formatting as we go until the end is reached */
   while(CLine != end)
     {
	eol();
	if (CLine != beg) indent_to(n);
	if (wrap_line1(1))
	  {
	     CLine = CLine->next;
	     indent_to(n);
	     continue;
	  }
	else if (CLine->next == end) break;

	next = CLine->next;
	if (next != end)
	  {
	     /* Now count the length of the word on the next line. */
	     CLine = next;
	     Point = 0;
	     trim_whitespace();
	     p = CLine->data;
	     while((*p > ' ') && (p - CLine->data < CLine->len)) p++;

	     CLine = CLine->prev;
	     eol();

	     col = calculate_column();
	     if ((p - next->data) + col < User_Vars.wrap_column - 1)
	       {
		  del();
		  ins(' ');
	       }
	     else CLine = CLine->next;
	  }
     }
   pop_spot();
   return(1);
}

int narrow_paragraph(void)
{
   int wrap, n;
   if (CBuf->modes != WRAP_MODE) return(0);
   get_current_indent(&n);
   wrap = User_Vars.wrap_column;
   if (wrap - n <= wrap/2) return(0);
   User_Vars.wrap_column -= n;
   text_format_paragraph();
   User_Vars.wrap_column = wrap;
   return(1);
}

int center_line(void)
{
   unsigned char *p, *pmax;
   int len;

   push_spot();
   (void) eol_cmd();
   p = CLine->data;
   pmax = p + CLine->len;

   while(p < pmax)
     {
	if (*p > ' ') break;
	p++;
     }
   if ((len = (int)(pmax - p)) < 0) len = 0;
   if ((len = (User_Vars.wrap_column - len) / 2) < 0) len = 0;
   indent_to(len);
   pop_spot();
   return(1);
}

int text_smart_quote(void)
{
   char c, last;
   int upd;

   if (Point) c = (char ) *(CLine->data + (Point - 1)); else c = 0;
   if (!(CBuf->modes & TEXT_MODE) || (c == '\\')) return ins_char_cmd();

   last = Last_Command_Char;
   if ((c == '(') || (c == '[') || (c == '{') || (c <= ' ') || !Point)
     Last_Command_Char = '`';
   else
     Last_Command_Char = '\'';

   upd = ins_char_cmd();
   if (last == '"') upd = ins_char_cmd();
   Last_Command_Char = last;
   return(upd);
}

#define upcase(ch) ((ch) &= 0xDF)
#define downcase(ch) ((ch) |= 0x20)

static char Word[60];

void define_word(char *w)
{
   strcpy(Word, w);
}


int forward_word()
{
   if (eolp()) return nextline(&Number_One);
   skip_whitespace();
   if (!skip_chars1(Word, 0)) skip_chars1(Word, 1);
   return(1);
}

int backward_word()
{
   if (!Point) return prevline(&Number_One);
   
   if (!bskip_chars1(Word, 0)) bskip_chars1(Word, 1);
   return(1);
}

void transform_region(char *what)
{
   int pnt, cap, tmpm;
   Line *line;
   unsigned char *p, *pmax;

   if (!check_region(&Number_One)) return;    /* spot pushed */

   pnt = Point;
   line = CLine;
   tmpm = 1; pop_mark(&tmpm);
   cap = 0;
   while(1)
     {
	p = CLine->data + Point;
	if (line != CLine) pmax = CLine->data + CLine->len;
	else pmax = CLine->data + pnt;
	while (p < pmax)
	  {
	     if (isalpha(*p))
	     switch(*what)
	       {
		  case 'u': if (*p > 'Z') upcase(*p); break;
		  case 'c': if (!cap)
		              {
				 cap = 1;
				 if (*p > 'Z') upcase(*p);
				 break;
			      }

		  case 'd': if (*p < 'a') downcase(*p); break;
		  default: if (*p < 'a') downcase(*p); else upcase(*p);
	       }
	     p++;
	  }
	register_change(0);
	if (line == CLine) break;
	CLine = CLine->next;
     }
   pop_spot();
   mark_buffer_modified(&Number_One);
}

void mark_word(void)
{
   push_mark();
   forward_word();
}

static void do_xword(char ch)
{
   mark_word();
   transform_region(&ch);
}

int upcase_word(void)
{
   do_xword('u');
   return(1);
}

int downcase_word(void)
{
   do_xword('d');
   return(1);
}

int capitalize_word(void)
{
   do_xword('c');
   return(1);
}

int delete_word(void)
{
   mark_word();
   delete_region();
   return(1);
}

