/*
 *  Copyright (c) 1992 John E. Davis  (davis@amy.tch.harvard.edu)
 *  All Rights Reserved.
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifdef ultrix
extern int system(char *);       /* used for slang function 'system' */
#else
extern int system(const char *);       /* used for slang function 'system' */
#endif


#ifdef MALLOC_DEBUG
FILE *DEBUG_OUT;

#include <alloc.h>

void FarFree(void far *buf)
{
   fprintf(DEBUG_OUT,"f: %ld\n", (long) buf);
   fflush(DEBUG_OUT);
   farfree(buf);
}

void *FarMalloc(unsigned long x)
{
   void *p;
   p = farmalloc(x);
   fprintf(DEBUG_OUT,"m: %ld %ld\n", (long) p, (long) x);
   fflush(DEBUG_OUT);
   return(p);
}

void *FarRealloc(void far *buf, unsigned long x)
{
   void far *p;
   p = farrealloc(buf, x);
   fprintf(DEBUG_OUT,"r: %ld %ld %ld\n", (long) p, (long) buf, (long) x);
   fflush(DEBUG_OUT);
   return(p);
}

void *FarCalloc(unsigned long n, unsigned long m)
{
   void far *p;
   p = farcalloc(n, m);
   fprintf(DEBUG_OUT,"c: %ld %ld %ld\n", (long) p, (long) n, (long) m);
   fflush(DEBUG_OUT);
   return(p);
}   
#endif

#ifdef msdos
#include <alloc.h>
#define FREE(buf)  farfree((void far *)(buf))
#define MALLOC(x) farmalloc((unsigned long) (x))
#define REALLOC(buf, n) farrealloc((void far *) (buf), (unsigned long) (n))
#define CALLOC(n, m) farcalloc((unsigned long) (n), (unsigned long) (m))

#define SLANG_SYSTEM_NAME "_IBMPC"
#else
#ifndef VMS
#include <malloc.h>
#define SLANG_SYSTEM_NAME "_UNIX"
#else
#define SLANG_SYSTEM_NAME "_VMS"
#endif
#ifndef VOID
#define VOID unsigned char
#endif
#define FREE(p) free((VOID *)(p))
#define MALLOC malloc
#define REALLOC realloc
#define CALLOC calloc
#endif

#include "slang.h"

#define LANG_MAX_NAME_LEN 33
/* maximum length + 1 of an identifier ==> 32 */

#define LANG_MAX_SYMBOLS 300
/* maximum number of global symbols--- slang builtin, functions, global vars */

typedef struct Lang_Name_Type
  {
     char name[LANG_MAX_NAME_LEN];
     Lang_Object_Type obj;
  }
Lang_Name_Type;

Lang_Name_Type Lang_Name_Table[LANG_MAX_SYMBOLS];
Lang_Name_Type *Lang_Local_Variable_Table;
int Local_Variable_Number;
#define MAX_LOCAL_VARIABLES 20

int Lang_Break = 0;
int Lang_Return = 0;
int Lang_Continue = 0;

/* this stack is used by the inner interpreter to execute top level
   interpreter commands which by definition are immediate so stack is
   only of maximum 4: the 2 blocks, else nnd terminator */
Lang_Object_Type Lang_Interp_Stack_Static[4];
Lang_Object_Type *Lang_Interp_Stack_Ptr = Lang_Interp_Stack_Static;
Lang_Object_Type *Lang_Interp_Stack = Lang_Interp_Stack_Static;

/* maximum size of run time stack */
#define LANG_MAX_STACK_LEN 1000
Lang_Object_Type Lang_Run_Stack[LANG_MAX_STACK_LEN];
int Lang_Run_Stack_Depth = -1;

/* Might want to increase this. */
#define MAX_LOCAL_STACK 200
Lang_Object_Type Local_Variable_Stack[MAX_LOCAL_STACK];

Lang_Object_Type *Local_Variable_Frame = Local_Variable_Stack;

volatile int Lang_Error = 0;

extern int inner_interp(Lang_Object_Type *addr);

int Lang_Defining_Function = 0;   /* true if defining a function */
Lang_Object_Type *Lang_Function_Body;
Lang_Object_Type *Lang_FBody_Ptr;
int Lang_FBody_Size = 0;

#define LANG_MAX_BLOCKS 30
/* max number of nested blocks--- was 10 but I once exceeded it! */

typedef struct Lang_Block_Type
  {
     int size;                         /* current nuber of objects malloced */
     Lang_Object_Type *body;           /* beginning of body definition */
     Lang_Object_Type *ptr;            /* current location */
  }
Lang_Block_Type;

int Lang_Defining_Block = 0;   /* true if defining a block */
Lang_Block_Type Lang_Block_Stack[LANG_MAX_BLOCKS];
Lang_Object_Type *Lang_Block_Body;
int Lang_BBody_Size;

int Lang_Block_Depth = -1;

Lang_Object_Type *Lang_Object_Ptr = Lang_Interp_Stack_Static;
/* next location for compiled obj -- points to interpreter stack initially */

/* useful macro to tell if string should be freed after its use. */
#define IS_DATA_STRING(obj)\
   ((((obj).type & 0xFF) == LANG_DATA) && (((obj).type >> 8) == STRING_TYPE))

int (*Lang_Error_Routine)(char *) = (int (*)(char *)) NULL;

/* array types */
typedef struct Array_Type
{
   int dim;			       /* # of dims (max 3) */
   int x,y,z;			       /* actual dims */
   long ptr;			       /* address of buffer */
   unsigned char type;		       /* int, float, etc... */
} Array_Type;

void lang_doerror(char *error)
{
   char err[80]; char *str;

   if (!Lang_Error) Lang_Error = -1;
   switch(Lang_Error)
     {
	case (UNDEFINED_NAME): str = "Undefined_Name"; break;
	case (SYNTAX_ERROR): str = "Syntax_Error"; break;
	case (STACK_OVERFLOW): str = "Stack_Overflow"; break;
	case (STACK_UNDERFLOW): str = "Stack_Underflow"; break;
	case (DUPLICATE_DEFINITION): str = "Duplicate_Definition"; break;
	case (TYPE_MISMATCH): str = "Type_Mismatch"; break;
	default: str = "Unknown Error.";
     }

   sprintf(err, "Lang_Error: %d: %s [%s]", Lang_Error, error, str);
   if (Lang_Error_Routine == NULL)
     fputs(err, stderr);
   else (*Lang_Error_Routine)(err);
}

int (*User_Load_File)(char *) = NULL; /* (int (*)(char *)) NULL; */

int lang_load_file(char *file)
{
   char buf[132], errbuf[255];
   int count = 0;
   FILE *fp;
   char *err;

   if (User_Load_File != NULL) return (*User_Load_File) (file);

   strcpy(buf, "Loading "); strcat(buf, file);
   puts(buf);

   if (NULL == (fp = fopen(file,"r")))
     {
	lang_doerror("Unable to load file.");
	return(0);
     }

   while(NULL != fgets(buf, 131, fp))
     {
	count++;
	err = interpret(buf);
	if (Lang_Error) 
	  {
	     sprintf(errbuf, "Error: line %d: %s", count, err);
	     puts(errbuf);
	     break;
	  }
     }

   fclose(fp);
   
   return( !Lang_Error );
}

int lang_pop(Lang_Object_Type *x)
{
   int depth = Lang_Run_Stack_Depth;

    if (depth < 0)
      {
	 Lang_Error = STACK_UNDERFLOW;
	 Lang_Run_Stack_Depth = -1;
	 return 0;
      }

    x->type = Lang_Run_Stack[depth].type;
    x->value = Lang_Run_Stack[depth].value;
    Lang_Run_Stack_Depth--;
    return(1);
}

void lang_push(Lang_Object_Type *x)
{
    int depth;

    depth = ++Lang_Run_Stack_Depth;

    /* flag it now */
    if (Lang_Run_Stack_Depth == LANG_MAX_STACK_LEN)
      Lang_Error = STACK_OVERFLOW;

    Lang_Run_Stack[depth].type = x->type;
    Lang_Run_Stack[depth].value = x->value;
}

void lang_free_branch(Lang_Object_Type *p)
{
   short type;

   while(1)
     {
        type = (p->type) & 0xFF;
	if (type == LANG_BLOCK)
	  {
	     lang_free_branch((Lang_Object_Type *) p->value);
	     FREE(p->value);
	  }
	else if (type == 0) break;
	p++;
     }
}

int lang_pop_integer(int *i)
{
   Lang_Object_Type obj;

   if (!lang_pop(&obj) || ((obj.type >> 8) != INT_TYPE))
     {
	if (!Lang_Error)
	  {
	     lang_push(&obj);
	     Lang_Error = TYPE_MISMATCH;
	  }
	return(0);
     }

   *i = (int) obj.value;
   return(1);
}

#ifdef FLOAT_TYPE
int lang_pop_float(float *x)
{
   Lang_Object_Type obj;
   unsigned char stype;

   if (!lang_pop(&obj)) return(0);
   stype = obj.type >> 8;

   if (stype == INT_TYPE) *x = (float) obj.value;
   else if (stype == FLOAT_TYPE) *x = *(float *) &obj.value;
   else
     {
	lang_push(&obj);
	Lang_Error = TYPE_MISMATCH;
	return(0);
     }
   return(1);
}

void lang_push_float(float x)
{
   Lang_Object_Type obj;

   obj.type = LANG_DATA | (FLOAT_TYPE << 8);
   obj.value = *(long *) &x;
   lang_push (&obj);
}

#endif

int pop_string(char **s, int *data)
{
   Lang_Object_Type obj;

   if (!lang_pop(&obj) || ((obj.type >> 8) != STRING_TYPE))
     {
	lang_push(&obj);
	if (!Lang_Error) Lang_Error = TYPE_MISMATCH;
	return(0);
     }

   *s = (char *) obj.value;
   /* return whether or not this should be freed after its use. */
   if ((obj.type & 0xFF) == LANG_DATA) *data = 1; else *data = 0;
   return(1);
}

void lang_push_integer(int i)
{
   Lang_Object_Type obj;

   obj.type = LANG_DATA | (INT_TYPE << 8);
   obj.value = (long) i;
   lang_push (&obj);
}

long lang_make_string(char *str)
{
   char *ptr;

   if (NULL == (ptr = (char *) MALLOC(strlen(str) + 1)))
     {
	lang_doerror("malloc error in lang_make_string.");
	return(NULL);
     }
   strcpy((char *) ptr, str);
   return((long) ptr);
}

void lang_push_string(char *t)
{
   Lang_Object_Type obj;

   if (NULL == (obj.value = (long) lang_make_string((char *) t))) return;
   obj.type = LANG_DATA | (STRING_TYPE << 8);
   lang_push(&obj);
}

void create_array(void)
{
   int dim, d, tstr_data;
   int x[3], size;
   char *tstr, t, *errstr = "Unable to create array.";
   long n, ptr;
   unsigned char type;
   Array_Type *at;
   Lang_Object_Type obj;

   if (!lang_pop_integer(&dim)) return;

   if (dim > 3)
     {
	lang_doerror("Array size not supported.");
	return;
     }

   d = dim;
   while (d--) if (!lang_pop_integer(&x[d])) return;

   if (!pop_string(&tstr, &tstr_data)) return;

   t = *tstr;
   if (tstr_data) FREE(tstr);

   switch (t)
     {
	case 'i': type = INT_TYPE; size = sizeof(int); break;
	case 's': type = STRING_TYPE; size = sizeof(char *); break;
#ifdef FLOAT_TYPE
	case 'f': type = FLOAT_TYPE; size = sizeof(float); break;
#endif
	default: lang_doerror("Unknown Array type."); return;
     }

   n = 1; d = dim;
   while (d--) n = n * x[d];

   if ((long) NULL == (ptr = (long) CALLOC(n, size)))
     {
	lang_doerror(errstr);
	return;
     }

   if (NULL == (at = (Array_Type *) MALLOC(sizeof(Array_Type))))
     {
	FREE(ptr);
	lang_doerror(errstr);
	return;
     }

   at->ptr = ptr;
   at->dim = dim;
   at->x = x[0]; at->y = x[1]; at->z = x[2];
   at->type = type;

   obj.type = LANG_DATA | (ARRAY_TYPE << 8);
   obj.value = (long) at;
   lang_push(&obj);
}

void free_array(void)
{
   Lang_Object_Type obj;

   if (!lang_pop(&obj)) return;

   if ((obj.type >> 8) != ARRAY_TYPE)
     {
	Lang_Error = TYPE_MISMATCH;
	return;
     }

   FREE( ((Array_Type *) obj.value)->ptr );
   FREE(obj.value);
}

Array_Type *pop_array(void)
{
   Lang_Object_Type obj;

   if (!lang_pop(&obj)) return(NULL);

   if ((obj.type >> 8) != ARRAY_TYPE)
     {
	Lang_Error = TYPE_MISMATCH;
	return(NULL);
     }

   return (Array_Type *) obj.value;
}

unsigned int compute_array_offset(Array_Type *at)
{
   int elem[3], el, x[3], d, dim;
   unsigned int off;

   dim = at->dim;
   x[0] = at->x; x[1] = at->y; x[2] = at->z;

   d = dim;

   while (d--)
     {
	if (!lang_pop_integer(&el)) return(0);
	if ((el > x[d]) || (el < 1))
	  {
	     lang_doerror("Array dims out of bounds.");
	     return(-1);
	  }
	elem[d] = el - 1;
     }

   off = 0;
   d = 0;
   while (d < dim)
     {
	off += off * x[d] + elem[d];
	d++;
     }
   return(off);
}

void array_getelem()
{
   Array_Type *at;
   unsigned char *p;
   unsigned int off;

   if (NULL == (at = pop_array())) return;

   off = compute_array_offset(at);
   if (Lang_Error) return;

   p = (unsigned char *) at->ptr;
   switch(at->type)
     {
	case INT_TYPE: lang_push_integer((int) *(((int *) p) + off)); break;
	case STRING_TYPE: lang_push_string((char *) *(((char **)p) + off)); break;
#ifdef FLOAT_TYPE
	case FLOAT_TYPE: lang_push_float((float) *(((float *)p) + off)); break;
#endif
	default: lang_doerror("Internal Error in array element.");
     }
   return;
}

void array_putelem()
{
   Array_Type *at;
   unsigned int off;
   int sdat, i, *ip;
   char *str, *newstr, **sp;
   unsigned char *p;
#ifdef FLOAT_TYPE
   float f, *fp;
#endif

   if (NULL == (at = pop_array())) return;

   off = compute_array_offset(at);
   if (Lang_Error) return;

   p = (unsigned char *) at->ptr;
   switch(at->type)
     {
	case INT_TYPE:
	  if (!lang_pop_integer(&i)) return;
	  ip = (int *) (off + (int *) p);
	  *ip = i; break;

	case STRING_TYPE:
	  if (!pop_string(&str, &sdat)) return;
	  newstr = (char *) lang_make_string(str);
	  if (sdat) FREE(str);
	  sp = (char **)(off + (char **) p);

	  if (NULL != *sp) FREE(*sp);
	  *sp = newstr;
	  break;

#ifdef FLOAT_TYPE
	case FLOAT_TYPE:
	  if (!lang_pop_float(&f)) return;
	  fp = off + (float *) p;
	  *fp = f;
	  break;
#endif
	default: lang_doerror("Corrupted Array.");
     }
   return;
}

/* builtin stack manipulation functions */

void do_pop(void)
{
   Lang_Object_Type x;
   if (!lang_pop(&x)) return;

   if (IS_DATA_STRING(x)) FREE(x.value);
}

int do_dup(void)
{
   Lang_Object_Type x;
   if (!lang_pop(&x)) return(0);
   lang_push(&x);
   if ((x.type >> 8) == STRING_TYPE)
     lang_push_string((char *) x.value); else lang_push (&x);
   return(1);
}


void do_strcat(void)
{
   char *a, *b, *c;
   int len;
   Lang_Object_Type obj;
   int adata, bdata;

   if (!pop_string(&b, &bdata) || !pop_string(&a, &adata)) return;

   len = strlen(a) + strlen(b);
   if (NULL == (c = (char *) MALLOC(len + 1)))
     {
	lang_doerror("Lang: Malloc error.");
	return;
     }
   strcpy(c, a);
   strcat(c, b);

   if (adata) FREE(a);
   if (bdata) FREE(b);

   /* instead of going throug push string, push it directly */
   obj.type = LANG_DATA | (STRING_TYPE << 8);
   obj.value = (long) c;
   lang_push(&obj);
}

/* returns the position of substrin in a string or null */
void do_issubstr(void)
{
   char *a, *b, *c;
   int adata, bdata, n;

   if (!pop_string(&b, &bdata) || !pop_string(&a, &adata)) return;

   if (NULL == (c = strstr(a, b))) n = 0; else n = 1 + (int) (c - a);

   if (adata) FREE(a);
   if (bdata) FREE(b);
   lang_push_integer (n);
}

/* returns to stack string at pos n to n + m of a */
void do_substr(void)
{
   char *a;
   int adata, n, m;
   char b[255];

   if (!lang_pop_integer(&m) || !lang_pop_integer(&n) || (!pop_string(&a, &adata))) return;

   *b = 0;
   if (m > 0)
     {
	strncpy(b, a + (n - 1), 254);
	if (m > 255) m = 255;
	b[m] = 0;
     }
   if (adata) FREE(a);
   lang_push_string(b);
}

void do_strsub(void)
{
   char *a;
   int adata, n, m;
   char b[255];

   if (!lang_pop_integer(&m) || !lang_pop_integer(&n) || (!pop_string(&a, &adata))) return;

   strncpy(b, a, 254);
   b[254] = 0;
   if (adata) FREE(a);
   if ((n < 1) || (n > 254)) n = 254;
   b[n-1] = (char) m;
   lang_push_string(b);
}

void do_strcmp(void)
{
   char *a, *b;
   int adata, bdata, i;

   if (!pop_string(&b, &bdata) || !pop_string(&a, &adata)) return;

   i = strcmp(a, b);

   if (adata) FREE(a);
   if (bdata) FREE(b);
   lang_push_integer (i);
}

void do_strlen(void)
{
   char *a;
   int adata, i;

   if (!pop_string(&a, &adata)) return;

   i = strlen(a);

   if (adata) FREE(a);
   lang_push_integer (i);
}

int do_isdigit(char *what)
{
   if ((*what >= '0') && (*what <= '9')) return(1); else return(0);
}

/* convert object to integer form */
void do_int(void)
{
   Lang_Object_Type x;
   int i;
   unsigned char stype;

   if (!lang_pop(&x)) return;
   stype = x.type >> 8;

   if (stype == INT_TYPE)
     {
	lang_push(&x);
	return;
     }

   else if (stype == STRING_TYPE)
     {
	i = (int) *(char *) x.value;
	if (IS_DATA_STRING(x)) FREE(x.value);
     }
#ifdef FLOAT_TYPE
   else if (stype == FLOAT_TYPE)
     {
	i = (int) *(float *) &x.value;
     }
#endif
   else
     {
	Lang_Error = TYPE_MISMATCH;
	return;
     }
   lang_push_integer(i);
}

/* convert integer to a sring of length 1 */
void do_char(void)
{
   char ch, buf[2];
   int x;

   if (!lang_pop_integer(&x)) return;

   ch = (char) x;
   buf[0] = ch;
   buf[1] = 0;
   lang_push_string((char *) buf);
}

/* format object into a string */
void do_string(void)
{
   Lang_Object_Type x;
   char buf[255];

   if (!lang_pop(&x)) return;

   if ((x.type >> 8) == STRING_TYPE)
     {
	lang_push(&x);
	return;
     }
#ifndef FLOAT_TYPE
   sprintf(buf, "%ld", (long) x.value);
#else
   sprintf(buf, "%.6g",  (float) *(float *) &x.value);
#endif
   lang_push_string((char *) buf);
}

void lang_push_variable(Lang_Object_Type *obj)
{
   unsigned char subtype;
   Lang_Object_Type new;
   long value;

   subtype = obj->type >> 8;
   value = obj->value;

   if (subtype == STRING_TYPE)
     {
	lang_push_string((char *) value);
	return;
     }
   new.type = obj->type;
   new.value = value;
   lang_push(&new);
}

/* local and global variable assignments */

/* value contains either the offset of data for local variables or
   location of object_type for global ones.  For strings, we have to be
   careful.  Literal (constant) strings which are already attached to these
   variables are not to be freed--- only those of type data (dynamic).
   There is no need to create new strings since they come from the stack.

   Note that strings appear on the stack in two forms:  literal and dynamic.
   Literal strings are constants.  Dynamic ones are created by, say, dup, etc.
   They are freed only by routines which eat them.  These routines must
   check to see if they are not literal types before freeing them.  The only
   other way they are freed is when they are on the local variable stack, e.g.,
   (assigned to local variables) and the function exits freeing them.

   Define Macro to do this:  (defined above)

#define IS_DATA_STRING(obj)\
   ((((obj).type & 0xFF) == LANG_DATA) && (((obj).type >> 8) == STRING_TYPE))
*/
void lang_do_eqs(Lang_Object_Type *obj)
{
   int i;
   unsigned char type;
   Lang_Object_Type *addr;
   long val;

   type = obj->type >> 8;

   switch(type)
     {
	case LANG_IEQS:
	  val = ((Lang_Name_Type *) (obj->value))->obj.value;
	  if (!lang_pop_integer(&i)) return;
	  *(int *) val = i;
	  return;
	case LANG_GEQS:
	  val = ((Lang_Name_Type *) (obj->value))->obj.value;
	  addr = (Lang_Object_Type *) val; break;
	default:
	  val = (long) obj->value;
	  addr = Local_Variable_Frame - (int) val;
     }

   /* is addr pointing at a volatile string type? */
   if (IS_DATA_STRING(*addr)) FREE(addr->value);

   lang_pop(addr);
}

/* pop a data item from the stack and return a pointer to it.
   Strings are not freed from stack so use another routine to do it.

   In addition, I need to make this work with the array types.  */

long *pop_pointer(unsigned short *type)
{
   Lang_Object_Type obj;
   long *val;
   int depth;

   if (!lang_pop(&obj)) return(NULL);

   /* use this because the stack is static but obj is not.
      do not even try to make it static either. See the intrinsic
      routine for details */
   depth = Lang_Run_Stack_Depth + 1;
   val = (long *) &Lang_Run_Stack[depth].value;
   *type = Lang_Run_Stack[depth].type;

   /* The assumption is that only data and literals are on stack */
   if ((*type >> 8) == STRING_TYPE) return((long *) *val);
   return((long *) val);
}

/* intrinsic functions are passed parameters by reference. The function
   pointer is given by obj->value. */
void lang_do_intrinsic(Lang_Object_Type *objf)
{
   long (*f)();
   void (*fv)();
   long ret;
   long *p1, *p2, *p3, *p4, *p5;
   unsigned short p1type, p2type, p3type, p4type, p5type;
   unsigned char type;
   Lang_Object_Type obj;
   int argc;
#ifdef FLOAT_TYPE
   float xf;
   float (*ff)();
   ff = (float (*)()) objf->value;
#endif

   f = (long (*)()) objf->value;
   fv = (void (*)()) objf->value;

   p5type = p4type = p3type = p2type = p1type = 0;
   argc = LANG_INTRINSIC_ARGC(*objf);
   type = LANG_INTRINSIC_TYPE(*objf);

   switch (argc)
     {
	case 5: p5 = pop_pointer(&p5type);
	case 4: p4 = pop_pointer(&p4type);
	case 3: p3 = pop_pointer(&p3type);
	case 2: p2 = pop_pointer(&p2type);
	case 1: p1 = pop_pointer(&p1type);
     }

   if (!Lang_Error) switch (argc)
     {

	case 0:
	  if (type == VOID_TYPE) (void) (*fv)();
#ifdef FLOAT_TYPE
	  else if (type == FLOAT_TYPE) xf = (*ff)();
#endif
	  else ret = (*f)();
	  break;

	case 1:
	  if (type == VOID_TYPE) (void) (*fv)(p1);
#ifdef FLOAT_TYPE
	  else if (type == FLOAT_TYPE) xf = (*ff)(p1);
#endif
	  else ret = (*f)(p1);
	  break;

	case 2:
	  if (type == VOID_TYPE) (void) (*fv)(p1, p2);
#ifdef FLOAT_TYPE
	  else if (type == FLOAT_TYPE) xf = (*ff)(p1, p2);
#endif
	  else ret = (*f)(p1, p2);
	  break;

	case 3:
	  if (type == VOID_TYPE) (void) (*fv)(p1, p2, p3);
#ifdef FLOAT_TYPE
	  else if (type == FLOAT_TYPE) xf = (*ff)(p1, p2, p3);
#endif
	  else ret = (*f)(p1, p2, p3);
	  break;

	case 4:
	  if (type == VOID_TYPE) (void) (*fv)(p1, p2, p3, p4);
#ifdef FLOAT_TYPE
	  else if (type == FLOAT_TYPE) xf = (*ff)(p1, p2, p3, p4);
#endif
	  else ret = (*f)(p1, p2, p3, p4);
	  break;

	case 5:
	  if (type == VOID_TYPE) (void) (*fv)(p1, p2, p3, p4, p5);
#ifdef FLOAT_TYPE
	  else if (type == FLOAT_TYPE) xf = (*ff)(p1, p2, p3, p4, p5);
#endif
	  else ret = (*f)(p1, p2, p3, p4, p5);
	  break;

	default:
	  lang_doerror("Function type not supported.");
	  Lang_Error = -1;
	  break;
     }

   if (((p5type & 0xFF) == LANG_DATA) && ((p5type >> 8) == STRING_TYPE)) FREE(p5);
   if (((p4type & 0xFF) == LANG_DATA) && ((p4type >> 8) == STRING_TYPE)) FREE(p4);
   if (((p1type & 0xFF) == LANG_DATA) && ((p1type >> 8) == STRING_TYPE)) FREE(p1);
   if (((p2type & 0xFF) == LANG_DATA) && ((p2type >> 8) == STRING_TYPE)) FREE(p2);
   if (((p3type & 0xFF) == LANG_DATA) && ((p3type >> 8) == STRING_TYPE)) FREE(p3);

   if (Lang_Error) return;

   if (type == VOID_TYPE) return;
   else if (type == STRING_TYPE)
     {
	lang_push_string((char *) ret);
	return;
     }
   else if (type == INT_TYPE)
     {
#ifdef msdos
	/* longs are 4 bytes and ints are 2 so try this: */
	obj.value = *(int *) &ret;
#else
	obj.value = ret;
#endif
     }
#ifdef FLOAT_TYPE
   else if (type == FLOAT_TYPE)	obj.value = *(long *) &xf;
#endif
   else Lang_Error = TYPE_MISMATCH;

   obj.type = LANG_DATA | (type << 8);
   lang_push (&obj);
}

void lang_do_loops(unsigned char type, long addr)
{
   int ctrl, first, last, i, dir, one = 0;
   Lang_Object_Type *obj1, *obj2, *block;

   block = (Lang_Object_Type *) addr;
   obj1 = (Lang_Object_Type *) block->value;

   switch (type)
     {
      case LANG_WHILE:

	/* we need 2 blocks: first is the control, the second is code */
	block++;
	if ((block->type & 0xFF) != LANG_BLOCK)
	  {
	     lang_doerror("Block needed for while.");
	     return;
	  }
	obj2 = (Lang_Object_Type *) block->value;

	while(!Lang_Error)
	  {
	     inner_interp(obj1);
	     if (Lang_Error || !lang_pop_integer(&ctrl)) return;
	     if (!ctrl) break;
	     inner_interp(obj2);
	     Lang_Continue = 0;
	     if (Lang_Break) break;
	  }
	break;

      case LANG_FOR:  /* 3 elements: first, last, step */
	if (!lang_pop_integer(&ctrl)) return;
	if (!lang_pop_integer(&last)) return;
	if (!lang_pop_integer(&first)) return;
	if (ctrl < 0) dir = -1; else dir = 1;

	for (i = first; i != last + dir; i += ctrl)
	  {
	     if (Lang_Error) return;
	     lang_push_integer(i);
	     inner_interp(obj1);
	     Lang_Continue = 0;
	     if (Lang_Break) break;
	  }
	break;

      case LANG_LOOP:
	if (!lang_pop_integer(&ctrl)) return;
      case LANG_FOREVER:
	if (type == LANG_FOREVER) one = 1;
	while (one || (ctrl-- > 0))
	  {
	     if (Lang_Error) break;
	     inner_interp(obj1);
	     Lang_Continue = 0;
	     if (Lang_Break) break;
	  }
	break;

      default:  lang_doerror("Unknown loop type.");
     }
   Lang_Break = Lang_Continue = 0;
}

void lang_do_ifs(Lang_Object_Type *addr)
{
   unsigned char type;
   long val;
   int test, doit;

   type = addr->type >> 8;
   if (!lang_pop_integer(&test)) return;

   addr--;

   doit = 1;         /* optimistic choice */
   switch (type)
     {
	case LANG_IF:  if (!test) doit = 0; break;
	case LANG_IFNOT: if (test) doit = 0; break;
	case LANG_ELSE: if (test) addr--; break;
	default:
	  lang_doerror("Unknown IF type!");
     }

   if ((addr->type & 0xFF) != LANG_BLOCK)
     {
	lang_doerror("Block needed.");
	return;
     }

   if (doit)
     {
	val = addr->value;
	inner_interp((Lang_Object_Type *) val);
     }
}

void lang_do_else(unsigned char type, long addr)
{
   long val;
   int test, status;
   char *str = NULL;
   Lang_Object_Type *obj, cobj;

   obj = (Lang_Object_Type *) addr;

   if (type == LANG_SWITCH)
     {
	if (!lang_pop(&cobj)) return;
	if (IS_DATA_STRING(cobj)) str = (char *) cobj.value;
     }
   
   while((obj->type & 0xFF) == LANG_BLOCK)
     {
	val = obj->value;
	if (type == LANG_SWITCH) 
	  {
	     if (str == NULL) lang_push(&cobj); else lang_push_string(str);
	  }
	
	status = inner_interp((Lang_Object_Type *) val);
	if (Lang_Error || Lang_Continue || Lang_Break) return;
	if (type == LANG_SWITCH)
	  {
	     if (status) break;
	  }
	
	else if (!lang_pop_integer(&test)) return;
	if (((type == LANG_ANDELSE) && !test)
	    || ((type == LANG_ORELSE) && test)) break;
	obj++;
     }
   if (type != LANG_SWITCH) lang_push_integer(test);
   else if (str != NULL) FREE(str);
   return;
}

/* inner interpreter */
int inner_interp(Lang_Object_Type *addr)
{
   Lang_Object_Type *frame;
   unsigned short type;
   unsigned char stype;
   int x, y, z, n_locals;
   long val, block = 0;
   Lang_Name_Type *name_entry;

#ifdef FLOAT_TYPE
   float xf, yf, zf;
#endif

   while(type = addr->type, type)
     {
	if ((Lang_Return) || Lang_Break)
	  {
	     Lang_Break = 1;
	     return(1);
	  }
	if (Lang_Continue) return(1);
	val = addr->value;
	stype = type >> 8;

	switch (type & 0xFF)
	  {
	   case LANG_INTRINSIC:
	     val = (long) &(((Lang_Name_Type *) val)->obj);
	     lang_do_intrinsic((Lang_Object_Type *) val);
	     break;
	   case LANG_BLOCK:
	     if (!block) block = (long) addr;
	     break;  /* don not know what to do yet */

	   case LANG_FUNCTION:
	     /* val is a reference to the name table, so get its obj: */
	     name_entry = (Lang_Name_Type *) val;
	     n_locals = (name_entry->obj.type) >> 8;

	     /* need loaded?  */
	     if (n_locals == 255)
	       {
		  if (!lang_load_file((char *) name_entry->obj.value)) break;
		  n_locals = (name_entry->obj.type) >> 8;
	       }

	     val = name_entry->obj.value;
	     /* set new stack frame */
	     frame = Local_Variable_Frame;
	     while(n_locals--)
	       {
		  Local_Variable_Frame++;
		  if (Local_Variable_Frame - Local_Variable_Stack > MAX_LOCAL_STACK)
		    {
		       lang_doerror("Local Variable Stack Overflow!");
		       return(0);
		    }

		  Local_Variable_Frame->type = 0;
	       }
	     inner_interp((Lang_Object_Type *) val);

	     /* free local variables.... */
	     while(Local_Variable_Frame > frame)
	       {
		  if (IS_DATA_STRING(*Local_Variable_Frame))
		    FREE (Local_Variable_Frame->value);
		  Local_Variable_Frame--;
	       }
	     Lang_Return = Lang_Break = 0;
	     break;

	    case LANG_LVARIABLE:      /* make val point to local stack */
	       val = (long) (Local_Variable_Frame - (int) val);
	       lang_push_variable((Lang_Object_Type *) val);
	       break;
	     case LANG_GVARIABLE:
	       val = ((Lang_Name_Type *) val)->obj.value;
	       lang_push_variable((Lang_Object_Type *) val);
	       break;
	     case LANG_IVARIABLE:
	       val = ((Lang_Name_Type *) val)->obj.value;
	       switch(stype)
	       {
		  case STRING_TYPE: lang_push_string((char *) val); break;
		  case INT_TYPE: lang_push_integer(*(int *) val); break;
		  default: lang_doerror("Unsupported Type!");
	       }

	       break;
             case LANG_LITERAL:        /* a constant */
	       lang_push(addr);
	       break;

	   case LANG_LABEL: 
	     if (!lang_pop_integer(&z) || !z) return(0);
	     break;

	     case LANG_DIRECTIVE:
	       if (stype & LANG_EQS_MASK)
	         {
		    lang_do_eqs((Lang_Object_Type *) addr);
		    break;
		 }

	       if (!block) lang_doerror("No Blocks!");
	       else if (stype & LANG_LOOP_MASK) lang_do_loops(stype, block);
	       else if (stype & LANG_IF_MASK) lang_do_ifs(addr);
	       else if (stype & LANG_ELSE_MASK) lang_do_else(stype, block);
	       else lang_doerror("Unknown directive!");
	       block = 0;
	       break;

	     case LANG_RETURN: Lang_Return = Lang_Break = 1; return(1);
	     case LANG_BREAK: Lang_Break = 1; return(1);
	     case LANG_CONTINUE: Lang_Continue = 1; return(1);

	   case LANG_UNARY: 
#ifndef FLOAT_TYPE
	     if (!lang_pop_integer(&z)) return(0);
	     switch (stype)
	       {
		  case LANG_NOT:  z = !z; break;
		  case LANG_BNOT:  z = ~z; break;
		  case LANG_CHS:  z = -z; break;
		  default: Lang_Error = -2; return(0);
	       }
	     lang_push_integer(z);
	     break;			/* unary */
#else
	     if (stype == LANG_CHS)
	       {
		  if (!lang_pop_float(&zf)) return(0);
		  zf = -zf;
		  lang_push_float(zf);
		  break;
	       }
	     else
	       {
		  if (! lang_pop_integer(&z)) return(0);
		  if (stype == LANG_NOT) z = !z;
		  else if (stype == LANG_BNOT) z = ~z;
		  else 
		    {
		       Lang_Error = -2;
		       return(0);
		    }
		  lang_push_integer(z);
	       }
	     
	     break;
#endif	     
	     case LANG_CMP:
	     case LANG_BINARY:
	       z = 0;
#ifndef FLOAT_TYPE
	       if (!lang_pop_integer(&y) || !lang_pop_integer(&x)) return(0);
	       switch (stype)
		 {
		    case LANG_EQ: if (x == y) z = 1; break;
		    case LANG_NE: if (x != y) z = 1; break;
		    case LANG_GT: if (x > y) z = 1; break;
		    case LANG_GE: if (x >= y) z = 1; break;
		    case LANG_LT: if (x < y) z = 1; break;
		    case LANG_LE: if (x <= y) z = 1; break;
		    case LANG_OR: if (x || y) z = 1; break;
		    case LANG_AND: if (x && y) z = 1; break;
		    case LANG_BAND: z = x & y; break;
		    case LANG_BXOR: z = x ^ y; break;
		    case LANG_BOR: z = x | y; break;
		    case LANG_PLUS: z = x + y; break;
		    case LANG_MINUS: z = x - y; break;
		    case LANG_TIMES: z = x * y; break;
		    case LANG_DIVIDE: z = x / y; break;   /* y == 0? */
		    case LANG_SHL: z = x << y; break;
		    case LANG_SHR: z = x >> y; break;
		    default:
		      Lang_Error = -1;
		      return(0);
		   }
		 lang_push_integer(z);
		 break;			/* binary */
#else /* FLOAT_TYPE */
	       if (!lang_pop_float(&yf) || !lang_pop_float(&xf)) return(0);
	       switch (stype)
		 {
		    /* case LANG_SHR, LANG_SHL needs done */
		    case LANG_EQ: if (xf == yf) z = 1; break;
		    case LANG_NE: if (xf != yf) z = 1; break;
		    case LANG_GT: if (xf > yf) z = 1; break;
		    case LANG_GE: if (xf >= yf) z = 1; break;
		    case LANG_LT: if (xf < yf) z = 1; break;
		    case LANG_LE: if (xf <= yf) z = 1; break;
		    case LANG_OR: if (xf || yf) z = 1; break;
		    case LANG_AND: if (xf && yf) z = 1; break;
		    case LANG_PLUS: zf = xf + yf; break;
		    case LANG_MINUS: zf = xf - yf; break;
		    case LANG_TIMES: zf = xf * yf; break;
		    case LANG_DIVIDE: zf = xf / yf; break;   /* y == 0? */
		    default:
		      Lang_Error = -1;
		      return(0);
		   }
		 if (type == LANG_CMP) lang_push_integer(z);
		 else lang_push_float(zf);
		 break;			/* binary */
#endif /* float */
	     default: lang_doerror("Run time error.");
	  }
	if (Lang_Error) break;
	addr++;
     }
   return(1);
}

Lang_Name_Type *lang_locate_name_in_table(char *name, Lang_Name_Type *table)
{
   char *nm;
   int max;

   if (table == Lang_Local_Variable_Table) max = MAX_LOCAL_VARIABLES;
   else max = LANG_MAX_SYMBOLS;

   while(max && (nm = table->name, *nm != 0))
     {
	if (!strcmp(nm,name)) break;
	table++;
	max--;
     }
   if (!max) return(NULL);
   return(table);
}

Lang_Name_Type *lang_locate_name(char *name)
{
   Lang_Name_Type *table;

   table = Lang_Local_Variable_Table;

   if (table != NULL) table = lang_locate_name_in_table(name, table);
   if ((table == NULL) || (*table->name == 0))
     {
	table = Lang_Name_Table;
	table = lang_locate_name_in_table(name, table);
     }
   return(table);
}

void lang_add_name(char *name, long addr, unsigned short type)
{
   Lang_Name_Type *entry;
   unsigned char stype;

   if (strlen(name) >= LANG_MAX_NAME_LEN)
     {
	lang_doerror("Name too long.");
	return;
     }

   if (NULL == (entry = lang_locate_name(name))) return;  /* table full */

   if (*entry->name != 0)
     {
	stype = entry->obj.type & 0xFF;
	if ((stype != LANG_INTRINSIC) && (stype != LANG_GVARIABLE)
	    && (stype != LANG_IVARIABLE))
	  {
	     /* free its function or block definition only if not autoloaded */
	     if (! ((stype == LANG_FUNCTION) && ((entry->obj.type >> 8) == 255)))
	       lang_free_branch((Lang_Object_Type *) entry->obj.value);

	     FREE(entry->obj.value);
	  }
     }
   else strcpy(entry->name, name);

   entry->obj.value = (long) addr;
   entry->obj.type = type;
}

void lang_autoload(char *name, char *file)
{
   unsigned short type;
   long f;

   type = LANG_FUNCTION | (255 << 8);
   f = lang_make_string(file);

   lang_add_name(name, f, type);
}

void lang_define_function(char *name)
{
   long addr;
   unsigned short type;

   Lang_Defining_Function = 0;
   addr = (long) Lang_Function_Body;
   type = LANG_FUNCTION | (Local_Variable_Number << 8);
   if (name != NULL) lang_add_name(name, addr, type);
   if (Lang_Local_Variable_Table != NULL) FREE(Lang_Local_Variable_Table);
   Lang_Local_Variable_Table = NULL;
   Local_Variable_Number = 0;

   /* terminate function */
   Lang_Object_Ptr->type = 0;
   Lang_Object_Ptr = Lang_Interp_Stack_Ptr;   /* restore pointer */
}

/* call inner interpreter or return for more */
void lang_try_now(void)
{
   Lang_Object_Type *old_stack, *new_stack, *old_stack_ptr, *old_int_stack_ptr;

   if (Lang_Defining_Function || Lang_Defining_Block)
     {
	Lang_Object_Ptr++;
	return;
     }

   /* This is the entry point into the inner interpreter.  As a result, it
      is also the exit point of the inner interpreter.  So it is necessary to
      clean up if there was an error.
    */

   (Lang_Object_Ptr + 1)->type = 0;  /* so next command stops after this */

   /* now before entering the inner interpreter, we make a new stack so that
      we are able to be reentrant */
   new_stack = (Lang_Object_Type *) CALLOC(4, sizeof(Lang_Object_Type));
   if (new_stack == NULL)
     {
	lang_doerror("malloc error.");
	return;
     }

   /* remember these values */
   old_int_stack_ptr = Lang_Interp_Stack_Ptr;
   old_stack_ptr = Lang_Object_Ptr;
   old_stack = Lang_Interp_Stack;

   /* new values for reentrancy */
   Lang_Interp_Stack_Ptr = Lang_Object_Ptr = Lang_Interp_Stack = new_stack;

   /* now do it */
   inner_interp(old_stack);

   /* we are back so restore old pointers */
   Lang_Interp_Stack_Ptr = old_int_stack_ptr;
   Lang_Object_Ptr = old_stack_ptr;
   Lang_Interp_Stack = old_stack;

   /* now free blocks from the current interp_stack */

   while(Lang_Object_Ptr != Lang_Interp_Stack)
     {
	/* note that top object is not freed since it was not malloced */
	Lang_Object_Ptr--;
	/* warning--- There is a memory leak here for blocks which are nested;
	   I only free the root block.  I should have a volatile block type! */
	FREE(Lang_Object_Ptr->value);
     }

   /* now free up the callocd stack. */
   FREE(new_stack);
}

int lang_exec(char *name)
{
   Lang_Name_Type *entry;
   short type;

   if (NULL == (entry = lang_locate_name(name))) return(0);  /* table full */
   if (*entry->name == 0) return(0);

   type = Lang_Object_Ptr->type = entry->obj.type;
   if (type == LANG_LVARIABLE)
     Lang_Object_Ptr->value = (long) entry->obj.value;
   else Lang_Object_Ptr->value = (long) entry;

   lang_try_now();
   return(1);
}

#define eqs(a,b) !strcmp(a,b)

int lang_try_binary(char *t)
{
   char ssub;
   unsigned char sub, type;
   ssub = 0;

   if (eqs(t, "+")) ssub = -LANG_PLUS;
   else if (eqs(t, "-")) ssub = -LANG_MINUS;
   else if (eqs(t, "*")) ssub = -LANG_TIMES;
   else if (eqs(t, "/")) ssub = -LANG_DIVIDE;
   else if (eqs(t, "==")) ssub = LANG_EQ;
   else if (eqs(t, "!=")) ssub = LANG_NE;
   else if (eqs(t, ">")) ssub = LANG_GT;
   else if (eqs(t, ">=")) ssub = LANG_GE;
   else if (eqs(t, "<")) ssub = LANG_LT;
   else if (eqs(t, "<=")) ssub = LANG_LE;
   else if (eqs(t, "and")) ssub = LANG_AND;
   else if (eqs(t, "or")) ssub = LANG_OR;
   else if (eqs(t, "shl")) ssub = LANG_SHL;
   else if (eqs(t, "shr")) ssub = LANG_SHR;
   else if (eqs(t, "xor")) ssub = LANG_BXOR;
   else if (eqs(t, "&")) ssub = LANG_BAND;
   else if (eqs(t, "|")) ssub = LANG_BOR;
   else return(0);

   if (ssub < 0)
     {
	ssub = -ssub;
	type = LANG_BINARY;
     }
   else type = LANG_CMP;
   sub = (unsigned char) ssub;

   Lang_Object_Ptr->type = type | (sub << 8);
   Lang_Object_Ptr->value = 0;         /* not used */

   lang_try_now();
   return(1);
}

int lang_try_unary(char *t)
{
   unsigned char ssub, type;

   if (eqs(t, "~")) ssub = LANG_BNOT;
   else if (eqs(t, "not")) ssub = LANG_NOT;
   else if (eqs(t, "chs")) ssub = LANG_CHS;
   else return(0);

   type = LANG_UNARY;

   Lang_Object_Ptr->type = type | (ssub << 8);
   Lang_Object_Ptr->value = 0;         /* not used */

   lang_try_now();
   return(1);
}

void lang_begin_function(void)
{
   if (Lang_Defining_Function || Lang_Defining_Block)
     {
	lang_doerror("Function nesting illegal.");
	return;
     }

   Lang_Defining_Function = 1;

   /* make initial size for 10 things */
   Lang_FBody_Size = 10;    /* 80 bytes */
   if (NULL == (Lang_Function_Body = (Lang_Object_Type *)
          CALLOC(Lang_FBody_Size, sizeof(Lang_Object_Type))))
     {
	lang_doerror("Calloc error defining function.");
	return;
     }
   /* function definitions should be done only at top level so it should be
      safe to do this: */
   Lang_Interp_Stack_Ptr = Lang_Object_Ptr;
   Lang_Object_Ptr = Lang_FBody_Ptr = Lang_Function_Body;
   return;
}

void lang_end_block(void)
{
   Lang_Object_Type *node, *branch;
   Lang_Block_Depth--;

   /* terminate the block */
   Lang_Object_Ptr->type = 0;

   branch = Lang_Block_Body;

   if (Lang_Block_Depth == -1)         /* done */
     {
	if (Lang_Defining_Function)
	  {
	     node = Lang_FBody_Ptr++;
	  }
	else node = Lang_Interp_Stack_Ptr;   /* on small stack */
     }
   else                                /* pop previous block */
     {
	Lang_BBody_Size = Lang_Block_Stack[Lang_Block_Depth].size;
	Lang_Block_Body = Lang_Block_Stack[Lang_Block_Depth].body;
	node = Lang_Block_Stack[Lang_Block_Depth].ptr;
     }

   node->type = LANG_BLOCK;
   node->value = (long) branch;
   Lang_Object_Ptr = node + 1;
   Lang_Defining_Block--;
}

void lang_begin_block(void)
{
   if (Lang_Block_Depth == LANG_MAX_BLOCKS - 1)
     {
	/* "Max Block Nesting exceeded." */
	Lang_Error = -1;
	return;
     }
   /* push the current block onto the stack */
   if (Lang_Block_Depth > -1)
     {
	Lang_Block_Stack[Lang_Block_Depth].size = Lang_BBody_Size;
	Lang_Block_Stack[Lang_Block_Depth].body = Lang_Block_Body;
	Lang_Block_Stack[Lang_Block_Depth].ptr = Lang_Object_Ptr;
     }

   /* otherwise this is first block so save function pointer */
   else if (Lang_Defining_Function) Lang_FBody_Ptr = Lang_Object_Ptr;
   else Lang_Interp_Stack_Ptr = Lang_Object_Ptr;

   Lang_BBody_Size = 5;    /* 40 bytes */
   if (NULL == (Lang_Block_Body = (Lang_Object_Type *)
                   CALLOC(Lang_BBody_Size, sizeof(Lang_Object_Type))))
      {
	 lang_doerror("Malloc error defining block.");
	 return;
      }
   Lang_Block_Depth++;
   Lang_Defining_Block++;
   Lang_Object_Ptr = Lang_Block_Body;
   return;
}

/* see if token is a directive, and add it to current block/function */
int try_directive(char *t, int *flag)
{
   unsigned char sub = 0;
   unsigned short type = LANG_DIRECTIVE;
   int flag_save;

   if (eqs(t, "while")) sub = LANG_WHILE;
   else if (eqs(t, "forever")) sub = LANG_FOREVER;
   else if (eqs(t, "loop")) sub = LANG_LOOP;
   else if (eqs(t, "for")) sub = LANG_FOR;
   else if (eqs(t, "if")) sub = LANG_IF;
   else if (eqs(t, "!if")) sub = LANG_IFNOT;
   else if (eqs(t, "else")) sub = LANG_ELSE;
   else if (eqs(t, "andelse")) sub = LANG_ANDELSE;
   else if (eqs(t, "orelse")) sub = LANG_ORELSE;
   else if (eqs(t, "switch")) sub = LANG_SWITCH;
   
   /* rest valid only if flag is zero */
   else if (*flag) return(0);
   else
     {
	if (eqs(t, "continue")) type = LANG_CONTINUE;
	else if (eqs(t, "break")) type = LANG_BREAK;
	else if (eqs(t, "return")) type = LANG_RETURN;
	else return(0);
	*flag = 1;
     }
   
   Lang_Object_Ptr->type = type | (sub << 8);
   Lang_Object_Ptr->value = 0;         /* not used */

   flag_save = *flag; *flag = 0;
   lang_try_now();
   *flag = flag_save;

   return(1);
}

long lang_make_object(void)
{
   Lang_Object_Type *obj;

   obj = (Lang_Object_Type *) MALLOC(sizeof(Lang_Object_Type));
   if (NULL == obj)
     {
	lang_doerror("Lang: malloc error.");
	return(0);
     }
   obj->type = 0;
   obj->value = 0;
   return (long) obj;
}

void interp_variable_eqs(char *name)
{
   Lang_Name_Type *v;
   Lang_Object_Type obj;
   unsigned short type;
   unsigned char stype;
   long value;

   v = lang_locate_name(name);
   if ((v == NULL) || *(v->name) == 0)
     {
	Lang_Error = UNDEFINED_NAME;
	return;
     }

   type = (v->obj.type) & 0xFF;

   if ((type != LANG_GVARIABLE) && (type != LANG_LVARIABLE)
       && (type != LANG_IVARIABLE))
     {
	Lang_Error = DUPLICATE_DEFINITION;
	return;
     }

   /* its value is location of object in name table */
   value = (long) v;

   if (type == LANG_IVARIABLE)
     {
	if ((v->obj.type >> 8) == STRING_TYPE)
	  {
	     lang_doerror("Illegal Assignment.");
	     return;
	  }

	stype = LANG_IEQS;
     }

   else if (type == LANG_GVARIABLE) stype = LANG_GEQS;
   else
     {
	stype = LANG_LEQS;
	value = v->obj.value;
     }

   if (Lang_Defining_Function || Lang_Defining_Block)
     {
	Lang_Object_Ptr->type = LANG_DIRECTIVE | (stype << 8);
	Lang_Object_Ptr->value = value;
	Lang_Object_Ptr++;
	return;
     }

   /* create an object with the required properties for next call */
   obj.type = LANG_DIRECTIVE | (stype << 8);
   obj.value = value;
   lang_do_eqs(&obj);
}

unsigned char is_number(char *t)
{
   char *p;

   p = t;
   if (*p == '-') p++;
   while ((*p >= '0') && (*p <= '9')) p++;
   if (t == p) return(STRING_TYPE);
   if (*p == 0) return(INT_TYPE);
#ifndef FLOAT_TYPE
   return(STRING_TYPE);
#else
   /* now down to float case */
   if (*p == '.')
     {
	p++;
	while ((*p >= '0') && (*p <= '9')) p++;
     }
   if (*p == 0) return(FLOAT_TYPE);
   if ((*p != 'e') && (*p != 'E')) return(STRING_TYPE);
   p++;
   if (*p == '-') p++;
   while ((*p >= '0') && (*p <= '9')) p++;
   if (*p != 0) return(STRING_TYPE); else return(FLOAT_TYPE);
#endif
}

/* a literal */
int interp_push_number(char *t)
{
   int i;
   unsigned char stype;
   long value = 0;
#ifdef FLOAT_TYPE
   float x;
#endif

   stype = is_number(t);
   if (stype == STRING_TYPE) return(0);
   if (stype == INT_TYPE)
     {
	sscanf(t, "%d", &i);
	value = (long) i;
     }

#ifdef FLOAT_TYPE
   else if (stype == FLOAT_TYPE)
     {
	sscanf(t, "%f", &x);
	value = *(long *) &x;
     }
#endif

   if (!Lang_Defining_Block && !Lang_Defining_Function)
     {
#ifdef FLOAT_TYPE
	if (stype == INT_TYPE)
	  {
#endif
	     lang_push_integer(i);
#ifdef FLOAT_TYPE
	  }
	else lang_push_float(x);
#endif
	return(1);
     }
   /* a literal */
   Lang_Object_Ptr->type = LANG_LITERAL | (stype << 8);
   Lang_Object_Ptr->value = value;
   Lang_Object_Ptr++;
   return(1);
}

/* only supports non negative integers, use 'chs' to make negative number */

void lang_check_space(void)
{
   int n;
   Lang_Object_Type *p;

   if (Lang_Interp_Stack_Ptr - Lang_Interp_Stack >= 3)
     {
	lang_doerror("Interpret stack overflow.");
	return;
     }

   if (Lang_Defining_Block)
     {
	n = (int) (Lang_Object_Ptr - Lang_Block_Body);
	if (n + 1 < Lang_BBody_Size) return;   /* extra for terminator */
	p = Lang_Block_Body;
     }
   else if (Lang_Defining_Function)
     {
	n = (int) (Lang_Object_Ptr - Lang_Function_Body);
	if (n + 1 < Lang_FBody_Size) return;
	p = Lang_Function_Body;
     }
   else return;

   /* enlarge the space by 5 objects */
   n += 5;
   if (NULL == (p = (Lang_Object_Type *) REALLOC(p, n * sizeof(Lang_Object_Type))))
     {
	lang_doerror("Lang: realloc failure.");
	return;
     }

   if (Lang_Defining_Block)
     {
	Lang_BBody_Size = n;
	n = (int) (Lang_Object_Ptr - Lang_Block_Body);
	Lang_Block_Body = p;
	Lang_Object_Ptr = p + n;
     }
   else
     {
	Lang_FBody_Size = n;
	n = (int) (Lang_Object_Ptr - Lang_Function_Body);
	Lang_Function_Body = p;
	Lang_Object_Ptr = p + n;
     }
}

int Lang_Defining_Variables = 0;

int lang_is_defined(char *what)
{
   Lang_Name_Type *n;
   n = lang_locate_name_in_table(what, Lang_Name_Table);
   if ((n == NULL) || (*n->name == 0)) return(0);
   return(1);
}

void lang_add_variable(char *name)
{
   Lang_Name_Type *table;
   long value;

   if (strlen(name) >= LANG_MAX_NAME_LEN)
     {
	lang_doerror("Name too long.");
	return;
     }

   if (Lang_Defining_Function)	       /* local variable */
     {
	table = Lang_Local_Variable_Table;
	if (!Local_Variable_Number)
	  {
	     table = (Lang_Name_Type *) CALLOC(MAX_LOCAL_VARIABLES, sizeof(Lang_Name_Type));
	     if (NULL == table)
	       {
		  lang_doerror("Lang: calloc error.");
		  return;
	       }
	     Lang_Local_Variable_Table = table;
	  }
	strcpy(table[Local_Variable_Number].name, name);
	table[Local_Variable_Number].obj.type = LANG_LVARIABLE;
	table[Local_Variable_Number].obj.value = Local_Variable_Number++;
     }
   else	if (!lang_is_defined(name))
     {
	if (0 == (value = lang_make_object())) return;
	lang_add_name(name, value, LANG_GVARIABLE);
     }
}

void interp_push_string(char *t)
{
   int len;

   /* strings come in with the quotes attached-- knock em off */
   if (*t == '"')
     {
	len = strlen(t) - 1;
	if (*(t + len) == '"') *(t + len) = 0;
	t++;
     }

   if (!Lang_Defining_Block && !Lang_Defining_Function)
     {
	lang_push_string(t);
	return;
     }

   /* a literal */
   Lang_Object_Ptr->type = LANG_LITERAL | (STRING_TYPE << 8);
   if (NULL == (Lang_Object_Ptr->value = (long) lang_make_string((char *) t)))
     return;
   Lang_Object_Ptr++;
}

/* if an error occurs, discard current object, block, function, etc... */
void lang_restart(void)
{
   int save = Lang_Error;
   Lang_Error = 1;

   Lang_Break = Lang_Continue = Lang_Return = 0;
   while(Lang_Defining_Block)
     {
	lang_end_block();
     }

   if (Lang_Defining_Function)
     {
	if (Lang_Function_Body != NULL)
	  {
	     lang_define_function(NULL);
	     lang_free_branch(Lang_Function_Body);
	     FREE(Lang_Function_Body);
	  }
	if (Local_Variable_Number) 
	  {
	     FREE(Lang_Local_Variable_Table);
	     Local_Variable_Number = 0;
	     Lang_Local_Variable_Table = NULL;
	  }
	
	Lang_Defining_Function = 0;
     }

   Lang_Error = save;
   /* --- warning--- I need to free things on the stack! */
   if (Lang_Error == STACK_OVERFLOW) Lang_Run_Stack_Depth = -1;
   Lang_Interp_Stack = Lang_Object_Ptr = Lang_Interp_Stack_Ptr = Lang_Interp_Stack_Static;
   Local_Variable_Frame = Local_Variable_Stack;
}

void lang_compile(char *t)
{
   static int flag = 0;
   int d = 0;
   
   lang_check_space();                 /* make sure there is space for this */

   if (!Lang_Error)
     {
	if (eqs(t,"{"))	
	  {
	     lang_begin_block();
	     d = 1;
	  }
	else
	  {
	     d = try_directive(t, &flag);
	     if ((!flag && d) || (flag && !d)) Lang_Error = SYNTAX_ERROR;
	  }
	flag = 0;
     }
   
   if (Lang_Error || d);  /* null... */
   else if (Lang_Defining_Variables)
     {
	if (eqs(t, "]")) Lang_Defining_Variables = 0;
	else lang_add_variable(t);
     }
   else if (Lang_Defining_Function == -1) lang_define_function(t);
   else if (*t == '"') interp_push_string(t);
   else if ((*t == ':') && (Lang_Defining_Block))
     {
	Lang_Object_Ptr->type = LANG_LABEL;
	Lang_Object_Ptr->value = 0;
	Lang_Object_Ptr++;
     }
   
   else if ((Lang_Defining_Function == 1) && eqs(t,")"))
     {
	if (Lang_Defining_Block) lang_doerror("Function nesting illegal.");
	else Lang_Defining_Function = -1;
     }

   else if (eqs(t,"{"))	
     {
	lang_begin_block();
	flag = 0; 
     }

   else if (eqs(t,"}"))	
     {
	lang_end_block();
	flag = 1; 
     }
   

   else if (eqs(t,"("))	lang_begin_function();

   else if (eqs(t, "[")) Lang_Defining_Variables = 1;
   else if (lang_exec(t));
   else if (lang_try_binary(t));
   else if (lang_try_unary(t));
   
   /* note that order here is important */
   else if (interp_push_number(t));
   else if (*t == '=') interp_variable_eqs(t + 1);
   else Lang_Error = SYNTAX_ERROR;

   if (Lang_Error) lang_restart(), flag = 0; 
}

#ifdef Future_Slang

void pretty_print_object(Lang_Object_Type *x)
{
   if ((x->type >> 8) == STRING_TYPE)
     {
	fprintf(stdout,"\"%s\"    ", (char *) x->value);
     }
   else if ((x->type >> 8)  == INT_TYPE)
       fprintf(stdout,"%ld    ", x->value);
#ifdef FLOAT_TYPE
   else
     if ((x->type >> 8)  == FLOAT_TYPE)
     fprintf(stdout,"%g    ", (float) *(float *) &x->value);
#endif
}

void print_stack(void)
{
   int i, depth;

   depth = Lang_Run_Stack_Depth;
   pretty_print_object(&Lang_Run_Stack[depth]);
   fputs("  (",stdout);
   for (i = depth - 1; i >= 0; i--) pretty_print_object(&Lang_Run_Stack[i]);
   fputs(")",stdout);
   fflush(stdout);
}
#endif

int extract_token(char **linep, char *word)
{
    char ch, *line;
    char *special = " \t+*/()[]{}:;";	/* note that '-' is not here! */

    int string;

    line = *linep;

    /* skip white space */
    while(ch = *line++, (ch == ' ') || (ch == '\t'));

    if ((!ch) || (ch == '\n'))
      {
	 *linep = line;
	 return(0);
      }

    *word++ = ch;
    if (ch == '"') string = 1; else string = 0;

    if (NULL == strchr(special, ch))
      while(ch = *line++, (ch != '\n') && (ch != 0) &&  (ch != '"'))
	{
	   if (string)
	     {
		if (ch == '\\')
		  {
		     /* word[i++] = ch; */  /* save the slash for other routines */
		     ch = *line++;
		     if ((ch == 0) || (ch == '\n')) break;
		     switch(ch)
		       {
			  case 'n': ch = '\n'; break;
			  case 'r': ch = '\r'; break;
			  case 't': ch = '\t'; break;
			  case 'e': ch = '\033'; break;  /* escape */
			  case 'b': ch = '\007'; break;  /* bell */
		       }
		  }
	     }
	   else if (NULL != strchr(special,ch))
	     {
		line--;
		break;
	     }

           *word++ = ch;
	}

   if ((!ch) || (ch == '\n')) line--;
   /* else if (string && (ch ==  '"'))
      {
	 *word++ = ch;
      }
      */
   if (string) *word++ = '"';

    *word = 0;
    *linep = line;
    return(1);
}

/* interprets line-- returns offset of last part of line evaluated */
char *interpret(line)
char *line;
{
   char token[255];
   char *ret;

   while(ret = line, extract_token(&line,token))
     {
	if (*token == ';') break;
	lang_compile(token);
	if (Lang_Error) break;
     }
   if (Lang_Error) lang_doerror("");
   return(ret);
}

/* Allow easy addition of new intrinsic functions */
void add_intrinsic(char *name, long f, unsigned char out, int argc)
{
   unsigned short type;
   unsigned char stype;

   stype = out | (argc << 4);
   type = LANG_INTRINSIC | (stype << 8);
   lang_add_name(name, f, type);
}

void add_variable(char *name, long addr, int vtype)
{
   unsigned short type;

   type = LANG_IVARIABLE | (vtype << 8);
   lang_add_name(name, addr, type);
}

static void lang_getenv_cmd(char *s)
{
   char *t;
   if (NULL == (t = getenv(s))) t = "";
   lang_push_string(t);
}

void lang_apropos(char *s)
{
   Lang_Name_Type *table = Lang_Name_Table;
   int max = LANG_MAX_SYMBOLS;
   int all = 0, n = 0;
   char *nm;
   Lang_Object_Type obj;

   if (*s == 0) all = 1;

   while(max && (nm = table->name, *nm != 0))
     {
	if ((*nm != 1) && (all || (NULL != strstr(nm, s))))
	  {
	     n++;
	     /* since string is static, make it literal */
	     obj.type = LANG_LITERAL | (STRING_TYPE << 8);
	     obj.value = (long) nm;
	     lang_push(&obj);
	     if (Lang_Error) return;
	  }
	table++;
	max--;
     }
   lang_push_integer(n);
}

void init_lang()
{
#ifdef MALLOC_DEBUG   
   if ((DEBUG_OUT = fopen("debug.out", "w")) == NULL) DEBUG_OUT = stderr;
#endif
   /* allow interpreter to call itself. */
   add_intrinsic("eval", (long) interpret, VOID_TYPE, 1);

   /* stack routines */
   add_intrinsic("pop", (long) do_pop, VOID_TYPE, 0);
   add_intrinsic("dup", (long) do_dup, VOID_TYPE, 0);

   /* string ops */
   add_intrinsic("isdigit", (long) do_isdigit, INT_TYPE, 1);
   add_intrinsic("strlen", (long ) do_strlen, VOID_TYPE, 0);
   add_intrinsic("strcat", (long ) do_strcat, VOID_TYPE, 0);
   add_intrinsic("strcmp", (long ) do_strcmp, VOID_TYPE, 0);
   add_intrinsic("is_substr", (long ) do_issubstr, VOID_TYPE, 0);
   add_intrinsic("strsub", (long ) do_strsub, VOID_TYPE, 0);
   add_intrinsic("substr", (long ) do_substr, VOID_TYPE, 0);
   add_intrinsic("string", (long ) do_string, VOID_TYPE, 0);
   add_intrinsic("char", (long ) do_char, VOID_TYPE, 0);
   add_intrinsic("int", (long ) do_int, VOID_TYPE, 0);
   add_intrinsic("getenv", (long ) lang_getenv_cmd, VOID_TYPE, 1);

   /* array ops: */
   add_intrinsic("create_array", (long) create_array, VOID_TYPE, 0);
   add_intrinsic("free_array", (long) free_array, VOID_TYPE, 0);
   add_intrinsic("aget", (long) array_getelem, VOID_TYPE, 0);
   add_intrinsic("aput", (long) array_putelem, VOID_TYPE, 0);
   /* misc */
   add_intrinsic("defined?", (long) lang_is_defined, INT_TYPE, 1);
   add_intrinsic("slapropos", (long) lang_apropos, VOID_TYPE, 1);
   add_intrinsic("evalfile", (long) lang_load_file, INT_TYPE, 1);
   add_intrinsic("autoload", (long) lang_autoload, VOID_TYPE, 2);
   add_intrinsic("system", (long) system, INT_TYPE, 1);

   /* variables */
   lang_add_variable(SLANG_SYSTEM_NAME);
}

#ifdef Future_Jed
int lang_run_hooks(char *hook, char *optional, int *ret)
#endif
int lang_run_hooks(char *hook, char *optional)
{
   if (!lang_is_defined(hook)) return(0);
   if (optional != NULL) lang_push_string(optional);
   interpret(hook);
#ifdef Future_Jed
   if (*ret) lang_pop_integer(ret);
#endif
   if (Lang_Error) return(0);
   return(1);
}
