/*
 *  Copyright (c) 1992 John E. Davis  (davis@amy.tch.harvard.edu)
 *  All Rights Reserved.
 */
#include <stdio.h>
#include <string.h>
#ifndef msdos
#ifndef VMS
#include <malloc.h>
#endif
#else
#include <alloc.h>
#endif
#include "buffer.h"
#include "keymap.h"
#include "file.h"
#include "ins.h"
#include "ledit.h"
#include "screen.h"
#include "window.h"
#include "display.h"
#include "search.h"
#include "misc.h"
#include "replace.h"
#include "paste.h"
#include "sysdep.h"
#include "cmds.h"
#include "text.h"
#include "slang.h"

#define JED_PROMPT "JED>"
Buffer *MiniBuffer;
Buffer *Last_Buffer;
char *Token_Pointer;
Buffer *The_MiniBuffer;     /* this never gets deleted since we may need it */
Window_Type *The_MiniWindow;

extern char *get_cwd(void);

User_Variable_Type User_Vars =
  {
      78,				/* wrap-column */
      8,				/* tabs */
      3,				/* c_indent */
      2,				/* c_brace */
     300,				/* max_hits */
  };

char Last_Search_Str[132];

int (*complete_open)(char *);
int (*complete_next)(char *);

static int Buf_List_Len;
Buffer *Buf_Context;

static char *Expect_File_Error = "Expecting filename.";
static char *Keymap_Error = "Unknown_Keymap";

int next_bufflist(char *buf)
{
   Buffer *this;
   while (1)
     {
	this = Buf_Context;
	if (this == MiniBuffer) return(0);
	Buf_Context = Buf_Context->next;
	if (!Buf_List_Len || !strncmp(buf, this->name, Buf_List_Len))
	  {
	     strcpy(buf, this->name);
	     return(1);
	  }
     }
}

int open_bufflist(char *buf)
{
   if ((Buf_Context = MiniBuffer) == NULL) return(0);
   Buf_Context = Buf_Context->next;
   Buf_List_Len = strlen(buf);
   return next_bufflist(buf);
}

char *what_buffer()
{
   return(CBuf->name);
}

int bufferp(char *name)
{
   if (NULL == find_buffer(name)) return(0); else return(1);
}

int insert_buffer_name(char *name)
{
   Buffer *buf;

   if (NULL != (buf = find_buffer(name)))
     {
	insert_buffer(buf);
	return(1);
     }
   else msg_error("Unable to find buffer.");
   return(0);
}

int replace_cmd(char *old, char *new, int *how)
{
   char msgstr[132];

   if (*old == 0)
     {
	msg_error("replace what?");
	return(0);
     }

   sprintf(msgstr,"Replace \"%s\" with \"%s\"? (y/n/!/q)", old, new);
   push_spot();
   backwchars(&Number_One);
   if (search(old, 1, 0))
     {
	if (*how == 0)
	  {
	     while(replace_next(old, new));
	     pop_spot();
	     return(1);
	  }

	while(1)
	  {
	     message(msgstr);
	     Window->trashed = 1;
	     update((Line *) NULL);
	     fflush(stdout);

	     /* This does not go through keyboard macro routines
	        on purpose! */
	     switch(getkey())
	       {
		  case 'Y': case 'y': replace_next(old, new); break;
		  case '!': while(replace_next(old, new)); break;
		  case 'q': case 'Q': return(1);
		  case 'N': case 'n': forwchars(&Number_One); break;
		  default: beep(); return(1);
	       }
	     if (!search(old, 1, 0)) break;
	  }
     }
   else pop_spot();

   if (*how) message("Done.");
   return(1);
}

int replace()
{
   static char old[80], new[80];
   int one = 1;

   complete_open = NULL;

   if (!read_from_minibuffer("Replace:", 0, old)) return(0);
   if (!read_from_minibuffer("With:", 0, new)) return(0);

   replace_cmd(old, new, &one);
   return(1);
}

int kill_buffer_cmd(char *name)
{
   Buffer *buf, *kill_buf, *scratch;
   Window_Type *w;
   int kill;

   if (!strcmp(name, "*scratch*"))
     {
	erase_buffer();
	return(1);
     }
   scratch = find_buffer("*scratch*");
   if (NULL == scratch)
     {
	scratch = make_buffer();
	strcpy(scratch->name, "*scratch*");
	strcpy(scratch->dir, CBuf->dir);
	*scratch->file = 0;
     }

   if (NULL == (kill_buf = find_buffer(name)))
     {
	msg_error("Buffer does not exist.");
	return(0);
     }

   kill = 1;
   if ((*kill_buf->name != ' ') && (kill_buf->flags & BUFFER_TRASHED))
     {
	message("Buffer modified. Kill, Save First, Abort: [KSA]");
	flush_message();
	/* This does not go through keyboard macro routines
	   on purpose! */
	switch (getkey())
	  {
	     case 'k': case 'K': kill = 1; break;
	     case 's': case 'S': kill = 2; break;
	     default: msg_error("Aborted."); return(0);
	  }
	message(" ");
     }

   if (kill == 2)
     {
	buf = CBuf;
	switch_to_buffer(kill_buf);
	write_buffer();
	switch_to_buffer(buf);
	if (*Error_Buffer) return(0);
     }

   buf = kill_buf->prev;
   while(buf != kill_buf)
     {
	if ((buf != scratch) && (*buf->name != ' ')) break;
	buf = buf->prev;
     }

   if (buf == kill_buf) buf = scratch;

   if (kill_buf == CBuf)
     {
	switch_to_buffer(buf);
	window_buffer(CBuf);
     }

   /* search through windows looking for the buffer and replace it */
   w = Window;
   Window = Window->next;
   while(w != Window)
     {
	if (kill_buf == Window->buffer)
	  {
	     window_buffer(buf);  /* I should choose a different one */
	  }
	Window = Window->next;
     }
   if (kill_buf == Last_Buffer) Last_Buffer = NULL;
   delete_buffer(kill_buf);
   return(0);
}

int write_buffer_cmd(char *filestr)
{
#ifdef VMS
   char *ff;
#endif
   char dir[255], *f, file[255];
   char msgstr[132];
   int n;

   strcpy(file, expand_filename(filestr));
   f = extract_file(file);

   if ((*f == 0) && (*CBuf->file == 0))
     {
	msg_error("Filename Required.");
	return(0);
     }

   strncpy(dir, file, (int) (f - file));
   dir[(int) (f - file)] = 0;
   if (*f == 0) f = CBuf->file;

   n = write_file_with_backup(dir, f);
   if (n >= 0)
     {
#ifdef VMS
	ff = f; while (*ff) if (*ff == ';') *ff = 0; else ff++;
#endif
	sprintf(msgstr,"Wrote %d lines to %s%s", n, dir, f);
	message(msgstr);
	CBuf->flags &= ~BUFFER_TRASHED;
	CBuf->flags |= AUTO_SAVE_BUFFER;
	CBuf->hits = 0;
	visit_file(dir, f);
	return(1);
     }
   else
     {
	sprintf(msgstr,"Error writing file %s%s", dir, f);
	msg_error(msgstr);
	return(0);
     }
}

#ifdef msdos
int show_memory()
{
   char buf[40];
   sprintf(buf, "%lu bytes free.", farcoreleft());
   message(buf);
   return(0);
}
#endif

int set_buffer(char *name)
{
    Buffer *buf;

    if ((name == NULL) || (*name == 0)) return(0);
    /* Last_Buffer = CBuf; */

    if (NULL == (buf = find_buffer(name)))
      {
	 buf = make_buffer();
	 strcpy(buf->dir, CBuf->dir);
	 *buf->file = 0;
	 switch_to_buffer(buf);
	 uniquely_name_buffer(name);
      }
    else switch_to_buffer(buf);
    return(1);
}

int insert_file_cmd()
{
   char filebuf[255], *f, *file;

   complete_open = sys_findfirst;
   complete_next = sys_findnext;

   if (*CBuf->dir == 0) strcpy(CBuf->dir, get_cwd());
   strcpy(filebuf, CBuf->dir);
   if (!read_from_minibuffer("Insert file:", 0, filebuf)) return(0);

   file = expand_filename(filebuf);
   f = extract_file(file);
   if ((*file == 0) || (*f == 0))
     {
	msg_error(Expect_File_Error);
	return(1);
     }

   if (insert_file(file) < 0) msg_error("Error inserting file.");
   return(1);
}

int get_yes_no(char *question)
{
   char buf[255], tmp[80];

   strcpy(buf, question);
   strcat(buf, "? (yes or no) ");

   while(1)
     {
	*tmp = 0;
	if (!read_from_minibuffer(buf, 0, tmp)) return(-1);
	if (!strcmp(tmp, "yes")) return(1);
	if (!strcmp(tmp, "no")) return(0);
	msg_error("yes or no!");
     }
}

int find_file_cmd(char *filestr)       /* filestr is const ! */
{
   char *f, *file, filebuf[255];
   char msgstr[132];
   Buffer *buf;
   int n;

   file = expand_filename(filestr);
   strcpy(filebuf, file);
   file = filebuf;
   f = extract_file(file);
   if ((*file == 0) || (*f == 0))
     {
	msg_error(Expect_File_Error);
	return(0);
     }

   /* search for the file in current buffer list */

   if (NULL != (buf = find_file_buffer(file)))
     {
	if (file_changed_on_disk(file))
	  {
	     if (get_yes_no("File changed on disk.  Read From Disk"))
	       {
		  if (*Error_Buffer) return(1);
		  n = what_line();
		  buf->flags  &= ~BUFFER_TRASHED;
		  kill_buffer_cmd(buf->name);
		  find_file_cmd(file);
		  goto_line(&n);
		  return(1);
	       }
	  }

	switch_to_buffer (buf);
	return(1);
/*	
	if (buf == CBuf) return(0);
	Last_Buffer = CBuf;
	switch_to_buffer(buf);
	window_buffer(buf); 
*/
     }

   buf = make_buffer();
/*   if (*CBuf->name != ' ') Last_Buffer = CBuf; */
   switch_to_buffer(buf);
   buffer_filename(file, NULL);

   n = read_file(file);
   switch(n)
     {
      case -2: msg_error("File not readable."); break;
      case -1: message("New file."); break;
      default:
	sprintf(msgstr,"%d lines read", n);
	message(msgstr);
     }

   CLine = CBuf->beg;
   Point = 0;
   if (CLine == NULL) make_line(25);
   set_file_modes();
   /* switch_to_buffer(this_buf); */
/*   window_buffer(CBuf); */

   return(1);
}

int find_file_in_window(char *file)
{
   int ret;
   Buffer *b = CBuf;
   
   ret = find_file_cmd(file);
   if ((b != CBuf) && (*CBuf->name != ' ')) Last_Buffer = CBuf;
   window_buffer(CBuf);
   return(ret);
}


void set_mode_cmd(char *mode)
{
   *CBuf->mode_str = 0;
   if (!strcmp("wrap", mode)) CBuf->modes = WRAP_MODE;
   else if (!strcmp("cmode", mode)) CBuf->modes = C_MODE;
   else if (!strcmp("nomode", mode)) CBuf->modes = NO_MODE;
   else	CBuf->modes = LANG_MODE;
   strncpy(CBuf->mode_str, mode, 12);
   CBuf->mode_str[12] = 0;
}

int do_search_cmd(int direct)
{

   char *s, *p, tmp[80], *str = Last_Search_Str;

   complete_open = NULL;

    if (direct == 1) p = "Search:"; else p = "Search backward:";
    if (*str == 0) s = NULL; else s = str;
    *tmp = 0;
    if (!read_from_minibuffer(p, s, tmp)) return(0);

    if (! *tmp) return(0);
    strcpy(str, tmp);
    if (direct == 1) forwchars(&Number_One);
    if (search(str, direct, 0)) return(1);
   
   /* I prefer having this generate an error since keyboard Macro will
      abort. */
    msg_error("Not Found!");
    return(0);
}

int search_forward_cmd()
{
   return do_search_cmd(1);
}

int search_backward_cmd()
{
   return do_search_cmd(-1);
}

/* create a minibuffer with window and switch to it. */
void create_minibuffer(void)
{
   Window_Type *w;
   MiniBuffer = The_MiniBuffer;
   
   /* I want to make Mini_Window->next = Current Window so that scroll other
      window routines will scroll it. */

   w = Window;
   do other_window(); while (Window->next != w);
   Window->next = The_MiniWindow;
   The_MiniWindow->next = w;
   
   Mini_Info.action_window = w;
   other_window();    /* now in minibuffer window */
   window_buffer(MiniBuffer);
   MiniBuffer->keymap = Mini_Map;
   switch_to_buffer(MiniBuffer);
   eob();
   bol();
   CLine->len = 0;
}

/* evaluate command in minibuffer and leave */
int exit_minibuffer()
{
   if (IS_MINIBUFFER)
     {
	Exit_From_MiniBuffer = 1;
     }
   return(0);
}

/* return 1 if minibuffer already exists otherwise returns 0 */
int select_minibuffer()
{
    Window_Type *w;

    /* Try to find active minibuffer and go there */
    w = Window;
    while (MiniBuffer != NULL)
      {
	 if (Window->top == Screen_Height) return(1);
	 other_window();
	 if (w == Window) exit_error("Internal Error:  no window!");
      }

    /* switchs to minibuffer too */
    create_minibuffer();
    return(0);
}

/* if cmd != NULL, insert it into minibuffer and then send the result to
   the appropriate routine. */

int ledit(void)
{
   char tmp[132];
   *tmp = 0;
   complete_open = NULL;
   if (!read_from_minibuffer(JED_PROMPT, 0, tmp)) return(0);
   Lang_Error = 0;
   interpret(tmp);
   if ((Lang_Error == -1) && KeyBoard_Quit)
     {
	msg_error("Quit!");
     }
   Lang_Error = 0;

   return(1);
}

int find_file()
{
   char tmp[255];
   
   if (*CBuf->dir == 0) strcpy(CBuf->dir, get_cwd());
   strcpy(tmp, CBuf->dir);

   complete_open = sys_findfirst;
   complete_next = sys_findnext;

   if (!read_from_minibuffer("Find file:", 0, tmp)) return(0);
   find_file_in_window(tmp);
   return(0);
}

int write_buffer()
{
    char tmp[255];

   complete_open = sys_findfirst;
   complete_next = sys_findnext;
   if (*CBuf->dir == 0) strcpy(CBuf->dir, get_cwd());
   strcpy(tmp, CBuf->dir);
   if (!read_from_minibuffer("write file:", 0, tmp)) return(0);
   write_buffer_cmd(tmp);
   return(0);
}

void switch_to_buffer_cmd(char *name)
{
   Buffer *this = CBuf;
   set_buffer(name);
   window_buffer(CBuf);
   if ((CBuf != this) && (*CBuf->name != ' ')) Last_Buffer = this;
}

void get_last_buffer(void)
{
   Buffer *b = CBuf->next;
   
   if ((NULL == buffer_visible(Last_Buffer))
       && (Last_Buffer != CBuf)) return;
   
   while (b != CBuf)
     {
	if ((*b->name) && (*b->name != ' ') && (NULL == buffer_visible(b)))
	  {
	     Last_Buffer = b;
	     return;
	  }
	b = b->next;
     }
   return;
}

int get_buffer()
{
   char tmp[255];
   *tmp = 0;
   if ((Last_Buffer == NULL) && (*CBuf->name != ' ')) Last_Buffer = CBuf;

   complete_open = open_bufflist;
   complete_next = next_bufflist;
   
   get_last_buffer();

   if (!read_from_minibuffer("Switch to buffer:", Last_Buffer->name, tmp)) return(0);
   switch_to_buffer_cmd(tmp);
   return(1);
}

int read_integer_from_minibuffer(char *prompt, int *x)
{
   char tmp[132];
   *tmp = 0;
   complete_open = NULL;
   if (!read_from_minibuffer(prompt, 0, tmp)) return(0);
   if (1 != sscanf(tmp, "%d", x))
     {
	msg_error("Expecting Integer.");
	return(0);
     }

   return(1);
}

int goto_line_cmd()
{
   int x;

   if (!read_integer_from_minibuffer("Goto line:", &x)) return(0);
   goto_line(&x);
   return(1);
}

int goto_column_cmd()
{
   int x;
   if (!read_integer_from_minibuffer("Goto column:", &x)) return(0);
   goto_column(&x);
   return(1);
}

int kill_buffer()
{
   char tmp[255];
   *tmp = 0;

   complete_open = open_bufflist;
   complete_next = next_bufflist;
   if (!read_from_minibuffer("Kill buffer:", (char *) CBuf->name, tmp)) return(0);
   kill_buffer_cmd(tmp);
   return(1);
}

int evaluate_cmd()
{
    return(!ledit());
}

void insert_string(char *s)
{
   ins_chars((unsigned char *) s, strlen(s));
}

/* This is weird, Ultrix cc will not compile if set_key comes before unset_key */

void unset_key(char *key)
{
   undefine_key(key, Global_Map);
}

void set_key(char *function, char *key)
{
   define_key(key, function, Global_Map);
}

void unset_key_in_keymap(char *key, char *map)
{
   Key_Type *kmap;

   if (NULL == (kmap = find_keymap(map)))
     {
	msg_error(Keymap_Error);
	return;
     }

   undefine_key(key, kmap);
}

int keymap_p(char *name)
{
   return ! (NULL == find_keymap(name));
}

void set_key_in_keymap(char *f, char *key, char *map)
{
   Key_Type *kmap;

   if (NULL == (kmap = find_keymap(map)))
     {
	msg_error(Keymap_Error);
	return;
     }

   define_key(key, f, kmap);
}

void pop_to_buffer(char *name)
{
   Window_Type *w, *action, *use_this;
   Line *line;
   int p;
   Buffer *b;
   
   if (!strcmp(name, " <mini>"))
     {
	select_minibuffer();
	return;
     }

   set_buffer(name);
   
   /* save position so we can pop back to it if buffer already exists in 
      window */
   
   line = CLine;
   p = Point;
   use_this = NULL;
   if (MiniBuffer != NULL)
     {
	action = Mini_Info.action_window;
     }
   else action = NULL;

   w = Window;
   /* find a window to use */
   do
     {
	if (w->top != Screen_Height)
	  {
	     if (action != NULL)
	       {
		  if (w != action) use_this = w;
	       }
	     else if (w != Window) use_this = w;

	     if (w->buffer == CBuf)
	       {
		  use_this = w;
		  break;
	       }
	  }
	w = w->next;
     }
   while (w != Window);

   b = CBuf;
   if (use_this != NULL)
     {
	while(Window != use_this) other_window();
     }
   else
     {
	if (action != NULL) while(Window != action) other_window();
	split_window();
     }
   
   switch_to_buffer(b);
   CLine = line;
   Point = p;
   window_buffer(b);
}

/* return number of windows */
int num_windows()
{
   Window_Type *w;
   int i = 0;

   w = Window->next;
   while (i++, w != Window) w = w->next;
   return(i);
}

void flush_message(void)
{
   do_dialog(Message_Buffer);
   goto_rc(Screen_Height, strlen(Message_Buffer) + 1);
   fflush(stdout);
   /* *Message_Buffer = 0; */
   Window->trashed = 1;
}

#ifdef unix

static char *Process_Error = "Unable to open process.";

int shell_command(char *cmd)
{
   FILE *pp;

   if (NULL == (pp = popen(cmd, "r")))
     {
	msg_error(Process_Error);
	return(0);
     }

   (void) read_file_pointer(pp);
   pclose(pp);
   return(1);
}

int pipe_region(char *cmd)
{
   FILE *pp;
   int n;

   if (NULL == (pp = popen(cmd, "w")))
     {
	msg_error(Process_Error);
	return(0);
     }

   n = write_region_to_fp(pp);
   pclose(pp);
   return(n);
}

#endif

/* Note that file could be freed from Slang during run of this routine
   so get it and store it !!*/
int load_file(char *file)
{
   FILE *fp;
   char buf[132], count, *lib, libfile[255];

   strcpy(libfile, file);

   if (NULL == (fp = fopen(file, "r")))
     {
	if (NULL != (lib = getenv("JED_LIBRARY")))
	  {
	     strcpy(libfile, lib);
	     fixup_dir(libfile);
	     strcat(libfile, file);
	  }

	if (NULL == (fp = fopen(libfile, "r")))
	  {
	     msg_error("Unable to load file.");
	     return(0);
	  }
     }
   file = libfile;
   count = 0;
   sprintf(buf, "loading %s...", file);
   message(buf);
   flush_message();
   while(NULL != fgets(buf, 131, fp))
     {
	count++;
	if (*buf == ';') continue;     /* since ';' is a comment */
	interpret(buf);
	if (Lang_Error || *Error_Buffer)
	  {
	     Lang_Error = 0;
	     break;
	  }
     }
   if (! *Error_Buffer)
     {
	sprintf(buf, "loading %s...done", file);
	message(buf);
	do_dialog(Message_Buffer);
     }

   fclose(fp);
   return(1);
}

void load_buffer()
{
   char buf[132];
   Line *l;
   unsigned char *offset;

   l = CBuf->beg;
   do
     {
	/* warning--- if the line is bigger than 132 then we are dead.
	   Also I should be able to send only the data without the strncpy
	   but the last line may not be null terminated. */

	strncpy(buf, (char *) l->data, l->len);  buf[l->len] = 0;
	offset = (unsigned char *) interpret(buf);
	if (Lang_Error || *Error_Buffer)
	  {
	     CLine = l;
	     Point = (int) (offset - (unsigned char *)buf);
	     (void) skip_whitespace();
	     break;
	  }
	l = l->next;
     }
   while(l != NULL);
   return;
}

int show_key()
{
   char msg[80], *str, *s;
   int kind; 
   
   message("Show key:");
   update((Line *) NULL);
   fflush(stdout);
   str = find_key(&kind);
   if (str != NULL)
     {
	if (kind) s = "internal"; else s = "Slang";
	sprintf(msg, "Key runs %s cmd \"%s\".", s, str);
     }
   else strcpy(msg, "Key is undefined.");
   message(msg);
   return(1);
}

int what_char()
{
   if (eobp()) return(0);
   return( (int) *(CLine->data + Point) );
}

void update_cmd()
{
   Window->trashed = 1;
   touch_window();
   update((Line *) NULL);
}

void call_cmd(char *str)
{
   int (*fp)();

   if (NULL == (fp = (int (*)()) find_function(str)))
     {
	msg_error("Function does not exist!");
     }
   else (void) (*fp)();
}

void copy_region_cmd(char *name)
{
   Buffer *buf;

   if (NULL != (buf = find_buffer(name)))
     {
	copy_region_to_buffer(buf);
     }
   else msg_error("Unable to find buffer.");
}

void screen_w80()
{
   narrow_width();
   change_screen_size(80, Screen_Height);
}

void screen_w132()
{
   wide_width();
   change_screen_size(132, Screen_Height);
}

/* only works on the current line */
char *buffer_substring(void)
{
   static char string[255];
   unsigned char *tmp, *p1, *p2;
   int tmpm;

   if (CBuf->marks == NULL)
     {
	p1 = CLine->data + Point;
	p2 = CLine->data + CLine->len;
     }
   else
     {
	p1 = CLine->data + CBuf->marks->point;
	p2 = CLine->data + Point;
	if (p2 < p1)
	  {
	     tmp = p1; p1 = p2; p2 = tmp;
	  }
	tmpm = 0; pop_mark(&tmpm);
     }
   if (p2 - p1 > 254) p2 = p1 + 254;
   tmp = (unsigned char *) string;
   while (p1 < p2) *tmp++ = *p1++;
   *tmp = 0;
   return(string);
}

int mini_complete()
{
   char *buf, *pl, *pb;
   char last[255];
   static char prev[255];
   int n;
   static int flag = 0;  /* when flag goes 0, we call open */

   if (complete_open == NULL) return ins_char_cmd();

   Point = 0;
   push_mark();
   eol();
   buf = buffer_substring();

   if ((Last_Command_Char == ' ') && (Last_Key_Function == (VOID *) mini_complete))
     {
	if (!flag || !(flag = (*complete_next)(buf)))
	  {
	     strcpy(buf, prev);
	     flag = (*complete_open)(buf);
	  }

	strcpy(last, buf);
	n = -1;
     }
   else
     {
	n = 0;
	strcpy(prev, buf);  /* save this search context */
     }

   if (!n && (flag = (*complete_open)(buf), flag))
     {
	strcpy(last, buf);
	do
	  {
	     n++;
	     pl = last;
	     pb = buf;
	     while (*pl && (*pl == *pb)) pl++, pb++;
	     if (*pl) *pl = 0;
	  }
	while(flag = (*complete_next)(buf), flag);
     }

   if (n)
     {
	Point = 0;
	push_mark();
	eol();
	delete_region();
	insert_string(last);
	if ((n == 1) && (Last_Key_Function == (VOID *) mini_complete))
	  message("[Sole Completion.]");
     }
   else msg_error("No Match!");

   return(1);
}

int markp(void)
{
   return (CBuf->marks != NULL);
}

int dup_mark(void)
{
   if (CBuf->marks == NULL) return(0);

   push_spot();
   CLine = CBuf->marks->line; Point = CBuf->marks->point;
   push_mark();
   pop_spot();
   return(1);
}

char *mini_read(char *prompt, char *def)
{
   static char buf[255];

   *buf = 0;
   complete_open = NULL;
   if (!read_from_minibuffer(prompt, def, buf)) return "";
   return(buf);
}

void send_string_to_term(char *str)
{
   fputs(str, stdout);
}

void get_buffer_info(void)
{
   lang_push_string(CBuf->file);
   lang_push_string(CBuf->dir);
   lang_push_string(CBuf->name);
   lang_push_integer(CBuf->flags);
}

void set_buffer_info(char *file, char *dir, char *name, int *flags)
{
   strcpy(CBuf->file, file);
   strcpy(CBuf->dir, dir);
   strcpy(CBuf->name, name);
   CBuf->flags = *flags;
}

void make_buffer_list(void)
{
   int n = 0;
   Buffer *b;

   b = CBuf;

   do
     {
	lang_push_string(b->name);
	b = b->next;
	n++;
     }
   while (b != CBuf);
   lang_push_integer(n);
}

char *what_mode(void)
{
   return CBuf->mode_str;
}


void init_minibuffer()
{
   Buffer *tmp;

   tmp = CBuf;
   The_MiniBuffer = make_buffer();
   The_MiniBuffer->modes = 0;
   strcpy(The_MiniBuffer->name, " <mini>");
   switch_to_buffer(The_MiniBuffer);  /* do some initializing */
   remake_line(128);
   switch_to_buffer(tmp);
   The_MiniWindow = create_window(Screen_Height, 1, 1, Screen_Width);
   The_MiniWindow->buffer = CBuf;
   define_word("0-9a-z");
   init_lang();

   /* use jed rouotines instead of default slang ones */
   Lang_Error_Routine = (int (*)()) msg_error;
   User_Load_File = (int (*)()) load_file;

#ifdef msdos
   add_intrinsic("coreleft", (long) show_memory,VOID_TYPE, 0);
#endif

#ifdef unix
   add_intrinsic("pipe_region", (long) pipe_region, INT_TYPE, 1);
   /*;; pipes region to CMD returning number of lines written. */
   add_intrinsic("shell_cmd", (long) shell_command, VOID_TYPE, 1);
   /*;; executes CMD in a subshell inserting output inter buffer at Point */
#endif

   add_intrinsic("append_region_to_file", (long) append_to_file, INT_TYPE, 1);
   /*;; appends a region to FILE returning number of lines written or -1
      ;; on error.  This does NOT modify a buffer visiting the file; however,
      ;; it does flag the buffer as being changed on disk. */
   add_intrinsic("autosave", (long) auto_save, VOID_TYPE, 0);
   /*;; autosave current buffer if marked for autosave */
   add_intrinsic("autosaveall", (long) auto_save_all, VOID_TYPE, 0);
   /*;; save all buffers marked for autosave */
   
   add_intrinsic("backward_paragraph", (long) backward_paragraph, VOID_TYPE, 0);
   /*;; move point past current paragraph.  Slang hook is_paragraph_seperator
    ;; is called (if defined) to determine if line is a paragraph seperator. */

   add_intrinsic("bfind", (long) backward_search_line, INT_TYPE, 1);
   /*;; returns TRUE if STRING found backward on current line */
   add_intrinsic("blank_rect", (long) blank_rectangle, VOID_TYPE, 0);
   /*;; blanks out rectangle defined by point and mark */
   add_intrinsic("bob", (long) bob,VOID_TYPE, 0);
   /*;; move to beginning of buffer */
   add_intrinsic("bobp", (long) bobp,INT_TYPE, 0);
   /*;; TRUE if at beginning of buffer */
   add_intrinsic("bol", (long) bol,VOID_TYPE, 0);
   /*;; Move point to beginning of line */
   add_intrinsic("bolp", (long) bolp,INT_TYPE, 0);
   /*;; TRUE if point is at the beginnning of the line */
   add_intrinsic("bsearch", (long) search_backward, INT_TYPE, 1);
   /*;; TRUE if STRING found backward */
   add_intrinsic("bskip_chars", (long) bskip_chars, VOID_TYPE, 1);
   /*;; skip backward chars in STRING.  See skip_word for definition of STRING */
   add_intrinsic("bskip_word", (long) backward_word, VOID_TYPE, 0);
   /*;; skip word backward.  For word definition, see define_word */
   add_intrinsic("buffer_list", (long) make_buffer_list,  VOID_TYPE, 0);
   /*;; returns a  list of buffers to the stack.  The top element of the 
   ;; stack is the number of buffers */
   add_intrinsic("bufferp", (long) bufferp, INT_TYPE, 1);
   /*;; returns  TRUE if BUFFER exists */
   add_intrinsic("bufsubstr", (long) buffer_substring, STRING_TYPE, 0);
   /*;; returns STRING from Mark to Point on the current line */
   add_intrinsic("call", (long) call_cmd, VOID_TYPE, 1);
   /*;; Execute a COMMAND which cannot be called  from slang  */
   add_intrinsic("check_region", (long) check_region, VOID_TYPE, 1);
   /*;; Signals Error if mark not set.  Exchanges point 
   ;; and mark to produce valid region.  A valid region is one with mark
   ;; earlier in the buffer than point.  Always call this if using a region
   ;; which requires point > mark.  Also, if argument is non-zero, spot is 
   ;; pushed. */
   add_intrinsic("copy_rect", (long) copy_rectangle, VOID_TYPE, 0);
   /*;; save a copy of rectangle defined by point and mark in rectangle 
    ;; buffer */
   add_intrinsic("copy_region", (long) copy_region_cmd, VOID_TYPE, 1);
   /*;; copies region to BUFFER */
   add_intrinsic("definekey", (long) set_key_in_keymap, VOID_TYPE, 3);
   /*;; Sets key sequence key in keymap keymap to invoke function f.
   ;; Usage: f key keymap definekey
   ;; See: setkey, undefinekey, make_keymap, use_keymap
   ;;*/
   add_intrinsic("define_word", (long) define_word, VOID_TYPE, 1);
   /*;; Only argument is a string which is an expression which defines 
     ;; a word.  Typically, it is a range of ascii values.  The default 
     ;; definition is: "a-z0-9"
     ;; To include a hyphen, make it the first character.  So for example,
     ;; "-i-n" defines a word to consist of letters 'i' to 'n'  and '-' */
     
   add_intrinsic("del", (long) del,VOID_TYPE, 0);
   /*;;  delete character at point */
   add_intrinsic("del_region", (long) delete_region, VOID_TYPE, 0);
   /*;;  deletes region.  Once this function is called, the region is GONE. */
   add_intrinsic("delbuf", (long) kill_buffer_cmd, VOID_TYPE, 1);
   /*;; deletes specified buffer name */
   add_intrinsic("delete_file",  (long) sys_delete_file, VOID_TYPE, 1);
   /*;; Deletes FILENAME */
   add_intrinsic("down", (long) nextline,INT_TYPE, 1);
   /*;; Move down INTEGER lines returning the number actually moved. */
   add_intrinsic("dupmark", (long) dup_mark, INT_TYPE, 0);
   /*;; If a mark is set, this function pushes an exact copy on the mark stack */
   add_intrinsic("enlargewin", (long) enlarge_window,VOID_TYPE, 0);
   /*;; Makes the current window bigger by one line. */
   add_intrinsic("eob", (long) eob,VOID_TYPE, 0);
   /*;; Move point to end of buffer. */
   add_intrinsic("eobp", (long) eobp,INT_TYPE, 0);
   /*;; returns TRUE if Point is at the end of the buffer */
   add_intrinsic("eol", (long) eol,VOID_TYPE, 0);
   /*;; Move Point to the end of the current line. */
   add_intrinsic("eolp", (long) eolp,INT_TYPE, 0);
   /*;; Returns TRUE if Point is at the end of a line. */
   add_intrinsic("erase_buffer", (long) erase_buffer, VOID_TYPE, 0);
   /*;; erases all text from the current buffer.
   ;; See: delbuf */
   add_intrinsic("error", (long) msg_error, VOID_TYPE, 1);
   /*;; Return to top level and display ERROR message. */
   add_intrinsic("evalbuffer", (long) load_buffer, VOID_TYPE ,0);
   /*;; evaluates a buffer as S-Lang code.  See: evalfile */
   add_intrinsic("exit_jed", (long) exit_jed, VOID_TYPE, 0);
   /*;; Exits JED.  If any buffers are modified, the user is queried 
     ;; about whether or not to save first.  Calls S-Lang hook "exit_hook"
     ;; if defined */ 
   add_intrinsic("ffind", (long) forward_search_line, INT_TYPE, 1);
   /*;; Returns TRUE if STRING is found forward current line. If found, Point
      ;; is moved to string */
   add_intrinsic("file_status", (long) file_status, INT_TYPE, 1);
 /*;; returns integer desecribing FILE:
   ;;     2 file is a directory
   ;;     1 file exists
   ;;     0 file does not exist.
   ;;    -1 no access.
   ;;    -2 path invalid
   ;;    -3 unknown error
   ;;*/
   add_intrinsic("filechgondsk", (long) file_changed_on_disk, INT_TYPE, 1);
   /*;; Returns true if FILE on disk is more recent than editor file */
   add_intrinsic("find_file", (long) find_file_in_window, INT_TYPE, 1);
   /*;; finds FILE in current window returning non zero if file found.
      ;; See Also: read_file */
   
   add_intrinsic("flush", (long) flush_message, VOID_TYPE, 0);
   /*;; Flush message buffer to screen immediately.  
   ;; Do not wait for top level. 
   ;; See: message, update */
   add_intrinsic("forward_paragraph", (long) forward_paragraph, VOID_TYPE, 0);
   /*;; move point past current paragraph.  Slang hook is_paragraph_seperator
    ;; is called (if defined) to determine if line is a paragraph seperator. */
   add_intrinsic("fsearch", (long) search_forward, INT_TYPE, 1);
   /*;; Search forward in buffer for STRING returning TRUE if found. */
   add_intrinsic("getbuf_info", (long) get_buffer_info,  VOID_TYPE, 0);
   /*;; Actually returns the following information on stack (from top): 
   ;;   flags
   ;;   buffer name
   ;;   directory associated with buf
   ;;   filename associated with buf (if any).
   ;; The flags are: 
   ;;    bit 0: buffer modified
   ;;    bit 1: auto save mode
   ;;    bit 2: file on disk modified
   ;; See also: setbuf_info
   ;;*/
   
   add_intrinsic("getkey", (long) getkey, INT_TYPE, 0);
   /*;; Read a key from input stream returning ASCII value read. */
   add_intrinsic("goto_column", (long) goto_column, VOID_TYPE, 1);
   /*;; Move Point to COLUMN inserting spaces and tabs if necessary. */
   add_intrinsic("goto_line", (long) goto_line,VOID_TYPE, 1);
   /*;; move Point to LINE. */
   add_intrinsic("indent_line", (long) indent_line, VOID_TYPE, 0);
   /*;; Indent line according to current mode. */
   add_intrinsic("insbuf", (long) insert_buffer_name, VOID_TYPE, 1);
   /*;; Insert BUFFER into current buffer at Point. */
   add_intrinsic("insert", (long) insert_string,VOID_TYPE, 1);
   /*;; Insert STRING at Point. */
   add_intrinsic("insert_file",  (long) insert_file, INT_TYPE, 1);
   /*;; This returns <= 0 if file not found. */

   add_intrinsic("insert_rect", (long) insert_rectangle, VOID_TYPE, 0);
   /*;; insert contents of previously deleted rectangle at Point. */
   add_intrinsic("keymap_p", (long)  keymap_p, INT_TYPE, 1);
   /*;; Returns TRUE if KEYMAP is defined. */
   add_intrinsic("kill_rect", (long) kill_rectangle, VOID_TYPE, 0);
   /*;; deletes rectangle defined  by point and mark.  The contents of 
      ;; the rectangle are saved in the rectangle buffer destroying previous
      ;; contents. */
   add_intrinsic("left", (long) backwchars,INT_TYPE, 1);
   /*;; move backward n chars returning number actually moved. */
   add_intrinsic("looking_at", (long) looking_at, INT_TYPE, 1);
   /*;; Returns 1 if Point is positioned on string s, otherwise returns 0.
   ;; See: ffind */
   add_intrinsic("make_keymap", (long) create_keymap, VOID_TYPE, 1);
   /*;; Creates a new keymap with name map.  The newly created keymap is an 
   ;; exact copy of the global map "global".
   ;; See: use_keymap, definekey, undefinekey */
   add_intrinsic("markp", (long) markp, INT_TYPE, 0);
   /*;; returns TRUE if mark is set. */
   add_intrinsic("message", (long) message, VOID_TYPE, 1);
   /*;; display STRING string in minibuffer. */
   add_intrinsic("narrow", (long) narrow_to_region, VOID_TYPE, 0);
   /*;; restrict editing to region of LINES defined by point and mark.  
      ;; Use 'widen' to remove the restriction. Be careful with this because
      ;; it currently does not remember a previous narrow. */
   add_intrinsic("nwindows", (long) num_windows,INT_TYPE, 0);
   /* return number of windows currently visible. */
   add_intrinsic("onewindow", (long) one_window,VOID_TYPE, 0);
   /*;; make current window the only one. */
   add_intrinsic("open_rect", (long) open_rectangle, VOID_TYPE, 0);
   /*;; insert a BLANK rectangle.  The rectangle  is defined by point and mark. */
   add_intrinsic("otherwindow", (long) other_window,VOID_TYPE, 0);
   /*;; move to nbext window */
   
   add_intrinsic("pop2buf", (long) pop_to_buffer, VOID_TYPE, 1);
   /*;; Pop up a window with BUFFER in it.  This will split the current
      ;; window if necessary */
   add_intrinsic("pop_mark", (long) pop_mark, INT_TYPE, 1);
   /*;; Pop last pushed mark off the mark stack. If argument is non zero, 
      ;; move point to position of mark first. */
   add_intrinsic("pop_spot", (long) pop_spot,VOID_TYPE, 0);
   /*;; return to point of previously pushed spot */
   add_intrinsic("push_mark", (long) push_mark,VOID_TYPE, 0);
   /*;; begin a region definition.  The region is defined by this mark and
      ;; the Point. */
   add_intrinsic("push_spot", (long) push_spot,VOID_TYPE, 0);
   /*;; push current point onto a stack.  This does not set the mark.  Use 
      ;; push_mark for that purpose. */
   add_intrinsic("quit_jed", (long) quit_jed, VOID_TYPE, 0);
   /*;; Quit JED saving no buffers, just get out! */
   
   add_intrinsic("read_mini", (long) mini_read, STRING_TYPE, 2);
   /*;; read from minibuffer with PROMPT and DEFAULT strings */
   add_intrinsic("read_file", (long) find_file_cmd, INT_TYPE, 1);
   /*;;  read FILE into its own buffer  returning non zero if file exists.
      ;; see find_file to read a file into a window. */
   
   add_intrinsic("replace", (long) replace_cmd, VOID_TYPE, 3);
   /*;; replace OLD_STRING with NEW_STRING using METHOD (integer).  If METHOD
     ;; is 0, replace all from point forward otherwise query user. */
   add_intrinsic("right", (long) forwchars,INT_TYPE, 1);
   /*;; Move forward N characters returning actual number moved. */
   add_intrinsic("self_ins", (long) ins_char_cmd, VOID_TYPE, 0);
   add_intrinsic("setbuf", (long) set_buffer, VOID_TYPE, 1);
   /*;; makes BUFFER the current buffer for editing.  This change only
      ;; lasts until top level is reached where buffer associated with 
      ;; current window is returned to. */
   add_intrinsic("setbuf_info", (long) set_buffer_info,  VOID_TYPE, 4);
   /*;; takes the following information (from bot of stack to top)
   ;;   filename associated with buf (if any).
   ;;   directory associated with buf
   ;;   buffer name
   ;;   flags
   ;; The flags are: 
   ;;    bit 0: buffer modified
   ;;    bit 1: auto save mode
   ;;    bit 2: file on disk modified
   ;; See also: getbuf_info
   ;;*/
   add_intrinsic("setkey", (long) set_key, VOID_TYPE, 2);
   /*;; Sets KEY to FUNCTION in "global" keymap.  For example,
      ;;   "redraw" "^L" setkey
      ;; sets control-L to call redraw function. */
   add_intrinsic("setmode", (long) set_mode_cmd, VOID_TYPE, 1);
   /*;; Set mode to s.  If s is "wrap", set mode to text-mode.  
   ;; If mode is "cmode", set mode for editing C code.  
   ;; If mode is "nomode" remove any current mode.
   ;; Any other value of mode is considered to be a user defined mode and the
   ;; mode string on the status line assumes its value. */
   add_intrinsic("showkey", (long) show_key, VOID_TYPE, 0);
   /*;; shows current key definition. */
   add_intrinsic("skip_chars", (long) skip_chars, VOID_TYPE, 1);
   /* skip past all characters in string s.
   ;; s is a string which contains ascii chars to skip, or a rang of ascii 
   ;; chars.  So for example, "- \t0-9ai-o_" will skip the hyphen, space, tab
   ;; numerals 0 to 9, letter a, letters i to o, and underscore.
   ;; See Also: bskip_chars, skip_white */
   
   add_intrinsic("skip_white", (long) skip_whitespace, VOID_TYPE, 0);
   /*;; Skip past whitespace.  This does not cross lines.  
   ;; See: skip_chars */
   add_intrinsic("skip_word", (long) forward_word, VOID_TYPE, 0);
   /*;; Move point past current word. */
   add_intrinsic("splitwindow", (long) split_window,VOID_TYPE, 0);
   /*;; Splits current window in half making two. */
   add_intrinsic("suspend", (long) sys_spawn_cmd, VOID_TYPE, 0);
   /*;; Suspend jed and return to calling process or spawn subprocess.
    ;; "suspend_hook" is called before suspension and "resume_hook" is called
    ;; after.  These are user defined S-Lang functions. */
   add_intrinsic("sw2buf", (long) switch_to_buffer_cmd, VOID_TYPE, 1);
   /*;; Switch to BUFFER.  If BUFFER does not exist, one is created with name
      ;; BUFFER */
   add_intrinsic("time", (long) get_time, STRING_TYPE, 0);
   /*;; return current date and time string */
   add_intrinsic("trim", (long) trim_whitespace,VOID_TYPE, 0);
   /*;; removes all whitespace around point */
   add_intrinsic("tt_send", (long) send_string_to_term, VOID_TYPE, 1);
   /*;; send STRING to terminal with no interpretation */
   add_intrinsic("undefinekey", (long) unset_key_in_keymap, VOID_TYPE, 2);
   /*;; Associate keymap name with current buffer.  By default the global 
   ;; keymap, "global", is used.  Use this function to override the default
   ;; behavior.
   ;;See: make_keymap. */
   add_intrinsic("ungetkey", (long) ungetkey, VOID_TYPE, 1);
   /*;; push ASCII value of character on input stream */

   add_intrinsic("unsetkey", (long) unset_key, VOID_TYPE, 1);
   /*;; remove definition of KEY from "global" keymap.  If KEY is a
      ;; prefix character for a keymap, the entire keymap definition is 
      ;; removed. */
   add_intrinsic("up", (long) prevline, INT_TYPE, 1);
   /*;; Move editing Point up n lines returning number actually moved.
   ;; See: down, left, right */
   add_intrinsic("update", (long) update_cmd, VOID_TYPE, 0);
   /*;; Update display.  If used in a slang routine, make sure update is
      ;; called again before return to top level */
   add_intrinsic("use_keymap", (long)  use_keymap, VOID_TYPE, 1);
   /*;; Asscoiate KEYMAP with buffer. */
   add_intrinsic("w132", (long) screen_w132, VOID_TYPE, 0);
   add_intrinsic("w80", (long) screen_w80, VOID_TYPE, 0);
   add_intrinsic("what_char", (long) what_char, INT_TYPE, 0);
   /*;; returns ASCII value of character point is on. */
   add_intrinsic("what_column", (long) calculate_column,INT_TYPE, 0);
   /*;; returns current column number expanding tabs, etc... */
   add_intrinsic("whatbuf", (long) what_buffer, STRING_TYPE, 0);
   /*;; returns buffer name */
   add_intrinsic("whatline", (long) what_line,INT_TYPE, 0);
   /*;; returns current line number */
   add_intrinsic("whatmode", (long) what_mode, STRING_TYPE, 0);
   /*;; returns buffer mode string. */
   add_intrinsic("whitespace", (long) insert_whitespace,VOID_TYPE, 1);
   /*;; inserts whitespace of length n using tabs and spaces.  If the global
   ;; variable  TAB is 0, only spaces are used. */
   add_intrinsic("widen", (long) widen, VOID_TYPE, 0);
   /*;; Opposite of Narrow.  See narrow for additional information. */
   add_intrinsic("write_buffer", (long) write_buffer_cmd, INT_TYPE, 1);
   /*;; writes buffer to FILE. Returns number of lines written or signals
      ;; error on failure. */
   add_intrinsic("write_region", (long) write_region, INT_TYPE, 1);
   /*;; write region to FILE.  Returns number of lines written or signals
      ;; error on failure. */
   add_intrinsic("xform_reg", (long) transform_region, VOID_TYPE, 1);
   /*;; takes a control string as argument:
     ;;    "u" upcase_region
     ;;    "d" downcase_region
     ;;    "c" Capitalize region
     ;;    anything _else will change case of region */
#ifdef msdos
   add_variable("COLOR", (long) &Attribute_Byte, INT_TYPE);
#endif
   add_variable("BLINK", (long) &Blink_Flag, INT_TYPE);
   add_variable("LAST_SEARCH", (long) Last_Search_Str, STRING_TYPE);
   add_variable("TAB", (long) &User_Vars.tab, INT_TYPE);
   add_variable("WRAP", (long) &User_Vars.wrap_column, INT_TYPE);
   add_variable("C_INDENT", (long) &User_Vars.c_indent, INT_TYPE);
   add_variable("C_BRACE", (long) &User_Vars.c_brace, INT_TYPE);
   add_variable("LAST_CHAR", (long) &Last_Command_Char, INT_TYPE);
   add_variable("MAX_HITS", (long) &User_Vars.max_hits, INT_TYPE);
   add_variable("CASE_SEARCH", (long) &Case_Sensitive, INT_TYPE);
   add_variable("POINT", (long) &Point, INT_TYPE);
   add_variable("ADD_NEWLINE", (long) &Require_Final_Newline, INT_TYPE);
   add_variable("META_CHAR", (long) &Meta_Char, INT_TYPE);
   /*;; When a character with the hi bit set is input, it gets mapped to
     ;; a two character sequence, The META_CHAR, followed by the
     ;; character with its hi bit off.  By default, META_CHAR is 27, the
     ;; escape character. */

   add_variable("DISPLAY_EIGHT_BIT", (long) &Display_Eight_Bit, INT_TYPE);
   /*;; if non zero, pass chars with hi bit set to terminal as is,
     ;; otherwise prefix with a `~' and pass char with hi bit off. */

   /* add_variable("SCRN_HEIGHT", (long) &Screen_Height, INT_TYPE);
      add_variable("SCRN_WIDTH", (long) &Screen_Width, INT_TYPE); */
}
