/*
 *  Copyright (c) 1992 John E. Davis  (davis@amy.tch.harvard.edu)
 *  All Rights Reserved.
 */

#include <stdio.h>
#include <setjmp.h>
#ifdef msdos
#include <alloc.h>
#include <process.h>
#else
#ifndef VMS
#include <malloc.h>
#endif
#endif
#include <string.h>

#include "buffer.h"
#include "window.h"
#include "file.h"
#include "ins.h"
#include "misc.h"
#include "paste.h"
#include "sysdep.h"

extern int Goal_Column;
extern jmp_buf Jump_Buffer;

typedef struct Buffer_List_Type
  {
      Buffer *buf;
      struct Buffer_List_Type *next;
  }
Buffer_List_Type;

Buffer_List_Type *Buffer_Stack;   /* a filo */

Buffer *CBuf;
Line *CLine;

int Number_One = 1;		       /* these should be const but  */
int Number_Zero = 0;		       /* some compilers complain */
int Number_Two = 2;

int Point = 0;

/* move point to top of buffer */
int bob()
{
    CLine = CBuf->beg;
    Point = 0;
    return(1);
}

int eob()                      /* point to end of buffer */
{
    CLine = CBuf->end;
    if (CLine == NULL)
      {
          Point = 0;
          return(0);
      }

    Point = CLine->len - 1;
    if (Point < 0)
      {
          Point = 0;
          return(1);
      }

    if (CLine->data[Point] != '\n') Point++;
    return(1);
}

int bol()
{
    Point = 0;
    Goal_Column = 1;
    return(1);
}

int eol()
{
    int newline = 1;
    if (CLine != NULL)
      {
          Point = CLine->len - 1;
          if (Point < 0)
            {
                Point = 0;
                return(0);
            }

          if ((CLine->data)[Point] != '\n') newline = 0;
          if (!newline) Point++;
      }

    else Point = 0;
    return(1);
}

int bobp()                /* returns 1 if top line of buffer, 0 otherwise */
{
    if ((CBuf->beg == CLine) && (Point == 0)) return(1); else return(0);
}

int eolp()
{
    int len;

    if (CLine == NULL) return(1);
    len = CLine->len;
    if (len == 0) return(1);
    if (Point < len - 1) return(0);

    /* Point is either len or len - 1 */

    if ((CLine->data)[len-1] == '\n') return(1);

    if (Point == len) return(1);
    Point = len - 1;
    return(0);
}

int eobp()
{
    if (CLine != CBuf->end) return(0);
    return(eolp());
}

int bolp()
{
    if (Point) return(0); else return(1);
}

/*  Attempt to goback n lines, return actual number. */
int prevline(int *n)
{
    int i = 0;
    Line *prev;

    Point = 0;     /* return to beginning of this line */
    while (i < *n)
      {
          prev = CLine->prev;
          while((prev != NULL) && (prev->len == 0)) prev = prev->prev;
          if (prev == NULL) break;
          i++;
          CLine = prev;
      }

    if (i) eol();
    return(i);
}

int nextline(int *n)
{
    int i = 0;
    Line *next;

    while(i < *n)
      {
          next = CLine->next;
          if (next == NULL) break;
          CLine = next;
          i++;
      }

    if (i) Point = 0;
    return(i);
}

/* The algorithm:  go to Point = len - 1.  Then forward a line counted as 1 */
int forwchars(int *np)
{
   int len, total = 0;
   unsigned char *p;
   Line *next;
   int n = *np;

    while(1)
      {
          len = CLine->len;
          /* if (Point == len) return(total); */
          p = CLine->data + Point;
          while(n && (Point < len) && (*p != '\n'))
            {
                Point++;
                n--;
                p++;
                total++;
            }
          if (!n) return(total);
          if (*p == '\n')
            {
                total++;
                n--;
            }

          if (NULL == (next = CLine->next))
            {
                if (*p == '\n') total--;
                return(total);
            }

          Point = 0;
          CLine = next;
      }
}

int backwchars(int *np)
{
   int total = 0;
   int n = *np;

    do
      {
          if (n <= Point)
            {
                total += n;
                Point = Point - n;
                return(total);
            }
          n = n - Point - 1;   /* Point + newline */
          total += Point + 1;   /* ditto */
      }
    while(prevline(&Number_One));

    return(total - 1);
}

Line *make_line1(int size)
{
   Line *new_line;
   unsigned char *data = NULL;
   char buff[80];
   unsigned int chunk;
   int memory_low = 0;

   /* chunk = (unsigned) (size + 15) & 0xFFF0; */    /* 16 byte chunks */
   chunk = (unsigned) (size + 7) & 0xFFF8;   /* 8 byte chunks */
   new_line = (Line *) MALLOC(sizeof(Line));
   if (new_line != NULL) do
     {
	if (NULL != (data = (unsigned char *) MALLOC(chunk))) break;
	chunk--;
	memory_low = 1;
     }
   while(chunk >= size);

   if ((new_line == NULL) || (data == NULL))
     {
	sprintf(buff, "Malloc Error in make_line: requested size: %d.", size);
	msg_error(buff);
	longjmp(Jump_Buffer, 1);
	/* exit_error(buff); */
     }
   if (memory_low) message("Memory is running low...");
   new_line->data = data;
   new_line->len = 0;
   new_line->space = chunk;
   return(new_line);
 }

/* adds a new link to list of lines at current point */
unsigned char *make_line(int size)
{
    Line *new_line;

    new_line = make_line1(size);
    /* if CLine is Null, then we are at the top of a NEW buffer.  Make this
       explicit. */
    if (CLine == NULL)
      {
          new_line -> prev = NULL;
          new_line -> next = NULL;
          CBuf -> beg = CBuf ->end = new_line;
      }
    else if (CLine == CBuf->end) /* at end of buffer */
      {
          CBuf->end  = new_line;
          new_line->next = NULL;
          new_line->prev = CLine;
          CLine->next = new_line;
      }
    else
      {
          new_line -> next = CLine -> next;
          if (CLine->next != NULL) CLine->next->prev = new_line;
          CLine->next = new_line;
          new_line->prev = CLine;
      }

    CLine = new_line;
    return(CLine->data);
}

void free_line(Line *line)
{
    FREE( (line->data));
    FREE( line);
}

/* deletes the line we are on and returns the prev one.  It does not
   delete the top line of the buffer.   Furthermore, it does not
   update any marks.  */

int delete_line()
{
    Line *n, *p, *this;

    p = CLine -> prev;
    if (p == NULL) return(1);

    n = CLine -> next;
    this = CLine;
    if (n == NULL)
      {
          CBuf->end = p;
          p->next = NULL;
      }
    else
      {
          p->next = n;
          n->prev = p;
      }

    free_line(this);
    CLine = p;
    return(0);
}

unsigned char *remake_line(int size)
{
   char buf[80];
   size = (unsigned) (size + 7) & 0xFFF8;   /* 8 byte chunks */
   /* size = (size + 15) & 0xFFF0; */
   if (NULL == (CLine->data = (unsigned char *) REALLOC(CLine->data, size)))
     {
	sprintf(buf, "remake_line: realloc error!, size = %d", size);
	exit_error(buf);
     }

   CLine->space = size;
   return(CLine->data);
}

void uniquely_name_buffer(char *try)
{
    Buffer *b;
    int exists, version = 0;
    char new[255];

    *CBuf->name = 0;
    strcpy(new, try);
    while (1)
      {
	  exists = 0;
	  b = CBuf->next;
	  while(b != CBuf)
	    {
		if (!strcmp(new, b->name)) exists = 1;
		b = b->next;
	    }
	  if (!exists)
	    {
		strcpy(CBuf->name,new);
		return;
	    }
	  version++;
	  sprintf(new, "%s<%d>", try, version);
      }
}

/* make a buffer and insert it in the list */
Buffer *make_buffer()
{
    Buffer *newB;
    void *p;

    p = MALLOC(sizeof(Buffer));
    if ((newB = (Buffer *) p) == NULL)
      {
          exit_error("make_buffer: malloc error");
       }
   newB->beg = newB->end = NULL;
   newB->beg1 = newB->end1 = NULL;
   newB->beg2 = newB->end2 = NULL;
   newB -> point = 0;
   newB->modes = newB->flags = 0;
   newB -> line = NULL;
   newB->marks = NULL;
   newB->spots = NULL;
   newB->keymap = Global_Map;
   newB->hits = 0;
   newB->m_time = 0;
   newB->c_time = sys_time();
   *newB->mode_str = *newB->file = *newB->dir = *newB->name = 0;
   if (CBuf == NULL)
     {
	newB->next = newB;
	newB->prev = newB;
     }
   else
     {
	newB->next = CBuf;
	newB->prev = CBuf->prev;
	CBuf->prev->next = newB;
	CBuf->prev = newB;
     }

   return(newB);
}

/* if there is a window attached to this buffer, then there are problems
   if we get to update() without attaching another one to it.  So
   beware! Always make sure CBuf is set too! */
void delete_buffer(Buffer *buf)
{
   Line *l,*n;
   Mark *m, *m1;

   widen_buffer(buf);
   if (buf -> beg != NULL) for (l = buf -> beg; l != NULL; l = n)
     {
	n = l -> next;
	FREE( l->data);
	FREE( l);
     }
   m = buf->marks;
   while (m != NULL)
      {
	 m1 = m->next;
	 FREE( m);
	 m = m1;
      }
   m = buf->spots;
   while (m != NULL)
     {
	m1 = m->next;
	FREE(m);
	m = m1;
     }

   buf->prev->next = buf->next;
   buf->next->prev = buf->prev;

   FREE( buf);
}

void goto_line(int *np)
{
   int n = *np;
   bob();
   if (n-- > 0) nextline(&n);
}

Line *dup_line(Line *l)
{
    Line *new;
    int n;
    unsigned char *p, *q;

    new = (Line *) MALLOC(sizeof(Line));
    if ((new == NULL) ||
        (NULL == (new->data = (unsigned char *) MALLOC(l->space))))
      {
          exit_error("Malloc Error in dup_line.");
      }
    new->next = l->next;
    new->prev = l->prev;
    new->len = l->len;
    n = new->space = l->space;
    p = new->data;
    q = l->data;

    while (n--) *p++ = *q++;
    return(new);
}

Buffer *find_buffer(char *name)
{
    Buffer *b;

    b = CBuf;
    do
      {
	  if (!strcmp(b->name, name)) return(b);
	  b = b->next;
      }
    while(b != CBuf);
    return(NULL);
}

int switch_to_buffer(Buffer *buf)
{
    /* unlink buf */
    /*  save this buffer position */
    CBuf->line = CLine;
    CBuf->point = Point;
   
    if (buf == CBuf) return(0);
    buf->prev->next = buf->next;
    buf->next->prev = buf->prev;

    buf->next = CBuf;
    buf->prev = CBuf->prev;
    CBuf->prev->next = buf;
    CBuf->prev = buf;


    /* now restore new buffer */
    CBuf = buf;
    CLine = CBuf->line;
    Point = CBuf->point;
    if (CLine == NULL)
      {
          make_line(25);
          Point = 0;
      }
    CBuf->line = CLine;
    CBuf->point = Point;
    return(1);
}

/* we assume that file has already been expanded--- that is, canonical. */
Buffer *find_file_buffer(char *file)
{
   Buffer *b;
   char *f;
   long n, m;

   f = extract_file(file);
   n = f - file;

   b = CBuf;
    do
      {
	 m = strlen(b->dir);
	 if ((m == n) && (!strcmp(b->file,f))
	     && (!strncmp(b->dir, file, (int) n)))
	   return(b);
          b = b->next;
      }
    while (b != CBuf);

    return(NULL);
}

/* take a dir and a filename, expand them then put in buffer structure */
void buffer_filename(char *dir, char *file)
{
   strcpy(CBuf->dir, dir);
   if ((file == NULL) || (*file == 0))
     {
	file = extract_file(CBuf->dir);
	strcpy(CBuf->file, file);
	*file = 0;
     }
   else strcpy(CBuf->file, file);
   uniquely_name_buffer(CBuf->file);
}

int get_percent()
{
    Line *line;
    unsigned long n;
    unsigned int m;

    line = CBuf->beg;
    n = 1;
    m = 1;
    while (line != NULL)
      {
	  if (line == CLine) break;
	  line = line->next;
	  m++;
      }
    n = (unsigned long) m;

    while (line != NULL)
      {
	  line = line->next;
	  m++;
      }
    m--;
    if ((m == 0) || (n == 1)) return(0);
    n = n * 100;

    return((int) (n / (long) m));
}

/* should use long for dos machines-- at least unsigned */
int what_line()
{
   int n;
   Line *line;

   n = 1;
   line = CBuf->beg;
   while (line != CLine)
     {
	line = line->next;
	n++;
     }
   return(n);
}

int erase_buffer()
{
   Line *beg, *this;

   CLine = CBuf->end;
   Point = 0;
   beg = CBuf->beg;
   while(CLine != beg)
     {
	update_marks(LDELETE, 1);
	this = CLine;
	CLine = CLine->prev;
	free_line(this);
     }
   CBuf->end = CLine;
   update_marks(CDELETE, CLine->len);
   CLine->len = 0;
   CLine->next = NULL;
   return(1);
}

void mark_buffer_modified(int *flag)
{
   unsigned long now;

   int status = CBuf->flags & BUFFER_TRASHED;

   if (*flag && status) return;

   if (*flag)
     {
	now = sys_time();
	CBuf->m_time = now;
	CBuf->flags |= BUFFER_TRASHED;
	return;
     }
   else CBuf->flags &= ~BUFFER_TRASHED;
}

