LISTING 1: ENGINE.C

/***************************************************************************
* ENGINE.C  -- Back end (NLM) for demonstration client-server record manager
*      Version 1.0    June 26 1992
*      by Michael Day
*      Copyright 1992, Michael Day. This work (and portions of it)
*      may be freely copied, altered, and distributed, either in source
*      or binary form, provided that it is not used for commercial purposes,
*      except as a teaching tool to demonstrate network programming.
*      This uses the Watcom C 386 compiler. See the makefile for more info.
*************************************************************************/

#include "engine.h"

/*------------------------ Global Variables ------------------------------*/
LONG requestSem, replySem, shutdownSem;       // semaphore handles
char dataFileName[_MAX_PATH + 1];             // data base file name
WORD engineSocket = ENGINE_SOCKET;            // primary dispatch socket
WORD initSocket = INIT_SOCKET;                //    initialization socket
int shutdown = 0, shutdownOK = 0;             // for a clean unload
FILE *fp;                                     // data base file handle
LONG opNodeSpinCount, opNodeHighSpinCount = 0;// information variables
LONG recordNumber = 0;                        // how many records in the file
LONG SAPHandle;                               // Service Advertising Protocol
LONG addInProgress = 0;                       // control competing add threads
ENGINE_OP_NODE opNodes[6];                    // thread control structures


/*************************************************************************
*     DoOperation
*
*    This routine is spawned as an asynchronous thread every time
*    the engine recieves a request packet from the client. The
*    parameter is a pointer to a structure which contains an op code
*    plus a copy of the request packet.
*
*    Because DoOperation is spawned as a thread, there can be multiple
*    instances of it executing at one moment. This allows the engine
*    to service many clients at once, with no single client taking a
*    disproportionate performance hit because of the other clients
*
**************************************************************************/
void DoOperation(ENGINE_OP_NODE *callBackNode)
{
    int ccode;

    switch(callBackNode->packet.operation)
    {
        case 0x07:    /* mark an existing record as deleted */

        case 0xf6:    /* edit an existing record            */
        
        case 0x02: { AddRecord(callBackNode); break; }

        case 0x03: { FindRecordKey(callBackNode); break; }

       default: break;
    }
    callBackNode->packet.operation = 0;
    ExitThread(EXIT_THREAD, 0);
}

/*************************************************************************
*     EngineMain
*
*    This is the primary dispatcher of work for the engine. It is started
*    as a thread by main. EngineMain has two purposes:
*              1) Listen for request packets
*              2) Start a worker thread to service each request packet
*
*    EngineMain can receive up to 6 request packets at the same instant
*    without losing a packet. Six packets is usually sufficient to service
*    over 100 clients. However, if the engine were dropping packets, it
*    could be altered to recieve more packets by increasing the number
*    of Event Control Block (ECB) structures.
*
*    When EngineMain receives a request packet, it copies that
*    packet to an ENGINE_OP_NODE structure and starts a worker thread.
*    It then recycles the ECB which received the packet, making it available
*    to receive further packets even before the worker thread is finished
*    executing the request. This makes it possible to handle large numbers
*    of packets without losing any.
*
*************************************************************************/
void EngineMain(void *param)
{
    int ccode, index;
    IPX_ECB listenECBs[6];         // ECBs for receiving packets
    IPX_ECB *thisECB, *tempECB;    // pointers to unlink ECBs from the
    IPX_ECB *queueHead = NULL;     // list of received packets
    IPX_HEADER listenHeaders[6];   // IPX packet headers
    IPX_HEADER *thisHeader;        // pointer for copying header to OpNode
    lPacket listenPackets[6];      // Buffers for received packets
    lPacket *thisPacket;           // pointer for copying packet to OpNode
    rec tempRec;                   //    buffer for updating data base file
                                   //    upon shutdown.
   if (param)
      ;

   printf("\nEngine Main Dispatcher started");

   /* fill out our listen ECBs and post them */

   for (index = 0; index < 6; index++)
   {
        opNodes[index].packet.operation = 0;
        listenECBs[index].queueHead = &queueHead;
        listenECBs[index].semHandle = requestSem;
        listenECBs[index].socket = ENGINE_SOCKET;
        listenECBs[index].fragCount = 2;
        listenECBs[index].fragList[0].fragAddress = &listenHeaders[index];
        listenECBs[index].fragList[0].fragSize = sizeof(IPX_HEADER);
        listenECBs[index].fragList[1].fragAddress = &listenPackets[index];
        listenECBs[index].fragList[1].fragSize = sizeof(lPacket);
        ccode = IpxReceive(0, &listenECBs[index]);
    }
    /* When we get here, all six ECBs are listening for IPX packets */
    /* sent to the engine by clients. */

    while(!shutdown)
    {
        /* At this point the EngineMain thread goes to sleep until */
        /* a client sends a request packet. When a request packet */
        /* comes in, the OS links the ECB which describes the packet */
        /* to a list of 'received ECBs.' Then, the OS wakes the */
        /* EngineMain thread up by signaling the requestSem semaphore. */

        ccode = WaitOnLocalSemaphore(requestSem);
        if (shutdown)
            break;

        /* IpxGetAndClearQ returns a linked list of all ECBs which */
        /* describe a received packet. The length of the list depends */
        /* on how many packets the engine has received. */

        thisECB = IpxGetAndClearQ(&queueHead);
        while(thisECB)
        {

            /* Traverse the list of ECBs which describe received packets */
            /* and  dispatch worker threads to perform the operations */
            /* they request. */

            thisHeader = (IPX_HEADER *)thisECB->fragList[0].fragAddress;
            thisPacket = (lPacket *)thisECB->fragList[1].fragAddress;

            /* Now we need to schedule a worker thread to do our operation. */
            /* First, find a free opNode and fill out the callback structure*/

            index = opNodeSpinCount = 0;
            while(1)
            {
                 if (index == 6)
                 {
                    ThreadSwitch();
                    index = 0;
                 }
                 if (opNodes[index].packet.operation == 0)
                 {
                    ThreadSwitch();
                    break;
                 }
                 index++;
                 opNodeSpinCount++;
            }
            if (opNodeSpinCount > opNodeHighSpinCount)
                opNodeHighSpinCount = opNodeSpinCount;

            /* Now that we have a free operation node, fill it out so */
            /* we can start a worker thread to perform the operation. */
            memcpy(&opNodes[index].packetHeader,thisHeader,
		                                       sizeof(IPX_HEADER));
            memcpy(&opNodes[index].packet, thisPacket, sizeof(lPacket));
        
            /* Dispatch the operation by starting a worker thread */
            ccode = BeginThread(DoOperation, NULL, 0x1000, &opNodes[index]);

            /* Now that we've dispatched the operation */
            /* we can re-post the ECB for further listening. This makes the */
            /* ECB available for receiving packets even before the worker */
            /* thread has performed the requested operation! */

            tempECB = thisECB;
            thisECB = thisECB->next;
            ccode = IpxReceive(0, tempECB);
            }
    }
    printf("\nEngine Main Dispatcher shutting down...");

    /* update the data base header and close the data file */
    ccode = fseek(fp, 0L, SEEK_SET);
    ccode = fread(&tempRec, sizeof(rec), 1, fp);
    tempRec.header.recordNumber = recordNumber;
    ccode = fwrite(&tempRec, sizeof(rec), 1, fp);

    fclose(fp);
    shutdownOK = 1;
    printf("Down");
    return;
}

/*************************************************************************
*     InitMain
*
*     This thread listens for query packets from clients. Clients send a
*     query packet when they want to establish a session with the engine.
*
*     When InitMain recieves a query packet from a client, it sends back
*     a response packet to the client. This allows the client to verify
*     that the engine is up and running before requesting any operations
*     of it.
*
*     Unlike EngineMain, InitMain is only capable of receiving one packet
*     at a given instant. However, this is no problem because clients
*     only send a query packet once to establish a session, after which
*     they send request packets exclusively.
*
*     InitMain is started as an asynchronous thread by main.
*
*************************************************************************/
void InitMain(void *param)
{
    IPX_ECB listenECB, sendECB;               // ECBs for send and receive
    IPX_HEADER listenHeader, sendHeader;      // IPX headers for send and rec.
    iPacket initPacket;                       // Packet buffer
    int ccode;
    unsigned long transportTime;              // Used for IpxGetLocalTarget

    if (param)
      ;

    /* Initialize the ECBs. We use the same packet buffer for */
    /* receiving AND sending. In this case, it's OK to do so, */
    /* because this thread only sends a packet after it has */
    /* received a packet, and it cannot receive further packets */
    /* until it has sent a packet. */

    listenECB.queueHead = sendECB.queueHead = NULL;
    listenECB.socket = sendECB.socket = INIT_SOCKET;
    listenECB.semHandle = replySem;
    listenECB.fragCount = 0x02;
    listenECB.fragList[0].fragAddress = &listenHeader;
    listenECB.fragList[0].fragSize = sizeof(IPX_HEADER);
    listenECB.fragList[1].fragAddress = &initPacket;
    listenECB.fragList[1].fragSize = sizeof(iPacket);

    sendECB.fragCount = 0x02;
    sendECB.fragList[0].fragAddress = &sendHeader;
    sendECB.fragList[0].fragSize = sizeof(IPX_HEADER);
    sendECB.fragList[1].fragAddress = &initPacket;
    sendECB.fragList[1].fragSize = sizeof(iPacket);

    sendHeader.packetType = 0x04;            // indicates an IPX packet

    while (!shutdown)
    {
        /* Start listening for query packets. */
        ccode = IpxReceive(0, &listenECB);

        /* Go to sleep until IPX receives a query packet/ */
        /* The OS will wake us up when a packet comes in. */
        
        ccode = WaitOnLocalSemaphore(replySem);

        /* We received a query packet! */

        printf("\nInit packet received");
        memcpy(&sendHeader.destNet, &listenHeader.sourceNet, 10);
        sendHeader.destSocket = INIT_SOCKET;

        /* Obtain the address of the nearest router that knows */
        /* the network path to the client which sent this packet. */

        ccode = IpxGetLocalTarget((unsigned char *)&listenHeader.sourceNet,
                                &sendECB, &transportTime);
        if (ccode)
            printf("\nError getting immediate address (InitMain): %#x",ccode);

        /* Put the name of the data base file in the response packet. */
        strcpy(initPacket.dataFilePath, dataFileName);

        /* Send a response packet back to the client. */
        ccode = IpxSend(0, &sendECB);
        if (ccode)
            printf("\nError sending init packet (InitMain): %#x", ccode);
        else
            printf("\nInit response packet sent");
    }
    return;
}
/**************************************************************************
*    AddRecord
*
*    This is the work-horse function of the entire NLM. AddRecord is called
*    by a worker thread (DoOperation) whenever a client makes a request
*    to do one of the following:
*
*                1) Add a new record
*                2) Edit (update) an existing record
*                3) Read an existing record
*                4) Mark a record as deleted
*
*    AddRecord does not distinguish between ops 2 and 3: the client handles
*    the difference. For example, when a client receives a record to edit,
*    it allows the user to enter a new data field, then makes an AddRecord
*    request. When the client receives a record to read, it merely
*    displays the record data and does not allow the client to edit it. 
*
**************************************************************************/
void AddRecord(ENGINE_OP_NODE *opNode)
{
    int ccode, index, duplicate = 0;
    IPX_ECB sendECB;                  // ECB for sending confirmation packet
    IPX_HEADER sendHeader;            // IPX header
    sPacket sendPacket;               // packet buffer for sending conf. pkt
    unsigned long transportTime;      // for use with IpxGetLocalTarget
    rec tempRec;                      // buffer for reading data base file
    
    if(opNode->packet.operation == 0x02)    /* client adding a new record */
    {
        printf("\nReceived AddRecord request...");

        /* Check to see if another worker thread is currently adding a */
        /* record. If one is, wait until it is finished adding the new */
        /* record. This is the only case where one worker thread must */
        /* wait on another. But it must do so in order to retain the */
        /* integrity of the check for duplicate record keys. We do this */
        /* using a global variable rather than record locks because */
        /* a global variable is more efficient and serves our purpuse */
        /* just as well in this case. */

        while(addInProgress)
            ThreadSwitch();

        /* Tell other worker threads I am adding a new record. */
        addInProgress = 1;

        /* Try to find a deleted record. This is kind of slow, and */
        /* would be greatly speeded by using an index file or a */
        /* auxiliary list of deleted records.*/

        printf("\nSeeking ...");
        ccode = fseek(fp, 0L, SEEK_SET);
        if (ccode)
        {
            printf("\nError seeking to first record (AddRecord)");
            return;
        }
        printf("\n");
        tempRec.header.status =  1L;
        opNode->packet.record.header.offset = 0L;

        /* Read the file. We are looking for two things: a free record */
        /* and also a duplicate record key. */
        while(tempRec.header.status)
        {
            fread(&tempRec, sizeof(rec), 1, fp);
            if (!strcmp(tempRec.header.key, opNode->packet.record.header.key)
                 && tempRec.header.status)
                    duplicate = 1;
            if (!tempRec.header.status)
                     break;
            opNode->packet.record.header.offset += sizeof(rec);
            if (opNode->packet.record.header.offset 
		 >= recordNumber * sizeof(rec))
                     break;
            ThreadSwitch();
        }
        /* If we read the entire file without finding a free record, */
        /* extend the file. But don't extend the file if the client */
        /* is trying to add a record with a duplicate key. */
        if (tempRec.header.status != 0 && !duplicate)
        {
            ccode = ExtendDataFile(0x64);
            if (ccode)
                return;
        }
        else
        {
            /* If we found a free record without extending the file, we */
            /* need to continue reading the record to search for a possible */
            /* duplicate key. */

            while( tempRec.header.offset <=
                      ((recordNumber + 1) * sizeof(rec)) && 
                    !duplicate)
            {
                fread(&tempRec, sizeof(rec), 1, fp);
                if (!strcmp(tempRec.header.key, 
		            opNode->packet.record.header.key)
                     && tempRec.header.status)
                        duplicate = 1;
                break;
            }
        }
        if (duplicate)
        {
            printf("\nDuplicate record key found");
            opNode->packet.record.header.status = 0x03;
        }
        else
        {
            /* We found a free record, and the client is not trying to */
            /* add a record with a duplicate key. Fill out the record */

            printf("filling record... ");
            opNode->packet.record.header.status = 0xffffffff;
            opNode->packet.record.header.hashkey = 0;
            opNode->packet.record.header.recordNumber =
		                           tempRec.header.recordNumber;
            opNode->packet.record.data.creationTime = \
                opNode->packet.record.data.lastReferenceTime = \
                opNode->packet.record.data.lastUpdateTime = time(NULL);
            memcpy(opNode->packet.record.data.nodeAddress,
                    &opNode->packetHeader.sourceNet, 10);

            /* Prepare for writing the record by seeking to the offset */
            /* of the free record. */
            ccode = fseek(fp, opNode->packet.record.header.offset, SEEK_SET);
            if (ccode)
            {
                printf("\nError seeking to new record offset (AddRecord)");
                return;
            }
        }
    }
    if (opNode->packet.operation == 0xf6)  /* updating an existing record */
    {
        /* The client wants to update an existing record. We don't need */
        /* to search the data file because the client has provided */
        /* us with the offset of the record. */

        printf("\nReceived UpdateRecord request...");

        /* Check for a bad record offset */
        if (opNode->packet.record.header.offset >= recordNumber * sizeof(rec))
        {
            printf("\nBad record offset received (UpdateRecord)");
            return;
        }

        /* Prepare for writing the updated record by seeking to its offset */
        ccode = fseek(fp, opNode->packet.record.header.offset, SEEK_SET);
        if (ccode)
        {
            printf("\nError seeking to record offset (UpdateRecord)");
            return;
        }
        /* obtain the current time and node address of the client */
        /* which is updating the record. */
        opNode->packet.record.data.lastUpdateTime = time(NULL);
        memcpy(opNode->packet.record.data.nodeAddress,
                &opNode->packetHeader.sourceNet, 10);
    }
    if (opNode->packet.operation == 0x07) /* mark record as deleted */
    {
        /* The client wants to mark an existing record as deleted. */
        /* We don't need to search the data file because the client */
        /* has provided us with the offset of the record. */

        printf("\nReceived DeleteRecord request...");

        /* Check for a bad offset */
        if (opNode->packet.record.header.offset >= recordNumber * sizeof(rec))
        {
            printf("\nBad record offset received (DeleteRecord)");
            return;
         }

        /* prepare for marking the record as deleted */
        /* by seeking to its offset */

        ccode = fseek(fp, opNode->packet.record.header.offset, SEEK_SET);
        if (ccode)
        {
            printf("\nError seeking to record offset (DeleteRecord)");
            return;
        }
        /* Indicate this is a deleted record. */
        opNode->packet.record.header.status = 0;
    }

    if (!duplicate)
    {
        /* This is were we actually write the new, updated, or deleted */
        /* record to the data base file. We use Transaction Tracking */
        /* To ensure failed or partial writes do not damage the integrity */
        /* of the data file. Transaction Tracking performs all necessary */
        /* record locks for us automatically. */

        printf("TTSBegin ...");
        ccode = TTSBeginTransaction();
        if (ccode && ccode != 0xff)
        {
            if(opNode->packet.operation == 0x02)    /* adding a new record */
                addInProgress = 0;
            return;
        }
        printf("writing ...");
        ccode = fwrite(&opNode->packet.record, sizeof(rec), 1, fp);
        if (ccode != 1)
        {
            TTSAbortTransaction();
            if(opNode->packet.operation == 0x02)    /* adding a new record */
                addInProgress = 0;
            return;
        }
        else
        {
           printf("TTSEnd\n");
           TTSEndTransaction(&opNode->packet.record.header.transactionNumber);
        }
        if(opNode->packet.operation == 0x02)    /* adding a new record */
            addInProgress = 0;

        if (ccode != 1)
        {
            printf("\nNew record not written to data base");
            return;
        }
        else
        {
            printf("\nNew record:");
            printf("\n\trecordNumber: %#lx",
		              opNode->packet.record.header.recordNumber);
            printf("\n\toffset: %#lx", opNode->packet.record.header.offset);
            printf("\nKey: %s", opNode->packet.record.header.key);
            printf("\nData: %s", opNode->packet.record.data.data);
        }
    }
    if (duplicate && opNode->packet.operation == 0x02)
        addInProgress = 0;

    /* Now set up a reply packet and send to the client. When the client */
    /* receives this packet, it infers that the operation was successful. */
    /* We send a smaller packet containing only the record header. */

    sendECB.queueHead = NULL;
    sendECB.semHandle = NULL;
    sendECB.socket = ENGINE_SOCKET;
    sendECB.fragCount = 0x02;
    sendECB.fragList[0].fragAddress = &sendHeader;
    sendECB.fragList[0].fragSize = sizeof(IPX_HEADER);
    sendECB.fragList[1].fragAddress = &sendPacket;
    sendECB.fragList[1].fragSize = sizeof(sPacket);

    /* Obtain the network address of the nearest router which knows the */
    /* path to the client. */
    ccode = IpxGetLocalTarget(
                        (unsigned char *)&opNode->packetHeader.sourceNet,
                        &sendECB,
                        &transportTime);

    memcpy(&sendHeader.destNet, &opNode->packetHeader.sourceNet, 12);
    sendHeader.packetType = 4;

    memcpy(&sendPacket.responseCode, &opNode->packet.responseCode, 3);
    memcpy(&sendPacket.header,&opNode->packet.record.header, sizeof(rHeader));

    ccode = IpxSend(0, &sendECB);

    printf("\nOperation Successful.");
    return;
}

/*************************************************************************
*    FindRecord
*
*    FindRecord is called by a worker thread whenever a client makes a
*    find record request. The client makes a find record request as the
*    prelude to a read record, edit record, or delete record request.
*
*    If the engine finds the record requested by the client, it sends a
*    packet back to the client containing the entire record.
*
*    If the engine does NOT find the record requested by the client, it
*    sends a null record back to the client.
*
*************************************************************************/
void FindRecordKey(ENGINE_OP_NODE *opNode)
{
    rec searchRec;                    // buffer for reading data base file
    int ccode;
    IPX_ECB sendECB;                  // ECB for sending confirmation packet
    IPX_HEADER sendHeader;
    lPacket sendPacket;
    unsigned long transportTime;

    /* We assume that the packet contains a record header with the key */
    /* filled out. We search on the key. Other possibilities include */
    /* searching on the record number, transaction number, and
    /* going directly to a record offset supplied by the client. */
    /* A production version of this application would supply some of */
    /* these more complex searching algorithms. */

    printf("\nReceived FindRecordKey request...");
    
    /* Seek to the beginning of the data file and search for the */
    /* record key. This is a relatively slow method. A production */
    /* version of the engine would use an indexing method to speed */
    /* searches for large data base files. */

    ccode = fseek(fp, 0L, SEEK_SET);
    if (ccode)
        return;

    while(1)
    {
        ccode = fread(&searchRec, sizeof(rec), 1, fp);
        if (ccode != 1)
            break;
        if (searchRec.header.status == 0) /* deleted record */
            continue;
        if (strcmp(opNode->packet.record.header.key, searchRec.header.key))
            continue;
        else
            break;
    }
    if (ccode != 1)        /* we didn't find the record */
        sendPacket.record.header.status = 0L;
    else
        memcpy(&sendPacket.record, &searchRec, sizeof(rec));

    /* Initialize the send ECB, IPX header, and packet buffer. */
    sendECB.queueHead = NULL;
    sendECB.semHandle = NULL;

    /* Obtain the network address of the nearest router which */
    /* knows the path to the client. */

    ccode = IpxGetLocalTarget(
                        (unsigned char *)&opNode->packetHeader.sourceNet,
                        &sendECB,
                        &transportTime);
    sendECB.socket = ENGINE_SOCKET;
    sendECB.fragCount = 2;
    sendECB.fragList[0].fragAddress = &sendHeader;
    sendECB.fragList[0].fragSize = sizeof(IPX_HEADER);
    sendECB.fragList[1].fragAddress = &sendPacket;
    if (sendPacket.record.header.status == 0)
        sendECB.fragList[1].fragSize = sizeof(sPacket);
    else
        sendECB.fragList[1].fragSize = sizeof(lPacket);
    
    memcpy(&sendHeader.destNet, &opNode->packetHeader.sourceNet, 12);
    sendHeader.packetType = 4;
    memcpy(&sendPacket.responseCode, &opNode->packet.responseCode, 3);
    
    /* Send the packet to the client. */
    ccode = IpxSend(0, &sendECB);

    if (sendPacket.record.header.status != 0L)
    {
        /* now update the lastReferenceTime field for the record */
        /* We use Transaction Tracking to ensure partial or failed */
        /* writes do not destroy the integrity of the data base file */
        /* Transaction tracking performs all necessary record locking */
        /* for us. */

        searchRec.data.lastReferenceTime = time(NULL);
        ccode = fseek(fp, searchRec.header.offset, SEEK_SET);
        if (ccode)
            return;
        ccode = TTSBeginTransaction();
        if (ccode && ccode != 0xff)
            return;
        ccode = fwrite(&searchRec, sizeof(rec), 1, fp);
        if (ccode != 1)
        {
            TTSAbortTransaction();
            return;
        }
        else
            TTSEndTransaction(&searchRec.header.transactionNumber);
       printf("Operation Successful");
    }
    else
        printf("Record not found");

    return;
}

/*************************************************************************
*    CreateDataFile
*
*    This function is called by main when it cannot find the data file.
*    When you load the NLM, you can specify an alternative data file
*    path using the command line. The main routine uses the data file
*    path--if you supply it--to search for the data base file.
*
*************************************************************************/
int CreateDataFile(char *netWarePath)
{
    int ccode;
    rec temp;
    BYTE extendedAttributes;

    fp = fopen(netWarePath, "w+b");
    if (fp == NULL)
        return(-1);

    printf("\nData file created ...");

    /* set up a special first record that will serve as a identifier */
    /* and special stamp we can use to track certain information */

    temp.header.status = 0x02L;  /* status of 2 means it is the file header */
    temp.header.offset = 0L;
    temp.header.recordNumber = 0x64;
    strcpy(temp.header.key, "NetWare data base engine");

    ccode = fwrite(&temp, sizeof(rec), 1, fp);
    if (ccode != 1)
    {
        fclose(fp);
        ccode = unlink(netWarePath);
        return(-1);
    }

    /* Now write 100 empty records to the data base file */
    memset(&temp, 0x01, sizeof(rec));
    temp.header.status = 0x0L;

    for( temp.header.recordNumber = 0; 
         temp.header.recordNumber < 0x64; 
         temp.header.recordNumber++)
    {
        temp.header.offset = ftell(fp);
        ccode = fwrite(&temp, sizeof(rec), 1, fp);
        if (ccode != 1)
        {
            printf("\nError extending data base file (CreateDataFile)");
            return(-1);
        }
    }
    fclose(fp);
    printf("file header written.");

    /* Set the transaction bit of the file's extended attribute byte. */
    /* This allows us to use Transaction Tracking Services (TTS) when */
    /* writing records to the file. */

    printf("\nSetting Transaction bit");

    ccode = GetExtendedFileAttributes(dataFileName, &extendedAttributes);

    if (ccode)
        printf("\nError getting extended attributes");

    extendedAttributes |= 0x10;        // turn transaction bit ON

    if (!ccode)
        ccode = SetExtendedFileAttributes(dataFileName, extendedAttributes);
    if (ccode)
        printf("\nError setting extended attributes");
    return(0);
}

/**************************************************************************
*    ExtendDataFile
*
*    This routine is called by AddRecord whenever it reads the entire
*    data base file without finding a free record. The parameter,
*    newRecords, determines by how many records the data file will be
*    extended.
*
**************************************************************************/
int ExtendDataFile(int newRecords)
{
    rec tempRec;
    int ccode, index;
    long oldOffset;

    oldOffset = ftell(fp);
    fclose(fp);
    fp = fopen(dataFileName, "ab");
    if (fp == NULL)
        return(-1);

    memset(&tempRec, 0x01, sizeof(rec));
    tempRec.header.status = 0x0L;
    for (index = 1; index <= newRecords; index++)
    {
        tempRec.header.offset = ftell(fp);
        tempRec.header.recordNumber =  (recordNumber + index);
        ccode = fwrite(&tempRec, sizeof(rec), 1, fp);
        if (ccode != 1)
        {
            printf("\nError extending data base file (ExtendDataFile)");
            return(-1);
        }
    }
    recordNumber += newRecords;
    ccode = fseek(fp, 0L, SEEK_SET);
    if (ccode)
    {
        printf("\nError reading the data base file header (ExtendDataFile)");
        return(-1);
    }
    tempRec.header.recordNumber = recordNumber;

    ccode = fwrite(&tempRec, sizeof(rec), 1, fp);
    if (ccode != 1)
    {
        printf("\nError writing updated header"
	       "to database file (ExtendDataFile)");
        return(-1);
    }
    printf("\n%#x New records successfully added to data file", newRecords);

    fclose(fp);
    fp = fopen(dataFileName, "ab+");
    if (fp == NULL)
    {
        printf("\nError re-opening data file (ExtendDataFile)");
        return(-1);
    }
    fseek(fp, oldOffset, SEEK_SET);
    return(0);
}

/**************************************************************************
*    OpenDataFile
*
*    This routine is called by main every time the engine is loaded.
*    It opens the file specified by the command line (if the user
*    specified one) or the default (SYS:$ENGINE$.DAT).
*
*    OpenDataFile reads the first record of the file to verify the
*    data base file signature and to determine how many records are
*    contained in the file.
*
**************************************************************************/
int OpenDataFile(char *netWarePath)
{
    int ccode;
    rec temp;

    fp = fopen(netWarePath, "r+b");
    if (fp == NULL)
        return(-1);

    printf("\nData file opened...");

    /* read the file header to verify it is our data file */

    ccode = fread(&temp, sizeof(rec), 1, fp);
    if (ccode != 1)
    {
        fclose(fp);
        return(-1);
    }

    /* now do checks to verify the file */

    if (temp.header.status != 0x02)
    {
        printf("\nBad status ID on file header");
        return(-1);
    }
    if (strcmp(temp.header.key, "NetWare data base engine"))
    {
        printf("Bad signature on file header");
        return(-1);
    }
    /* initialize the global variable recordNumber */
    recordNumber = temp.header.recordNumber;

    /* return with the file pointer located just after the file header */
    printf("file signature verified.");
    return(0);
}

/**************************************************************************
*    UnloadCleanUp
*
*    This routine was registered with the OS in main using the signal API.
*
*    The OS calls this routine whenever a user UNLOADS the engine NLM.
*    When UnloadCleanUp is called by the OS, all engine threads are still
*    alive. This gives us a chance to kick alive any threads which are
*    asleep, letting them clean up before they die.
*
**************************************************************************/
void UnloadCleanUp(int sig)
{
    sig = sig;

    shutdown = 1;
    SignalLocalSemaphore(replySem);
    SignalLocalSemaphore(requestSem);
    SignalLocalSemaphore(shutdownSem);
    return;
}

/**************************************************************************
*    ShutdownCleanUp
*
*    This routine was registered with the OS in main using the atexit API.
*
*    The OS calls this routine when the NLM is about to die. When
*    ShutdownCleanUp is called, all threads are dead. At this point all we
*    can do is free OS resources allocated during the life of the NLM.
*
**************************************************************************/
void ShutdownCleanUp(void)
{
    int ccode;

    ccode = CloseLocalSemaphore(replySem);
    ccode = CloseLocalSemaphore(shutdownSem);
    ccode = CloseLocalSemaphore(requestSem);
    ccode = IpxCloseSocket(ENGINE_SOCKET);
    ccode = IpxCloseSocket(INIT_SOCKET);
    ccode = ShutdownAdvertising(SAPHandle);
    return;
}

/*************************************************************************/
int main(int argc, char **argv)
{
    int ccode;
    DIR *dirStruct, *tempDir;
    char server[_MAX_SERVER + 1];
    char volume[_MAX_VOLUME + 1];
    char directories[_MAX_DIR + 1];
    BYTE extendedAttributes;

    /* Check to see that TTS is enabled on the server. */
    ccode = TTSIsAvailable(); 
    if (ccode != 0xff)
    {
        printf("\nTTS is currently disabled ...");
        printf("\nPlease enable it and re-load ENGINE.NLM");
        return(ccode);
    }
    if (argc < 2)
        strcpy(dataFileName, "$engine$.dat");
    else
        strncpy(dataFileName, argv[1], _MAX_PATH);

    /* check to ensure we have a valid path or, if we have a */
    /* partial path, to obtain the server and volume names */

    ccode = ParsePath(dataFileName, server, volume, directories);

    /* if the pathname was invalid, return with a -1 */
    if (ccode)
    {
         printf("\nInvalid path for data file... Exiting");
         return(-1);
    }
    /* now look for the file to see if it already exists */
    dirStruct = tempDir = opendir(dataFileName);

    readdir(tempDir);
    if (tempDir == NULL)
        ccode = CreateDataFile(dataFileName);

    ccode = OpenDataFile(dataFileName);

    if (ccode == -1)      /* err creating or opening the file */
    {
        closedir(dirStruct);
        printf("\nError opening or creating the data file... Exiting");
        return(ccode);
    }
    closedir(dirStruct);    /* close the dir node */

    /* open up the Engine's IPX socket and two semaphores for thread control*/
    IpxOpenSocket(&engineSocket);
    IpxOpenSocket(&initSocket);
    printf("\nIPX Sockets opened.");

    requestSem = OpenLocalSemaphore(0L);
    replySem = OpenLocalSemaphore(0L);

    /* open up a semaphore that will tell when to shut down */
    shutdownSem = OpenLocalSemaphore(0L);

    printf("\nNetWare Semaphores opened.");

    /* Now start advertising the ENGINE over the network */
    SAPHandle = AdvertiseService(0x88, "Data_Base_Engine");
    printf("\nSAP Handle: %#08lx", SAPHandle);

    /* Now we need to register functions that will execute */
    /* if somebody unloads ENGINE.NLM at the server console. */
    signal(SIGTERM, UnloadCleanUp);
    atexit(ShutdownCleanUp);

    printf("\nExit procedures registered with OS.");

    /* now that we have the file open and ready to go, we can */
    /* start the main thread group */
    ccode = BeginThread(InitMain, NULL, 0x1000, NULL);

    ccode = BeginThreadGroup(EngineMain, NULL, 0x2000, NULL);
    if (ccode)
        return(-1);

    /* Go to sleep until it's time to shut down. */
    WaitOnLocalSemaphore(shutdownSem);

    while(!shutdownOK)
        ThreadSwitch();    /* give worker threads a chance to cleanup */
    return(0);
}

