//-------------------------------------------------------------------------
//			ActionMedia II Programmer's Toolkit
//			
//			Windows Motion Capture Sample Program
//
// Application:	AvkCapt.exe
// Module Name:	acrect.c
//
// description:	Convert a Windows RECT's physical screen coordintes into 
//				an AVK BOX according to the AVK View's resolution.
//
// Copyright Intel Corp. 1991, 1992
// All Rights Reserved.
//
//-------------------------------------------------------------------------
//
// Exported functions from this module:
//
//		WinRect2AvkBox
//
//-------------------------------------------------------------------------
#include <windows.h>
#define	ACRECT_NOEXTERNS	1
#include "avkcapt.h"

//-------------------------------------------------------------------------
//	Macro to scale physical screen resolution to AVK View resolution.
//	The physical screen coordinate is multiplied by the conversion
//	delta which is arrived at by dividing the AVK View resolution by the
//	physical screen resolution. The conversion is done in double precision and
//	the resulting value, truncated to an integer, is the AVK View coordinate.
//	
//	Usage is as follows:
//
//		xAvkRes = SCREEN2AVK(xPhysRes, xDelta)
//		yAvkRes = SCREEN2AVK(yPhysRex, yDelta)
//
//-------------------------------------------------------------------------

#define	SCREEN2AVK(r, d)	(int)((double)(r) * (double)(d))

//-------------------------------------------------------------------------
//	Macro to truncate to the actual view (Windows coords can be off the 
//	screen, but AVK coords can't.
//-------------------------------------------------------------------------

#define	TRUNC(x, y)			(max(min((x), (y) - 1), 0))

//-------------------------------------------------------------------------	 
//FUNCTION:																	 
//	
//	VOID WinRect2AvkBox(pWinRect, pAvkBox, pView)
//
//PARMS IN:
//
//	RECT	*pWinRect;		Window's rectangle with window coordinates
//								as absolute screen coordinates (NOT 
//								relative to the window frame!).
//	BOX		*pAvkBox;		Box to be set to the AVK coordinates that
//								correspond to pWinRect's screen coordinates
//	VIEW	*pView;			View control structure.
//
//DESCRIPTION:
//
//	Converts absolute screen coordinates to AVK plane coordinates.  The
//	screen coordinates are based on the physical screen resolution (e.g.,
//	640x480 for VGA or 1024x768 for XGA.  The coordinate system is the
//	lower right quadrant of a Cartesian coordinate system where x=0,y=0 
//	is the upper left corner of the screen and x=xres-1,y=yres-y is the 
//	lower right corner.  AVK plane coordinates are determined by the
//	x and y resolutions used to create the current View in AvkViewCreate().
//	They are also based on the lower right Cartesian quadrant.
//
//	The conversion algorithm is:
//
//		Xavk = (int)((double)Xphys * ((double)Xviewres / (double)Xphysres))
//		
//-------------------------------------------------------------------------

VOID
WinRect2AvkBox(RECT *pWinRect, BOX *pAvkBox, VIEW *pView)
{
	//	If the View's screen coordinates are set, then we have initialized
	//	the View structur and we can set the destination box coordinates.

	if (pView->cxScreen)
	{
		//	Scale the Windows coordinates of the Windows rectangle to
		//	AVK coordinates.

		pAvkBox->x1 = SCREEN2AVK(pWinRect->left, 	pView->xDelta);
		pAvkBox->y1 = SCREEN2AVK(pWinRect->top, 	pView->yDelta);
		pAvkBox->x2 = SCREEN2AVK(pWinRect->right, 	pView->xDelta);
		pAvkBox->y2 = SCREEN2AVK(pWinRect->bottom, pView->yDelta);

		//	Truncate the AVK coordinates to the actual screen size (the 
		//	source Windows coords may have been offscreen).
								   
		pAvkBox->x1 = TRUNC(pAvkBox->x1, pView->cxView);
		pAvkBox->y1 = TRUNC(pAvkBox->y1, pView->cyView);
		pAvkBox->x2 = TRUNC(pAvkBox->x2, pView->cxView);
		pAvkBox->y2 = TRUNC(pAvkBox->y2, pView->cyView);

		//	If the bitmap format is a 9-bit subsampled format, we align
		//	the destination box on 4-pixel boundaries and size it to
		//	multiples of 4 pixels to avoid edge effects from trying to 
		//	interpolate imcomplete U and V plane data.

		if (pView->BmFmt == AVK_YUV9)
		{
			//	Align the the upper left coordinates by rounding down to 
			//	a multiple of 4 pixels.

			pAvkBox->x1 &= ~0x03;
			pAvkBox->y1 &= ~0x03;

			//	Make the width and depth of the box multiples of 4 by 
			//	rounding the lower right coordinates up to one less 
			//	than a multiple of 4.

			pAvkBox->x2 |=  0x03;
			pAvkBox->y2 |=  0x03;
		}
	}

}

