/**************************************************************************************************
*
*		Title:	XPX_OPEN.C
*		Copyright (c) January 1992, Ryu Consulting, 916/722-1939
*		Written by Rahner James
*
*		This file contains all the functions that support openning and closing an IPX/SPX stream
*
**************************************************************************************************/

#define	_XPX_OPEN_C_
#include	<dos.h>
#include	<malloc.h>
#include	<stdlib.h>
#include	<time.h>
#include	"network.h"


/**************************************************************************************************
*
*									Global Data
*
**************************************************************************************************/

XPX_STREAM_T far *_First_Stream = NULL;		// -> first stream structure allocated by this application
int _Total_Open_Streams = 0;				// Number of open streams
SPX_PACKET_T far *_First_Nomatch = NULL;	// -> first received packet that does not match a stream address
SPX_PACKET_T far *_Last_Nomatch = NULL;		// -> last received packet that does not match a stream address
int _Total_Nomatchs = 0;					// Count of the packets of nonmatched packets
int _Wait_Timeout = 5 * 18;					// Number of system ticks to wait for a response
char	_Asynch_Flag = 0;					// Set to !0 if all writes will be asynchronous
char	_Ignore_Nomatch = 1;				// Set to 0 if non matching packets should be kept

int _Default_Talkers = DEFAULT_TALKERS;		// Number of talker packets created when a stream is opened
int _Default_Listeners = DEFAULT_LISTENERS;	// Number of listening packets created when a stream is opened
void (far *_Default_Talk_ESR)() = talk_esr;	// -> default talking ESR
void (far *_Default_Listen_ESR)() = listen_esr;	// -> default listening ESR


/**************************************************************************************************
*
*	XPX_STREAM_T far *FIND_STREAM( NET_ADDRESS_T far *NP )
*	Finds an open stream that is currently communicating with the destination network
*
*	Given:
*		NP -> full network address to find
*
*	Returns:
*		Stream structure address if found, NULL if not
*
**************************************************************************************************/
XPX_STREAM_T far *find_stream( NET_ADDRESS_T far *np )
{
	XPX_STREAM_T far *rv;

	for ( rv=_First_Stream ; rv != NULL ; rv=rv->next )
	{
		if ( farcmpw(&rv->dest,np,sizeof(NET_ADDRESS_T)) == 0 )
			break;
	}

	return rv;
}


/**************************************************************************************************
*
*	int CREATE_TALKER( XPX_STREAM_T far *SP )
*	Allocates and creates a talking IPX_PACKET structure for a stream
*
*	Given:
*		SP -> stream structure to add a talker to
*
*	Returns:
*		0 if the talker was created properly
*
**************************************************************************************************/
int create_talker( XPX_STREAM_T far *sp )
{
	SPX_PACKET_T far *ip;
	int packet_size;

/*
** Allocate and initialize a buffer for the packet
*/
	packet_size = (sp->open_flags & XPX_SPX) ? sizeof(SPX_PACKET_T) : sizeof(IPX_PACKET_T);
	if ( (ip=halloc(packet_size,1)) == NULL )
		return ERR_MEMORY;
	farset( ip, 0, packet_size );

	packet_size = (sp->open_flags & XPX_SPX) ? sizeof(SPX_T) : sizeof(IPX_T);
	if ( _Asynch_Flag == 0 )
	{
		if ( (ip->default_buffer=halloc(MAX_PACKET_SIZE-packet_size,1)) == NULL )
		{
			hfree( ip );
			return ERR_MEMORY;
		}
		ip->default_size = MAX_PACKET_SIZE - packet_size;
	}

	ip->function = _Default_Talk_ESR;
	ip->socket = sp->dest.socket;
	farmove( ip->dest_address, sp->local_target, NODE_ADDRESS_SIZE );
	ip->hdr = (SPX_T far *)&ip->checksum;
	ip->size_hdr = packet_size;
	ip->type = (sp->open_flags & XPX_SPX) ? SPX_TYPE : IPX_TYPE;
	farmove( &ip->dest, &sp->dest, sizeof(NET_ADDRESS_T) );

/*
** Add the new packet descriptor to the list of allocated and free packets
*/
	ip->parent = sp;
	_disable();
	ip->next_allocated = (SPX_PACKET_T far *)sp->first_allocated;
	sp->first_allocated = (SPX_PACKET_T far *)ip;
	if ( sp->last_allocated == NULL )
		sp->last_allocated = (SPX_PACKET_T far *)ip;
	ip->next_sibling = sp->first_free;
	sp->first_free = (SPX_PACKET_T far *)ip;
	_enable();
	++sp->total_talkers;
	++sp->free_count;

	return 0;
}


/**************************************************************************************************
*
*	int CREATE_LISTENER( XPX_STREAM_T far *SP )
*	Allocates and creates a listening xPX_PACKET structure for a stream
*
*	Given:
*		SP -> stream structure to add a listener to
*
*	Returns:
*		0 if the listener was created properly
*
**************************************************************************************************/
int create_listener( XPX_STREAM_T far *sp )
{
	SPX_PACKET_T far *ip;
	int packet_size;

/*
** Allocate and initialize a buffer for the packet
*/
	packet_size = (sp->open_flags & XPX_SPX) ? sizeof(SPX_PACKET_T) : sizeof(IPX_PACKET_T);
	if ( (ip=halloc(packet_size,1)) == NULL )
		return ERR_MEMORY;
	farset( ip, 0, packet_size );

	packet_size = (sp->open_flags & XPX_SPX) ? sizeof(SPX_T) : sizeof(IPX_T);
	if ( (ip->default_buffer=halloc(MAX_PACKET_SIZE-packet_size,1)) == NULL )
	{
		hfree( ip );
		return ERR_MEMORY;
	}
	ip->default_size = MAX_PACKET_SIZE - packet_size;

	ip->function = _Default_Listen_ESR;
	ip->socket = sp->dest.socket;
	ip->fragment_count = 2;
	ip->hdr = (SPX_T far *)&ip->checksum;
	ip->size_hdr = packet_size;
	ip->buffer_ptr = ip->default_buffer;
	ip->buffer_size = ip->default_size;

/*
** Add the new packet descriptor to the list of allocated packets
*/
	ip->parent = sp;
	_disable();
	ip->next_allocated = sp->first_allocated;
	sp->first_allocated = (SPX_PACKET_T far *)ip;
	if ( sp->last_allocated == NULL )
		sp->last_allocated = (SPX_PACKET_T far *)ip;
	_enable();
	++sp->total_listeners;

/*
** Now send the packet to IPX or SPX to listen like a good little chunk o' code
*/
	return ((sp->open_flags & XPX_SPX)?spx_add_listener:ipx_add_listener)( (ECB_HEADER_T far *)ip );
}


/**************************************************************************************************
*
*	int DESTROY_PACKET( SPX_PACKET_T far *PP )
*	Stops a packet and frees its allocated memory
*
*	Given:
*		PP -> packet to destroy
*
*	Returns:
*		Always returns 0
*
**************************************************************************************************/
int destroy_packet( SPX_PACKET_T far *pp )
{
	if ( pp != NULL )
	{
/*
** First, need to cancel the packet's communication ticket if it is being used
*/
		if ( pp->in_use != IU_DONE )
			ipx_cancel_ecb( (ECB_HEADER_T far *)pp );

/*
** Next, we just need to free up the memory
*/
		hfree( pp );
	}

	return 0;
}


/**************************************************************************************************
*
*	int XPX_CLOSE( XPX_STREAM_T far *SP )
*	Closes a stream and all the packets associated with it
*
*	Given:
*		SP -> stream structure to close
*
*	Returns:
*		0 if the stream was closed properly
*		ERR_POINTER if bad pointer provided to this function
*
**************************************************************************************************/
int xpx_close( XPX_STREAM_T far *sp )
{
	int	i, rv;
	SPX_PACKET_T far *pp, far *np;
	XPX_STREAM_T far *nsp;

#ifdef DEBUG_MODE
	if ( (sp=find_stream(&sp->dest)) == NULL )
		return ERR_NOT_OPEN;
#endif

/*
** Get rid of the talkers and listeners
*/
	for ( i=sp->total_talkers+sp->total_listeners, pp=sp->first_allocated ; i && (pp != NULL) ; --i, pp=np )
	{
		np = pp->next_allocated;
		if ( (rv=destroy_packet(pp)) != 0 )
			return rv;
	}

/*
** Get rid of the current stream consciousness
*/
	if ( _First_Stream == sp )
	{
		_disable();
		_First_Stream = sp->next;
		_enable();
	}
	else
	{
		for ( nsp=_First_Stream ; nsp != NULL ; nsp=nsp->next )
		{
			if ( nsp->next == sp )
			{
				_disable();
				nsp->next = sp->next;
				_enable();
				break;
			}
		}
	}
	hfree( sp );

	return 0;
}


/**************************************************************************************************
*
*	int XPX_OPEN( NET_ADDRESS_T *DP, int FLAGS, XPX_STREAM_T far **STREAM) )
*	Opens an IPX/SPX data stream pipe to some destination address
*
*	Given:
*		DP -> full address of the destination to communicate with
*		FLAGS = XPX_ flags used to define the attributes of the stream
*		STREAM -> variable in which to place the pointer to the allocated stream structure
*
*	Returns:
*		0 if the stream was opened OK
*		ERR_SOCKET_FULL if the socket table was full
*		ERR_MEMORY if not enough memory
*		ERR_IPX_TIMEOUT if timeout on wait
*		ERR_POINTER if bad pointer provided to this function
*		ERR_PARAMETER if bad flag content
*
*	Note:
*		If dp->network == 0, the current physical network will be assumed
*		If dp->node == 0xFFFFFFFFFF, a broadcast stream will be opened
*		If dp->socket == 0, a socket will be selected by IPX and used
*
**************************************************************************************************/
int xpx_open( NET_ADDRESS_T *dp, int flags, XPX_STREAM_T far **stream )
{
	int				rv, i;
	unsigned long	end_time;
	XPX_STREAM_T	far *sp;

/*
** First, check the passed pointers and parameters
*/
	if ( (dp == NULL) || (stream == NULL) )
		return ERR_POINTER;
	if ( (flags & XPX_RW) == 0 )
		return ERR_PARAMETER;
	if ( (flags & XPX_SPX) && isbroadcast(dp->node) )
		return ERR_NO_SUPPORT;

/*
** Setup the definition for our stream
*/
	if ( (sp=halloc(sizeof(XPX_STREAM_T),1)) == NULL )
		return ERR_MEMORY;
	farset( sp, 0, sizeof(XPX_STREAM_T) );
	farmove( &sp->dest, dp, sizeof(NET_ADDRESS_T) );
	if ( (rv=ipx_open_socket(&sp->dest.socket)) != 0 )
	{
		if ( rv != ERR_SOCKET_OPEN )
		{
			hfree( sp );
			return rv;
		}
	}
	if ( find_stream(&sp->dest) != NULL )
	{
		hfree( sp );
		return ERR_OPEN;
	}
	if ( (rv=ipx_target(sp->local_target,&sp->dest)) < 0 )
	{
		hfree( sp );
		return rv;
	}

/*
** Create the talkers if need be
*/
	if ( flags & XPX_WRITE )
	{
		for ( i=0 ; i<_Default_Talkers ; ++i )
		{
			if ( (rv=create_talker(sp)) != 0 )
			{
				xpx_close( sp );
				return rv;
			}
		}
	}

/*
** Create the listeners if that is the way it should be
*/
	if ( flags & XPX_READ )
	{
		for ( i=0 ; i<_Default_Listeners ; ++i )
		{
			if ( (rv=create_listener(sp)) != 0 )
			{
				xpx_close( sp );
				return rv;
			}
		}
	}

/*
** Include our newly created stream definition in the stream list
*/
	_disable();					// No interruptions, please
	sp->next = _First_Stream;
	_First_Stream = sp;
	_enable();					// Annoyances allowed

/*
** Wait for a message if bwana wanted us to wait
*/
	++_Total_Open_Streams;
	if ( flags & XPX_SPX )
	{
/*
** Take care of SPX establish connection
*/
		if ( (rv=xpx_write(sp,NULL,0,NULL)) != 0 )		// Note: SPX write of 0 bytes does an establish connection
		{
			xpx_close( sp );
			return rv;
		}
	}

/*
** See if we need to wait for our remote friend to talk to us
*/
	if ( (flags & (XPX_WAIT | XPX_READ)) == (XPX_WAIT | XPX_READ) )
	{
		end_time = clock() + _Wait_Timeout;
		while ( (sp->total_receptions == 0) && (clock() < end_time) )
			;
		if ( sp->total_receptions == 0 )
		{
			xpx_close( sp );
			return ERR_IPX_TIMEOUT;
		}
	}

	*stream = sp;
	return 0;
}
