/* bob.h - bob definitions */
/*
	Copyright (c) 1991, by David Michael Betz
	All rights reserved
*/

#include <stdio.h>
#include <ctype.h>

/* limits */
#define TKNSIZE		50	/* maximum token size */
#define SMAX		500	/* runtime stack size */
#define CMAX		32767	/* code buffer size */

/* useful definitions */
#define TRUE		1
#define FALSE		0

/* token definitions */
#define T_NOTOKEN	-1
#define T_EOF		0
#define T_STRING	1
#define T_IDENTIFIER	2
#define T_NUMBER	3
#define T_CLASS		4
#define T_STATIC	5
#define T_IF		6
#define T_ELSE		7
#define T_WHILE		8
#define T_RETURN	9
#define T_FOR		10
#define T_BREAK		11
#define T_CONTINUE	12
#define T_DO		13
#define T_NEW		14
#define T_NIL		15
#define T_LE		16	/* '<=' */
#define T_EQ		17	/* '==' */
#define T_NE		18	/* '!=' */
#define T_GE		19	/* '>=' */
#define T_SHL		20	/* '<<' */
#define T_SHR		21	/* '>>' */
#define T_AND		22	/* '&&' */
#define T_OR		23	/* '||' */
#define T_INC		24	/* '++' */
#define T_DEC		25	/* '--' */
#define T_ADDEQ		26	/* '+=' */
#define T_SUBEQ		27	/* '-=' */
#define T_MULEQ		28	/* '*=' */
#define T_DIVEQ		29	/* '/=' */
#define T_CC		30	/* '::' */
#define T_MEMREF	31	/* '->' */
#define _TMAX		31

/* stack manipulation macros */
#define check(n)	 { if (sp - (n) < stkbase) stackover(); }
#define chktype(o,t)	 { if (sp[o].v_type != t) badtype(o,t); }
#define push(x,t,f)	 (--sp, sp->v_type = (t), sp->v.f = (x))
#define push_integer(x)	 push(x,DT_INTEGER,v_integer)
#define push_class(x)	 push(x,DT_CLASS,v_class)
#define push_object(x)	 push(x,DT_OBJECT,v_object)
#define push_bytecode(x) push(x,DT_CODE,v_bytecode)

/* macros to set values */
#define set(s,x,t,f)	  ((s)->v_type = (t), (s)->v.f = (x))
#define set_integer(s,x)  set(s,x,DT_INTEGER,v_integer)
#define set_class(s,x)	  set(s,x,DT_CLASS,v_class)
#define set_object(s,x)   set(s,x,DT_OBJECT,v_object)
#define set_code(s,x)     set(s,x,DT_CODE,v_code)
#define set_bytecode(s,x) set(s,x,DT_BYTECODE,v_bytecode)
#define set_var(s,x)      set(s,x,DT_VAR,v_var)
#define set_string(s,x)	  set(s,x,DT_STRING,v_string)
#define set_vector(s,x)	  set(s,x,DT_VECTOR,v_vector)
#define set_file(s,x)	  set(s,x,DT_FILE,v_fp)
#define set_nil(s)	  ((s)->v_type = DT_NIL)

/* string value structure */
typedef struct string {
  int s_length;			/* length */
  unsigned char s_data[1];	/* data */
} STRING;

/* value descriptor structure */
typedef struct value {
  int v_type;			/* data type */
  union {			/* value */
    struct class *v_class;
    struct value *v_object;
    struct value *v_vector;
    struct string *v_string;
    struct value *v_bytecode;
    struct dict_entry *v_var;
    int (*v_code)();
    long v_integer;
    FILE *v_fp;
  } v;
} VALUE;

/* object fields */
#define OB_CLASS	0	/* class */
#define _OBSIZE		1

/* class definition structure */
typedef struct class {
    char *cl_name;			/* class name */
    struct class *cl_base;		/* base class */
    struct dictionary *cl_members;	/* data member */
    struct dictionary *cl_functions;	/* function members */
    int cl_size;			/* object size */
} CLASS;

/* dictionary structure */
typedef struct dictionary {
    CLASS *di_class;			/* associated class */
    struct dict_entry *di_contents;	/* dictionary entries */
} DICTIONARY;

/* dictionary entry structure */
typedef struct dict_entry {
    DICTIONARY *de_dictionary;		/* backpointer to dictionary */
    char *de_key;			/* symbol name */
    int de_type;			/* symbol type */
    VALUE de_value;			/* symbol value */
    struct dict_entry *de_next;		/* next entry */
} DICT_ENTRY;

/* symbol types */
#define ST_CLASS	1	/* class definition */
#define ST_DATA		2	/* data member */
#define ST_SDATA	3	/* static data member */
#define ST_FUNCTION	4	/* function member */
#define ST_SFUNCTION	5	/* static function member */

/* data types */
#define _DTMIN		0
#define DT_NIL		0
#define DT_CLASS	1
#define DT_OBJECT	2
#define DT_VECTOR	3
#define DT_INTEGER	4
#define DT_STRING	5
#define DT_BYTECODE	6
#define DT_CODE		7
#define DT_VAR		8
#define DT_FILE		9
#define _DTMAX		9

/* function argument structure */
typedef struct argument {
    char *arg_name;		/* argument name */
    struct argument *arg_next;	/* next argument */
} ARGUMENT;

/* literal structure */
typedef struct literal {
    VALUE lit_value;		/* literal value */
    struct literal *lit_next;	/* next literal */
} LITERAL;

/* opcodes */
#define OP_BRT		0x01	/* branch on true */
#define OP_BRF		0x02	/* branch on false */
#define OP_BR		0x03	/* branch unconditionally */
#define OP_NIL		0x04	/* load top of stack with nil */
#define OP_PUSH		0x05	/* push nil onto stack */
#define OP_NOT		0x06	/* logical negate top of stack */
#define OP_NEG		0x07	/* negate top of stack */
#define OP_ADD		0x08	/* add top two stack entries */
#define OP_SUB		0x09	/* subtract top two stack entries */
#define OP_MUL		0x0A	/* multiply top two stack entries */
#define OP_DIV		0x0B	/* divide top two stack entries */
#define OP_REM		0x0C	/* remainder of top two stack entries */
#define OP_BAND		0x0D	/* bitwise and of top two stack entries */
#define OP_BOR		0x0E	/* bitwise or of top two stack entries */
#define OP_XOR		0x0F	/* bitwise xor of top two stack entries */
#define OP_BNOT		0x10	/* bitwise not of top two stack entries */
#define OP_SHL		0x11	/* shift left top two stack entries */
#define OP_SHR		0x12	/* shift right top two stack entries */
#define OP_LT		0x13	/* less than */
#define OP_LE		0x14	/* less than or equal to */
#define OP_EQ		0x15	/* equal to */
#define OP_NE		0x16	/* not equal to */
#define OP_GE		0x17	/* greater than or equal to */
#define OP_GT		0x18	/* greater than */
#define OP_INC		0x19	/* increment */
#define OP_DEC		0x1A	/* decrement */
#define OP_LIT		0x1B	/* load literal */
#define OP_RETURN	0x1C	/* return from interpreter */
#define OP_CALL		0x1D	/* call a function */
#define OP_REF		0x1E	/* load a variable value */
#define OP_SET		0x1F	/* set the value of a variable */
#define OP_VREF		0x20	/* load a vector element */
#define OP_VSET		0x21	/* set a vector element */
#define OP_MREF		0x22	/* load a member variable value */
#define OP_MSET		0x23	/* set a member variable */
#define OP_AREF		0x24	/* load an argument value */
#define OP_ASET		0x25	/* set an argument value */
#define OP_TREF		0x26	/* load a temporary variable value */
#define OP_TSET		0x27	/* set a temporary variable */
#define OP_TSPACE	0x28	/* allocate temporary variable space */
#define OP_SEND		0x29	/* send a message to an object */
#define OP_DUP2		0x2A	/* duplicate top two elements on the stack */
#define OP_NEW		0x2B	/* create a new class object */

/* external variables */
extern VALUE *stkbase,*sp,*fp,*stktop;

/* external routines */
extern CLASS *newclass();
extern VALUE *newobject();
extern VALUE *newvector();
extern STRING *newstring();
extern STRING *makestring();
extern DICTIONARY *newdictionary();
extern DICT_ENTRY *findentry();
extern DICT_ENTRY *addentry();
extern char *copystring();
extern char *getmemory();
