
#include <stdio.h>
#include <ctype.h>
#include "ga.h"
#include "table.h"
#include "param.h"
#include "util.h"

/***
*       GASystem
*       Mike Morrow
*       September 1989
***/

static void exevset();
static void exevshow();


static int param_exe_gen = 0;		/* generation count */

static CONST PARAMSPEC exeparams[] =
{
	{"GEN", (void *) &param_exe_gen,	MODEINT,	FLAG_RO},	


	{(char *) 0, 	(void *) 0, (MODE) 0,	0}
};


static void exerun(), exequit(), exedump();

static CONST PARAMSPEC execmds[] =
{
	{"GO", 	(void *) exerun, 	0,		0},
	{"QUIT", (void *) exequit,	0,		0},
	{"DUMP", (void *) exedump,	0,		0},



	{(char *) 0, 	(void *) 0, (MODE) 0,	0}
};


/**
*	This routine inserts information about the parameter vars
*	and commands into the symbol table.
**/
exepinit()
{
	CONST PARAMSPEC *p;
	
	for(p = execmds; p->param_name; p++)
		paramins(p, SYMCMD);
		
	for(p = exeparams; p->param_name; p++)
		paramins(p, SYMVAR);
}



/**
*       This is the command interpreter.  s is a string containing the
*       user input.
**/

void exec(s)
char *s;
{
	TBL_PTR t;
	char *cmd, remainder[80], *tail;
	
	s = eatblanks(s);
	strupper(s);
	
					/* get first part of line -- command or var name */
	cmd = s;
	if(! *cmd)
		return ;	/* null command */
	
					/* remainder of line are arguments to command */
	while(isalnum(*s++)) /* blank */ ;

	s--;	
	strcpy(remainder, s);
	*s = '\0';
	tail = eatblanks(remainder);


	/**
	*	Look up this word in the symbol table.  We'll learn if it's a
	*	variable name, a command, or undefined.
	**/
	t = tbl_find(cmd);
	if(! t)
	{
		warning("No such command/variable");
		return ;
	}
	
	if(t->symtype == SYMCMD)
	{
		void (*fcn)();

		fcn = t->symdata;
		fcn(tail);
	}
	else if(t->symtype == SYMVAR && *tail == '=')
	{
		tail++;
		tail = eatblanks(tail);
		exevset(t, tail);
	}
	else if(t->symtype == SYMVAR)
	{
		exevshow(t);
	}
}
		

/**
*       Set the variable referenced by t to the value implied
*       in the "tail" parameter.
**/

static void exevset(t, tail)
TBL_PTR t;
char *tail;
{
	int n;
	FIT_TYPE fit;

	if(t->symflags & FLAG_RO)
	{
		warning("Can't set that variable");
		return ;
	}

	if(t->symflags & FLAG_INIT)
		poplfree();
		
	switch(t->symmode)
	{
		case MODEINT:
			if(sscanf(tail, "%d", &n) != 1)
			{
				warning("error in scanning RHS of '='");
				return ;
			}
			* ((int *) t->symdata) = n;
			break;
			
		case MODEFIT:		/* OOPS, we assume MODEFIT is long int -- bad stuff */
			if(sscanf(tail, "%ld", &fit) != 1)
			{
				warning("error in scanning RHS of '='");
				return ;
			}
			* ((FIT_TYPE *) t->symdata) = fit;
			break;
			
		default:
			fatal("strange mode in exevset");
	}
				
	if(t->symflags & FLAG_INIT)
	{
		param_exe_gen = 0;
		poplinit();
		popleval();
	}
}


/**
*       Show the value of the variable implied by t.
**/

static void exevshow(t)
TBL_PTR t;
{
	switch(t->symmode)
	{
		case MODEINT:
			printf("%d\n", * ((int *) t->symdata));
			break;
			
		case MODEFIT:
			printf("%ld\n", * ((FIT_TYPE *) t->symdata));
			break;
			
		default:
			fatal("strange mode in exerun");
			
	}
}


/**
*       Do s generations
**/
static void exerun(s)
char *s;
{
	unsigned int i, n;
	

	if(*s)
		n = atoi(s);
	else
		n = 1;
		

	for(i = 0; i < n; i++, param_exe_gen++)
	{
		printf("Generation %d\n", param_exe_gen);
		fflush(stdout);
		poplrepro();
		poplcross();
		poplmutate();
		popleval();
	}
}

/**
*       Dump s members of the population.
**/
static void exedump(s)
char *s;
{
	int n;
	
	if(s)
		n = atoi(s);
	else
		n = 0;
	
	popldump(n);
}


/**
*       Normal exit.
**/
static void exequit(s)
char *s;
{
	exit(0);
}
