//--------------------------------------------------------------------------
//
//      TICK.CPP: Once-a-second tick demo for DOS TSR class.
//      (c) J.English 1993 (je@unix.brighton.ac.uk)
//
//      This example makes an irritating beep every second after the
//      hotkey is pressed.  Try removing the call to "sync()" and see
//      what happens.
//
//--------------------------------------------------------------------------

#include <iostream.h>
#include <stdlib.h>
#include <dos.h>
#include <ctype.h>
#include "tsr.h"

//----- Error messages for TSR status values -----
const char* status_msg [] = {
    "success",
    "DOS version 3 or higher required",
    "multiple TSR instances declared in program", 
    "can't allocate stack",
    "can't allocate multiplex function code",
    "already loaded",
    "can't make program resident",
    "startup error"
};

//----- Error messages for unload result -----
const char* unload_msg [] = {
    "unloaded",
    "not loaded",
    "can't be unhooked",
    "can't be unloaded",
    "environment can't be unloaded"
};

//----- The derived TSR class -----
class Ticker : public TSR
{
  private:
    virtual void main (int hotkey);     // body of clock TSR
    int active;                         // true if ticking
  public:
    Ticker () :                         // constructor
        TSR ("TSR ticking demo"),       // ... call base class constructor
        active (0)                      // ... no ticking initially
        { }                             // ... nothing else needed
};

//----- The main program -----
int main (int argc, char** argv)
{
    Ticker ticker;                      // create TSR instance

    switch (argc)
    {
      case 1:                           // invoked by TICK -- load it
        cout << "Installing " << ticker.name() << " in memory\n";
        cout << "Press both shift keys to activate and deactivate\n";

        ticker.run (TSR::LSHIFT + TSR::RSHIFT, 18);

        cout << "Error installing " << ticker.name() << ": "
             << status_msg [ticker.status()] << "\n";
        break;

      case 2:                           // invoked by TICK/U -- unload it
        if (argv[1][0] == '/' && toupper (argv[1][1]) == 'U')
        {   cout << ticker.name() << " "
                 << unload_msg [ticker.unload()] << "\n";
            break;
        }

      default:                          // give usage message otherwise
        cout << "Usage: TICK [/U]\n";
        return -1;
    }
    return ticker.status ();
}

//----- Resident part of TSR -----
void Ticker::main (int hotkey)
{
    if (hotkey)
    {   active = !active;               // hotkey toggles visibility
        sync ();                        // 1-second timeslice starts now
    }
    if (!active)
        return;                         // nothing to do if inactive

    sound (880);                        // peep for 100 ms
    delay (100);
    nosound ();
}
