/* -------------------------------------------------------------------- */
/* String++ Version 3.00                                       04/10/93 */
/*                                                                      */
/* Enhanced string class for Turbo C++/Borland C++.                     */
/* Copyright 1991-1993 by Carl W. Moreland                              */
/*                                                                      */
/* strng.h                                                              */
/* -------------------------------------------------------------------- */

#ifndef _STRNG_H
#define _STRNG_H

#ifndef __CLSTYPES_H
#include <ClsTypes.h>
#endif

#ifndef __SORTABLE_H
#include <Sortable.h>
#endif

#ifndef __STRING_H
#include <string.h>
#endif

const char LEFT   = 0;
const char CENTER = 1;
const char RIGHT  = 2;
const char NOCLIP = 0;
const char NOTRIM = 0;
const char CLIP   = 1;
const char TRIM   = 2;
const char WHITESPACE = 0;

#pragma option -Vo-
#if defined( __BCOPT__ ) && !defined( _ALLOW_po )
#pragma option -po-
#endif

_CLASSDEF(ostream)
_CLASSDEF(RegExp)
_CLASSDEF(String)

class _CLASSTYPE String : public Sortable
{
protected:
  unsigned strLen;			// length of the String
  char*    strPtr;			// pointer to the String contents
  unsigned bufferLen;			// length of the buffer

public:
  String();				// default constructor;
  String(const char c,			// initialize with a character,
         const unsigned n = 1);		//   optional # of characters
  String(const char* p);		// initialize with a char *
  String(const char* p,			// initialize with a char *,
         const unsigned pos,		//   optional starting char
         const unsigned len = 10000);	//   optional # of chars
  String(const String& s);		// initialize with another String
  String(const String& s,		// initialize with another String,
         const unsigned pos,		//   optional starting char
         const unsigned len = 10000);	//   optional # of chars
  String(const int n);			// initialize with an integer
  String(const long n);			// initialize with a long int
  String(const float n,			// initialize with a float
         const char* format = "");	//   and a format specifier
  String(const double n,		// initialize with a double
         const char* format = "");	//   and a format specifier

 ~String(void);

protected:
  static unsigned strMinLength;		// minimum memory allocated
  static unsigned strIncLength;		// incremental memory allocated
  static String*  findIn;		// these are for FindFirst, etc.
  static String&  findStr;
  static unsigned findPos;
  static String   fpFormat;

  virtual void SetStr(const char c,
                      const unsigned n = 1);
  virtual void SetStr(const char* p);
  virtual void SetStr(const char* p,
                      const unsigned pos,
		      const unsigned len = 10000);
  virtual void SetStr(const String& s);
  virtual void SetStr(const String& s,
                      const unsigned pos,
		      const unsigned len = 10000);
  virtual void AddStr(const char c);
  virtual void AddStr(const char* p);
  virtual void AddStr(const String& s);
  virtual unsigned GetSize(unsigned n);
  void ltos(const long n);
  void dtos(const double n, const char* format);

public:
  void SetMinLength(unsigned len = 16) const { strMinLength = len; }
  void SetIncLength(unsigned len = 8)  const { strIncLength = len; }
  unsigned SetSize(unsigned len);
  void Minimize(void);				// minimize bufferLen
  void SetFloatFormat(const char*) const; 	// floating point format

  virtual operator const char() const        { return strPtr[0]; }
  virtual operator const char*() const       { return strPtr; }
  virtual operator char*() const             { return strPtr; }
  virtual const char  operator *() const     { return strPtr[0]; }
  virtual const char* operator()() const     { return strPtr; }
  const char* operator()(unsigned pos) const { return strPtr+pos; }
  String      operator()(unsigned pos, unsigned len) const;
  char* ptr(void) const                      { return strPtr; }
  char* ptr(unsigned pos) const              { return strPtr+pos; }

  virtual int Length(void) const { return strLen; }
  virtual int Len(void)    const { return strLen; }
  String& toUpper(void);			// convert to uppercase
  String& toLower(void);			// convert to lowercase
  int&    Value(int& n) const;			// int value of str
  long&   Value(long& n) const;			// long int value of str
  float&  Value(float& n) const;		// float value of str
  double& Value(double& n) const;		// double value of str

  String& Left(unsigned len);			// left   len chars
  String& Right(unsigned len);			// right  len chars
  String& Mid(unsigned pos,			// middle len chars from pos
              unsigned len);
  String& Justify(char type,			// justify str
                  unsigned len,
		  char mode = CLIP|TRIM);
  String& Trim(int mode = CENTER,		// delete whitespace
               char ch = WHITESPACE);

  String& Insert(unsigned pos,			// insert substring
                 const String& s);
  String& Delete(unsigned pos,			// delete substring
                 unsigned len = 1);
  String& Replace(unsigned pos,
                  unsigned len,
                  const String& to);
  int     Replace(const String& from,		// substitute from -> to
                  const String& to,
	          unsigned count = 10000);
  int     Replace(const RegExp& from,
                  const String& to,
                  unsigned count = 10000);
  char*   Copy(char*&) const;			// copy str to char*

  int     Index(const String& t) const;		// position of t in str
  int     Index(const RegExp& t) const;		// position of t in str
  String  SubStr(unsigned pos,			// substring at position pos
                 unsigned len = 10000) const;
  int     Split(String*& a,			// split into an array a on
                const String& fs) const;	//   field separator fs
  int     Split(String*& a,			// split into an array a on
                const RegExp& fs) const;	//   field separator fs
  int     Sub(const String& from,		// substitute from -> to
              const String& to,
	      unsigned count = 10000);
  int     Sub(const RegExp& from,		// substitute from -> to
              const String& to,
              unsigned count = 10000);

  int     FindFirst(const String& s) const;	// first occurance of s
  int     FindNext (void) const;		// next occurance of s
  int     FindPrev (void) const;		// previous occurance of s
  int     FindLast (const String& s) const;	// last occurance of s

  virtual String& operator=(const char c);	// str1 = char
  virtual String& operator=(const char* p);	// str1 = char*
  virtual String& operator=(const String& s);	// str1 = str
  virtual String& operator=(const int n);	// str1 = int
  virtual String& operator=(const long n);	// str1 = long
  virtual String& operator=(const float n);	// str1 = float
  virtual String& operator=(const double n);	// str1 = double
  String& operator+=(const char c);		// str1 += char
  String& operator+=(const char* p);		// str1 += char*
  String& operator+=(const String& s);		// str1 += str
  String& operator*=(const unsigned n);		// str1 *= n
  char&   operator[](const unsigned i) const;	// ch = str[i] or str[i] = ch

  friend String operator+(const String& s1, const String& s2);
  friend String operator+(const String& s,  const char* p);
  friend String operator+(const char* p,    const String& s);
  friend String operator*(const String& s,  const unsigned n);
  friend String operator*(const unsigned n, const String& s);

  virtual String& operator<<(const char c);	// s << char
  virtual String& operator<<(const char* p);	// s << char*
  virtual String& operator<<(const String& s);	// s << str
  virtual String& operator<<(const int n);	// s << int
  virtual String& operator<<(const long n);	// s << long
  virtual String& operator<<(const float n);	// s << float
  virtual String& operator<<(const double n);	// s << double

  virtual int isEqual(const Object _FAR &) const;
  virtual int isLessThan(const Object _FAR &) const;
  virtual classType isA() const { return stringClass; }
  virtual char _FAR *nameOf() const { return "String"; }
  virtual hashValueType hashValue() const;
  virtual void printOn(ostream _FAR &) const;
};

typedef String string;			// compatibility with older versions

/* ----- Awk-style functions ------------------------------------------ */

inline int length(const String& s) { return s.Len(); }
int    index(const String& s, const String& t);
String substr(const String& s, unsigned pos, unsigned len = 10000);
int    split(const String& s, String*& a, const String& fs);
int    gsub(const String& from, const String& to,
            String& str, unsigned count = 10000);
inline int sub(const String& from, const String& to, String& str) {
  return gsub(from, to, str, 1);
}

// Regular Expression versions - requires regexp.cpp

extern int index(const String& s, const RegExp& t);
extern int split(const String& s, String*& a, const RegExp& fs);
extern int gsub(const RegExp& from, const String& to,
                String& str, unsigned count = 10000);
inline int sub(const RegExp& from, const String& to, String& str) {
  return gsub(from, to, str, 1);
}

/* ----- Other C-style functions -------------------------------------- */

String toupper(const String& s);
String tolower(const String& s);
String left(const String& s, unsigned len);
String right(const String& s, unsigned len);
String mid(const String& s, unsigned pos, unsigned len);
String justify(const String& s, char type, unsigned len, char mode=CLIP|TRIM);
String trim(const String& s, int mode = CENTER);

/* ----- Inline functions --------------------------------------------- */

inline String& String::operator=(const int n) {
  operator=((long)n);
  return *this;
}

inline String& String::operator=(const float n) {
  operator=((double)n);
  return *this;
}

inline String& String::operator<<(const int n) {
  operator<<((long)n);
  return *this;
}

inline String& String::operator<<(const float n) {
  operator<<((double)n);
  return *this;
}

inline int String::Replace(const String& from,
                           const String& to, unsigned count) {
  return gsub(from, to, *this, count);
}

inline int String::Replace(const RegExp& from,
                           const String& to, unsigned count) {
  return gsub(from, to, *this, count);
}

inline int operator==(const String& s1, const String& s2) {
  return strcmp(s1, s2) == 0;
}

inline int operator!=(const String& s1, const String& s2) {
  return strcmp(s1, s2) != 0;
}

inline int operator<(const String& s1, const String& s2) {
  return strcmp(s1, s2) < 0;
}

inline int operator>(const String& s1, const String& s2) {
  return strcmp(s1, s2) > 0;
}

inline int operator<=(const String& s1, const String& s2) {
  return strcmp(s1, s2) <= 0;
}

inline int operator>=(const String& s1, const String& s2) {
  return strcmp(s1, s2) >= 0;
}

inline int operator==(const String& s, const char* p) {
  return strcmp(s, p) == 0;
}

inline int operator!=(const String& s, const char* p) {
  return strcmp(s, p) != 0;
}

inline int operator<(const String& s, const char* p) {
  return strcmp(s, p) < 0;
}

inline int operator>(const String& s, const char* p) {
  return strcmp(s, p) > 0;
}

inline int operator<=(const String& s, const char* p) {
  return strcmp(s, p) <= 0;
}

inline int operator>=(const String& s, const char* p) {
  return strcmp(s, p) >= 0;
}

inline int operator==(const char* p, const String& s) {
  return strcmp(p, s) == 0;
}

inline int operator!=(const char* p, const String& s) {
  return strcmp(p, s) != 0;
}

inline int operator<(const char* p, const String& s) {
  return strcmp(p, s) < 0;
}

inline int operator>(const char* p, const String& s) {
  return strcmp(p, s) > 0;
}

inline int operator<=(const char* p, const String& s) {
  return strcmp(p, s) <= 0;
}

inline int operator>=(const char* p, const String& s) {
  return strcmp(p, s) >= 0;
}

inline int operator==(const String& s, const char c) {
  return *s == c;
}

inline int operator!=(const String& s, const char c) {
  return *s != c;
}

inline int operator<(const String& s, const char c) {
  return *s < c;
}

inline int operator>(const String& s, const char c) {
  return *s > c;
}

inline int operator<=(const String& s, const char c) {
  return *s <= c;
}

inline int operator>=(const String& s, const char c) {
  return *s >= c;
}

inline int String::Index(const String& t) const {
  return index(*this, t);
}

inline int String::Index(const RegExp& t) const {
  return index(*this, t);
}

inline String String::SubStr(unsigned p, unsigned n) const {
  return substr(*this, p, n);
}

inline int String::Split(String*& a, const String& fs) const {
  return split(*this, a, fs);
}

inline int String::Split(String*& a, const RegExp& fs) const {
  return split(*this, a, fs);
}

inline int String::Sub(const String& from, const String& to, unsigned count) {
  return gsub(from, to, *this, count);
}

inline int String::Sub(const RegExp& from, const String& to, unsigned count) {
  return gsub(from, to, *this, count);
}

extern const String STR_NULL;		// a global NULL string
extern const String& StrNull;		// same string, different name
extern const String& Str;		// same string, different name

#if defined( __BCOPT__ ) && !defined( _ALLOW_po )
#pragma option -po.
#endif
#pragma option -Vo.

#endif
