/*
 *                              SCREEN.C
 *
 *                         Screen i/o Handler
 *
 *                           Written for the
 *
 *                              Datalight
 *                           Microsoft V 5.x
 *                                TurboC
 *                                  &
 *                               Zortech
 *
 *                             C Compilers
 *
 *            Copyright (c) John Birchfield 1987, 1988, 1989
 */

#include <dos.h>
#include "screen.h"
/*
 *	defines for the bits returned in regs.x.cflag and by int86 () ...
 */

#define ZERO_BIT       0x040
#define CARRY_BIT          1
#define SIGN_BIT       0x080
#define DIRECTION_FLAG 0x400
#define OVERFLOW_BIT   0x800
static union REGS regs;



#define VIDEO		0x10    /* interrupt for dealing with screen */
#define MODE		 0      /* code for setting new screen mode */
#define SETCURTYP	 1      /* code for setting new cursor type */
#define SETCURSOR	 2      /* code for addressing cursor */
#define GETCURSOR	 3      /* code for reading cursor location */
#define READLP		 4      /* code for reading light pen position */
#define SETPAGE		 5      /* code to select active page */
#define SCROLLUP	 6      /* code to scroll screen up */
#define SCROLLDN	 7      /* code to scroll screen nown */
#define READCH		 8      /* code to read a character from screen */
#define WRITEACH	 9      /* code to write char and attributes */
#define WRITECH		10      /* code to write character only */
#define SETPAL		11      /* code to set new setpal or border */
#define WDOT		12      /* code to write a dot */
#define RDOT		13      /* code to read a dot */
#define WRITETTY	14      /* code to write as if teletype */
#define STATE		15      /* code to find current screen status */


/*
 *	note- make 25 for ms-dos and 24 for cp/m as cp/m steals the bottom
 *	line.
 */

int     Scr_Rows = 25;  /* current number of rows */
int     Scr_Cols = 80;  /* current number of columns */
char    Scr_Mode = 0;   /* current screen mode */
char    Scr_Page = 0;   /* current page */
char    Scr_ATTR = 7;   /* current attributes for screen 7 is white letters
                         * on black    */
char    Scr_Window_Top = 0;     /* first line to scroll */



/*
 *	SCREEN_INIT - screen_init must be called before any use of any
 *	           other routine unless the starting mode is 80X25
 *             character mode (3,4 or 7). Must be called for
 *	           monocrome (mode 7) for scr_curson to set a proper
 *	           cursor.
 *
 *	Usage:      screen_init ();
 */

int 
screen_init (void)
{
    regs.h.ah = STATE;
    int86 (VIDEO, &regs, &regs);
    Scr_Mode = regs.h.al;
    Scr_Cols = regs.h.ah;
    Scr_Page = regs.h.bh;
    Scr_ATTR = (regs.h.al < 4 || regs.h.al == 7) ? 7 : 0;
    return (regs.h.al);
}




/*
 *	SCREEN_SMODE - set a new screen mode
 *
 *	Usage:        screen_smode (new mode);
 */

int 
screen_smode (char newmode)
{
    regs.h.al = newmode;
    regs.h.ah = MODE;
    int86 (VIDEO, &regs, &regs);
    screen_init ();
}



/*
 *	ROWCOL - sets cursor at any location.
 *
 *	Usage:       rowcol (new row, new column);
 */

int 
rowcol (int row, int col)
{
    regs.h.dl = col;
    regs.h.dh = row;
    regs.h.bh = Scr_Page;
    regs.h.ah = SETCURSOR;
    int86 (VIDEO, &regs, &regs);
}



/*
 *  G_ROWCOL - returns the current screen row and column
 *	           the row is the high order 8 bits and the
 *	           column is the low order.
 *
 *	Usage:     rowcol = g_rowcol ();
 *	           cur_row = rowcol >> 8;
 *	           cur_col = rowcol && 0xFF;
 */

int 
g_rowcol (void)
{
    regs.h.ah = GETCURSOR;
    regs.h.bh = Scr_Page;
    int86 (VIDEO, &regs, &regs);
    return (regs.x.dx);
}




/*
 *	CLRSCRN - clear entire screen
 *
 *	Usage:    clrscrn ();
 */

int 
clrscrn (void)
{
    regs.h.al = 0;
    regs.x.cx = 0;
    regs.h.dh = Scr_Rows - 1;
    regs.h.dl = Scr_Cols - 1;
    regs.h.bh = Scr_ATTR;
    regs.h.ah = SCROLLUP;
    int86 (VIDEO, &regs, &regs);
}




/*
 *	CLEOL - clear to End Of Line
 *
 *	Usage:     cleol ();
 */

int 
cleol (void)
{
    regs.h.bh = Scr_Page;
    regs.h.ah = GETCURSOR;
    int86 (VIDEO, &regs, &regs);
    regs.h.cl = Scr_Cols - regs.h.dl;
    regs.h.ch = 0;
    regs.h.al = ' ';
    regs.h.bl = Scr_ATTR;
    regs.h.bh = Scr_Page;
    regs.h.ah = WRITEACH;
    int86 (VIDEO, &regs, &regs);
}




/*
 *	CLEOP - clear to End Of Page
 *
 *	Usage:    cleop ();
 */

int 
cleop (void)
{
    cleol ();
    regs.h.ah = GETCURSOR;
    regs.h.bh = Scr_Page;
    int86 (VIDEO, &regs, &regs);
    regs.h.al = 0;
    if ((regs.h.ch = regs.h.dh + 1) < Scr_Rows - 1)
    {
        regs.h.cl = 0;
        regs.h.dh = Scr_Rows - 1;
        regs.h.dl = Scr_Cols - 1;
        regs.h.bh = Scr_ATTR;
        regs.h.ah = SCROLLUP;
        int86 (VIDEO, &regs, &regs);
    }
}



/*
 *	SCROLL_UP - Scroll the screen up. The window is scrolled
 *	            up nlines lines. A zero nlines will clear the
 *	            window. Top left of the screen is 0,0.
 *
 *	Usage:      SCROLL_UP (nlines, start_row, start_col, end_row, end_col);
 */

int 
scroll_up (int nlines, 
           int start_row, int start_col,
		   int end_row,   int end_col)
{
    regs.h.al = nlines;
    regs.h.ch = start_row;
    regs.h.cl = start_col;
    regs.h.dh = end_row;
    regs.h.dl = end_col;
    regs.h.bh = Scr_ATTR;
    regs.h.ah = SCROLLUP;
    int86 (VIDEO, &regs, &regs);
}



/*
 *	SCROLL_DN - scroll the screen down. the window is scrolled
 *	            down nline lines. A zero nline will clear the
 *	            window. Top left of the screen in 0,0.
 *
 *	Usage:      scroll_dn (nlines,start_row, start_col, end_row, end_col);
 */

int 
scroll_dn (int nlines,
           int start_row, int start_col,
		   int end_row,   int end_col)
{
    regs.h.al = nlines;
    regs.h.ch = start_row;
    regs.h.cl = start_col;
    regs.h.dh = end_row;
    regs.h.dl = end_col;
    regs.h.bh = Scr_ATTR;
    regs.h.ah = SCROLLDN;
    int86 (VIDEO, &regs, &regs);
}





/*
 *	SCREEN_CO - write a character to the screen. this
 *	            routine increments the cursor position
 *	            after writing. normal C88 puts and printf
 *	            statements can also be used to write to the
 *	            screen.
 *
 *	Usage:   screen_co (character);
 */

int 
screen_co (char ch)
{
    regs.h.al = ch;
    regs.h.bh = Scr_Page;
    regs.h.ah = WRITETTY;
    int86 (VIDEO, &regs, &regs);
}




/*
 *	SINP - screen input (read character from the screen).
 *
 *	Usage:     character = sinp ();
 */

int 
sinp (void)
{
    regs.h.bh = Scr_Page;
    regs.h.ah = READCH;
    int86 (VIDEO, &regs, &regs);
    return ((int) (regs.h.al) ? regs.h.al : ' ');
}





/*
 *	CURSOR_OFF - turn cursor off.
 *
 *	Usage:        cursor_off ();
 */

int 
cursor_off (void)
{
    if (Scr_Mode < 4 || Scr_Mode == 7)
    {
        regs.x.cx = 0x0f00;
        regs.h.ah = SETCURTYP;
        int86 (VIDEO, &regs, &regs);
    }
}




/*
 *	CURSOR_ON - turn cursor back on.
 *
 *	Usage:       cursor_on ();
 */

int 
cursor_on (void)
{
    if (Scr_Mode == 7)
        regs.x.cx = 0x0C0D;
    else
    if (Scr_Mode < 4)
        regs.x.cx = 0x0607;
    else
        return;
    regs.h.ah = SETCURTYP;
    int86 (VIDEO, &regs, &regs);
}





/*
 *	APUTS - write a string in the current attribute to the screen.
 *
 *	Usage:      aputs (attr, "Write this out\n");
 */

int 
aputs (int a, char *s)
{
    while (*s)
        aput (a, *s++);
}




/*
 *	APUT - write a string and attributes to the screen.
 *	           the cursor is moved normally
 *
 *	Usage:     aput (INVERSE, 'x');
 */

int 
aput (int attr, char ch)
{
    switch (ch)
    {
        default:
            regs.h.al = ch;
            regs.h.bl = attr;
            regs.x.cx = 1;
            regs.h.bh = Scr_Page;
            regs.h.ah = WRITEACH;
            int86 (VIDEO, &regs, &regs);
            regs.h.ah = GETCURSOR;
            int86 (VIDEO, &regs, &regs);
            if (!(++regs.h.dl < Scr_Cols))
            {
                regs.h.dl = 0;
                if (!(++regs.h.dh < Scr_Rows))
                {
                    scroll_up (1, 0, 0, 23, 79);
                    regs.h.dl = 0;
                    regs.h.dh = 23;
                }
            }
            regs.h.bh = Scr_Page;
            regs.h.ah = SETCURSOR;
            int86 (VIDEO, &regs, &regs);
            break;
        case 10:
            regs.h.ah = GETCURSOR;
            int86 (VIDEO, &regs, &regs);
            regs.h.dl = 0;
            if (!(++regs.h.dh < Scr_Rows - 2))
            {
                scroll_up (0, 0, 0, Scr_Rows - 2, 79);
                regs.h.dl = 0;
                regs.h.dh = Scr_Rows - 2;
            }
            regs.h.bh = Scr_Page;
            regs.h.ah = SETCURSOR;
            int86 (VIDEO, &regs, &regs);
            break;
        case 13:
            regs.h.ah = GETCURSOR;
            int86 (VIDEO, &regs, &regs);
            regs.h.dl = 0;
            regs.h.bh = Scr_Page;
            regs.h.ah = SETCURSOR;
            int86 (VIDEO, &regs, &regs);
            break;
    }
}
