/*
   Module:  input.c
   Date:    3/9/92
   Version: 1.0b
   Author:  Dave Lutz
   Email:   lutz@psych.rochester.edu
   Copyright: 1992 University of Rochester, Psychology Dept.

   Disclaimer:  This software is distributed free of charge.  As such, it
                comes with ABSOLUTELY NO WARRANTY.  The user of the software
                assumes ALL RISKS associated with its use.

                Your rights to modify and/or distribute this software are
                outlined in the file SEND7221.DOC.

   Purpose: This module provides the input routines for the PC -> HP7221
            controller.  It provides the routines needed by the controller
            to open the input file and read data.

   Functions Provided:

            openin
            closein
            fillbuff

   Functions Required:

            none

*/




#include <fcntl.h>
#include <sys\types.h>
#include <sys\stat.h>
#include <stdlib.h>
#include <malloc.h>
#include <io.h>
#include "retcodes.h"
#include "input.h"


/*
   Function: openin
   Purpose:  Allocate the input buffer and open the input file for reading.

   Pre: name is a pointer to an ascii string containing the name of the 
        input file.
        pltfpp is a pointer to storage for a pointer to a PLTFILE.

   Post:  The input buffer is allocated.
          The input file is opened for reading.
          If the buffer can't be allocated, *pltfpp is set to NULL and
          NOBUFF is returned.
          If the input file can't be opened, *pltfpp is set to NULL and
          BADOPEN is returned.
          Otherwise, TRUE is returned.
*/

int openin (name,pltfpp)
   char *name;
   PLTFILE **pltfpp;
{
   /* first try to allocate a PLTFILE */
   *pltfpp = (PLTFILE *) malloc (sizeof (PLTFILE));
   if (*pltfpp == (PLTFILE *)NULL)
      return (NOBUFF);

   /* now try to allocate the buffer */
   (*pltfpp)->buf = (char *) malloc ((size_t)(BUFFSIZE*sizeof(char)));  
   if ((*pltfpp)->buf == (char *)NULL) {
      free ((void *)*pltfpp);
      *pltfpp = (PLTFILE *)NULL;
      return (NOBUFF);
   }

   /* now attempt to open the input file */
   (*pltfpp)->fd = open (name,O_RDONLY|O_BINARY);
   if ((*pltfpp)->fd < 0) {
      free ((void *)(*pltfpp)->buf);
      free ((void *)*pltfpp);
      *pltfpp = (PLTFILE *)NULL;
      return (BADOPEN);
   }

   /* now set buf_ctr to zero so the first read will cause the buffer to be 
      filled from the input file
   */
   (*pltfpp)->buf_ctr = 0;
   (*pltfpp)->eof=FALSE;
   return (TRUE);
}



/*
   Function: closein
   Purpose:  Discard the input buffer, close the input file, and deallocate 
             the input buffer.

   Pre:  pltfpp is a pointer to a pointer to a PLTFILE.
         If *pltfpp does not point to an open PLTFILE, it is NULL.

   Post:  If *pltfpp = NULL, TRUE is returned.
          Otherwise, the contents of the input buffer are discarded, the 
          input file is closed, and the input buffer is deallocated.
          If an error occurs during the above process, FALSE is returned.
*/

int closein(pltfpp)
   PLTFILE **pltfpp;
{
   if (*pltfpp == (PLTFILE *)NULL)
      return (TRUE);
   free ((void *)(*pltfpp)->buf);
   if (close((*pltfpp)->fd) < 0) {
       free ((void *)*pltfpp);
       return (FALSE);
   }
   free ((void *)*pltfpp);
   return (TRUE);
}



/*
   Function: fillbuff
   Purpose:  Read a block of data from the input file, filling the buffer
             in the PLTFILE.

   Pre:  pltfp is a pointer to a PLTFILE that has been opened for reading.

   Post: The input file is read to refill the buffer in PLTFILE.
         If end_of_file has already been reached, 0 is returned.
         If an error occures, -1 is returned.
         Otherwise, the number of bytes read from the file is returned.
*/

int fillbuff (pltfp)
   PLTFILE *pltfp;
{
   if (pltfp->eof == TRUE)
      return (0);

   switch (pltfp->buf_ctr = read(pltfp->fd,pltfp->buf, (unsigned)BUFFSIZE)) {
      case 0:
         pltfp->eof = TRUE;
         return (0);
         break;
      case -1:
         return (-1);
         break;
      default:
         /* pltfp->buf_ctr was set by the read funtion */
         break;
   }
   return (pltfp->buf_ctr);
}
