/*
 * grafix --- halofake.c
 *
 * implement halo-like transformations and calls
 *
 * Written 4/87 by Scott Snyder (ssnyder@romeo.caltech.edu or @citromeo.bitnet)
 *
 */

#include "macros.h"
#include "graf.h"
#include <math.h>

/* graphics parameters */

STATIC unsigned h_xsize, h_ysize, h_colormax;

/* transformation parameters */

STATIC float w_x_offs = 0;
STATIC float w_y_offs = 0;
STATIC float w_x_mult = 1;
STATIC float w_y_mult = 1;
STATIC float v_x_offs = 0;
STATIC float v_y_offs = 0;
STATIC float v_x_mult = 1;
STATIC float v_y_mult = 1;

/* state variables */

STATIC float cur_wx, cur_wy;
STATIC unsigned cur_c;

#define smashcolor(c) ((c) > h_colormax ? h_colormax : c)
#define swap(a, b) {int tmp; tmp=a; a=b; b=tmp;}

/*****************************************************************************
 *                          transformation macros                            *
 *****************************************************************************/

#define tr_x_wtov(x) (((x) + w_x_offs) * w_x_mult)
#define tr_x_vton(x) ((x) * v_x_mult + v_x_offs)
#define tr_x_ntod(x) ((x) * (h_xsize-1))

#define tr_x_dton(x) (((float)(x)) / (h_xsize-1))
#define tr_x_ntov(x) (((x) - v_x_offs) / v_x_mult)
#define tr_x_vtow(x) ((x) / w_x_mult - w_x_offs)

#define tr_x_wton(x) tr_x_vton( tr_x_wtov (x))
#define tr_x_ntow(x) tr_x_vtow( tr_x_ntov (x))
#define tr_x_dtow(x) tr_x_ntow( tr_x_dton (x))


#define tr_y_wtov(y) (((y) + w_y_offs) * w_y_mult)
#define tr_y_vton(y) ((y) * v_y_mult + v_y_offs)
#define tr_y_ntod(y) ((y) * (h_ysize-1))

#define tr_y_dton(y) (((float)(y)) / (h_ysize-1))
#define tr_y_ntov(y) (((y) - v_y_offs) / v_y_mult)
#define tr_y_vtow(y) ((y) / w_y_mult - w_y_offs)

#define tr_y_wton(y) tr_y_vton( tr_y_wtov (y))
#define tr_y_ntow(y) tr_y_vtow( tr_y_ntov (y))
#define tr_y_dtow(y) tr_y_ntow( tr_y_dton (y))


#define tr_r_wtov(r) ((r) * w_x_mult)
#define tr_r_vton(r) ((r) * v_x_mult)
#define tr_r_ntod(r) tr_x_ntod(r)

#define tr_r_wton(r) tr_r_vton( tr_r_wtov (r))

/*****************************************************************************
 *                        private utility functions                          *
 *****************************************************************************/

int tr_x(float);
STATIC int tr_x(x)
float x;
{
  return( tr_x_ntod( tr_x_wton (x)));
}

int tr_y(float);
STATIC int tr_y(y)
float y;
{
  return( tr_y_ntod( tr_y_wton (y)));
}

int tr_r(float);
STATIC int tr_r(r)
float r;
{
  return( tr_r_ntod( tr_r_wton (r)));
}

/*****************************************************************************
 *                          user-callable routines                           *
 *****************************************************************************/

/* coordinate transforms */

void mapdton(dx, dy, nx, ny)
int   *dx, *dy;
float *nx, *ny;
{
  *nx = tr_x_dton(*dx);
  *ny = tr_y_dton(*dy);
}

void mapdtow(dx, dy, wx, wy)
int   *dx, *dy;
float *wx, *wy;
{
  *wx = tr_x_dtow(*dx);
  *wy = tr_y_dtow(*dy);
}

void mapntod(nx, ny, dx, dy)
float *nx, *ny;
int   *dx, *dy;
{
  *dx = tr_x_ntod(*nx);
  *dy = tr_y_ntod(*ny);
}

void mapntow(nx, ny, wx, wy)
float *nx, *ny;
float *wx, *wy;
{
  *wx = tr_x_ntow(*nx);
  *wy = tr_y_ntow(*ny);
}

void mapwtod(wx, wy, dx, dy)
float *wx, *wy;
int   *dx, *dy;
{
  *dx = tr_x(*wx);
  *dy = tr_y(*wy);
}

void mapwton(wx, wy, nx, ny)
float *wx, *wy;
float *nx, *ny;
{
  *nx = tr_x_wton(*wx);
  *ny = tr_y_wton(*wy);
}

/* routines controlling transformations */

void setworld(x1,y1,x2,y2)
float *x1,*y1,*x2,*y2;
{
  w_x_offs = -*x1;
  w_y_offs = -*y2;

  if (*x2 - *x1 != 0.0)
    w_x_mult = 1.0/(*x2 - *x1);
  else
    w_x_mult = 1e15;

  if (*y1 - *y2 != 0.0)
    w_y_mult = 1.0/(*y1 - *y2);
  else
    w_y_mult = 1e15;
}

void setviewport(x1, y1, x2, y2, bdr, bck)
float *x1,*x2,*y1,*y2;
int *bdr,*bck;
{
  int dx1, dx2, dy1, dy2;

  v_x_offs = *x1;
  v_y_offs = *y1;

  if (*x2 - *x1 != 0.0)
    v_x_mult = (*x2 - *x1);
  else
    v_x_mult = 1e-15;

  if (*y2 - *y1 != 0.0)
    v_y_mult = (*y2 - *y1);
  else
    v_y_mult = 1e-15;

  dx1 = tr_x_ntod(*x1);
  dx2 = tr_x_ntod(*x2);
  dy1 = tr_y_ntod(*y1);
  dy2 = tr_y_ntod(*y2);

  if (*bdr >= 0) {
    if (dx1 > dx2) swap(dx1, dx2);
    if (dy1 > dy2) swap(dy1, dy2);

    g_setclip(dx1-1, dy1-1, dx2+1, dy2+1);
    g_box(dx1-1, dy1-1, dx2+1, dy2+1, smashcolor(*bdr));
  }

  g_setclip(dx1, dy1, dx2, dy2);

  if (*bck >= 0)
    g_clear(smashcolor(*bck));
}

/* drawing routines */

void movabs(wx, wy)
float *wx, *wy;
{
  cur_wx = *wx;
  cur_wy = *wy;
}

void movrel(wx, wy)
float *wx, *wy;
{
  cur_wx += *wx;
  cur_wy += *wy;
}

void lnabs(wx, wy)
float *wx, *wy;
{
  g_line(tr_x(cur_wx), tr_y(cur_wy), tr_x(*wx), tr_y(*wy), cur_c);
  cur_wx = *wx;
  cur_wy = *wy;
}

void lnrel(wx, wy)
float *wx, *wy;
{
  g_line(tr_x(cur_wx),     tr_y(cur_wy),
         tr_x(cur_wx+*wx), tr_y(cur_wy+*wy), cur_c);
  cur_wx += *wx;
  cur_wy += *wy;
}

void cir(r)
float *r;
{
  int dx, dy, dr;

  if (*r > 0.0) {
    dx = tr_x(cur_wx);
    dy = tr_y(cur_wy);
    dr = tr_r(*r);
    g_circle(dx, dy, dr, cur_c);
  }
}

void box(wx1, wy1, wx2, wy2)
float *wx1, *wx2, *wy1, *wy2;
{
  g_box(tr_x(*wx1), tr_y(*wy1), tr_x(*wx2), tr_y(*wy2), cur_c);
}

/* mode setting stuff */

setcolor(c)
int *c;
{
  cur_c = smashcolor(*c);
}

setxor(m)
int *m;
{
  g_setxor(*m);
}

/* get parameters - must be called AFTER g_open */

void halo_init()
{
  struct g_info info;

  g_info(&info);
  h_xsize = info.xsize;
  h_ysize = info.ysize;
  h_colormax = info.colormax;

  w_x_offs = w_y_offs = 0;
  w_x_mult = w_y_mult = 1;
  v_x_offs = v_y_offs = 0;
  v_x_mult = v_y_mult = 1;

  cur_wx = cur_wy = 0;
  cur_c = 0;
}
