                                                                      
/*
**  Snapshot Load / Save Functions.
**
**  Copyright 1993-97 by Paul D. Burgin. All rights reserved.
*/

#include "stdio.h"
#include "conio.h"
#include "dos.h"
#include "dir.h"
#include "string.h"
#include "ctype.h"
#include "key.h"

#include "build.h"
#include "types.h"
#include "extern.h"
#include "6809cpu.h"
#include "6809regs.h"
#include "macros.h"
#include "snapshot.h"

/* Version of PC-Dragon PAK file extensions written by this emulator. */
#define PCD_PAK_EXT_VER		0x21

extern outfile_status		open_outfile(unsigned char *fname,
								FILE **fptr, unsigned char *dev);

/* Name of last snapshot file loaded/saved. */
unsigned char snapshot_name[MAXPATH];

/* Snap header info etc. */
pak_format			snap_info;
unsigned int		snap_length								= 0xffff; /*Rogue*/
boolean				load_debug_state,	load_ds_default		= TRUE;
boolean				snap_load_options,	snap_opt_default	= TRUE;
unsigned char		in_ver, version,	version_default		= V1_4;
boolean				load_extensions,	load_ext_default	= TRUE;
portability			write_port								= this_rom;
FILE				*sfp;
int					ch;
boolean				snap_eof;
unsigned char		pak_checksum;
unsigned char		this_version = ((VER_MAJOR << 4) | (VER_MINOR));
boolean				compress = FALSE, compressed;
unsigned char		algorithm, zerocode, ffcode, othercode;
unsigned char		comp_stock, comp_byte;

void checksum_registers(void)
{
	unsigned char	*snap_ptr = (unsigned char *)&snap_info.p3;
	unsigned int	chk_cnt;

	for (chk_cnt = 0; chk_cnt < sizeof(pak_part3); chk_cnt++)
		pak_checksum += *snap_ptr++;
}

boolean read_bytes(void *snap_ptr, unsigned int count)
{
	unsigned char *sp = snap_ptr;

	while (count > 0)
	{
		if (!snap_eof)
		{
			if ((ch = fgetc(sfp)) != EOF)
				*sp++ = ch;
			else
				snap_eof = TRUE;
		}
		count--;
	}
	return (snap_ptr != sp);
}

void write_bytes(void *snap_ptr, unsigned int count)
{
	unsigned char *sp = snap_ptr;

	while (count > 0)
	{
		fputc(*sp++, sfp);
		count--;
	}
}

void init_snap_info(void)
{
	snap_info.p2.debug_dump_flag			= 0;
	snap_info.p2.debug_disassemble_reg		= 1;
	snap_info.p3.snap_zero1					= 0;
	snap_info.p5.snap_write_protect			= 0;
	snap_info.p5.u1.snap_disk_directory[0]	= 2;
	strcpy(snap_info.p5.u2.snap_disk_name[0], "DISK0.DSK");
	strcpy(snap_info.p5.u2.snap_disk_name[1], "DISK1.DSK");
	snap_info.p5.snap_pak_directory[0]		= 2;
	strcpy(&snap_info.p5.snap_pak_directory[1], "\\COCO");
	snap_info.p5.snap_speed					= 0;
	snap_info.p5.snap_sound					= 0xff;
	snap_info.p6.snap_joystick_limits[0] =
		snap_info.p6.snap_joystick_limits[1] =
		snap_info.p6.snap_joystick_limits[4] =
		snap_info.p6.snap_joystick_limits[5] = 0;
	snap_info.p6.snap_joystick_limits[2] =
		snap_info.p6.snap_joystick_limits[3] =
		snap_info.p6.snap_joystick_limits[6] =
		snap_info.p6.snap_joystick_limits[7] = 1;
	snap_info.p6.snap_clock = ((arch == tandy) ? 0x4da7 : 0x5d2f);
	snap_info.p6.snap_drive_mode			= 0;
	snap_info.p6.snap_volume				= 0;
}

void put_snap_info(void)
{
	unsigned char farcnt;

	strncpy(snap_info.p1.snap_name, snapshot_name, 32);

	snap_info.p2.debug_dump_address			= dump_address;

	snap_info.p3.snap_pcr					= pcr.u_value;
	snap_info.p3.snap_x						= x_reg.u_value;
	snap_info.p3.snap_y						= y_reg.u_value;
	snap_info.p3.snap_u						= u_reg.u_value;
	snap_info.p3.snap_s						= s_reg.u_value;
	snap_info.p3.snap_dp					= dp_reg.u_value;
	snap_info.p3.snap_b						= b_reg.u_value;
	snap_info.p3.snap_a						= a_reg.u_value;
	snap_info.p5a.snap_8086_flags			= (0x7206 | (c_flag) |
												(h_flag << 4) | (z_flag << 6) |
												(n_flag << 7) | (v_flag << 11));
	snap_info.p5a.snap_6809_flags			= get_cc_reg();

	snap_info.p5b.snap_low32k_read_seg		= 0;
	snap_info.p5b.snap_low32k_write_seg		= 0;
	snap_info.p5b.snap_high32k_read_seg		= (mapmode1 ? 0 : 0x0800);
	snap_info.p5b.snap_high32k_write_seg	= (mapmode1 ? 0 : 0x1000);

	snap_info.p5c.snap_new_page_status		= 0xd4;
	snap_info.p5c.snap_new_rom_status		= (mapmode1 ? 0xdf : 0xde);
	snap_info.p5c.snap_new_ff02
		= snap_info.p7.snap_ff02			= memory[0xff02];
	snap_info.p5c.snap_new_ff03
		= snap_info.p7.snap_ff03			= memory[0xff03];
	snap_info.p5c.snap_new_ff22
		= snap_info.p7.snap_ff22			= memory[0xff22];

	snap_info.p5d.snap_video_base1			=
		snap_info.p7.snap_video_base2		= new_screen_base;
	snap_info.p5d.snap_video_end1			=
		snap_info.p7.snap_video_end2		= new_screen_end() + 1;

	snap_info.p5.u1.snap_disk_directory[1]	= 0;
	snap_info.p5.u2.snap_disk_name[2][0]	= 0;
	snap_info.p5.u2.snap_disk_name[3][0]	= 0;

	snap_info.p5.snap_crlf					= 0;
	snap_info.p5.snap_keyboard_mode			= 0;
	snap_info.p5.snap_left_joystick			=
		snap_info.p5.snap_right_joystick	= (realjoy ? 2 : 0);
	snap_info.p5.snap_lower_case			= (lower_case ? 6400 : 0);
	snap_info.p5.snap_artifact				= get_artifact();

	snap_info.p5.snap_dragon_rom			= ((arch == tandy) ? 0 : 0xff);

	/* Sound selection is 255 for old snaps, but 3 for V1.4 with SB. */
	if ((snap_info.p5.snap_sound > 0x03) && (version == V1_4))
		snap_info.p5.snap_sound &= 0x03;
	else if ((snap_info.p5.snap_sound != 0x00) && (version != V1_4))
		snap_info.p5.snap_sound = 0xff;

	/* V1.4 only items. */
	snap_info.p6.snap_cas_mode				= 0;
	snap_info.p6.snap_cas_directory[0]		= 2;
	strcpy(&snap_info.p6.snap_cas_directory[1], "\\PCDRAGON\\CASSETTE");
	strcpy(snap_info.p6.snap_cas_name, "CASSETTE.CAS");

	/* PC-Dragon specific items. */
	snap_info.p5.u1.pi.pcd_signature		= 0xBAAB;
	snap_info.p5.u1.pi.pcd_pak_version		= version;
	snap_info.p5.u1.pi.pcd_ext_version		= PCD_PAK_EXT_VER;
	snap_info.p5.u1.pi.pcd_emu_version		= this_version;
	snap_info.p5.u1.pi.pcd_state_only		= (write_port << 2);
	snap_info.p5.u1.pi.pcd_arch				= arch;
	snap_info.p5.u1.pi.pcd_rom_start		= (unsigned char)(ROM_START >> 8);

	snap_info.p5.u1.pi.pcd_irq_cycles		= irq_cycles;
	snap_info.p5.u1.pi.pcd_screen_size		= new_screen_size;
	snap_info.p5.u1.pi.pcd_pias[0]			= memory[0xff00];
	snap_info.p5.u1.pi.pcd_pias[1]			= memory[0xff01];
	snap_info.p5.u1.pi.pcd_pias[2]			= memory[0xff08];
	snap_info.p5.u1.pi.pcd_pias[3]			= memory[0xff0a];
	snap_info.p5.u1.pi.pcd_pias[4]			= memory[0xff20];
	snap_info.p5.u1.pi.pcd_pias[5]			= memory[0xff21];
	snap_info.p5.u1.pi.pcd_pias[6]			= memory[0xff28];
	snap_info.p5.u1.pi.pcd_pias[7]			= memory[0xff23];
	snap_info.p5.u1.pi.pcd_pias[8]			= memory[0xff2a];
	snap_info.p5.u1.pi.pcd_last_shift		= last_shift;
	memcpy(snap_info.p5.u1.pi.pcd_filemem,filemem,8);

	snap_info.p5.u1.pi.pcd_last_text_base	= last_text_base;
	memcpy(snap_info.p5.u1.pi.pcd_breakpoints,breakpoint,12);
	snap_info.p5.u1.pi.pcd_break_op			= break_op;
	snap_info.p5.u1.pi.pcd_diss_pcr			= diss_pcr;
	snap_info.p5.u1.pi.pcd_fill_start		= fill_start;
	snap_info.p5.u1.pi.pcd_fill_end			= fill_end;
	snap_info.p5.u1.pi.pcd_fill_value		= fill_value;
	snap_info.p5.u1.pi.pcd_left_window		= left_window;
	snap_info.p5.u1.pi.pcd_temp_break		= temp_breakpoint;
	snap_info.p5.u1.pi.pcd_breakvalue		= breakvalue;
	snap_info.p5.u1.pi.pcd_brk_control		= (in_debug << 4)
												| (edit_ascii << 1)
												| await_breakpoint;
	snap_info.p5.u1.pi.pcd_breaktype		= breaktype;
	snap_info.p5.u1.pi.pcd_intern			= intern_break;
	snap_info.p5.u1.pi.pcd_dump_address		= dump_address;

	snap_info.p5.u2.po.pcd_bitpack1				= cursortype
													| (new_int9_set << 2)
													| (install_breakpoints << 3)
													| (digital_joysticks << 4)
													| (printfile << 5)
													| (printsimple << 6)
													| (ask_filenames << 7);
	snap_info.p5.u2.po.pcd_back_ary				= (back_ary[0] << 4)
													| back_ary[1];
	snap_info.p5.u2.po.pcd_fore_ary				= (fore_ary[0] << 4)
													| fore_ary[1];
	memcpy(snap_info.p5.u2.po.pcd_vmode_ary,vmode_ary,24);
	snap_info.p5.u2.po.pcd_border_ary[0][0]		= border_ary[0][0];
	snap_info.p5.u2.po.pcd_border_ary[0][1]		= border_ary[0][1];
	snap_info.p5.u2.po.pcd_border_ary[1][0]		= border_ary[1][0];
	snap_info.p5.u2.po.pcd_border_ary[1][1]		= border_ary[1][1];
	snap_info.p5.u2.po.pcd_irq_rate				= irq_rate;
	for (farcnt = 0; farcnt < 16; farcnt++)
		snap_info.p5.u2.po.pcd_service_mem[farcnt] = memory[0xffe0+farcnt];
	snap_info.p5.u2.po.pcd_comspeed				= (unsigned char)comspeed;
	snap_info.p5.u2.po.pcd_lpt_and_swapping		= (lpt << 4) | swapping
													| (ignore_illegal_opcodes << 3);
	snap_info.p5.u2.po.pcd_hardjoy_resolution	= hardjoy_resolution;
}

void get_snap_info(unsigned int this_snap_len)
{
	boolean			load_options = snap_load_options;
	boolean			pcd_extensions;
	unsigned char	ext_ver;
	unsigned char	farcnt;
	portability		in_port = this_rom;

	/* Determine whether to load the PC-Dragon extensions. */
	/* File checksum and signature must be correct, and    */
	/* loading of extensions must be enabled by user.      */
	pcd_extensions = (load_extensions
			& (snap_info.p5.u1.pi.pcd_signature == 0xBAAB)
			& (snap_info.p5.u1.pi.pcd_checksum == pak_checksum));
	if (pcd_extensions)
	{
		switch (snap_info.p5.u1.pi.pcd_state_only & 0x03)
		{
			case 1:	load_options = FALSE;		break;
			case 2:	load_options = TRUE;		break;
		}
		switch (snap_info.p5.u1.pi.pcd_state_only >> 4)
		{
			case 1:	load_debug_state = FALSE;	break;
			case 2:	load_debug_state = TRUE;	break;
		}
	}

	pcr.u_value		= snap_info.p3.snap_pcr;
	x_reg.u_value	= snap_info.p3.snap_x;
	y_reg.u_value	= snap_info.p3.snap_y;
	u_reg.u_value	= snap_info.p3.snap_u;
	s_reg.u_value	= snap_info.p3.snap_s;
	dp_reg.u_value	= snap_info.p3.snap_dp;
	b_reg.u_value	= snap_info.p3.snap_b;
	a_reg.u_value	= snap_info.p3.snap_a;
	far_set_cc_reg(snap_info.p5a.snap_6809_flags);

	if (in_ver != V1_4)
	{
		mapmode1 = (snap_info.p5b.snap_high32k_read_seg == 0);
		memory[0xff22] = snap_info.p7.snap_ff22;
		memory[0xff02] = snap_info.p7.snap_ff02;
		memory[0xff03] = snap_info.p7.snap_ff03;
	}
	else
	{
		mapmode1 = (snap_info.p5c.snap_new_rom_status == 0xdf);
		memory[0xff02] = snap_info.p5c.snap_new_ff02;
		memory[0xff03] = snap_info.p5c.snap_new_ff03;
		memory[0xff22] = snap_info.p5c.snap_new_ff22;
	}

	new_screen_base = snap_info.p5d.snap_video_base1;
	switch (snap_info.p5d.snap_video_end1 - new_screen_base)
	{
		case 512:	new_screen_size = 0; break;
		case 1024:	new_screen_size = 1; break;
		case 2048:	new_screen_size = 2; break;
		case 1536:
		case 4608:	new_screen_size = 3; break;
		case 3072:	new_screen_size = 4; break;
		case 9216:	new_screen_size = 5; break;
		case 6144:
		default:	new_screen_size = 6; break;
	}

	if (load_options)
	{
		if (hasjoystick)
			realjoy = (snap_info.p5.snap_left_joystick == 2)
				|| (snap_info.p5.snap_left_joystick == 4);

		lower_case = (snap_info.p5.snap_lower_case > 0);
		set_artifact(snap_info.p5.snap_artifact);

		snap_length			= this_snap_len;
		compress			= compressed;
		version				= in_ver;
	}

	/* PC-Dragon specific items. */
	if (pcd_extensions)
	{
		ext_ver			= snap_info.p5.u1.pi.pcd_ext_version;
		if (ext_ver >= 0x21)
			in_port			= (snap_info.p5.u1.pi.pcd_state_only >> 2) & 0x03;
		irq_cycles		= snap_info.p5.u1.pi.pcd_irq_cycles;
		new_screen_size	= snap_info.p5.u1.pi.pcd_screen_size;
		memory[0xff00]	= snap_info.p5.u1.pi.pcd_pias[0];
		memory[0xff01]	= snap_info.p5.u1.pi.pcd_pias[1];
		memory[0xff08]	= snap_info.p5.u1.pi.pcd_pias[2];
		memory[0xff0a]	= snap_info.p5.u1.pi.pcd_pias[3];
		memory[0xff20]	= snap_info.p5.u1.pi.pcd_pias[4];
		memory[0xff21]	= snap_info.p5.u1.pi.pcd_pias[5];
		memory[0xff28]	= snap_info.p5.u1.pi.pcd_pias[6];
		memory[0xff23]	= snap_info.p5.u1.pi.pcd_pias[7];
		memory[0xff2a]	= snap_info.p5.u1.pi.pcd_pias[8];
		last_shift		= snap_info.p5.u1.pi.pcd_last_shift;
		memcpy(filemem,snap_info.p5.u1.pi.pcd_filemem,8);
		for (farcnt = 0; farcnt < 16; farcnt++)
			memory[0xffe0+farcnt] = snap_info.p5.u2.po.pcd_service_mem[farcnt];

		if (load_debug_state)
		{
			last_text_base		= snap_info.p5.u1.pi.pcd_last_text_base;
			memcpy(breakpoint,snap_info.p5.u1.pi.pcd_breakpoints,12);
			break_op			= snap_info.p5.u1.pi.pcd_break_op;
			diss_pcr			= snap_info.p5.u1.pi.pcd_diss_pcr;
			fill_start			= snap_info.p5.u1.pi.pcd_fill_start;
			fill_end			= snap_info.p5.u1.pi.pcd_fill_end;
			fill_value			= snap_info.p5.u1.pi.pcd_fill_value;
			left_window			= snap_info.p5.u1.pi.pcd_left_window;
			temp_breakpoint		= snap_info.p5.u1.pi.pcd_temp_break;
			breakvalue			= snap_info.p5.u1.pi.pcd_breakvalue;
			in_debug			= snap_info.p5.u1.pi.pcd_brk_control >> 4;
			edit_ascii			= (snap_info.p5.u1.pi.pcd_brk_control >> 1) & 0x01;
			await_breakpoint	= snap_info.p5.u1.pi.pcd_brk_control & 0x01;
			breaktype			= snap_info.p5.u1.pi.pcd_breaktype;
			intern_break		= snap_info.p5.u1.pi.pcd_intern;
			dump_address		= snap_info.p5.u1.pi.pcd_dump_address;
		}

		if (load_options)
		{
			write_port			= in_port;
			cursortype			= snap_info.p5.u2.po.pcd_bitpack1 & 0x03;
			_setcursortype(cursortype);

			if (new_int9_set)
				Set_Old_Int9();
			new_int9_set		= (snap_info.p5.u2.po.pcd_bitpack1 >> 2) & 0x01;
			if (new_int9_set)
				Set_New_Int9();

			install_breakpoints	= (snap_info.p5.u2.po.pcd_bitpack1 >> 3) & 0x01;
			digital_joysticks	= (snap_info.p5.u2.po.pcd_bitpack1 >> 4) & 0x01;
			change_printfile((snap_info.p5.u2.po.pcd_bitpack1 >> 5) & 0x01);
			printsimple			= (snap_info.p5.u2.po.pcd_bitpack1 >> 6) & 0x01;
			ask_filenames		= (snap_info.p5.u2.po.pcd_bitpack1 >> 7) & 0x01;
			back_ary[0]			= snap_info.p5.u2.po.pcd_back_ary >> 4;
			back_ary[1]			= snap_info.p5.u2.po.pcd_back_ary & 0x0f;
			fore_ary[0]			= snap_info.p5.u2.po.pcd_fore_ary >> 4;
			fore_ary[1]			= snap_info.p5.u2.po.pcd_fore_ary & 0x0f;
			memcpy(vmode_ary,snap_info.p5.u2.po.pcd_vmode_ary,24);
			border_ary[0][0]	= snap_info.p5.u2.po.pcd_border_ary[0][0];
			border_ary[0][1]	= snap_info.p5.u2.po.pcd_border_ary[0][1];
			border_ary[1][0]	= snap_info.p5.u2.po.pcd_border_ary[1][0];
			border_ary[1][1]	= snap_info.p5.u2.po.pcd_border_ary[1][1];
			irq_rate			= snap_info.p5.u2.po.pcd_irq_rate;
			should_be_seconds	= (double)irq_rate / 89.0;
			set_comspeed((unsigned int)snap_info.p5.u2.po.pcd_comspeed);
			change_lpt(snap_info.p5.u2.po.pcd_lpt_and_swapping >> 4);
			swapping			= snap_info.p5.u2.po.pcd_lpt_and_swapping & 0x07;
			ignore_illegal_opcodes = (snap_info.p5.u2.po.pcd_lpt_and_swapping >> 3) & 0x01;
			hardjoy_resolution	= snap_info.p5.u2.po.pcd_hardjoy_resolution;
		}
	}

	/* Patch Dragon 32 snapshots loaded into a      */
	/* Dragon 64 such that the 32K ROM is selected. */
	if ((arch == dragon64) &&
		((!pcd_extensions) || (snap_info.p5.u1.pi.pcd_arch != dragon64)))
	{
		memory[0xff22] |= 0x04;
	}

	/* Check for incompatibilities of snapshot files with current setup. */
	if (pcd_extensions && (snap_info.p5.u1.pi.pcd_pak_version != in_ver))
		selection_box(4,15,0,FALSE,FALSE,FALSE,99,
			"    WARNING!","",
			"Mismatched .PAK",
			"format version!"); /* unlikely that this is ever detected */

	if (in_port == this_rom)
	{
		if ((pcd_extensions && (snap_info.p5.u1.pi.pcd_arch != arch))
			|| (!pcd_extensions &&
				(snap_info.p5.snap_dragon_rom != ((arch == tandy) ? 0 : 0xff))))
			selection_box(3,18,0,FALSE,FALSE,FALSE,99,
				"     WARNING!","",
				"ROM file mismatch!");
	}
	else if (in_port == any_dragon) /* only true with pcd extensions */
	{
		if ((snap_info.p5.u1.pi.pcd_arch == tandy) || (arch == tandy))
			selection_box(3,18,0,FALSE,FALSE,FALSE,99,
				"     WARNING!","",
				"ROM file mismatch!");
	}

	/* Check for newer emulator version or extensions. */
	if (pcd_extensions && (snap_info.p5.u1.pi.pcd_ext_version > this_version))
		selection_box(4,18,0,FALSE,FALSE,FALSE,99,
			"     WARNING!","",
			"Newer PCD snapshot",
			"extensions found!!");
	else if (pcd_extensions && (snap_info.p5.u1.pi.pcd_emu_version > this_version))
		selection_box(4,18,0,FALSE,FALSE,FALSE,99,
			"      -INFO-","",
			"Snapshot was saved",
			"by newer emulator!");

	so_backlog_cnt = backlog_cnt = 0;
	execute_vmode(TRUE);
}

unsigned char getmembyte(void)
{
	if (comp_stock > 0)
	{
		comp_stock--;
		return(comp_byte);
	}
	else
		comp_byte = fgetc(sfp);

	if (compressed)
	{
		if (comp_byte == zerocode)
		{
			comp_stock	= fgetc(sfp);
			comp_byte	= 0;
		}
		else if (comp_byte == ffcode)
		{
			comp_stock	= fgetc(sfp);
			comp_byte	= 0xff;
		}
		else if (comp_byte == othercode)
		{
			comp_stock	= fgetc(sfp);
			comp_byte	= fgetc(sfp);
		}
	}
	return (comp_byte);
}

boolean load_snapshot(unsigned char *sname, boolean *eb)
{
	unsigned long		dump_len;
	unsigned int		local_snap_len;
	unsigned int		dump_add, dump_start;
	unsigned char far	*mem = memory;
	unsigned char		snap_filename[MAXPATH+4+9] = "snapshot\\";
	boolean				included_ext = TRUE;
	boolean				extra_bytes;

	pak_checksum	= 0;
	in_ver			= version;

	strcpy(&snap_filename[9], sname);
	if (strext(&snap_filename[9]) == NULL)
	{
		included_ext = FALSE;
		strcat(&snap_filename[9],".pak");
	}

	if ((sfp = fopen(&snap_filename[9],"rb")) == NULL)
	{
		if ((sfp = fopen(snap_filename,"rb")) == NULL)
		{
			if (!included_ext)
			{
				*strext(&snap_filename[9]) = '\0';
				if (arch == dragon32)
					strcat(&snap_filename[9],".d32");
				else if (arch == dragon64)
					strcat(&snap_filename[9],".d64");
				else
					strcat(&snap_filename[9],".cco");

				if ((sfp = fopen(&snap_filename[9],"rb")) == NULL)
				{
					if ((sfp = fopen(snap_filename,"rb")) == NULL)
					{
						if (!included_ext)
						{
							*strext(&snap_filename[9]) = '\0';
							if (arch != tandy)
							{
								strcat(&snap_filename[9],".d96");
								if ((sfp = fopen(&snap_filename[9],"rb")) == NULL)
								{
									if ((sfp = fopen(snap_filename,"rb")) == NULL)
										return FALSE;
								}
							}
							else
								return FALSE;
						}
						else
							return FALSE;
					}
				}
			}
			else
				return FALSE;
		}
	}

	local_snap_len				= fgetc(sfp) + (fgetc(sfp) << 8);
	dump_start = dump_add		= fgetc(sfp) + (fgetc(sfp) << 8);

	dump_len = (unsigned long)local_snap_len;
	if (dump_len == 0L)
		dump_len = 0x10000;
	if (dump_len > 0xff00L)
		local_snap_len = 0xff00;

	compressed = (dump_add == COMPRESSED_SIG);
	if (compressed)
	{
		dump_add	= fgetc(sfp) + (fgetc(sfp) << 8);
		in_ver		= fgetc(sfp);
		algorithm	= fgetc(sfp);
		if (algorithm > 0)
			selection_box(6,13,0,FALSE,FALSE,FALSE,99,
				" FATAL ERROR",
				" ===========","",
				"   Unknown",
				" compression",
				"   method!");
		zerocode	= fgetc(sfp);
		ffcode		= fgetc(sfp);
		othercode	= fgetc(sfp);
	}
	comp_stock = 0;

	while (dump_len > 0)
	{
		if (dump_add == 0x8000)
		{
			if (arch != dragon32)
			{
				mem = hiram;
				dump_add = 0;
			}
			else
			{
				goto excess_data;
			}
		}
		else if (dump_add == 0xff00)
		{
excess_data:
			while (dump_len > 1L)
			{
				ch = getmembyte();
				pak_checksum += ch;
				dump_len--;
			}
			dump_add = SNAP_JUNK;
		}
		mem[dump_add++] = ch = getmembyte();
		pak_checksum += ch;
		dump_len--;
	}
	snap_eof = (ch == EOF);

	/* Prepare to receive new info. */
	put_snap_info();

	/* Read in the snapshot information. */
	extra_bytes = read_bytes(&snap_info.p1, sizeof(pak_part1));

	if (in_ver == V1_2)
		read_bytes(&snap_info.p2, sizeof(pak_part2));

	read_bytes(&snap_info.p3, sizeof(pak_part3));

	if (in_ver == V1_2)
		read_bytes(&snap_info.p4, sizeof(pak_part4));

	if (in_ver == V1_4)
		read_bytes(&snap_info.p5e, sizeof(pak_part5e));
	read_bytes(&snap_info.p5a, sizeof(pak_part5a));
	if (in_ver != V1_4)
		read_bytes(&snap_info.p5b, sizeof(pak_part5b));
	else
		read_bytes(&snap_info.p5c, sizeof(pak_part5c));
	read_bytes(&snap_info.p5d, sizeof(pak_part5d));
	if (in_ver != V1_4)
		read_bytes(&snap_info.p5e, sizeof(pak_part5e));
	read_bytes(&snap_info.p5, sizeof(pak_part5));

	if (in_ver == V1_4)
		read_bytes(&snap_info.p6, sizeof(pak_part6));
	else
		read_bytes(&snap_info.p7, sizeof(pak_part7));

	fclose(sfp);

	/* Update checksum for registers. */
	checksum_registers();

	/* Install new info. */
	get_snap_info(local_snap_len);

	strcpy(snapshot_name,sname);

	/* Pass back extra_bytes flag if requested. */
	if (eb != NULL)
	{
		*eb = extra_bytes;

		/* Cause FIRQ and override extra_bytes */
		/* flag if snapshot starts at $C000.   */
		if (dump_start == 0xc000)
		{
			firq();
			*eb = TRUE;
		}
	}

	return(TRUE);
}

boolean save_snapshot(unsigned char *sname)
{
	unsigned char		snap_filename[MAXPATH+4+9] = "snapshot\\";
	boolean				opened = FALSE;

	strcpy(&snap_filename[9], sname);
	if (strext(&snap_filename[9]) == NULL)
		strcat(&snap_filename[9],".pak");

	if ((strchr(&snap_filename[9],'\\') == NULL)
		&& (strchr(&snap_filename[9],'/') == NULL))
	{
		switch (open_outfile(snap_filename,&sfp,NULL))
		{
			case OUT_DECLINED:	/* Don't check other directory - it would    */
								/* be confusing whether it succeeded or not! */
								return FALSE;

			case OUT_OPEN:		opened = TRUE;
								break;
		}
	}

	if (!opened)
	{
		switch (open_outfile(&snap_filename[9],&sfp,NULL))
		{
			case OUT_DECLINED:	return FALSE;

			case OUT_FAILED:	selection_box(2,14,0,FALSE,FALSE,FALSE,99,
									"Failed to open",
									"snapshot file!");
								return FALSE;
		}
	}

	/* Cannot actually write data now (emulator might be in the middle */
	/* of an instruction). Operation will be completed at the end of   */
	/* a fetch/execute cycle (just before the next IRQ).               */
	snapshot_in_progress = TRUE;

	strcpy(snapshot_name,sname);
	return TRUE;
}

void snap_block(unsigned int block_len, unsigned char far *block)
{
	unsigned int	block_add = 0;
	unsigned char	repeat;

	while (block_len > 0)
	{
		ch = block[block_add++];
		pak_checksum += ch;
		block_len--;

		if (compress)
		{
			if ((ch == zerocode)
				|| (ch == ffcode)
				|| (ch == othercode)
				|| ((block_len >= 2) && (block[block_add] == ch) && (block[block_add + 1] == ch)))
			{
				repeat = 0;
				while ((block[block_add] == ch) && (block_len > 0) && (repeat < 255))
				{
					repeat++;
					pak_checksum += ch;
					block_add++;
					block_len--;
				}
				if (ch == 0)
					fputc(zerocode, sfp);
				else if (ch == 0xff)
					fputc(ffcode, sfp);
				else
					fputc(othercode, sfp);

				fputc(repeat, sfp);
				if ((ch != 0) && (ch != 0xff))
					fputc(ch, sfp);
			}
			else
				fputc(ch, sfp);
		}
		else
			fputc(ch, sfp);
	}
}

void complete_snapshot(void)
{
	unsigned int		dump_len;
	unsigned int		dump_add;
	unsigned char far	*mem;

	pak_checksum = 0;

	fputc(snap_length & 0x00ff, sfp);
	fputc(snap_length >> 8, sfp);

	if (compress)
	{
		fputc(COMPRESSED_SIG & 0xff, sfp);
		fputc(COMPRESSED_SIG >> 8, sfp);
	}

	fputc(0, sfp);
	fputc(0, sfp);

	if (compress)
	{
		unsigned int usage_array[256];

		/* Calculate the three least used bytes. */
		for (dump_add = 0; dump_add < 256; dump_add++)
			usage_array[dump_add] = 0;

		dump_len	= snap_length;
		dump_add	= 0;
		mem			= memory;
		while (dump_len > 0)
		{
			if ((dump_add == 0x8000) && (arch != dragon32))
			{
				mem = hiram;
				dump_add = 0;
			}
			++usage_array[mem[dump_add++]];
			dump_len--;
		}

		zerocode	= 0;
		ffcode		= 1;
		othercode	= 2;

		for (dump_add = 3; dump_add < 256; dump_add++)
		{
			if (usage_array[dump_add] < usage_array[zerocode])
				zerocode = dump_add;
			else if (usage_array[dump_add] < usage_array[ffcode])
				ffcode = dump_add;
			else if (usage_array[dump_add] < usage_array[othercode])
				othercode = dump_add;
		}

		fputc(version, sfp);
		fputc(0, sfp);
		fputc(zerocode, sfp);
		fputc(ffcode, sfp);
		fputc(othercode, sfp);
	}

	if (snap_length > 0x8000)
	{
		snap_block(0x8000, memory);
		snap_block(snap_length - 0x8000,
			(arch == dragon32) ? (&memory[0x8000]) : hiram);
	}
	else
		snap_block(snap_length, memory);

	/* Prepare info for snap header. */
	put_snap_info();

	/* Update checksum for registers. */
	checksum_registers();
	snap_info.p5.u1.pi.pcd_checksum = pak_checksum;

	/* Write the snapshot information. */
	write_bytes(&snap_info.p1, sizeof(pak_part1));

	if (version == V1_2)
		write_bytes(&snap_info.p2, sizeof(pak_part2));

	write_bytes(&snap_info.p3, sizeof(pak_part3));

	if (version == V1_2)
		write_bytes(&snap_info.p4, sizeof(pak_part4));

	if (version == V1_4)
		write_bytes(&snap_info.p5e, sizeof(pak_part5e));
	write_bytes(&snap_info.p5a, sizeof(pak_part5a));
	if (version != V1_4)
		write_bytes(&snap_info.p5b, sizeof(pak_part5b));
	else
		write_bytes(&snap_info.p5c, sizeof(pak_part5c));
	write_bytes(&snap_info.p5d, sizeof(pak_part5d));
	if (version != V1_4)
		write_bytes(&snap_info.p5e, sizeof(pak_part5e));
	write_bytes(&snap_info.p5, sizeof(pak_part5));

	if (version == V1_4)
		write_bytes(&snap_info.p6, sizeof(pak_part6));
	else
		write_bytes(&snap_info.p7, sizeof(pak_part7));

	fclose(sfp);
}
