/* MBRSCAN.CPP  Version 1.0 95/06/26
                        1.1 99/05/05 LBA access
  Master Boot Record SCAN - Partition Table Analyzer
  Copyright (C) 1995,1999 by Nobumichi Kozawa

  Please refer GPL.TXT including this package for the GNU GPL details.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#define VER "MBRSCAN V1.1 Copyright (C) 1995,1999 by Nobumichi Kozawa"

#include <io.h>
#include <share.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <dos.h>
#include <dir.h>
#include <conio.h>
#include <fstream.h>
#include <process.h>
#include "mbrscan.h"

#define CYL(r1, r2) (((r1 >> 6) << 8)+r2)
#define SEC(r) (r % 0x40)
#define EXT(r) (5 == r || 0xf == r || 0x85 == r)  // Partition ID == Extended ?
// Calculate SECTor address from CCHHSS and geometry. xhh = number of heads, xss = number of sectors
#define CSECT(cc, hh, ss, xhh, xss) (((unsigned long)cc * (unsigned long)xhh + (unsigned long)hh) * (unsigned long)xss + (unsigned long)ss - 1ul)

struct freespace {
  unsigned long start;
  unsigned long end;
};

struct patlist {
  int lba;  // 0 = non-LBA, 1 = LBA
  int depth;  
  int startsec;  // start sector
  int cyll;      // cylinder low
  int cylhsec;   // cylinder high + sector
  int head;
  int partid;    // partition id
  int drv;
  unsigned long abssect; // abs base sector
  unsigned long extsect; // ext sector
  int pte;       // partition table entry
};

struct diskparm {
   int drv; // drive id, 0x80 - 0x8n
   unsigned int lba; // 0=non-LBA, 1=LBA
   unsigned int cc;  // # of Cyl
   unsigned int hh;  // # of Heads
   unsigned int ss;  // # of Sec/trac
   unsigned long blks;  // # of sectors
};  
 
void mbrdisp(struct patlist *plist, unsigned char ptable[16][4][16], int sd, int st);
void partdisp(unsigned char *ptable);
char* ppartid(int partid);
void pbiosperm(unsigned char *buff);
void getparm(struct diskparm *drv);
void int1308(struct diskparm *drv);
void int1348(struct diskparm *drv);
unsigned long getnum(unsigned char *buff, int ofs, int len);
char* int13err(int rc);
void resetdisk(int drv);
void setdap(int sects,char *dap,unsigned char *buff, unsigned long bptr);
void partmap3(unsigned char ptable[16][4][16], int dep, struct diskparm *disk, int sm, int sf);
void lbafree(struct freespace free[16], unsigned long absstart, unsigned long absend);
void chkgeo(unsigned char ptable[16][4][16], struct diskparm *disk);

/**************
 ***  main  ***
 **************/

void main(int argc, char *argv[]) {
   int drv = 0x80, i, sd = 0, sm = 0, sf = 0, st = 0, fnlba = 0;
   struct patlist plist;
   struct diskparm disk;
   unsigned char ptable[16][4][16];  // partition tables
   //                   |   |  -- 16 bytes per entry
   //                   |   +---- 4 entries per a sector
   //                   +-------- up to 16 partition tables

   puts(VER);
   if (argc > 1) {
     if ((('-' == argv[1][0] || '/' == argv[1][0]) && '?' == argv[1][1]) || '?' == argv[1][0]) {
       cout << "\nMBRSCAN comes with ABSOLUTELY NO WARRANTY;";
       cout << "\nThis is free software, and you are welcome to redistribute it";
       cout << "\nunder certain conditions: See GPL.TXT for details.\n";
       cout << "\nUsage: MBRSCAN [-opt][drive]\n";
       cout << "\n       opt: sd = Suppress Dump";   
       cout << "\n            st = Suppress Partition Table Report";
       cout << "\n            sm = Suppress Partition Map";
       cout << "\n            sf = Suppress Free Space Report";
       cout << "\n       drive: 0 = FD-A";
       cout << "\n              1 = FD-B";
       cout << "\n              80 = 1st HDD";
       cout << "\n              81 = 2nd HDD ...";
       cout << "\n              default = 80 (1st HDD)\n";
       exit(99);
     }
     for (i = 1; i < argc; i++) {
       if ('-' == argv[i][0] || '/' == argv[i][0]) {
         if (0 == strcmpi(argv[i]+1,"sd")) sd = 1;
         else if (0 == strcmpi(argv[i]+1,"st")) st = 1;
         else if (0 == strcmpi(argv[i]+1,"sm")) sm = 1;
         else if (0 == strcmpi(argv[i]+1,"sf")) sf = 1;          
         else if (0 == strcmpi(argv[i]+1,"fnlba")) fnlba = 1;
       } else sscanf(argv[i],"%x",&drv);
     }
   }

   if (fnlba) {   // hidden option :-)
     disk.drv = 0 - drv;
     puts("Forced non-LBA mode");
   } else disk.drv = drv;             

   getparm(&disk);            

   if (0 == disk.lba) puts("Non-LBA mode");
   else puts("LBA mode");
   printf("DRV:%02X ",drv);
   if (0 != disk.cc) printf(" CC=%04Xh(%u), HH=%02Xh(%u), SS=%02Xh(%u), BLKS=%lXh(%lu)",disk.cc,disk.cc,disk.hh,disk.hh,disk.ss,disk.ss,disk.blks,disk.blks);
   puts("");
   plist.lba = disk.lba;
   plist.depth = 1;
   plist.startsec = 0;  // --> MBR
   plist.cyll = 0;      // |
   plist.cylhsec = 1;   // +-> MBR
   plist.head = 0;      // |
   plist.drv = drv;
   plist.abssect = 0;
   plist.extsect = 0;
   while (plist.lba >= 0) { 
     mbrdisp(&plist, ptable, sd, st);
   }

   if (0 == disk.lba && 0 != disk.cc) chkgeo(ptable, &disk);
   partmap3(ptable, plist.depth, &disk, sm, sf);
}

// Cheks geometry parms with 1st partition table entry
void chkgeo(unsigned char ptable[16][4][16], struct diskparm *disk) {
  unsigned int sect, correct = 0;                                                         

  if (1 + ptable[1][0][5] != disk->hh) {
    disk->hh = ptable[1][0][5] + 1;
    printf("MBRS008 Geometry Unmatched. Assuming Head Counts = %02X.\n", disk->hh);
    correct = 1;
  }
  if ((sect = SEC(ptable[1][0][6])) != disk->ss) {
    printf("MBRS009 Geometry Unmatched. Assuming Number of Sector = %02X.\n", sect);
    disk->ss = sect;
    correct = 1;    
  }
  if (correct) {
    disk->blks = (unsigned long) disk->cc * disk->hh * disk->ss;
    printf("MBRS010 Number of Blocks re-calculated. Blocks = %02lX(%ul)\n",disk->blks,disk->blks);
  }  
}

// Partition Map display
void partmap3(unsigned char ptable[16][4][16], int dep, struct diskparm *disk, int sm, int sf) {
   unsigned long base = 0, ext = 0, start, sects, absstart, absend, sectpertrack;   
   struct freespace prim_free[16], ext_free[16];
   for (int i = 0; i < 16; i++) prim_free[i].start = prim_free[i].end = ext_free[i].start = ext_free[i].end = 0;
   if (0 != disk->cc) {   // if Geometry saved
     prim_free[0].start = disk->ss;  // first partition starts from 2nd track
     prim_free[0].end = disk->blks - 1;
   } else {  
     prim_free[0].start = SEC(ptable[1][0][6]); // assuming track size from end sector
     prim_free[0].end = 0xffffffff;  // set max value due to unknown actual size
   }
   sectpertrack = prim_free[0].start; // should be !

   unsigned int xhh, xss;  // maximum number of head and sector
   if (0 == disk->cc) {
     xhh = ptable[1][0][5];
     xss = SEC(ptable[1][0][6]);
     printf("\nMBRS011 Geometry was not saved. Assuming HH = %02X, SS = %02X\n", xhh + 1, xss);
   } else {
     xhh = disk->hh - 1;
     xss = disk->ss;
   }

   if (!sm && 0 == disk->lba) {  // non-LBA map
     puts("\n[ PARTITION MAP ]\n");                      
     puts("        START      END");
     puts("      CCC-HH-SS CCC-HH-SS   START    BLOCKS");
     puts("      ========= =========  ======== ========");

     for (i = 1; i <= dep; i++) {
        for (int j = 0; j < 4; j++) {
           if (0 != ptable[i][j][4]) {
             printf("%d-%d : ", i, j);
             printf("%03X-%02X-%02X %03X-%02X-%02X  %08lX %08lX %s\n", 
                    CYL(ptable[i][j][2],ptable[i][j][3]), ptable[i][j][1], SEC(ptable[i][j][2]), 
                    CYL(ptable[i][j][6],ptable[i][j][7]), ptable[i][j][5], SEC(ptable[i][j][6]), 
                    getnum(ptable[i][j], 8, 4), getnum(ptable[i][j], 12, 4), ppartid(ptable[i][j][4]));
             if (0 != ptable[i][j][1] && ptable[i][j][1] != 1) puts("MBRS012 START HEAD is usually 0x00 or 0x01.");
             if (1 != SEC(ptable[i][j][2])) puts("MBRS013 START SECT should be 0x01.");
             if (xhh != ptable[i][j][5]) printf("MBRS014 END HEAD should be %02X.\n", xhh);
             if (xss != SEC(ptable[i][j][6])) printf("MBRS015 END SECT should be %02X.\n", xss);
           }
         }
      }
   }
             
   // LBA map
   if (!sm) {
     puts("\n[ PARTITION MAP - LBA ]\n");                         
     puts("          ABSOLUTE          PART-TABLE");
     puts("       START     END      START    BLOCKS");
     puts("      ======== ========  ======== ========");
   }
               
   unsigned long ultemp, maxchs = CSECT(0x3ff,xhh,xss,(xhh+1),xss);
   for (i = 1; i <= dep; i++) {            // number of partitions
      for (int j = 0; j < 4; j++) {        // partition table entry
         if (0 != ptable[i][j][4]) {       // Not Empty ?
           if (!sm) printf("%d-%d : ", i, j);
           start = getnum(ptable[i][j], 8, 4);
           sects = getnum(ptable[i][j], 12, 4);                               
           if (EXT(ptable[i][j][4])) {     // Partition ID == Extended Partition ?
             if (1 == i) {                 // Primary partition ?
               absstart = ext = base = start; // set base address
               ext_free[0].start = absstart + sectpertrack;  // set free space area
               ext_free[0].end = absstart + sects - 1;       // for extended partition
             } else absstart = ext = start + base; // set extended base address
           } else if (1 == i) absstart = start;    // primary partition ?
           else absstart = ext + start;
           absend = absstart + sects - 1;
           if (!sm) printf("%08lX %08lX  %08lX %08lX  %s\n", absstart, absend, start, sects, ppartid(ptable[i][j][4]));
           if (maxchs >= absstart) {
             if ((ultemp = CSECT(CYL(ptable[i][j][2],ptable[i][j][3]),ptable[i][j][1],SEC(ptable[i][j][2]),(xhh+1),xss)) != absstart) 
               printf("MBRS016 START LBA and CCHHSS unmatched. Calculated Block Address = %08lX\n", ultemp);
           }
           if (maxchs >= absend) {
             if ((ultemp = CSECT(CYL(ptable[i][j][6],ptable[i][j][7]),ptable[i][j][5],SEC(ptable[i][j][6]),(xhh+1),xss)) != absend) 
               printf("MBRS017 END LBA and CCHHSS unmatched. Calculated Block Address = %08lX\n", ultemp);
           }                                                
           // checking free space
           if (1 == i) lbafree(prim_free, absstart, absend);
           else lbafree(ext_free, absstart, absend);
         }
      }
   }
   
   int flg = 1;
   if (!sf) {
     puts("\n[ FREE SPACE ]\n"); 
     for (i = 0; i < 16; i++) {           
       if (0 == prim_free[i].start) break;
       if (0 == disk->cc && 0xffffffff == prim_free[i].end) printf("MBRS005 Geometry was not saved. Free space after %08lX is unknown.\n",prim_free[i].start);
       else printf("Primary : %08lX - %08lX\n", prim_free[i].start, prim_free[i].end);
       flg = 0;
     }
     for (i = 0; i < 16; i++) {           
       if (0 == ext_free[i].start) break;
       printf("Extended: %08lX - %08lX\n", ext_free[i].start, ext_free[i].end);
       flg = 0;
     }      
     if (flg) puts("None.");
   }
}
                                 
// free space checker
void lbafree(struct freespace free[16], unsigned long absstart, unsigned long absend) {
  int done = 0;
  unsigned long tmpstart, tmpend, tmp;

  for (int i = 0; i < 16; i++) {
    if (done) {
      if (tmpstart) {  // needs to shift ?
        // swap
        tmp = free[i].start; free[i].start = tmpstart; tmpstart = tmp;
        tmp = free[i].end; free[i].end = tmpend; tmpend = tmp;
      } else break;    // complete
    } else {
      if (free[i].start <= absstart && absstart < free[i].end) { 
        if (free[i].start < absend && absend <= free[i].end) {
          // found the area to be occupied
          done = 1;
          if (free[i].start == absstart && absend == free[i].end) free[i].start = free[i].end = 0; // just fit !
          else if (free[i].start == absstart && absend < free[i].end) {  // more space after this area
            free[i].start = absend + 1; tmpstart = tmpend = 0;
          } else if (free[i].start < absstart && absend < free[i].end) {  // spaces before and after the area
            tmpend = free[i].end; free[i].end = absstart - 1; tmpstart = absend + 1;
          } else if (free[i].start < absstart && absend == free[i].end) { // free space exists before the area
            free[i].end = absstart - 1; tmpstart = tmpend = 0;
          }
        } else printf("MBRS004 ERROR(lbafree): Overlap! FREE:%08lX-%08lX SECTS:%08lX-%08lX\n",free[i].start,free[i].end,absstart,absend);
      }
    } 
  }
}

// dump boot record and scan partition table
void mbrdisp(struct patlist *plist, unsigned char ptable[16][4][16], int sd, int st)
{
   union REGS inr, outr;
   struct SREGS seg;
   unsigned char buff[1024], buff2[512], partid, dap[16];
   int dp = plist->depth;
   int lba = plist->lba;
   unsigned long sects;

   plist->lba = -1;           // set flag to found 2nd extended partition in a sector
   memdump mdump;             // memory dump class
   // read boot record
   if (0 == lba) {              // non-LBA
     inr.h.ah = 0x02;           // Read sector
     inr.h.al = 0x01;           // # of sectors
     inr.h.ch = plist->cyll;              // CYL low
     inr.h.cl = plist->cylhsec;           // CYL high + SEC
     inr.h.dh = plist->head;              // HEAD
     seg.es = FP_SEG(buff);     // buffer
     inr.x.bx = FP_OFF(buff);
   } else {                     // LBA
     setdap(1,dap,buff,plist->abssect);
     inr.h.ah = 0x42;
     seg.ds = FP_SEG(dap);
     inr.x.si = FP_OFF(dap);
   }
   inr.h.dl = plist->drv;           // Drive
   int86x(0x13, &inr, &outr, &seg);
   if (0 != outr.h.ah) {
     printf("MBRS007 Disk read error ! %s\n", int13err(outr.h.ah));
     exit(1);
   }           
   if (!sd) {
     if (1 == dp) printf("\n1 - Master Boot Record");
     else printf("\n%d - %s", dp, ppartid(plist->partid));
     if (0 == lba) printf(" (%04Xh-%02Xh-%02Xh)",CYL(plist->cylhsec, plist->cyll), plist->head, SEC(plist->cylhsec));
     else printf(" (%08lXh)", plist->abssect);
     if (1 != dp) printf(" (Partition Table Entry: %d-%d)\n", plist->depth - 1, plist->pte);
     else puts("");
     mdump.disp(buff, 512);
   }
   int i;             
   // partition table
   for (i = 0; i < 4; i++) {
     partid = buff[4+0x1be+i*16];     // partition ID
     mdump.offset = 0x1be + i * 16;
     if (!st) {
       printf("\n%d - Partition table entry = %d  << %s >>\n", dp, i, ppartid(partid));
       mdump.disp(buff + mdump.offset, 16);
       puts("       ST HH CS CC ID hh cs cc START-SEC   NUMBER-SEC");
       partdisp(buff + mdump.offset);
     }
     if (dp < 16) memcpy(ptable[dp][i], buff + mdump.offset, 16); // copy ptable
     mdump.offset = 0;
     if (0 != partid) {  // not a empty partition ?
       if (0 == lba) {
         inr.h.ah = 0x02;           // Read sector
         inr.h.al = 0x01;           // # of sectors
         inr.h.ch = buff[3+0x1be+i*16];   // CYL low
         inr.h.cl = buff[2+0x1be+i*16];   // CYL high + SEC
         inr.h.dh = buff[1+0x1be+i*16];   // HEAD
         seg.es = FP_SEG(buff2);     // buffer
         inr.x.bx = FP_OFF(buff2);
       } else {
         setdap(1,dap,buff2,plist->abssect + (sects = getnum(buff,8+0x1be+i*16,4)));
         inr.h.ah = 0x42;
         seg.ds = FP_SEG(dap);
         inr.x.si = FP_OFF(dap);
       }  
       inr.h.dl = plist->drv;           // Drive
       if (EXT(partid)) {  // partition ID = extended ?
         if (0 > plist->lba) {  // is this 1st time ?
           plist->lba = lba;    // setup plist for next partition table
           plist->depth = dp+1;
           plist->cyll = inr.h.ch;
           plist->cylhsec = inr.h.cl;
           plist->head = inr.h.dh;
           plist->partid = partid;  
           plist->abssect = sects + plist->extsect;
           plist->pte = i;      // just remember origin
           if (1 == dp) plist->extsect = sects;
         } else puts("MBRS006 Ooops! Second extension found in a partition table.");
       } else {
         int86x(0x13, &inr, &outr, &seg);
         if (!sd) {
           printf("\n%d-%d %s Boot Record", dp, i, ppartid(partid));
           if (0 == lba) printf(" (%04Xh-%02Xh-%02Xh)\n",CYL(inr.h.cl, inr.h.ch), inr.h.dh, SEC(inr.h.cl));
           else printf(" (%08lXh)\n",plist->abssect + sects);
           mdump.disp(buff2, 512);
         }
       }
     }
   }
}
                                                       
// partition table entry display
void partdisp(unsigned char *ptable)
{
  printf("   +0  %02X        STATUS:                ", ptable[0]);
  switch (ptable[0]) {
    case 0 : puts("Inactive"); break;
    case 0x80 : puts("Active"); break;
    default : puts("Unknown"); break;
  }
  printf("   +1  %02X %02X %02X", ptable[1], ptable[2], ptable[3]);
  int cyl, sec;
  sec = SEC(ptable[2]);
  cyl = CYL(ptable[2], ptable[3]);
  printf("  START CYL-HEAD-SECTOR: %03Xh-%02Xh-%02Xh (%u-%u-%u)\n",
         cyl, ptable[1], sec, cyl, ptable[1], sec);
  printf("   +4  %02X        PARTITION ID:          %s\n", ptable[4], ppartid(ptable[4]));
  printf("   +5  %02X %02X %02X", ptable[5], ptable[6], ptable[7]);
  sec = SEC(ptable[6]);
  cyl = CYL(ptable[6], ptable[7]);
  printf("  END CYL-HEAD-SECTOR:   %03Xh-%02Xh-%02Xh (%u-%u-%u)\n",
         cyl, ptable[5], sec, cyl, ptable[5], sec);
  printf("   +8  %02X%02X%02X%02X", ptable[11], ptable[10], ptable[9], ptable[8]);
  unsigned long abssect = 0;
  for(int i = 11; i > 7; i--) abssect = (abssect << 8) + ptable[i];
  printf("  START SECTOR:          %lu\n",abssect);
  printf("   +C  %02X%02X%02X%02X", ptable[15], ptable[14], ptable[13], ptable[12]);
  abssect = 0;
  for(i = 15; i > 11; i--) abssect = (abssect << 8) + ptable[i];
  printf("  NUMBER OF SECTORS:     %lu\n", abssect);
}
             
// set Disk Address Packet
void setdap(int sects, char *dap,unsigned char *buff, unsigned long bptr)
{
   dap[0] = 0x10;   // DAP size
   dap[1] = 0;      // Reserved
   dap[2] = sects;  // Number of blocks to transfer
   dap[3] = 0;      // Reserved
   dap[4] = FP_OFF(buff) & 0xff;   // buffer
   dap[5] = FP_OFF(buff) >> 8;
   dap[6] = FP_SEG(buff) & 0xff;
   dap[7] = FP_SEG(buff) >> 8;
   for (int i = 0; i < 4 ; i++) dap[8+i] = (bptr >> (8 * i)) & 0xff; // LBA
   dap[12] = dap[13] = dap[14] = dap[15] = 0;
}

void getparm(struct diskparm *drv)
{
   union REGS inr, outr;
   struct SREGS seg;

   inr.h.ah = 0x41;          // Check INT13 Extension Present
   inr.h.dl = abs(drv->drv);           // Drive
   inr.x.bx = 0x55aa;       // Sign
   int86x(0x13, &inr, &outr, &seg);
   if (0xaa55 != outr.x.bx) drv->lba = 0; // Non-LBA
   else drv->lba = 1;                     // LBA mode
   if (drv->drv < 0) drv->lba = 0;
   if (0 == drv->lba) int1308(drv);
   else int1348(drv);
}   

void int1308(struct diskparm *drv)
{
   union REGS inr, outr;
   struct SREGS seg;

   resetdisk(abs(drv->drv));
   inr.h.ah = 0x08;          // Get drive parm
   inr.h.dl = abs(drv->drv);      // Drive
   int86x(0x13, &inr, &outr, &seg);
   if (outr.x.cflag != 0)
   {
      printf("MBRS001 Error(int1308) Get Geometry failed. RC=%X %s\n",outr.h.ah,int13err(outr.h.ah));
      drv->cc = 0;
   }
   else
   {
      drv->hh = outr.h.dh + 1;
      drv->ss = SEC(outr.h.cl);
      drv->cc = CYL(outr.h.cl, outr.h.ch) + 1;
      drv->blks = (unsigned long) drv->cc * drv->hh * drv->ss;
   } /* endif */
}

void int1348(struct diskparm *drv)
{
   union REGS inr, outr;
   struct SREGS seg;
   unsigned char buff[256];
   int retry = 7;

   do {
      resetdisk(drv->drv);
      inr.h.ah = 0x48;          // Get drv parm
      inr.h.dl = drv->drv;      // Drive
      seg.ds = FP_SEG(buff);     // buffer
      inr.x.si = FP_OFF(buff);
      int86x(0x13, &inr, &outr, &seg);
   } while (outr.x.cflag != 0 && --retry > 0); /* enddo */
   if (outr.x.cflag != 0) {
     drv->cc = 0;
     printf("MBRS002 Error(int1348) Get Geometry Failed. RC=%X %s\n",outr.h.ah,int13err(outr.h.ah));
   } else {
      drv->cc = getnum(buff,4,4);
      drv->hh = getnum(buff,8,4);
      drv->ss = getnum(buff,12,4);
      drv->blks = getnum(buff,16,4);
   }
   
}

void resetdisk(int drv)
{
   union REGS inr, outr;
   struct SREGS seg;

   inr.h.ah = 0;          // Get drv parm
   inr.h.dl = drv;      // Drive
   int86x(0x13, &inr, &outr, &seg);
   if (outr.x.cflag != 0) printf("MBRS003 Disk reset error! DRV=%X RC=%X %s\n",drv,outr.h.ah,int13err(outr.h.ah));
}
   
unsigned long getnum(unsigned char *buff, int ofs, int len)
{
   unsigned long result;
   int i = 0;
   
   for (i=ofs+len-1; i>=ofs ; i--) 
   {
      result= (result << 8) + buff[i];
   } /* endfor */
   return result;
}

char* ppartid(int partid)
{
  char* p;

  switch (partid) {
    case 0 : p = "Empty"; break;
    case 1 : p = "DOS 12-bit FAT"; break;
    case 2 : p = "XENIX root"; break;
    case 3 : p = "XENIX usr"; break;
    case 4 : p = "DOS 16-bit <32M"; break;
    case 5 : p = "Extended"; break;
    case 6 : p = "DOS 16-bit >=32M"; break;
    case 7 : p = "OS/2 HPFS"; break;
    case 8 : p = "AIX"; break;
    case 9 : p = "AIX bootable"; break;
    case 0xa : p = "OS/2 Boot Manag"; break;
    case 0xb : p = "Win95 FAT32"; break;
    case 0xc : p = "Win95 FAT32 (LBA)"; break;
    case 0xd : p = "Win95 FAT16"; break;
    case 0xe : p = "Win95 FAT16 (LBA)"; break;
    case 0xf : p = "Win95 Extended"; break;
    case 0x17 : p = "NTFS"; break;
    case 0x40 : p = "Venix 80286"; break;
    case 0x51 : p = "Novell?"; break;
    case 0x52 : p = "Microport"; break;
    case 0x63 : p = "GNU HURD"; break;
    case 0x64 :
    case 0x65 : p = "Novell Netware"; break;
    case 0x75 : p = "PC/IX"; break;
    case 0x80 : p = "Old MINIX"; break;
    case 0x81 : p = "Linux/MINIX"; break;
    case 0x82 : p = "Linux swap"; break;
    case 0x83 : p = "Linux native"; break;
    case 0x85 : p = "Linux Extended"; break;
    case 0x93 : p = "Amoeba"; break;
    case 0x94 : p = "Amoeba BBT"; break;
    case 0xa5 : p = "BSD/386"; break;
    case 0xa6 : p = "OpenBSD"; break;
    case 0xa7 : p = "NEXTSTEP"; break;
    case 0xb7 : p = "BSDI fs"; break;
    case 0xb8 : p = "BSDI swap"; break;
    case 0xc7 : p = "Syrinx"; break;
    case 0xdb : p = "CP/M"; break;
    case 0xe1 : p = "DOS access"; break;
    case 0xe3 : p = "DOS R/O"; break;
    case 0xf2 : p = "DOS secondary"; break;
    case 0xff : p = "BBT"; break;
    default : p = "UNKOWN";
  }
  return p;
}

char* int13err(int rc)
{
  char* p;

  switch (rc)
  {
    case 0x00 : p = "successful completion"; break;
    case 0x01 : p = "invalid function in AH or invalid parameter"; break;
    case 0x02 : p = "address mark not found"; break;
    case 0x03 : p = "disk write-protected"; break;
    case 0x04 : p = "sector not found/read error"; break;
    case 0x05 : p = "reset failed (hard disk)"; break;
    case 0x06 : p = "disk changed (floppy)"; break;
    case 0x07 : p = "drive parameter activity failed (hard disk)"; break;
    case 0x08 : p = "DMA overrun"; break;
    case 0x09 : p = "data boundary error (attempted DMA across 64K boundary or >80h sectors)"; break;
    case 0x0A : p = "bad sector detected (hard disk)"; break;
    case 0x0B : p = "bad track detected (hard disk)"; break;
    case 0x0C : p = "unsupported track or invalid media"; break;
    case 0x0D : p = "invalid number of sectors on format (PS/2 hard disk)"; break;
    case 0x0E : p = "control data address mark detected (hard disk)"; break;
    case 0x0F : p = "DMA arbitration level out of range (hard disk)"; break;
    case 0x10 : p = "uncorrectable CRC or ECC error on read"; break;
    case 0x11 : p = "data ECC corrected (hard disk)"; break;
    case 0x20 : p = "controller failure"; break;
    case 0x31 : p = "no media in drive (IBM/MS INT 13 extensions)"; break;
    case 0x32 : p = "incorrect drive type stored in CMOS (Compaq)"; break;
    case 0x40 : p = "seek failed"; break;
    case 0x80 : p = "timeout (not ready)"; break;
    case 0xAA : p = "drive not ready (hard disk)"; break;
    case 0xB0 : p = "volume not locked in drive (INT 13 extensions)"; break;
    case 0xB1 : p = "volume locked in drive (INT 13 extensions)"; break;
    case 0xB2 : p = "volume not removable (INT 13 extensions)"; break;
    case 0xB3 : p = "volume in use (INT 13 extensions)"; break;
    case 0xB4 : p = "lock count exceeded (INT 13 extensions)"; break;
    case 0xB5 : p = "valid eject request failed (INT 13 extensions)"; break;
    case 0xBB : p = "undefined error (hard disk)"; break;
    case 0xCC : p = "write fault (hard disk)"; break;
    case 0xE0 : p = "status register error (hard disk)"; break;
    default : p = "UNKOWN";
  }
  return p;
}

memdump::memdump()
{
  code = 'A';
  offset = 0;
}

void memdump::clearbuff(void)
{
  int i;

  for (i = 0; i < 81; i++) linebuff[i] = ' ';
}

void memdump::disp(unsigned char* mem, unsigned long length)
{
  unsigned long lp;
  unsigned char buff[5];
  unsigned char sbuff[] ="  [                  ]";

  clearbuff();

  for (lp = 0; lp < length; lp++) {
    if (linebuff[0] == ' ') sprintf(linebuff, "%04X:  ", lp + offset);
    if (7 == lp % 16) sprintf(buff, "%02X-", mem[lp]);
    else sprintf(buff, "%02X ", mem[lp]);
    strcat(linebuff, buff);
    sbuff[4 + (lp % 16)] = (mem[lp] >= 0x20) ? mem[lp] : ' ';

    if (15 == (lp % 16)) {
      strcat(linebuff, sbuff);
      puts(linebuff);
      clearbuff();
    }
  }
}

