// ------------------------------- //
// -------- Start of File -------- //
// ------------------------------- //
// ----------------------------------------------------------- // 
// C++ Source Code File Name: vbsocket.cpp
// C++ Compiler Used: MSVC, BCC32, GCC, HPUX aCC, SOLARIS CC
// Produced By: Doug Gaer   
// File Creation Date: 09/20/1999
// Date Last Modified: 08/10/2000
// Copyright (c) 2000 Douglas M. Gaer
// ----------------------------------------------------------- // 
// ------------- Program Description and Details ------------- // 
// ----------------------------------------------------------- // 
/*
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.
 
This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.
 
You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  
USA

The vbSocket class is used to create TCP/IP sockets. The vbSocket
class supports stream sockets and datagram sockets It includes
several low-level functions needed by derived classes to establish
communication end-points and transfer data.
*/
// ----------------------------------------------------------- // 
#include<string.h>
#include <fcntl.h>
#include "vbsocket.h"

#if defined(__UNIX__) && !defined(__BSD_UNIX__)  
#include <sys/utsname.h> // Used for host name information
#endif // __UNIX__

// NOTE: This array must contain the same number of exceptions as the
// vbsSocketError enumeration. 
const int vbsMaxSocketExceptionMessages = 22;
const char *vbsSocketExceptionMessages[vbsMaxSocketExceptionMessages] = {
  "Socket exception: No exception reported",         // NO_ERROR
  "Socket exception: Invalid exception code",        // INVALID_ERROR_CODE 
  "Socket exception: Error accepting remote socket", // ACCEPT_ERROR
  "Socket exception: Could not bind socket",         // BIND_ERROR
  "Socket exception: Could not connect socket",      // CONNECT_ERROR
  "Socket exception: Could not resolve hostname",    // HOSTNAME_ERROR
  "Socket exception: Initialization error",          // INIT_ERROR
  "Socket exception: Listen error",                  // LISTEN_ERROR
  "Socket exception: Unknown protocol requested",    // PROTOCOL_ERROR
  "Socket exception: Receive error",                 // RECEIVE_ERROR
  "Socket exception: Unknown service requested",     // SERVICE_ERROR
  "Socket exception: Unknown socket type requested", // SOCKETTYPE
  "Socket exception: Transmit error",                // TRANSMIT_ERROR

  // WIN32 exception messages (Transparent to UNIX applications)
  "WinSock exception: WSACleanup operation failed",        // WINSOCKCLEANUP
  "WinSock exception: Version specified is not supported by this DLL",
                                                           // WINSOCKEINVAL
  "WinSock exception: Could not initialize Winsock",       // WINSOCKINIT
  "WinSock exception: Network subsystem is not ready",     // WINSOCKNETWORK
  "WinSock exception: Requested version in not supported", // WINSOCKVERSION
  
  // Variable block exception messages
  "Socket exception: Variable block acknowledgment error", // BLOCKACK
  "Socket exception: Bad variable block header",           // BLOCKHEADER
  "Socket exception: Bad variable block size",             // BLOCKSIZE
  "Socket exception: Variable block synchronization error" // BLOCKSYNC
};

vbSocket::vbSocket()
// Socket constructor that performs no initialization other then
// setting default values for the socket data members.
{
  address_family = AF_INET;      // Default address family
  socket_type = SOCK_STREAM;     // Default socket type
  protocol_family = IPPROTO_TCP; // Default protocol family
  port_number = vbsDEFAULT_PORT; // Default port number
  vbsocket = -1;
  remote_socket = -1;
  bytes_read = bytes_moved = 0;
  is_connected = 0;
  is_bound = 0;
  socket_error = vbSOCKET_NO_ERROR;
  socket_version = vbSOCKET_WSAVER_ONEONE;
}

vbSocket::vbSocket(vbsSocket_t st, vbsPort_t port, char *hostname) 
// Socket constructor used to initialize the socket according to the
// socket type. A hostname name should only be specified for client
// sockets.
{
  vbsocket = -1;
  remote_socket = -1;
  bytes_read = bytes_moved = 0;
  is_connected = 0;
  is_bound = 0;
  socket_error = vbSOCKET_NO_ERROR;
  socket_version = vbSOCKET_WSAVER_ONEONE;

  // Initialize the socket. NOTE: Any errors detected during initialization
  // will be recorded in the socket_error member.
  if(InitSocketLibrary() == 0) InitSocket(st, port, hostname);
}

vbSocket::~vbSocket()
{
  Close();
}

vbsSocket_t vbSocket::Socket()
// Create a socket. Returns a valid socket descriptor or
// -1 if the socket cannot be initialized.
{
  vbsocket = socket(address_family, socket_type, protocol_family);
#if defined (__WIN32__)
  // The SOCKET type is unsigned in the WinSock library
  if(vbsocket == INVALID_SOCKET) // Defined as (SOCKET)(~0)
#elif defined (__UNIX__)
  if(vbsocket < 0)
#else
#error You must define a target platform:\n  __WIN32__ or __UNIX__
#endif
    {
      socket_error = vbSOCKET_INIT_ERROR;
      return -1;
    }

  return vbsocket;
}

vbsSocket_t vbSocket::InitSocket(vbsSocket_t st, vbsPort_t port,
				 char *hostname)
// Create a and initialize a socket according to the socket type. NOTE: all
// ports below 1024 are reserved. The "hostname" variable is an optional
// parameter that allows clients to specify a server name. Returns a valid
// socket descriptor or -1 if the socket cannot be initialized.
{
  address_family = AF_INET;
  port_number = port;

  if(st == SOCK_STREAM) {
    socket_type = SOCK_STREAM;
    protocol_family = IPPROTO_TCP;
  }
  else if(st == SOCK_DGRAM) {
    socket_type = SOCK_DGRAM;
    protocol_family = IPPROTO_UDP;
  }
  else {
    socket_error = vbSOCKET_SOCKETTYPE_ERROR;
    return -1;
  }

  // Put the server information into the server structure.
  sin.sin_family = address_family;

  if(hostname) {
    // Get the server's Internet address
    hostnm = gethostbyname(hostname);
    if(hostnm == (struct hostent *) 0) {
      socket_error = vbSOCKET_HOSTNAME_ERROR;
      return -1;
    }
    // Put the server information into the client structure.
    sin.sin_addr.s_addr = *((unsigned long *)hostnm->h_addr);
  }
  else   
    sin.sin_addr.s_addr = INADDR_ANY; // Use my IP address

  // The port must be put into network byte order.
  // htons()--"Host to Network Short" 
  // htonl()--"Host to Network Long" 
  // ntohs()--"Network to Host Short" 
  // ntohl()--"Network to Host Long" 
  sin.sin_port = htons(port_number);

  // Create a TCP/IP stream socket
  if(Socket() < 0) {
    socket_error = vbSOCKET_INIT_ERROR;
    return -1;
  }

  return vbsocket;
}

int vbSocket::Bind()
// Bind the socket to a name so that other processes can
// reference it and allow this socket to receive messages.
// Returns -1 if an error occurs.
{
  int rv = bind(vbsocket, (struct sockaddr *)&sin, sizeof(sin));
  if(rv >= 0) {
    is_bound = 1;
  } 
  else {
    socket_error = vbSOCKET_BIND_ERROR;
    is_bound = 0;
  }
  return rv;
}

int vbSocket::Connect()
// Connect the socket to a client or server. On the client side
// a connect call is used to initiate a connection.
// Returns -1 if an error occurs.
{
  int rv = connect(vbsocket, (struct sockaddr *)&sin, sizeof(sin));
  if(rv >= 0) {
    is_connected = 1; 
  }
  else {
    socket_error = vbSOCKET_CONNECT_ERROR;
    is_connected = 0;
  }
  return rv;
}

int vbSocket::Recv(void *buf, int bytes, int flags)
// Receive a block of data from the bound socket and do not return
// until all the bytes have been read. Returns the total number of 
// bytes received or -1 if an error occurs.
{
  return Recv(vbsocket, buf, bytes, flags);
}

int vbSocket::Send(const void *buf, int bytes, int flags)
// Send a block of data to the bound socket and do not return
// until all the bytes have been written. Returns the total number 
// of bytes sent or -1 if an error occurs.
{
  return Send(vbsocket, buf, bytes, flags);
}

int vbSocket::Recv(vbsSocket_t s, void *buf, int bytes, int flags)
// Receive a block of data from a specified socket and do not return
// until all the bytes have been read. Returns the total number of
// bytes received or -1 if an error occurs.
{
  bytes_read = 0;           // Reset the byte counter
  int num_read = 0;         // Actual number of bytes read
  int num_req = (int)bytes; // Number of bytes requested 
  char *p = (char *)buf;    // Pointer to the buffer

  while(bytes_read < bytes) { // Loop until the buffer is full
    if((num_read = recv(s, p, num_req-bytes_read, flags)) > 0) {
      bytes_read += num_read;   // Increment the byte counter
      p += num_read;            // Move the buffer pointer for the next read
    }
    if(num_read < 0) {
      socket_error = vbSOCKET_RECEIVE_ERROR;
      return -1; // An error occurred during the read
    }
  }
  
  return bytes_read;
}

int vbSocket::Send(vbsSocket_t s, const void *buf, int bytes, int flags)
// Send a block of data to a specified socket and do not return
// until all the bytes have been written. Returns the total number of
// bytes sent or -1 if an error occurs.
{
  bytes_moved = 0;           // Reset the byte counter
  int num_moved = 0;         // Actual number of bytes written
  int num_req = (int)bytes;  // Number of bytes requested 
  char *p = (char *)buf;     // Pointer to the buffer

  while(bytes_moved < bytes) { // Loop until the buffer is empty
    if((num_moved = send(s, p, num_req-bytes_moved, flags)) > 0) {
      bytes_moved += num_moved;  // Increment the byte counter
      p += num_moved;            // Move the buffer pointer for the next read
    }
    if(num_moved < 0) {
      socket_error = vbSOCKET_TRANSMIT_ERROR;
      return -1; // An error occurred during the read
    }
  }

  return bytes_moved;
}

int vbSocket::RemoteRecv(void *buf, int bytes, int flags)
// Receive a block of data from a remote socket in blocking mode.
// Returns the total number of bytes received or -1 if an error occurs.
{
  return Recv(remote_socket, buf, bytes, flags);
}

int vbSocket::RemoteSend(const void *buf, int bytes, int flags)
// Send a block of data to a remote socket and do not return
// until all the bytes have been written. Returns the total 
// number of bytes received or -1 if an error occurs.
{
  return Send(remote_socket, buf, bytes, flags);
}

void vbSocket::ShutDown(int how)
// Used to close and un-initialize a full-duplex socket.
{
  bytes_moved = 0;
  bytes_read = 0;
  is_connected = 0;
  is_bound = 0;

  // NOTE 05/25/2000: The last error is no longer reset in following
  // a close call to allow the that appllication can close the socket
  // and catch the last exception reported if the socket has to be
  // closed after a Bind(), Listen(), or any other initialization
  // failure.
  // socket_error = vbSOCKET_NO_ERROR;
 
  // 0 = User is no longer interested in reading data
  // 1 = No more data will be sent
  // 2 = No data is to be sent or received
  if(vbsocket != -1) shutdown(vbsocket, how);
  if(remote_socket != -1) shutdown(remote_socket, how);

  vbsocket = -1;
  remote_socket = -1;
}

void vbSocket::ShutDown(vbsSocket_t s, int how)
// Used to close and un-initialize the specified full-duplex socket.
{
  if(s != -1) shutdown(s, how);
  s = -1;
}

void vbSocket::ShutDownSocket(int how)
// Used to close a full-duplex server side socket.
{
  if(vbsocket != -1) shutdown(vbsocket, how);
  vbsocket = -1;
}

void vbSocket::ShutDownRemoteSocket(int how)
// Used to close a full-duplex client side socket.
{
  if(remote_socket != -1) shutdown(remote_socket, how);
  remote_socket = -1;
}

void vbSocket::Close()
// Close any and un-initialize any bound sockets.
{
  bytes_moved = 0;
  bytes_read = 0;
  is_connected = 0;
  is_bound = 0;

  // NOTE 05/25/2000: The last error is no longer reset in following
  // a close call to allow the that appllication can close the socket
  // and catch the last exception reported if the socket has to be
  // closed after a Bind(), Listen(), or any other initialization
  // failure.
  // socket_error = vbSOCKET_NO_ERROR;
#if defined (__WIN32__)
  if(vbsocket != -1) closesocket(vbsocket);
  if(remote_socket != -1) closesocket(remote_socket);
#elif defined (__UNIX__)
  if(vbsocket != -1) close(vbsocket);
  if(remote_socket != -1) close(remote_socket);
#else
#error You must define a target platform: __WIN32__ or __UNIX__
#endif
}

void vbSocket::Close(vbsSocket_t s)
// Close the specified socket
{
#if defined (__WIN32__)
  if(s != -1) closesocket(s);
#elif defined (__UNIX__)
  if(s != -1) close(s);
#else
#error You must define a target platform: __WIN32__ or __UNIX__
#endif
  s = -1;
}

void vbSocket::CloseSocket()
// Close the server side socket
{
#if defined (__WIN32__)
  if(vbsocket != -1) closesocket(vbsocket);
#elif defined (__UNIX__)
  if(vbsocket != -1) close(vbsocket);
#else
#error You must define a target platform: __WIN32__ or __UNIX__
#endif
  vbsocket = -1;
}

void vbSocket::CloseRemoteSocket()
// Close the client socket
{
#if defined (__WIN32__)
  if(remote_socket != -1) closesocket(remote_socket);
#elif defined (__UNIX__)
  if(remote_socket != -1) close(remote_socket);
#else
#error You must define a target platform: __WIN32__ or __UNIX__
#endif
  remote_socket = -1;
}

int vbSocket::Listen(int max_connections)
// Listen for connections if configured as a server.
// The "max_connections" variable determines how many
// pending connections the queue will hold. Returns -1
// if an error occurs.
{
  int rv = listen(vbsocket,         // Bound socket
		  max_connections); // Number of connection request queue
  if(rv < 0) socket_error = vbSOCKET_LISTEN_ERROR;
  return rv;
}

vbsSocket_t vbSocket::Accept()
// Accept a connect from a remote socket. An Accept() 
// call blocks the server until the a client requests 
// service. Returns a valid socket descriptor or -1 
// if an error occurs.
{
  int addr_size = sizeof(remote_sin); // Length of client address
  remote_socket = accept(vbsocket, (struct sockaddr *)&remote_sin, &addr_size);

#if defined (__WIN32__)
  // The SOCKET type is unsigned in the WinSock library
  if(remote_socket == INVALID_SOCKET) // Defined as (SOCKET)(~0)
#elif defined (__UNIX__)
  if(remote_socket < 0)
#else
#error You must define a target platform: __WIN32__ or __UNIX__
#endif
    {
      socket_error = vbSOCKET_ACCEPT_ERROR;
      return -1;
    }

  return remote_socket;
}

int vbSocket::RecvFrom(void *buf, int bytes, int flags)
// Receive a block of data from a remote datagram socket 
// and do not return until all the bytes have been read. 
// Returns the total number of bytes received or -1 if 
// an error occurs.
{
  return RecvFrom(vbsocket, &remote_sin, buf, bytes, flags);
}

int vbSocket::SendTo(void *buf, int bytes, int flags)
// Send a block of data to a datagram socket and do not return
// until all the bytes have been written. Returns the total number
// of bytes sent or -1 if an error occurs.
{
  return SendTo(vbsocket, &sin, buf, bytes, flags);
}

int vbSocket::GetSockName()
// Retrieves the current name for the socket descriptor.
// It is used on a bound and/or connected socket and returns the
// local association. This function is especially useful when a
// connect call has been made without doing a bind first in which
// case this function provides the only means by which you can
// determine the local association which has been set by the system.
// Returns -1 if an error occurs.
{
  int namelen = sizeof(vbsSocketAddress);
  int rv = getsockname(vbsocket, (struct sockaddr *)&sin, &namelen);
  if(rv < 0) socket_error = vbSOCKET_HOSTNAME_ERROR;
  return rv;
}

int vbSocket::GetServByName(char *name, char *protocol)
// Get service information corresponding to a service name and protocol.
// Returns -1 if an unknown service or protocol is requested.
{
  // If the "protocol" pointer is NULL, getservbyname returns
  // the first service entry for which the name matches the s_name
  // or one of the s_aliases. Otherwise getservbyname matches both
  // the name and the proto.
  sp = getservbyname(name, protocol);
  if(sp == 0) {
    socket_error = vbSOCKET_PROTOCOL_ERROR;
    return -1;
  }
  sin.sin_port = sp->s_port;
  return 0;
}

int vbSocket::GetServByPort(int port, char *protocol)
// Get service information corresponding to a port number and protocol.
// Returns -1 if an unknown service or protocol is requested.
{
  // If the "protocol" pointer is NULL, getservbyport returns the
  // first service entry for which the port matches the s_port.
  // Otherwise getservbyport matches both the port and the proto.
  sp = getservbyport(port, protocol);
  if(sp == 0) {
    socket_error = vbSOCKET_PROTOCOL_ERROR;
    return -1;
  }
  sin.sin_port = sp->s_port;
  return 0;
}

int vbSocket::GetPortNumber()
// Return the port number actually set by the system. Use this function
// after a call to vbSocket::GetSockName();
{
  return ntohs(sin.sin_port);
}

int vbSocket::GetRemotePortNumber()
// Return the port number of the client socket.
{
  return ntohs(remote_sin.sin_port);
}

vbsAddressFamily vbSocket::GetAddressFamily()
// Returns the address family of this socket
{
  return sin.sin_family;
}

vbsAddressFamily vbSocket::GetRemoteAddressFamily()
// Returns the address family of the remote socket.
{
  return remote_sin.sin_family;  
}

int vbSocket::GetHostName(char *hs)
// Pass back the host name of this machine in the "hs" variable.
// A memory buffer equal to "vbsMAX_NAME_LEN" must be pre-allocated
// prior to using this function. Return -1 is an error occurs.
{
  // Prevent crashes if memory has not been allocated
  if(!hs) hs = new char[vbsMAX_NAME_LEN]; 
  int rv = gethostname(hs, vbsMAX_NAME_LEN);
  if(rv < 0) socket_error = vbSOCKET_HOSTNAME_ERROR;
  return rv;
}

int vbSocket::GetRemoteHostName(char *hs)
// Pass back the host name the client in the "hs" variable.
// A memory buffer equal to "vbsMAX_NAME_LEN" must be pre-allocated
// prior to using this function. Return -1 is an error occurs.
{
  char *s = inet_ntoa(remote_sin.sin_addr);
  if(s == 0) {
    socket_error = vbSOCKET_HOSTNAME_ERROR;
    return -1;
  }

  // Prevent crashes if memory has not been allocated
  if(!hs) hs = new char[vbsMAX_NAME_LEN]; 

  strcpy(hs, s);
  return 0;
}

int vbSocket::RawRead(void *buf, int bytes, int flags)
// Receive a block of data from the bound socket. NOTE: A
// raw read may return before all the bytes have been read.
// Returns -1 if an error occurs.
{
  return RawRead(vbsocket, buf, bytes, flags);
}

int vbSocket::RawWrite(const void *buf, int bytes, int flags)
// Send a block of data to the bound socket. NOTE: A raw write
// may return before all the bytes have been written. Returns -1
// if an error occurs.
{
  return RawWrite(vbsocket, buf, bytes, flags);
}

int vbSocket::RawRead(vbsSocket_t s, void *buf, int bytes, int flags)
// Receive a block of data from a specified socket. NOTE: A
// raw read may return before all the bytes have been read.
// Returns -1 if an error occurs.
{
  bytes_read = 0;
  bytes_read = recv(s, (char *)buf, bytes, flags);
  if(bytes_read < 0) socket_error = vbSOCKET_RECEIVE_ERROR;
  return bytes_read;
}

int vbSocket::RawWrite(vbsSocket_t s, const void *buf, int bytes,
		      int flags)
// Send a block of data to a specified socket. NOTE: A raw write
// may return before all the bytes have been written. Returns -1 if an
// error occurs.
{
  bytes_moved = 0;
  bytes_moved = send(s, (char *)buf, bytes, flags);
  if(bytes_moved < 0) socket_error = vbSOCKET_TRANSMIT_ERROR;
  return bytes_moved;
}

int vbSocket::RawRemoteRead(void *buf, int bytes, int flags)
// Receive a block of data from a remote socket.  NOTE: A
// raw read may return before all the bytes have been read.
// Returns -1 if an error occurs.
{
  return RawRead(remote_socket, buf, bytes, flags);
}

int vbSocket::RawRemoteWrite(const void *buf, int bytes, int flags)
// Send a block of data to a remote socket. NOTE: A raw write
// may return before all the bytes have been written. Returns
// -1 if an error occurs.
{
  return RawWrite(remote_socket, buf, bytes, flags);
}

int vbSocket::RawReadFrom(void *buf, int bytes, int flags)
// Receive a block of data from a remote datagram socket. NOTE: 
// A raw read may return before all the bytes have been read.
// Returns -1 if an error occurs.
{
  return RawReadFrom(vbsocket, &remote_sin, buf, bytes, flags);
}

int vbSocket::RawWriteTo(void *buf, int bytes, int flags)
// Send a block of data to a datagram socket. NOTE: A raw write
// may return before all the bytes have been written. Returns -1
// if an error occurs.
{
  return RawWriteTo(vbsocket, &sin, buf, bytes, flags);
}

const char *vbSocket::SocketExceptionMessage()
// Returns a null terminated string that can
// be use to log or print a socket exception.
{
  if((int)socket_error > vbsMaxSocketExceptionMessages)
    socket_error = vbSOCKET_INVALID_ERROR_CODE;
  
  // Find the corresponding message in the exception array
    int error = (int)socket_error;
  return vbsSocketExceptionMessages[error];
}

void vbSocket::GetClientInfo(char *client_name, int &r_port)
// Get the client's host name and port number. NOTE: This
// function assumes that a block of memory equal to the
// vbsMAX_NAME_LEN constant has already been allocated.
{
  int rv = GetRemoteHostName(client_name);
  if(rv < 0) {
    char *unc = "UNKNOWN";
    for(unsigned i = 0; i < vbsMAX_NAME_LEN; i++) client_name[i] = '\0';
    strcpy(client_name, unc);
  }
  r_port = GetRemotePortNumber();
}

int vbSocket::RecvFrom(vbsSocket_t s, vbsSocketAddress *sa, void *buf,
		       int bytes, int flags)
// Receive a block of data from a remote datagram socket 
// and do not return until all the bytes have been read. 
// Returns the total number of bytes received or -1 if 
// an error occurs.
{
  int addr_size = sizeof(vbsSocketAddress); // Length of client address
  bytes_read = 0;           // Reset the byte counter
  int num_read = 0;         // Actual number of bytes read
  int num_req = (int)bytes; // Number of bytes requested 
  char *p = (char *)buf;    // Pointer to the buffer

  while(bytes_read < bytes) { // Loop until the buffer is full
    if((num_read = recvfrom(s, p, num_req-bytes_read, flags, \
			    (struct sockaddr *)sa, &addr_size)) > 0) {
      bytes_read += num_read;   // Increment the byte counter
      p += num_read;            // Move the buffer pointer for the next read
    }
    if(num_read < 0) {
      socket_error = vbSOCKET_RECEIVE_ERROR;
      return -1; // An error occurred during the read
    }
  }
  return bytes_read;
}

int vbSocket::SendTo(vbsSocket_t s, vbsSocketAddress *sa, void *buf,
		     int bytes, int flags)
// Send a block of data to a datagram socket and do not return
// until all the bytes have been written. Returns the total number
// of bytes sent or -1 if an error occurs.
{
  int addr_size = sizeof(vbsSocketAddress); // Length of address
  bytes_moved = 0;             // Reset the byte counter
  int num_moved = 0;           // Actual number of bytes written
  int num_req = (int)bytes;    // Number of bytes requested 
  char *p = (char *)buf;       // Pointer to the buffer

  while(bytes_moved < bytes) { // Loop until the buffer is full
    if((num_moved = sendto(s, p, num_req-bytes_moved, flags, \
			   (const struct sockaddr *)sa, addr_size)) > 0) {
      bytes_moved += num_moved;  // Increment the byte counter
      p += num_moved;            // Move the buffer pointer for the next read
    }
    if(num_moved < 0) {
      socket_error = vbSOCKET_TRANSMIT_ERROR;
      return -1; // An error occurred during the read
    }
  }
  return bytes_moved;
}

int vbSocket::RawReadFrom(vbsSocket_t s, vbsSocketAddress *sa, void *buf,
		  int bytes, int flags)
// Receive a block of data from a remote datagram socket. NOTE: 
// A raw read may return before all the bytes have been read.
// Returns -1 if an error occurs.
{
  bytes_read = 0;
  int addr_size = sizeof(vbsSocketAddress); // Length of client address
  bytes_read = recvfrom(s, (char *)buf, bytes, flags,
			(struct sockaddr *)sa, &addr_size);
  if(bytes_read < 0) socket_error = vbSOCKET_RECEIVE_ERROR;
  return bytes_read;
}

int vbSocket::RawWriteTo(vbsSocket_t s, vbsSocketAddress *sa, void *buf,
			 int bytes, int flags)
// Send a block of data to a datagram socket. NOTE: A raw write
// may return before all the bytes have been written. Returns -1
// if an error occurs.
{
  bytes_moved = 0;
  int addr_size = sizeof(vbsSocketAddress); // Length of address
  bytes_moved = sendto(s, (char *)buf, bytes, flags,
		       (const struct sockaddr *)sa, addr_size);
  if(bytes_moved < 0) socket_error = vbSOCKET_TRANSMIT_ERROR;
  return bytes_moved;
}

int vbSocket::InitSocketLibrary()
// Perform any platform specific initialization required
// before network communication can be established.
// Returns zero if no errors occur, -1 if an error occurs
// and the error can be mapped internally, or a value greater
// then zero if an error occurs and the error cannot be
// determined.
{
#if defined (__WIN32__)
  // Initialize the WinSock DLL with the specified version.
  WORD wVersionRequested;
  switch(socket_version) { 
    // NOTE: An application can successfully use a Windows Sockets
    // DLL if there is any overlap in the version ranges. 
    case vbSOCKET_WSAVER_ONEZERO:
      // Included for test purposes only
      wVersionRequested = MAKEWORD(1, 0);
      break;
    case vbSOCKET_WSAVER_ONEONE:
      // Windows 95A/B/C, 98, 98SE, NT4.0 compatiable
      wVersionRequested = MAKEWORD(1, 1);
      break;
    case vbSOCKET_WSAVER_TWOZERO:
      // Windows 98, 98SE, 2000 compatiable
      wVersionRequested = MAKEWORD(2, 0);
      break;
    case vbSOCKET_WSAVER_TWOTWO:
      // Windows 98SE, 2000 compatiable
      wVersionRequested = MAKEWORD(2, 2);
      break;
    default:
      // Should work on all WIN32 platforms
      wVersionRequested = MAKEWORD(1, 1);
      break;
  }

  // Initialize the WinSock DLL. Every Windows Sockets application
  // MUST make a WSAStartup call before issuing any other Windows
  // Sockets API calls. 
  int rv = WSAStartup(wVersionRequested, &socket_data);
  if(rv != 0) {
    switch(rv) {
      // NOTE: Application calls to WSAGetLastError to determine the WinSock
      // error code cannot be here used since the Windows Sockets DLL may
      // not have established the client data area where the "last error"
      // information is stored. 
      case WSASYSNOTREADY:
	// Network subsystem is not ready for network communication
	socket_error = vbSOCKET_WINSOCKNETWORK_ERROR;
	return -1;
      case WSAVERNOTSUPPORTED:
	// The requested WinSock version is not supported
	socket_error = vbSOCKET_WINSOCKVERSION_ERROR;
	return -1;
      case WSAEINVAL:
	// The WinSock version specified is not supported by this DLL	
	socket_error = vbSOCKET_WINSOCKEINVAL_ERROR;
	return -1;
      default:
	socket_error = vbSOCKET_WINSOCKINIT_ERROR;
	return rv; // Can't map this error so return the WIN32 error code
    }
  }

  // Confirm that the WinSock DLL supports the requested version
  if(socket_data.wVersion != wVersionRequested) return -1;
  switch(socket_version) { 
    case vbSOCKET_WSAVER_ONEZERO:
      if(LOBYTE(socket_data.wVersion ) != 1 ||
	 HIBYTE(socket_data.wVersion ) != 0) {
	socket_error = vbSOCKET_WINSOCKVERSION_ERROR;
	return -1; 
      }
      break;
    case vbSOCKET_WSAVER_ONEONE:
      if(LOBYTE(socket_data.wVersion ) != 1 ||
	 HIBYTE(socket_data.wVersion ) != 1) {
	socket_error = vbSOCKET_WINSOCKVERSION_ERROR;
	return -1; 
      }
      break;
    case vbSOCKET_WSAVER_TWOZERO:
      if(LOBYTE(socket_data.wVersion ) != 2 ||
	 HIBYTE(socket_data.wVersion ) != 0) {
	socket_error = vbSOCKET_WINSOCKVERSION_ERROR;
	return -1; 
      }
      break;
    case vbSOCKET_WSAVER_TWOTWO:
      if(LOBYTE(socket_data.wVersion ) != 2 ||
	 HIBYTE(socket_data.wVersion ) != 2) {
	socket_error = vbSOCKET_WINSOCKVERSION_ERROR;
	return -1; 
      }
      break;
    default:
      socket_error = vbSOCKET_WINSOCKVERSION_ERROR;
      return -1; 
  }
#endif

  return 0;
}

int vbSocket::ReleaseSocketLibrary()
// Perform any platform specific operations required
// to release the socket library and free any resources
// associated with it. Returns -1 if any errors occur.
{
#if defined (__WIN32__)
  // All WIN32 applications or DLLs are required to perform a successful
  // WSAStartup() call before they can use Windows Sockets services. When
  // finished the application or DLL must call WSACleanup() to deregister
  // itself from a Windows Sockets implementation. Any pending blocking or
  // asynchronous calls issued by any thread in this process are canceled
  // without posting any notification messages, or signaling any event
  // objects. NOTE: There must be a call to WSACleanup() for every successful
  // call to WSAStartup() made by a task. Only the final WSACleanup() for that
  // task does the actual cleanup. The preceding calls decrement an internal
  // reference count in the Windows Sockets DLL. In multithreaded environments
  // WSACleanup() terminates Windows Sockets operations for all threads.

  // Will return zero if the operation was successful. Otherwise the value
  // SOCKET_ERROR is returned, and a specific error number may be retrieved
  // by calling int WSAGetLastError(void);
  if(WSACleanup() != 0) {
    socket_error = vbSOCKET_WINSOCKCLEANUP_ERROR;
    return -1;
  }
#endif
  return 0;
}
// ----------------------------------------------------------- // 
// ------------------------------- //
// --------- End of File --------- //
// ------------------------------- //

