// ------------------------------- //
// -------- Start of File -------- //
// ------------------------------- //
// ----------------------------------------------------------- // 
// C++ Source Code File Name: vb_debug.cpp 
// Compiler Used: MSVC, BCC32, GCC, HPUX aCC, SOLARIS CC
// Produced By: Doug Gaer 
// File Creation Date: 02/04/1997  
// Date Last Modified: 08/10/2000
// Copyright (c) 1997, 2000 Douglas M. Gaer
// ----------------------------------------------------------- // 
// ------------- Program Description and Details ------------- // 
// ----------------------------------------------------------- // 
/*
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.
 
This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  
USA

vbDatabase file recovery functions.
*/
// ----------------------------------------------------------- // 
#include <iostream.h>
#include <iomanip.h>
#include <string.h>
#include "vbdbase.h"
#include "vbstat32.h"

// Program name and version number
char *VersionNumber;
const char *ProgramName = "vb_debug";

class vbDatabaseDebugManager : public vbDatabase
{
public:
  vbDatabaseDebugManager() { }
  ~vbDatabaseDebugManager() { }

public:
  vbDatabaseError BlindOpen(const char *fname);
  FAU VBSearch(FAU Offset); 
};

// Type definition used to defined reference counted VBD file pointers
typedef RefCount<vbDatabaseDebugManager> vbDebugPtr;

// Function prototypes for display menu
void ClearInputStream(istream &s);
int Quit();
void Menu(const vbDebugPtr &f);
void PausePrg();
void Version();

// Function prototypes for VBD utilities 
void FindVB(const vbDebugPtr &f, int verbose = 0, int walk = 0);
void AnalyzeHeader(const vbDebugPtr &f);
void DisplayStats(const vbDebugPtr &f);
void Rebuild(const vbDebugPtr &f, int update_ver = 0);

vbDatabaseError vbDatabaseDebugManager::BlindOpen(const char *fname)
// Open the file in read only mode without throwing a wrong file type
// exception.
{
  // Close any open files
  if(Close() != vbDBASE_NO_ERROR) return vbd_error;
  
  fp = vbdFPTROpen(fname, vbDBASE_READONLY);
    
  if(fp == 0) {
    vbd_error = vbDBASE_FILE_OPEN_ERROR;
#ifdef __CPP_EXCEPTIONS__
    throw CFileOpenError();
#else
    return vbd_error;
#endif
  }
  else {
    ready_for_reading = 1;
    ready_for_writing = 0;
    is_open = 1;
    is_ok = 1;
    strcpy(file_name, fname);
    last_operation = vbDBASE_WRITE;
    
    if(ReadFileHdr()!= vbDBASE_NO_ERROR) return vbd_error;

    // VBD version 1 compatibility mode
    int vbd_ver = 2;
    switch(file_header.vbd_ver) {
      case 1024:
	vbd_ver = 1;
      	break;
      case 1025:
	vbd_ver = 1;
      	break;
      case 1027:
	vbd_ver = 1;
	break;
      case 1031:
	vbd_ver = 1;
	break;
      case 1033:
	vbd_ver = 1;
	break;
      case 1034:
	vbd_ver = 1;
	break;
      default:
	break;
    }

    if(vbd_ver == 1) {
      vbDatabase::VBDVersion = file_header.vbd_ver;
      vbDatabase::VBDInternalCheckWord = 0x0000fefe;
      memmove(vbDatabase::VBDSignature, "VBDFILE", 7); 
    }

    // Test file type, checking the revision letter
    if(memcmp(file_header.vbd_sig, vbDatabase::VBDSignature, 7) == 0) {
      // Set the revision letter according to the file header
      char revision[8];
      memmove(revision, file_header.vbd_sig, 8);
      rev_letter = revision[7];
    }
  }

  // Ensure that true end of file is stored in the file header. 
  vbINT32 filesize;
  filesize = FileSize(fname);
  if(file_header.vbd_eof < filesize) {
    file_header.vbd_eof = filesize;
    if(Flush() != vbDBASE_NO_ERROR) return vbd_error;
  }
  
  return vbd_error = vbDBASE_NO_ERROR;
}

FAU vbDatabaseDebugManager::VBSearch(FAU Offset)
// Search through the VBD file until a valid VB is found.
// The search starts at the beginning  of the file or the
// offset value. Returns 0 if no valid VB is found in the
// file.
{
  vbBlockHeader vb;
  FAU file_address = (FAU)0;

  FAU StaticEOF = FileSize(file_name);

  if(!Offset)
    file_address = FileHeaderSize(); // If no Offset, start after VBD header
  else {
    file_address = file_address + Offset; // Offset the starting address 

    if(file_address >= StaticEOF) // Prevent offsetting past EOF
      return 0; // Invalid address
  }
  
  while(1) {
    if((FAU)(file_address + VBHeaderSize()) >= StaticEOF || !IsOK()) 
      return (FAU)0;

    if(Read(&vb, sizeof(vbBlockHeader), file_address) != vbDBASE_NO_ERROR)
      return (FAU)0;
    if(vb.block_check_word != vbDatabase::VBDInternalCheckWord)
      file_address++; // Loop through the file byte by byte
    else
      break; // Found valid block
  }
  return file_address;
}

void DisplayStats(const vbDebugPtr &f)
{
  vbFileHeader fh;

  f->Read(&fh, sizeof(vbFileHeader), 0);
  if(memcmp(fh.vbd_sig, vbDatabase::VBDSignature, 7)) { // Test file type
    cout << endl;
    cout << "VBD File Header is damaged or does not exist" << endl;
    cout << endl;
    return;
  }

  VBDStats((const vbDatabasePtr &)f);
}

void AnalyzeHeader(const vbDebugPtr &f)
{
  vbFileHeader fh;
  vbBlockHeader vb;
  int errors = 0;
  
  f->Read(&fh, sizeof(vbFileHeader), 0);
  cout << endl;
  cout << "Analyzing VBD file header..." << endl;
  cout << endl;

  cout << "Checking VBD signature..." << endl;
  if (memcmp(fh.vbd_sig, vbDatabase::VBDSignature, 7)) { // Test file type
    cout << endl;
    cout << "VBD File Header is damaged or does not exist" <<endl;
    cout << endl;
    FindVB(f, 0);
    return;
  }
  else if (memcmp(fh.vbd_sig, vbDatabase::VBDSignature, 8)) {
    // Check revision letters
    char revision[8];
    revision[8] = 0; // Ensure null termination
    char rev_letter;
    cout << endl;
    cout << "NOTE: The VBD file revision letters do not match." << endl;
    memmove(revision, vbDatabase::VBDSignature, 8);
    rev_letter = revision[7];
    if(rev_letter == 0)
      cout << "Current VBD File Revision = ZERO" << endl;
    else
      cout << "Current VBD File Revision = " << rev_letter << endl;
    memmove(revision, fh.vbd_sig, 8);
    rev_letter = revision[7];
    if(rev_letter == 0)
      cout << "Current VBD File Revision = ZERO" << endl;
    else
      cout << "The file reads: " << rev_letter << endl;
    cout << "Backward compatibility rules will be enforced." << endl;
    cout << endl;
  }

  cout << "Checking for End of File errors..." << endl;
  FAU StaticEOF = f->FileSize(f->vbDatabaseName());
  if(StaticEOF > fh.vbd_eof) {
    cout << endl;
    cout << "End of file error in file: " << f->vbDatabaseName() << endl;
    cout << "The actual length is longer then the allocated length!" << endl;
    cout << endl;
    errors++;
  }

  cout << "Checking the HeapStart value..." << endl;
  f->Read(&vb, sizeof(vbBlockHeader), fh.vbd_hs_fptr);
  if(vb.block_check_word != vbDatabase::VBDInternalCheckWord) {
    cout << endl;
    cout << "Bad HeapStart value in file: " << f->vbDatabaseName() << endl;
    cout << "No variable block found at file address: "
	 << fh.vbd_hs_fptr << endl;
    cout << endl;
    errors++;
  }

  cout << "Checking the FreeSpace value..." << endl;
  if(fh.vbd_fs_fptr != (vbINT32)0) {
    f->Read(&vb, sizeof(vbBlockHeader), fh.vbd_fs_fptr);
    if(vb.block_check_word != vbDatabase::VBDInternalCheckWord) {
      cout << endl;
      cout << "Bad FreeSpace value in file: " << f->vbDatabaseName() << endl;
      cout << "No variable block found at file address: "
	   << fh.vbd_fs_fptr << endl;
      cout << endl;
      errors++;
    }
  }

  cout << "Checking the HighestVB value..." << endl;
  f->Read(&vb, sizeof(vbBlockHeader), fh.vbd_hb_fptr);
  if (vb.block_check_word != vbDatabase::VBDInternalCheckWord) {
    cout << endl;
    cout << "Bad HeapStart value in file: " << f->vbDatabaseName() << endl;
    cout << "No variable block found at file address: "
	 << fh.vbd_hb_fptr << endl;
    cout << endl;
    errors++;
  }
  
  cout << "Checking the VBD version number..." << endl;
  if(fh.vbd_ver != (vbINT32)vbDatabase::VBDVersion) {
    cout << endl;
    cout << "NOTE: The version numbers do not match." << endl;
    cout << "Current VBD File Version = " << vbDatabase::VBDVersion << endl;
    cout << "The file reads: " << fh.vbd_ver << endl;
    cout << "Backward compatibility rules will be enforced." << endl;
    cout << endl;
  }

  if(errors) {
    cout << endl;
    cout << "VBD file header has errors!" << endl;
  }
  else {
    cout << endl;
    cout << "VBD file header checks good." << endl;
  }
  cout << endl;
}

void FindVB(const vbDebugPtr &f, int verbose, int walk)
// Serach the file for all variable block.
{
  vbBlockHeader vb;
  int count = 0;
  int badvb = 0;
  
  cout << endl;
  cout << "Searching file for all variable blocks..." << endl;
  
  FAU StaticEOF = f->FileSize(f->vbDatabaseName());
  FAU addr = f->VBSearch(0); // Search the entire file
  
  if(addr == (FAU)0) {
    cout << endl;
    cout << "No variable blocks found in file: "
	 << f->vbDatabaseName() << endl;
    cout << endl;
    return;
  }
  
  while(1) { 
    if(addr >= StaticEOF) break;
    f->Read(&vb, sizeof(vbBlockHeader), addr);

    if (vb.block_check_word == vbDatabase::VBDInternalCheckWord) {
      count++; // Increment the variable block count
      if(verbose) {
	VBStats((const vbDatabasePtr &)f, (addr+f->VBHeaderSize()));
	if(walk) {
	  char c;
	  cout << endl;
	  cout << "Press enter to continue or enter 'X' to exit >";
	  cin.clear();
	  cin.get(c);
	  switch(c) {
	    case 'x' : case 'X' :
	      cout << endl;
	      return;
	    default:
	      break;
	  }
	}
      }
      addr = addr + vb.block_length; // Goto the next variable block
    }
    else {
      badvb++;
      cout << endl;
      cout << "Found bad variable block at address " << addr << endl;
      cout << "Searching for next good variable block..." << endl;
      addr = f->VBSearch(addr); // Search for the next good block
      if(!addr) {
	cout << "None found!" << endl;
	cout << endl;
	return;
      }
    }
  }
  cout << endl;
  cout << "Found " << count << " good varible blocks." << endl;
  if(badvb) cout << "Found " << badvb << " bad variable blocks." << endl;
  cout << endl;
}

void Rebuild(const vbDebugPtr &f, int update_ver)
{
  cout << endl;
  if(update_ver) 
    cout << "Rebuilding VBD file and updating to latest version..." << endl;
  else
    cout << "Rebuilding damaged VBD file..." << endl;
  cout << endl;

  char buf[255];
  int retry = 3;
  
  while(1) { // Loop until a good file name is found
    cout << "Enter new name of file to build >";
    cin.getline(buf, sizeof(buf));
    cout << endl;
    if(!*buf) { // Return if nothing is entered
      Menu(f);
      return;
    }
    if(vbDatabase::Exists(buf)) {
      cout << "File already exists!" << endl << endl;
       retry--;
      if(!retry) {
	Menu(f);
	return;
      }
    }
    else
      break;
  }

  vbFileHeader fh;
  vbBlockHeader vb;
  int errors = 0;
  
  // Analyze the VBD file header to determine if the file has
  // a pre-allocated static area
  __LWORD__ static_area;
  f->Read(&fh, sizeof(vbFileHeader), 0);
  if(CheckError((const vbDatabasePtr &)f) != 0) return;
  char rev_letter = f->GetRevLetter();

  
  // Check the VBD vbFileHeader's signature. 06/08/2000: Modified
  // to ignore the revision letter. If the header is damaged the
  // current VBD revision will be used.
  if (memcmp(fh.vbd_sig, vbDatabase::VBDSignature, 7)) {
    rev_letter = vbDatabaseRevisionLetter;
    errors++; // Header is damaged and cannot be read
  }
  
  if(!errors) { // Check the HeapStart value 
    f->Read(&vb, sizeof(vbBlockHeader), fh.vbd_hs_fptr);
    if(CheckError((const vbDatabasePtr &)f) != 0) return;
    if (vb.block_check_word != vbDatabase::VBDInternalCheckWord)
      errors++;
  }

  // If no errors, calculate the the size of the static area.
  // 06/08/2000: Modified to calculate the correct heap size.
  if(!errors) static_area = fh.vbd_hs_fptr - f->FileHeaderSize();

  FAU StaticEOF = f->FileSize(f->vbDatabaseName());
  FAU addr = f->VBSearch(0); // Search the entire file
  
  if(addr == (FAU)0) {
    cout << endl;
    cout << "No variable blocks found in file: "
	 << f->vbDatabaseName() << endl;
    PausePrg();
    return;
  }

  // Create the new file
  vbDebugPtr NewFile(new vbDatabaseDebugManager);

  // Get the previous state of all the version specific information
  __LWORD__ prev_ver = vbDatabase::VBDVersion;
  char prev_sig[8];
  memmove(prev_sig, vbDatabase::VBDSignature, 8);
  __LWORD__ prev_check_word = vbDatabase::VBDInternalCheckWord;

  if(update_ver) {
    vbDatabase::VBDVersion = vbDatabaseVersionNumber;
    memmove(vbDatabase::VBDSignature, "VBDBASE", 7);
    vbDatabase::VBDInternalCheckWord = vbCheckWord;
    rev_letter = vbDatabaseRevisionLetter;
    NewFile->Create(buf, static_area, rev_letter);
    vbDatabase::VBDVersion = prev_ver;
    memmove(vbDatabase::VBDSignature, prev_sig, 8);
    vbDatabase::VBDInternalCheckWord = prev_check_word;
  }
  else {
    NewFile->Create(buf, static_area, rev_letter);
  }
  
  int count = 0;
  int badvb = 0;

  char *v;

  if(static_area) { // Write the static area data
    v = new char[static_area];

    // 06/08/2000: Modified to calculate the correct header size
    f->Read(v, static_area, f->FileHeaderSize());
    NewFile->Write(v, static_area, f->FileHeaderSize());
    delete v;
  }
  
  FAU ObjectLength;
  vbINT32 checksum;
    
  while(1) { 
    if(addr >= StaticEOF) break;

    f->Read(&vb, sizeof(vbBlockHeader), addr);
    if(CheckError((const vbDatabasePtr &)f) != 0) return;
    if((vb.block_check_word == vbDatabase::VBDInternalCheckWord)) {
      if(vb.block_status == vbNormalBlock) { // Only copy normal blocks
	ObjectLength = f->ObjectLength(addr + f->VBHeaderSize());
	v = new char[(int)ObjectLength];
	if(!v) {
	  cout << "Memory allocation error!" << endl;
	  return;
	}
	f->Read(v, ObjectLength);
	if(CheckError((const vbDatabasePtr &)f) != 0) return;
	if((__SBYTE__)vb.block_status == vbNormalBlock) {
	  count++; // Increment the variable block count
	  if(update_ver) {
	    vbDatabase::VBDInternalCheckWord = vbCheckWord;
	    NewFile->Write(v, ObjectLength, NewFile->Alloc(ObjectLength));
	    vbDatabase::VBDInternalCheckWord = prev_check_word;
	  }
	  else 
	    NewFile->Write(v, ObjectLength, NewFile->Alloc(ObjectLength));
	  if(CheckError((const vbDatabasePtr &)NewFile) != 0) return;
	  switch(rev_letter) {
	    case '\0': case ' ': 
	      break;

	    default: // Default to rev A and higher
	      checksum = calcCRC32(v, ObjectLength);
	      NewFile->Write(&checksum, sizeof(checksum));
	      if(CheckError((const vbDatabasePtr &)NewFile) != 0) return;
	      break;
	  }
	}
	delete v;
      }
      addr = addr + vb.block_length; // Goto the next variable block
    }
    else {
      badvb++;
      addr = f->VBSearch(addr); // Search for the next good block
      if(!addr) break; 
    }
    
  }

      cout << endl;

  cout << "Wrote " << count << " good variable blocks to file: "
       << NewFile->vbDatabaseName() << endl;

  if(static_area)
    cout << "Wrote " << static_area << " bytes of Static area data."
	 << endl;
 
  if(badvb) 
    cout << "Did not write " << badvb << " bad variables found in file: "
	 << f->vbDatabaseName() << endl;

  if(errors) {
    cout << f->vbDatabaseName() << " file header is damaged!" << endl;
    cout << "No header information was copied to "
	 << NewFile->vbDatabaseName() << endl;
  }

  NewFile->Close();
  cout << endl;
}

void Menu(const vbDebugPtr &f)
{
  cout << endl;
  cout << "Analyzing file: " << f->vbDatabaseName() << endl;
  cout << "Enter the letter of your selection at the prompt." << endl;
  cout << endl;
  cout << "(A, a)    - Analyze the VBD file header" << endl;
  cout << "(D, d)    - Dump every variable block" << endl;
  cout << "(F, f)    - Find every variable block in the file" << endl;
  cout << "(H, h, ?) - Displays this menu" << endl;
  cout << "(Q, q)    - Quit this program" << endl;
  cout << "(r)       - Rebuild a damaged VBD file" << endl;
  cout << "(R)       - Rebuild and update to current version/revision" 
       << endl;
  cout << "(S, s)    - Display VBD file statistics" << endl;
  cout << "(W, s)    - Walk through every variable block" << endl;
  cout << endl;
}

void ClearInputStream(istream &s)
// Used to clear istream
{
  char c;
  s.clear();
  while(s.get(c) && c != '\n') { ; }
}

void PausePrg()
{
  cout << endl;
  cout << "Press enter to continue..." << endl;
  cin.get();
}

int Quit()
{
  // Cleanup and exit the porgram
  cout << "Exiting..." << endl;
  return 0;
}

void Version()
{
  cout << endl;
  cout << ProgramName << " vbDatabase file recovery program." << endl;
  EchoVBDVersion();
  cout << endl;
}

int main(int argc, char **argv)
{
  // Display the program version information and exit the program
  if(argc >= 2) {
    if(strcmp(argv[1], "version") == 0) {
      Version();
      return 0;
    }
  }

  if(argc < 2) {
    cout << endl;
    Version();
    cout << "Usage: " << ProgramName << " infile.vbd" << endl;
    cout << "Usage: " << ProgramName << " infile.vbd (command)" << endl;
    cout << endl;
    return 1;
   }

  vbDebugPtr f(new vbDatabaseDebugManager);   
  const char *fname = argv[1];
  if(!vbDatabase::Exists(fname)) {
    cout << "The specified file does not exist!" << endl;
    cout << "Exiting..." << endl;
    return 1;
  }
  else {
    f->BlindOpen(fname);
    if(CheckError((const vbDatabasePtr &)f) != 0) return 1;
  }

  char key;
  if(argc <= 2) Menu(f); // Not processing a command
  int rv = 1;

  while(rv) {
    if(argc > 2) { // Process a single command and exit the loop
      key = *(argv[2]);
      rv = 0;
    }
    else {
      if (!cin) { 
	ClearInputStream(cin);
	if (!cin) {
          cout << "Input stream is broken" << endl;
          return 0;
	}
      }
      cout << '>';
      cin >> key;
      if (!cin) continue;
    }
    switch(key) {
      case 'a' : case 'A' :
	if(argc <= 2) ClearInputStream(cin);
	AnalyzeHeader(f);
	break;
      case 'f' : case 'F' :
	if(argc <= 2) ClearInputStream(cin);
	FindVB(f);
	break;
      case 'd' : case 'D' :
	if(argc <= 2) ClearInputStream(cin);
	FindVB(f, 1);
	break;
      case 'h' : case 'H' :
	Menu(f);
	break;
      case '?' :
	Menu(f);
	break; 
      case 'q' : case 'Q' :
	rv = Quit();
	break;
      case 'r' : 
	if(argc <= 2) ClearInputStream(cin);
	Rebuild(f);
	break;
      case 'R' :
	if(argc <= 2) ClearInputStream(cin);
	Rebuild(f, 1);
	break;
      case 's' : case 'S' :
	if(argc <= 2) ClearInputStream(cin);
	DisplayStats(f);
	break;
      case 'w' : case 'W' :
	if(argc <= 2) ClearInputStream(cin);
	FindVB(f, 1, 1);
	break;
      default:
        cout << "Unrecognized command" << endl;
    }
  }

  return 0;
}
// ----------------------------------------------------------- //
// ------------------------------- //
// --------- End of File --------- //
// ------------------------------- //

