// ------------------------------- //
// -------- Start of File -------- //
// ------------------------------- //
// ----------------------------------------------------------- // 
// C++ Source Code File Name: vthreadt.cpp
// Compiler Used: MSVC, BCC32, GCC, HPUX aCC, SOLARIS CC
// Produced By: Doug Gaer   
// File Creation Date: 03/25/2000
// Date Last Modified: 08/11/2000
// Copyright (c) 2000 Douglas M. Gaer
// ----------------------------------------------------------- // 
// ------------- Program Description and Details ------------- // 
// ----------------------------------------------------------- // 
/*
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.
 
This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  
USA

The vbThread_t class is a data structure used to initialize and
store thread variables prior to and after thread creation.
*/
// ----------------------------------------------------------- //   
#include "vthreadt.h"
#include "vbthread.h"
#include "vbmutex.h"
#include "vbcond.h"

// --------------------------------------------------------------
// Constants used for thread status and debug functions
// --------------------------------------------------------------
// NOTE: This array must contain the same number of exceptions as the
// vbThreadState enumeration. 
const int vbMaxThreadStateMessages = 7;
const char *vbThreadStateMessages[vbMaxThreadStateMessages] = {
  "Thread state: INVALID",  // vbTHREAD_STATE_INVALID
  "Thread state: CANCELED", // vbTHREAD_STATE_CANCELED
  "Thread state: EXITED",   // vbTHREAD_STATE_EXITED
  "Thread state: NEW",      // vbTHREAD_STATE_NEW
  "Thread state: RUNNING",  // vbTHREAD_STATE_RUNNING
  "Thread state: SUSPENED", // vbTHREAD_STATE_SUSPENDED 
  "Thread state: WAITING"   // vbTHREAD_STATE_WAITING
};

// NOTE: This array must contain the same number of exceptions as the
// vbThreadPriority enumeration. 
const int vbMaxThreadPriorityMessages = 4;
const char *vbThreadPriorityMessages[vbMaxThreadPriorityMessages] = {
  "Thread priority: INVALID",
  "Thread priority: LOW",
  "Thread priority: NORMAL",
  "Thread priority: HIGH"
};

// NOTE: This array must contain the same number of exceptions as the
// vbThreadPriorityClass enumeration. 
const int vbMaxThreadPriorityClassMessages = 4;
const char *vbThreadPriorityClassMessages[vbMaxThreadPriorityClassMessages] = {
  "Thread priority class: INVALID",
  "Thread priority class: OTHER",    
  "Thread priority class: FIFO",
  "Thread priority class: ROUND ROBIN"
};

// NOTE: This array must contain the same number of exceptions as the
// vbThreadError enumeration.
const int vbMaxThreadExceptionMessages = 15;
const char *vbThreadExceptionMessages[vbMaxThreadExceptionMessages] = {
  "Thread exception: No exception reported",             // NO_ERROR
  "Thread exception: Invalid exception code",            // INVALID_CODE 
  "Thread exception: Error canceling thread",            // CANCEL_ERROR
  "Thread exception: Error closing thread",              // CLOSE_ERROR 
  "Thread exception: Error creating thread",             // CREATE_ERROR
  "Thread exception: Thread is already running",         // RUNNING_ERROR
  "Thread exception: Thread cannot be executed",         // EXECUTE_ERROR 
  "Thread exception: Error setting scheduling policy",   // POLICY_ERROR 
  "Thread exception: Error setting priority",            // PRIORITY_ERROR
  "Thread exception: Error resuming thread",             // RESUME_ERROR
  "Thread exception: Scheduling error",                  // SCHED_ERROR
  "Thread exception: Contention scope error",            // SCOPE_ERROR
  "Thread exception: Error setting the stack size",      // STACK_ERROR
  "Thread exception: Could not set the thread state",    // STATE_ERROR
  "Thread exception: Error suspending thread"            // SUSPEND_ERROR
};

// NOTE: This array must contain the same number of exceptions as the
// vbThreadType enumeration.
const int vbMaxThreadTypeMessages = 2;
const char *vbThreadTypeMessages[vbMaxThreadTypeMessages] = {
  "Thread type: DETACHED", // vbTHREAD_TYPE_DETACHED,
  "Thread type: JOINABLE"  // vbTHREAD_TYPE_JOINABLE
};
// --------------------------------------------------------------

vbThread_t::vbThread_t(vbThreadType t)
{
  // Use the API's default stack size. NOTE: The WIN32 Interface
  // will used the will use the same the stack size specified for
  // the main thread when the stack_size value is set to zero.
  stack_size = 0; 

  thread_type = t;
  thread_error = vbTHREAD_NO_ERROR;
  thread_state = vbTHREAD_STATE_NEW;
  thread_priority = vbTHREAD_PRIORITY_NORMAL;
  thread_priority_class = vbTHREAD_PRIORITY_CLASS_OTHER;
  
  // Set the thread exit value to zero initially. This value will
  // be set when the thread's entry function returns.
  thread_exit_code = (vbThreadExitCode)0;

  // NOTE: The thread ID will be set when the thread is created.
  // Not all UNIX variants treat the threads IDs as an integer type.
  // thread_id = 0;
  
  // Initialize the vbThread base class pointer. NOTE: This
  // pointer will be set by the vbThread class when a vbThread_t
  // object is passed to any of the vbThread::CreateThread()
  // functions.
  entry = 0;

  // Initialize the additional thread parameter pointer. NOTE: This
  // value is an optional and should only be accessed if was thread
  // parameter is being used with this thread.
  thread_parm = 0;
  
  // Set the class ID and object ID to -1 initially. These values are
  // optional and must be set by the class that create the thread. A
  // value of -1 indicates that this value has not been set.
  thread_oid = vbThreadObjectID(-1); 
  thread_cid = vbThreadClassID(-1); 

  suspend_lock_t = new vbMutex; // Lock used to serialize access to suspend_t
  suspend_t = new vbCondition;  // Thread suspend condition
}

vbThread_t::~vbThread_t()
{
  if(suspend_lock_t) delete suspend_lock_t;
  if(suspend_t) delete suspend_t;
}

const char *vbThread_t::ThreadExceptionMessage()
// Returns a null terminated string that can
// be used to log or print a thread exception.
{
  int error = (int)thread_error;
  if(error > vbMaxThreadExceptionMessages) error = vbTHREAD_INVALID_CODE;
  // Find the corresponding message in the exception array
  return vbThreadExceptionMessages[error];
}

const char *vbThread_t::ThreadTypeMessage()
// Returns a null terminated string that can
// be used to log or print the thread type.
{
  int type = (int)thread_type;
  if(type > vbMaxThreadTypeMessages) type = vbTHREAD_TYPE_JOINABLE;
  return vbThreadTypeMessages[type];
}

const char *vbThread_t::ThreadPriorityMessage()
// Returns a null terminated string that can
// be used to log or print the thread's priority.
{
  // The use of a switch case statement allows values to be assigned
  // to the priority members of the vbThreadPriority enumeration.
  switch(thread_priority) {
    case vbTHREAD_PRIORITY_LOW:
      return vbThreadPriorityMessages[1];
    case vbTHREAD_PRIORITY_NORMAL:
      return vbThreadPriorityMessages[2];
    case vbTHREAD_PRIORITY_HIGH:
      return vbThreadPriorityMessages[3];
    default:
      break; // Unknown priority
  }
  return vbThreadPriorityMessages[0];
}

const char *vbThread_t::ThreadPriorityClassMessage()
// Returns a null terminated string that can
// be used to log or print the thread's priority class.
{
  int type = (int)thread_priority_class;
  if(type > vbMaxThreadPriorityClassMessages)
    type = vbTHREAD_PRIORITY_CLASS_INVALID;
  return vbThreadPriorityClassMessages[type];
}

const char *vbThread_t::ThreadStateMessage()
// Returns a null terminated string that can
// be used to log or print the thread's state.
{
  int state = (int)thread_state;
  if(state > vbMaxThreadStateMessages) state = vbTHREAD_STATE_INVALID;
  return vbThreadStateMessages[state];
}

int operator==(const vbThread_t &a, const vbThread_t &b)
// Compare the thread IDs vbThread_t "a" and "b". Returns
// true if the IDs are equal. This function was added
// for UNIX variants that do not use an integer type for
// thread IDs.
{
#if defined (__WIN32__)
  return a.thread_id == b.thread_id;
#elif defined (__POSIX__)
  // NOTE: Undefined behaviour will result If both vbThread_t
  // objects do not contain valid thread IDs.
  if(pthread_equal(a.thread_id, b.thread_id)) return 1;
  return 0; // The thread IDs are not equal
#else 
#error You must define a native API: __WIN32__ or __UNIX__
#endif
}

int operator!=(const vbThread_t &a, const vbThread_t &b)
// Compare the thread IDs vbThread_t "a" and "b". Returns
// true if the IDs are not equal. This function was 
// added for UNIX variants that do not use an integer type
// for thread IDs.
{
#if defined (__WIN32__)
  return a.thread_id != b.thread_id;
#elif defined (__POSIX__)
  // NOTE: Undefined behaviour will result If both vbThread_t
  // objects do not contain valid thread IDs.
  if(pthread_equal(a.thread_id, b.thread_id) != 0) return 1;
  return 0; // The thread IDs not equal
#else 
#error You must define a native API: __WIN32__ or __UNIX__
#endif
}
// ----------------------------------------------------------- // 
// ------------------------------- //
// --------- End of File --------- //
// ------------------------------- //
