/* example 2: using flat real mode to access conventional memory structures

   This program demostrates how to access the video memory using flat
   pointers, write programs that don't use extended memory and use some of
   the macros #defined in flat.h.
 */

#include <dos.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>

#include "../flat.h"

/* prototypes, the functions are explained below */
void set_video_mode (int m);
int bget_key (void);

/* NOTE: the gfx functions are specific for mode 0x13 (320x200x256) */
void setup13 (void);
void put_pixel (int x, int y, int c);
int get_pixel (int x, int y);

void put_char (int x, int y, int b, int c);
void print_string (int x, int y, const char *s, int c);
int gprintf (int x, int y, int c, const char *fmt,...);

/* you already know */
void errexit (const char *,...);

/* flat pointers to the video memory and the bios 8x8 char 0x00-0x7f
   font
 */
flatptr vidmem, biosfont;

char *errmsg[] =
{
  NULL,
  "80386 or better CPU not detected",
  "CPU is in V86-mode",
  "Extended memory couldn't be allocated",
  "Unable to lock extended memory"
};

int
main (void)
{
  int foo;

  /* Tell init_flat not to use extended memory, by setting the maximum
     amount of memory to allocate to zero.
   */
  em_max = 0;

  foo = init_flat ();

  if (foo)
    errexit (errmsg[foo]);

  /* switch to mode 0x13 */
  set_video_mode (0x13);

  /* set up the pointers */
  setup13 ();

  /* write some stuff to the screen to demonstrate it works */
  gprintf (8, 8, 12,
           "This string is written to the video\n"
           "memory using flat pointers\n"
           "\n"
           "Spreadsheet\n"
           "-------------------------------------\n"
           "\n"
           "5\t11\t534\t8472\n"
           "\n"
           "What\ta\tnice\ttext!\n"
    );

  /* wait for keystroke */
  bget_key ();

  /* set 80x25x16 text mode, hopefully default */
  set_video_mode (0x03);

  exit_flat ();
  return (0);
}

/* errexit -- exit with (possibly) formatted error message */
void
errexit (const char *fmt,...)
{
  va_list v;
  char s[513];
  va_start (v, fmt);

  vsprintf (s, fmt, v);
  fprintf (stderr, "%s\n", s);
  exit_flat ();
  exit (1);
}

/* set_video_mode -- set a specific video mode */
void
set_video_mode (int m)
{
  /* should be quite portable */
  union REGS r;

  r.x.ax = m;
  int86 (0x10, &r, &r);
}

/* bget_key -- wait for keystroke */
int
bget_key (void)
{
  union REGS r;

  r.h.ah = 0;
  return (int86 (0x16, &r, &r));
}

/* setup13 -- setup pointers to be used while mode 0x13 is active */
void
setup13 (void)
{
  /* Vidmem should point to the video memory,
     (MaKe_ a FlatPoinTeR)
   */
  vidmem = MK_FPTR (0xa000, 0);

  /* biosfont to the bios 8x8 font for characters 0x00-0x7f. */
  biosfont = MK_FPTR (0xf000, 0xfa6e);
}

/* put_pixel -- write a pixel to the screen */
void
put_pixel (int x, int y, int c)
{
  /* I hope no explanations are necessary */
  pokeb32 (vidmem + 320 * y + x, c);
}

#if 0                           /* not used */
/* get_pixel -- read a pixel's color value */
int
get_pixel (int x, int y)
{
  return (peekb32 (vidmem + 320 * y + x));
}
#endif

/* put_char -- write a char from the bios 8x8 font to the screen
   This function is written for readability, not speed or size.
 */
void
put_char (int x, int y, int b, int c)
{
  flatptr f = biosfont + 8 * b, fend;
  int foo, byte;
  fend = f + 8;

  /* read eight bytes for one character */
  for (; f < fend; ++f, ++y)
    {
      /* read the current */
      byte = peekb32 (f);

      /* display one line */
      for (foo = 0; foo < 8; ++foo)
        {
          /* write it only, if the highest bit is a 1 */
          if (byte & 128)
            put_pixel (x + foo, y, c);  /* bitarray */

          /* go to next bit */
          byte <<= 1;
        }
    }
}

/* print_string -- write a string to the video memory using put_char
   This performs interpretation on '\t' and '\n'.
 */
void
print_string (int x, int y, const char *s, int c)
{
  /* the original starting x */
  int x1 = x;

  /* write every character */
  while (*s)
    {
      switch (*s)
        {
          /* go to next cell dividable by 64 (=eight characters) */
        case '\t':
          x = (x & (~63)) + 64;
          break;

          /* advance to next line */
        case '\n':
          x = x1;
          y += 8;
          break;

        default:
          put_char (x, y, *s, c);
          x += 8;
        }
      /* perform clipping on the right edge */
      if (x > 320 - 8)
        {
          x = x1;
          y += 8;
        }
      ++s;
    }
}

/* gprintf -- write a printf style string to the screen */
int
gprintf (int x, int y, int c, const char *f,...)
{
  int ret_val;
  va_list v;
  char s[513];

  va_start (v, f);

  ret_val = vsprintf (s, f, v);

  if (EOF != ret_val)
    print_string (x, y, s, c);
  return (ret_val);
}
