// DrMatrix.hpp
// Include this file in all modules that use DrMatrix classes

#ifndef __STDLIB_H
#include <stdlib.h>
#endif

#ifndef __IOSTREAM_H
#include <iostream.h>
#endif


// Enum data types
enum V_ALIGN {V_HORIZ,V_VERT};
enum BOOL    {FALSE,TRUE};


// Vector template class definition
template <class Type> class Vector
{
	friend ostream& operator<<(ostream&,Vector&);
	friend istream& operator>>(istream&,Vector&);
	friend Vector   operator* (Type,Vector&);

	public:
		// Constructors and desctructor
		Vector(int n,V_ALIGN vtype=V_HORIZ);
		Vector(const Vector&);
		~Vector() {delete [] data;};

		// Member operators
		Type&  operator()(int k);
		Vector operator+(Vector& b);
		Vector operator-(Vector& b);
		Vector operator-();
		Type   operator*(Vector& b);
		Vector& operator=(Vector& b);
		BOOL   operator==(Vector& b);

		// Member functions
		Vector T();  // transpose
		int lead();  // leading element
		V_ALIGN align() {return _vtype;};
		int len() {return _len;};
		Type abs();	 // absolute value (distance from origin)

	private:
		Type *data;
		int _len;
		V_ALIGN _vtype;
};


// Matrix template class definition
template <class Type> class Matrix
{
	friend ostream& operator<<(ostream&,Matrix&);
	friend istream& operator>>(istream&,Matrix&);
	friend Matrix   operator* (Type a,  Matrix&);

	public:
		// Constructors and destructor
		Matrix(int n,int m);
		Matrix(const Matrix&);
		~Matrix() {delete [] data;};

		// Member operators
		inline Type& operator()(int i,int j);
		Matrix operator+(Matrix& b);
		Matrix operator-(Matrix& b);
		Matrix operator-();
		Matrix operator*(Matrix& b);
		Matrix& operator=(Matrix& b);
		BOOL operator==(Matrix& b);

		// Mathematical functions
		Matrix T();				// transpose
		Type tr();				// trace
		Type det();				// determinant
		int lead(int n);		// leading element of nth row
		Matrix step();			// return equivalent, stepped matrix
		Matrix canon();			// return the equivalent canonized matrix
		int rank();				// return the rank of the matrix
		Matrix inv();			// return the inverse matrix

		// Property functions - return true if matrix satisfies property
		BOOL diag();			// diagonal matrix
		BOOL iden();			// identity matrix
		BOOL scalar();			// scalar matrix (equal to kI, for real k)
		BOOL sym();				// symmetric matrix (m(i,j)==m(j,i))
		BOOL antisym();			// antisymmetric matrix (m(i,j)==-m(j,i))
		BOOL square();			// square matrix (nrows==ncols)
		BOOL stepped();			// stepped matrix
		BOOL reg();				// regular (reversible) matrix
		BOOL sing();			// singular (non-reversible) matrix

		// Organizational functions
		Vector<Type> row(int n); // returns nth row in matrix
		Vector<Type> col(int m); // returns mth column in matrix
		int nrows() {return _nrows;};
		int ncols() {return _ncols;};

	private:
		int _nrows,_ncols;
		Type* data;
};
