/* GETLINES:        -- TEXT FILE UTILITY --
 *
 *  Given a list of line numbers within a specifed text file
 *  display the corresponding lines.
 *
 *  Note that sorted input lists process much faster!
 *  Previously scanned lines require program to rewind
 *  the file stream and recount the line numbers.
 *
 *  This program will compile with Turbo/Borland C++, UNIX ANSI C,
 *  GNU C, and VAX/VMS C.
 *
 *  Developed by Jason Mathews
 *	NASA/Goddard Space Flight Center
 *	<mathews@nssdca.gsfc.nasa.gov>
 *
 *  Copyright (C) 1994 by Jason Mathews.  Permission is granted to any
 *  individual or institution to use, copy or modify this software so long
 *  as it is not sold for profit, provided this copyright notice is retained.
 *
 *  Modification history:
 *
 *    20-May-94  Original version.
 *
 *****************************************************************************/

#include <stdio.h>
#include <stdlib.h>

#if defined __MSDOS__
#  include <io.h>
#elif defined unix
#  include <unistd.h>
#  define ALT_SW
#elif defined vms
#  include <unixio.h>
#endif /* ?__MSDOS__ */

/* define alternate switch for non-unix systems */
#ifndef unix
#  define ALT_SW *argv[i]=='/' ||
#endif

int  interactive;    /* flag indicating interactive mode */
int  errcode    = 0; /* return error code */
char bShowLines = 0; /* boolean flag to show line numbers */
char buf[512];
char copyright[] = "GETLINES v1.0 (C) 1994 Jason Mathews\n";

void usage(void)
{
    fprintf(stderr, copyright);
    fprintf(stderr, "Show specified lines of text file given line number(s)\n\n");
    fprintf(stderr, "Usage: GETLINES [filename] [-ln] [-cn] [-n]\n\n");
    fprintf(stderr, "  filename = File to search\n");
    fprintf(stderr, "  -ln      = Display line number n, where n >= 1\n");
    fprintf(stderr, "  -cn      = Display n lines (default=1)\n");
    fprintf(stderr, "  -n       = Prefix line with line numbers\n\n");
    fprintf(stderr, "Default input accepted from stdin.\n\n");
    fprintf(stderr, "Examples: > GETLINES foo.dat.c -n < lines.dat\n");
    fprintf(stderr, "          > GETLINES bar.c -l100 -c10\n");
    exit(1);
}

/*
 * Purpose:	Search text file for specified line number
 *
 * Returns:	0 if successful, otherwise 1 if error
 */
int getline( FILE* fp, int target_line )
{
  static int linenum = 0;
   /* if input line out of order then rewind file */
    if (target_line < linenum)
    {
        rewind(fp);
        linenum = 0;
    }
    while (linenum < target_line)
    {
        if (fgets(buf, sizeof(buf), fp)==0)
        {
            fprintf(stderr, "%s: Invalid line %d: file has only %d lines\n",
                    (interactive ? "Warning" : "Error"), target_line, linenum);
	    if (interactive)
	    {
		rewind(fp);
		linenum = 0;
	    }
            return !interactive;
        }
        linenum++;
    }

    if (bShowLines) printf("%-7d", linenum);
    printf("%s", buf);
    return 0;
}

/*
 *  Scan text file using line numbers read from stdin.
 */
void scan( FILE* fp )
{
 int target_line;
 char* status = interactive ? "Warning" : "Error";
 char linestr[80];

 do {
     if (interactive)
     {
  moreInput:
	fprintf(stderr, "Line number to extract: ");
     }
     if (!gets(linestr)) break; /* eof? */
     if ((target_line = atoi(linestr)) < 1)
     {
	fprintf(stderr, "%s: Illegal line number %s\n", status, linestr);
	if (interactive) goto moreInput;
	errcode = 1; /* error */
	break;
     }
#ifdef vms
     if (interactive) fprintf(stderr, "\n");
#endif
 } while (!getline(fp, target_line));
}

int main( int argc, char**argv )
{
  FILE* fp;
  char *filename = NULL;
  int i;
  int linenum = 0;
  int count   = 1;

  for (i=1; i < argc; i++)
  {
    if (ALT_SW *argv[i] == '-')
    {
	switch (argv[i][1]) {
	    case 'n':
		bShowLines = 1;
		break;
            case 'l':
		if ((linenum = atoi(argv[i]+2)) < 1) goto Arg_Error;
		break;
            case 'c':
		if ((count = atoi(argv[i]+2)) < 1) goto Arg_Error;
		if (!linenum) linenum = 1;
		break;
	    default:
	    Arg_Error:
		fprintf(stderr, "Error: Illegal parameter or value %s\n\n", argv[i]);
	    case 'h':
	    case '?':
		usage();
	}
    }
    else filename = argv[i];
  } /* for each arg */

  /* interactive mode if input is from keyboard, otherwise piped input */
  interactive = isatty(fileno(stdin));

  if (!filename)
  {
	if (interactive) fprintf(stderr, "Filename to search: ");
	gets(filename = buf);
  }

  fp = fopen(filename, "r");
  if (!fp)
  {
	fprintf(stderr, "Error: Cannot open ");
	perror(filename);
	usage();
  }

  if (linenum != 0)
  {
	interactive = 0; /* disable interative mode */
	while (count && !getline(fp, linenum++)) count--;
  }
  else scan(fp);

  fclose(fp);
  return errcode;
}
