

#include "PrnStr.h"
#include "oprnstr.h"
#include <Bios.h>

#define ESC '\x1b'

prnbuf::prnbuf()
	{
	xfd = EOF;
	isBinary = 0;
	char* p = new char[B_size];
	if( p )
		{
		setb( p, p+B_size,1 );	// ~streambuf() will delete.
		setp(base(),ebuf());
		setg(0,0,0);
		}
	}

prnbuf::prnbuf(int f)
	{
	xfd = EOF;
	isBinary = 0;

	// Check that port is supported.

	unsigned u = ((unsigned)(biosequip()) >> 14) & 0x0003;
	// u is the number of printers supported by BIOS.

	if( f >= 0 && f < u )
		{

		// Make sure printer is there.

		unsigned u = biosprint( _PRINTER_STATUS, 0, f );
		if( (u & pfSelected) == 0 )
			return;
		xfd = f;
		}

	char* p = new char[B_size];
	if( p )
		{
		setb(p, p+B_size, 1);	// ~streambuf() will delete.
		setp(base(),ebuf());
		setg(0,0,0);
		}
	}

prnbuf::prnbuf(int f, char* buf, int len)
	{
	xfd = EOF;
	isBinary = 0;

	// Check that port is supported.

	unsigned u = ((unsigned)(biosequip()) >> 14) & 0x0003;
	if( f >= 0 && f < u )
		{

		// Make sure printer is there.

		unsigned u = biosprint( _PRINTER_STATUS, 0, f );
		if( (u & pfSelected) == 0 )
			return;
		xfd = f;
		}

	setbuf(buf, len);
	setp(base(),ebuf());
	setg(0,0,0);
	}

prnbuf::~prnbuf()
	{
	if( xfd >= 0 )
		sync();		// flush the buffer.
	}

void prnbuf::init()
	{
	if( xfd < 0 )
		return;
	biosprint( _PRINTER_INIT, 0, xfd );
	}

prnbuf* prnbuf::attach(int f)
	{
	// See if supported by equipment list.

	unsigned u = ((unsigned)(biosequip()) >> 14) & 0x0003;
	if( f >= 0 && f < u )
		{

		// Make sure printer is there.

		unsigned u = biosprint( _PRINTER_STATUS, 0, f );
		if( (u & pfSelected) == 0 )
			return 0;
		}
	else
		return 0;

	// If already attached to a port, flush.

	if( xfd >= 0 && sync() == EOF )
		return 0;	// do NOT un-attach if flush fails.

	xfd = f;        // assumed to be valid (IE on, not out of paper...)
	isBinary = 0;
	char *b = base();       // buffer address
	if( ! b )
		{
		b = new char[B_size];
		if( b )
			{
			setb( b, b+B_size, 1 );	// ~streambuf() will delete.
			setp(base(),base());
			setg(0,0,0);
			}
		}
	return this;
	}

int prnbuf::overflow(int c)
	{
	unsigned st;

	if( xfd < 0 )
		return EOF;

	if( unbuffered() || !base() )
		{
		if( c != EOF )
			{
			char b = c;
			st = biosprint( _PRINTER_WRITE, b, xfd );
			if( (st & (pfOutOfPaper | pfIoError | pfTimeout)) != 0 )
				return EOF;

			if( b == '\n' && !isBinary )
				{
				st = biosprint( _PRINTER_WRITE, '\r', xfd );
				if( (st & (pfOutOfPaper | pfIoError | pfTimeout)) != 0 )
					return EOF;
				}
			}
		}
	else	// flush.
		{
		char *p = pbase();
		char *ep = pptr();
		while( p < ep )
			{
			st = biosprint( _PRINTER_WRITE, *p, xfd );
			if( (st & (pfOutOfPaper | pfIoError | pfTimeout)) != 0 )
				{
				setp( p, ebuf() );
				return EOF;
				}
			if( *p == '\n' && !isBinary )
				{
				st = biosprint( _PRINTER_WRITE, '\r', xfd );
				if( (st & (pfOutOfPaper | pfIoError | pfTimeout)) != 0 )
					return EOF;
				}
			p++;
			}
		setp( base(), ebuf() );
		if( c != EOF )
			sputc(c);
		}
	return 1;
	}

int prnbuf::underflow()
	{
	if( xfd < 0 )
		return EOF;

	return biosprint( _PRINTER_STATUS, 0, xfd );
	}


int prnbuf::sync()
	{
	if( xfd < 0 )	return EOF;
	return overflow(EOF);
	}


prnstreambase::prnstreambase()
		: buf()
	{
	ios::init(&buf);
	}

prnstreambase::prnstreambase(int f)
		: buf(f)
	{
	ios::init(&buf);
	if( buf.fd() < 0 )
		clear(ios::badbit);
	else
		clear(ios::goodbit);
	}

prnstreambase::prnstreambase(int f, char* b, int len)
		: buf(f, b, len)
	{
	ios::init(&buf);
	if( buf.fd() < 0 )
		clear(ios::badbit);
	else
		clear(ios::goodbit);
	}

prnstreambase::~prnstreambase()
	{
	}

void prnstreambase::attach(int f)
	{
	if( buf.attach(f) )
		clear(ios::goodbit);
	else
		clear(ios::badbit);
	}

void prnstreambase::setbuf(char* b, int len)
	{
	if( buf.setbuf(b, len) )
		clear(ios::goodbit);
	else
		setstate(ios::failbit);
	}

// Define Generic Printer Stream Class !!!! this won't work for PostScript
//  Printers

prnstream::prnstream(int psflag) :
		prnstreambase(),
		oprnstream(psflag)
	{
	}

prnstream::prnstream(int f, int psflag) :
		prnstreambase(f),
		oprnstream(psflag)
	{
	}

prnstream::prnstream(int f, char* b, int len, int psflag) :
		prnstreambase(f, b, len),
		oprnstream(psflag)
	{
	}

prnstream::~prnstream()
	{
	}

// *******************************************************************
// define the manipulators for text attributes for the Printer classes
// *******************************************************************

oprnstream _FAR & _Cdecl bold(oprnstream _FAR & pr)	{
	Printer& prn = (Printer&) pr;
	prn.setBold();
	return pr;
}

oprnstream _FAR & _Cdecl normal(oprnstream _FAR & pr)	{
	Printer& prn = (Printer&) pr;
	prn.setNormal();
	return pr;
}

oprnstream _FAR & _Cdecl italics(oprnstream _FAR & pr)	{
	Printer& prn = (Printer&) pr;
	prn.setItalics();
	return pr;
}

oprnstream _FAR & _Cdecl compressed(oprnstream _FAR & pr)	{
	Printer& prn = (Printer&) pr;
	prn.setCompressed();
	return pr;
}

oprnstream _FAR & _Cdecl reset(oprnstream _FAR & pr)	{
	Printer& prn = (Printer&) pr;
	prn.Reset();
	return pr;
}

oprnstream _FAR & _Cdecl font(oprnstream _FAR & pr, int fnt)	{
	Printer& prn = (Printer&) pr;
	prn.setFont(fnt);
	return pr;
}

PMANIP(int) font(int fnt)	{
	return PMANIP (int) (font, fnt);
}

oprnstream _FAR & _Cdecl pitch(oprnstream _FAR & pr, int pit)	{
	Printer& prn = (Printer&) pr;
	prn.setPitch(pit);
	return pr;
}

PMANIP(int) pitch(int pit)	{
	return PMANIP (int) (pitch, pit);
}


oprnstream& tabs(oprnstream& pr, String& tabstr)	{
	Printer& prn = (Printer&) pr;
	prn.setTabs(tabstr);
	return pr;
}

PMANIP(StrRef) tabs(String& tabstr)	{
	return PMANIP(StrRef) (tabs, tabstr);
}

/***********************************************************************
			Epson LQ/EX type printer


***********************************************************************/


// now define the different printers
// first an Epson, the codes are taken from an EX800 manual

_Cdecl	EpsonPrinter::EpsonPrinter() : Printer(FALSE)	{
};

_Cdecl  EpsonPrinter::EpsonPrinter(int f) : Printer(f, FALSE) {
};

_Cdecl  EpsonPrinter::EpsonPrinter(int f, char _FAR *buf, int len) :
			Printer(f, buf, len, FALSE) {
};

_Cdecl	EpsonPrinter::~EpsonPrinter() {
	EndJob();
};

// Set Printer Margins and Page Size
void EpsonPrinter::setPageSize(PageSize ps)	{
	switch (ps)	{
		case Legal :
			PageSizeX = 8.5 * POINTS_PER_INCH;
			PageSizeY = 14.0 * POINTS_PER_INCH;
			break;
		case Letter :
		default		:
			PageSizeX = 8.5 * POINTS_PER_INCH;
			PageSizeY = 14.0 * POINTS_PER_INCH;
			break;
	}
}

void EpsonPrinter::setMargins(float tm, float bm, float lm, float rm)	{
	BM = bm * POINTS_PER_INCH;
	TM = PageSizeY - (tm * POINTS_PER_INCH);
	LM = lm * POINTS_PER_INCH;
	RM = PageSizeX - (rm * POINTS_PER_INCH);
}

void EpsonPrinter::Reset()	{
	init();
	*this << ESC << "\x40";
}

void EpsonPrinter::EndJob()	{
	*this << "\f";
}

void EpsonPrinter::setBold()	{
	*this << ESC << "\x45";
}

void EpsonPrinter::setNormal()	{
	*this << ESC << "\x21" << '\x00';
}

void EpsonPrinter::setItalics()	{
	*this << ESC << "\x34";
}

void EpsonPrinter::setTabs(String& tabs)	{
	*this << ESC << "\x44" << tabs << '\x00';
}

void EpsonPrinter::setCompressed()	{
	*this << ESC << "\x0f";
}

void EpsonPrinter::setFont(int font)	{
	switch(font)	{
		case 0 :	// Draft mode
			*this << ESC << "\x78" << '\x00' << endl;
			break;
		case 1 :    // NLQ Roman
			*this << ESC << "\x78\x01" << ESC << "\x6b" << '\x00' << endl;
			break;
		case 2 :    // NLQ Sans Serif
			*this << ESC << "\x78\x01" << ESC << "\x6b" << '\x01' << endl;
		default : break;
	}
}

void EpsonPrinter::setPitch(int pitch)	{
	// for the Epson 00 = Pica, 01 = elite
	if ((pitch < 2) && (pitch >= 0))
		*this << ESC << "\x21" << (char) pitch ;
}


/***********************************************************************
			HP Laserjet II type printer


***********************************************************************/

// next we'll do a HP Printer, codes are from a Laserjet II manual

_Cdecl	HPPrinter::HPPrinter() : Printer(FALSE)	{
	fontsize = 12;
	currentFont = 0;
	setPageSize(Letter);
	setMargins(0.25, 0.25, 0.25, 0.25);     // 0.25" margins all around
};

_Cdecl  HPPrinter::HPPrinter(int f) : Printer(f, FALSE) {
	fontsize = 12;
	currentFont = 0;
	setPageSize(Letter);
	setMargins(0.25, 0.25, 0.25, 0.25);     // 0.25" margins all around
};

_Cdecl  HPPrinter::HPPrinter(int f, char _FAR *buf, int len) :
			Printer(f, buf, len, FALSE) {
	fontsize = 12;
	currentFont = 0;
	setPageSize(Letter);
	setMargins(0.25, 0.25, 0.25, 0.25);     // 0.25" margins all around
};

_Cdecl	HPPrinter::~HPPrinter() {
	EndJob();
};

// Set Printer Margins and Page Size
void HPPrinter::setPageSize(PageSize ps)	{
	switch (ps)	{
		case Legal :
			PageSizeX = 8.5 * POINTS_PER_INCH;
			PageSizeY = 14.0 * POINTS_PER_INCH;
			*this << ESC << "&l3A";
			break;
		case Letter :
		default		:
			PageSizeX = 8.5 * POINTS_PER_INCH;
			PageSizeY = 11.0 * POINTS_PER_INCH;
			*this << ESC << "&l2A";
			break;
	}
}

void HPPrinter::setMargins(float tm, float bm, float lm, float rm)	{
	BM = bm * POINTS_PER_INCH;
	TM = PageSizeY - (tm * POINTS_PER_INCH);
	LM = lm * POINTS_PER_INCH;
	RM = PageSizeX - (rm * POINTS_PER_INCH);
//	*this << ESC << "&l" << tm << "E";
}

void HPPrinter::Reset()	{
	init();
	*this << ESC <<  "E";
	setFont(currentFont);
	setPitch(fontsize);
}

void HPPrinter::EndJob()	{
	*this << "\f";
}

void HPPrinter::setBold()	{
	*this << ESC << "(s3B";
}

void HPPrinter::setNormal()	{
	*this << ESC << "(s0b0S";
}

void HPPrinter::setItalics()	{
	*this << ESC << "(s1S";
}

void HPPrinter::setTabs(String& tabs)	{
//  don't know how so do nothing
//	*this << ESC << "\x44" << tabs << '\x00';
}

void HPPrinter::setCompressed()	{
	*this << ESC << "&k2S";
}

void HPPrinter::setFont(int font)	{
	currentFont = font;
	*this << ESC << "(10U" << ESC << "(s1p" << fontsize << "v0s0b";
	switch(font)	{
		case 0 :	// Courier
			*this << "3T";
			break;
		case 1 :    // CG Times Roman
			*this << "4101T";
			break;
		case 2 :    // Universal
			*this << "4148T";
		default : break;
	}
}

void HPPrinter::setPitch(int pitch)	{
	fontsize = pitch;
	setFont(currentFont);
}


/***********************************************************************
			PostScript  type printer


***********************************************************************/


// next we'll do a PS Printer, codes are from a PostScript manual



// **** This part is not finished yet ****

_Cdecl	PSPrinter::PSPrinter() : Printer(TRUE)	{
	fontsize = 12;
	setPageSize(Letter);
	setMargins(0.3, 0.3, 0.3, 0.3);     // 0.3" margins all around
};

_Cdecl  PSPrinter::PSPrinter(int f) : Printer(f, TRUE) {
	fontsize = 12;
	setPageSize(Letter);
	setMargins(0.3, 0.3, 0.3, 0.3);     // 0.25" margins all around
};

_Cdecl  PSPrinter::PSPrinter(int f, char _FAR *buf, int len) :
				Printer(f, buf, len, TRUE) {
	fontsize = 12;
	setPageSize(Letter);
	setMargins(0.3, 0.3, 0.3, 0.3);     // 0.25" margins all around
};

PSPrinter::~PSPrinter(void)	{
	EndJob();
}

void PSPrinter::init() {
	// initialize the printer
	setHdrFlag(TRUE);
	*this << "%%!PS-Adobe-2.0\n%%CreationDate:(8/15/95)\n"
		  << "%%Creator:BEMAX\n"
		  << "%%End Comments\n"
		  << "%%BeginProcSet\n"
		  << "/fs " << typesize << " def\n"
		  << "/sizefont {fs scalefont} def % define fontsize\n"
		  << "/sfs {/fs exch store\n} def % store new fontsize\n"
		  << "  % calculate width of char\n"
		  << "/cwidth 72.0 fs div def\n"
		  << "/inch {72.0 mul} def     % define inch\n"
		  << "/TM " << TM << " def     % Top Margin\n"
		  << "/LM " << LM << " def     % Left Margin\n"
		  << "/BM " << BM << " def     % Bottom Margin\n"
		  << "/RM " << RM << " def     % Right Margin\n"
		  << "/Hx LM def /Hy TM def /URx " << PageSizeX << " def /URy "
			<< PageSizeY << " def\n"
		  << "/LLx 0 def\n /LLy 0 def % define boundaries\n"
		  << "/incline {Hy fs 1.1 mul sub} def %dec return new line position\n"
		  << "/nextline {/Hx LM store /Hy incline store Hx Hy moveto\n} def % move to next line\n"
		  << "/np {showpage\nrestore\nsave\n /Hy TM store /Hx LM "
		  <<   " store Hx Hy moveto\n} def %start new Page\n"
		  << "% decide new page or new line\n";
	*this << "/nl {incline BM lt {np} {nextline} ifelse} def\n "
		  << "/tablen 8 def\n"
		  << "/sf {fa exch get sizefont setfont} def\n"
		  << "/fa [0 /Courier findfont\n"
		  << "/Courier-Bold findfont\n"
		  << "/Courier-Oblique findfont\n"
		  << "/Courier-BoldOblique findfont\n"
		  << "] def %% font array\n"
		  << "/tabwidth { tablen cwidth mul } def % define tabbing \n"
		  << "/tab {currentpoint pop LM sub cvi tabwidth cvi mod tabwidth exch sub abs"
		  << " 0 rmoveto\n} def % calculate tabs\n"
		  << "%%EndProcSet\n%%EndProlog\n"
		  << "12 sfs 1 sf  % set initial font and size\n"
		  << "save\nLM TM moveto \n(";
	setHdrFlag(FALSE);
}

// Set Printer Margins and Page Size
void PSPrinter::setPageSize(PageSize ps)	{
	switch (ps)	{
		case Legal :
			PageSizeX = 8.5 * POINTS_PER_INCH;
			PageSizeY = 14.0 * POINTS_PER_INCH;
			break;
		case Letter :
		default		:
			PageSizeX = 8.5 * POINTS_PER_INCH;
			PageSizeY = 11.0 * POINTS_PER_INCH;
			break;
	}
}

void PSPrinter::setMargins(float tm, float bm, float lm, float rm)	{
	if (tm < 0.3) tm = 0.3;
	if (bm < 0.3) bm = 0.3;
	if (lm < 0.3) lm = 0.3;
	if (rm < 0.3) rm = 0.3;
	BM = bm * POINTS_PER_INCH;
	TM = PageSizeY - (tm * POINTS_PER_INCH);
	LM = lm * POINTS_PER_INCH;
	RM = PageSizeX - (rm * POINTS_PER_INCH);
}

void PSPrinter::Reset()	{
	init();
}

void PSPrinter::EndJob() {
	setHdrFlag(TRUE);           // set header flag to prevent parsing of PS code
	*this << ") show\nshowpage\nrestore\n%%EndJob\n%%Trailer\n";
	setHdrFlag(FALSE);
}

void PSPrinter::setBold()	{
	setHdrFlag(TRUE);
	*this << ") show\n2 sf (";
	setHdrFlag(FALSE);
}

void PSPrinter::setNormal()	{
	setHdrFlag(TRUE);
	*this << ") show\n1 sf (";
	setHdrFlag(FALSE);
}

void PSPrinter::setItalics()	{
	setHdrFlag(TRUE);
	*this << ") show\n3 sf (";
	setHdrFlag(FALSE);
}

void PSPrinter::setTabs(String& tabs)	{
//  don't know how so do nothing
//	*this << ESC << "\x44" << tabs << '\x00';
}

void PSPrinter::setCompressed()	{
//	*this << ESC << "&k2S";
}

void PSPrinter::setFont(int font)	{
/*	switch(font)	{
		case 0 :	// Courier
			*this << ESC << "(s3T";
			break;
		case 1 :    // NLQ Roman
			*this << ESC << "(s3T";
			break;
		case 2 :    // NLQ Sans Serif
			*this << ESC << "(s3T";
		default : break;
	}
*/
}

void PSPrinter::setPitch(int pitch)	{
	fontsize = pitch;
	setHdrFlag(TRUE);
	*this << ") show\n" << fontsize << " sfs\n 1 sf (";
	setHdrFlag(FALSE);
}

