/*******************************************************

    The CalcPlus Class Library Version 1.0,
    Author: Vladimir Schipunov, Copyright (C) 1996

    This library is free software. Permission to use,
    copy, modify and redistribute the CalcPlus library
    for any purpose is hereby granted without fee,
    provided that the above copyright notice appear
    in all copies.

*******************************************************/

#ifndef __CALCTYPE_H
#define __CALCTYPE_H
class istream;
class ostream;

//
//  Type identificators, C++ RTTI and templates
//  are not used for compatibility with older compilers
//

enum {
    idNil,
    idBool,
    idLong,
    idDouble,
    idString,
    idArray,
    idDate,
    idUserType
};

//
//  CType is the root class for all types of data built in interpreter.
//  All names of the classes based on CType begin with letter 'C'.
//  Class derived from CType should redefine several methods of CType.
//
//  Pure virtual methods:
//
//      int         type() - returns type identifier
//      int         size() - size of data pointed by methods ptr && data
//      const void* data() - constant pointer to the data buffer
//      void*       ptr()  - nonconstant pointer to the same data buffer
//      const char* name() - string with the name of type
//      CType*      copy() - makes copy of the object
//

class CType
{
public:
    int id;
    CType( int a ) : id( a ){}
    CType( const CType& t ) : id( t.id ){}
    virtual ~CType(){}
    virtual const void *data() const = 0;
    virtual size() const = 0;
    virtual void *ptr() = 0;
    virtual CType& operator =( const CType& t );
    virtual CType* copy() const = 0;
    virtual void print( ostream& ) const = 0;   // prints on a stream
    virtual void get( istream& ) = 0;           // reads from stream
    virtual void read( char* );                 // reads from char buffer
    virtual compare( const CType& ) const;      // 0 - the same
    virtual const char *name() const = 0;
    virtual type() const = 0;
    virtual const char *s( char *p = 0 ) const; // conversion to char*
    virtual empty() const;                      // nonzero if empty
};

extern ostream& operator << ( ostream& o, const CType& t );
extern istream& operator >> ( istream& i, CType& t );

//
//  Nil values are assigned to all uninitialized variables,
//  this is also value of expressions which cannot be calculated.
//

class CNil : public CType
{
public:
    char dummy;
    CNil() : CType( idNil) {}
    const void* data() const { return &dummy; }
    void *ptr() { return &dummy; }
    size() const { return sizeof( dummy ); }
    const char* name() const { return "Nil"; }
    type() const { return idNil; }
    empty() const { return 1; }
    void get( istream& ){}
    void print( ostream& ) const;
    CType* copy() const { return new CNil; }
};

//
//  Boolean type.
//

class CBool : public CType
{
public:
    int n;
    CBool( int a = 0 ) : CType( idBool), n( a ){}
    const void* data() const { return &n; }
    void* ptr() { return &n; }
    size() const { return sizeof(n); }
    const char* name() const { return "Bool"; }
    type() const { return idBool; }
    empty() const { return !n; }
    void get( istream& );
    void print( ostream& ) const;
    CType* copy() const { return new CBool( n ); }
    CBool& operator=( int k ) { n = k; return *this; }
    CType& operator =( const CType& t ){ return CType::operator =(t); }
    operator int() const { return n; }
};

//
//  Long type. There is no 2-bytes integer type,
//  all integer expressions have values of type Long.
//

class CLong : public CType
{
public:
    long n;
    CLong( long a = 0 ) : CType( idLong ), n( a ){}
    const void* data() const { return &n; }
    void* ptr() { return &n; }
    size() const { return sizeof(n); }
    const char* name() const { return "Number"; }
    type() const { return idLong; }
    empty() const { return n==0L; }
    void get( istream& );
    void print( ostream& ) const;
    CType *copy() const { return new CLong( n ); }
    operator long() const { return n; }
    CType& operator =( const CType& t );
    CType& operator =( long k ){ n = k; return *this; }
};

class CDouble : public CType
{
public:
    double n;
    CDouble( double a = 0 ) : CType( idDouble ), n( a ){}
    const void* data() const { return &n; }
    void* ptr() { return &n; }
    size() const { return sizeof(n); }
    const char* name() const { return "Float"; }
    type() const { return idDouble; }
    empty() const { return n==0; }
    void get( istream& );
    void print( ostream& ) const;
    CType *copy() const { return new CDouble( n ); }
    operator double() const { return n; }
};

//
//  String may be zero-terminated (in this case field
//  zero is set (to non-zero;)) or not zero-terminated.
//

class CString : public CType
{
public:
    char *p;
    int n, zero;

    CString( const char* = "", int = 0 );
    CString( const CString& );
    CString( int );
    ~CString(){ delete[] p; }
    const void* data() const { return p; }
    void* ptr() { return p; }
    size() const { return n-zero; }
    const char* name() const { return "String"; }
    type() const { return idString; }
    operator const char*() const { return p; }
    virtual const char *s( char *p = 0 ) const;
    CType *copy() const { return new CString( *this ); }
    void get( istream& );
    void print( ostream& ) const;
    CType& operator =( const CType& t );
    CType& operator =( const char* t );
    CString& right( char = ' ' );
    CString& left ( char = ' ' );
    operator==(const char*s) const;
    char& operator[]( int ) const;
};

//
//  Array type represents is the collection of values which
//  may be indexed by array of CString's.
//

class CArray : public CType
{
public:
    CType **array;
    CArray *structure;
    int n;
    const char* name() const { return "Array"; }
    type() const { return idArray; }
    const void* data() const { return 0; }
    void* ptr() { return 0; }
    size() const { return n; }
    empty() const { return !n; }
    CType* copy() const { return new CArray( *this ); }
    void get( istream& ){}
    void print( ostream& ) const;

    CArray() : CType( idArray ), array( 0 ),
        n( 0 ), structure( 0 ){}
    CArray( const CArray& );
    ~CArray();
    void add( CType* );
    void clear();
    compare( const CType& ) const;
    CType& operator=(const CType& );
    CArray& operator=(const CArray&);
    CType& operator []( int );
    CType& operator []( const char* );
    index( const char* ) const;
    remove( int );
};

//
//  Date may be converted to number.
//  Static field MonthIsFirst shows the style of date:
//  dd/mm/yy or mm/dd/yy. Day is first by default.
//

class CDate : public CType
{
public:
    int d, m, y;

    const char* name() const    { return "Date"; }
    type() const                { return idDate; }
    const void* data() const    { return &d; }
    void* ptr()                 { return &d; }
    size() const        { return sizeof(d)+sizeof(m)+sizeof(y); }
    empty() const       { return d==0 && m==0 && y==0; }
    CType *copy() const { return new CDate( *this ); }

    void get( istream& );
    void print( ostream& ) const;
    void set( long );
    long i() const;
    static MonthIsFirst;

    CDate( long n ) : CType( idDate ) { set( n ); }
    CDate( int a1, int a2, int a3 ) :
        CType( idDate ), y( a3 ),
        d( MonthIsFirst ? a2 : a1 ),
        m( MonthIsFirst ? a1 : a2 )
    {}
    CDate( const CDate& dt ) :
        CType( idDate ),
        d( dt.d ),
        m( dt.m ),
        y( dt.y )
    {}
    CDate();
    ~CDate(){}
};

#endif
