#include <stdio.h>
#include <fcntl.h>
#include <bios.h>
#include <time.h>
#include "ibmcom.h"

/* Space Mouse Parameters */
#define POSITION_MASK 0xfff00000
#define ROTATION_MASK 0xffffc000
#define ROTATION_OFFSET 1992

/* These are commands for the bioscom function  */

#define SETUP_PORT 0
#define XMT_BYTE   1
#define D8_BITS    0x03
#define NO_PARITY  0x00
#define BAUD110    0x00

int tractorID;            /* serial port for the tractor */

unsigned char mbuff[20];  /* internal data buffer for Space Mouse */

/* Mouse commands */

unsigned char m_init[]=  {'*','R' ,0};
unsigned char m_diag[]=  {'*',0x05,0};
unsigned char m_euler[]= {'*','G' ,0};
unsigned char m_report[]={'*','d' ,0};
unsigned char m_quat[]=  {'*','Q' ,0};

/*******************************************************************/
/* Routine to initialize the serial port                           */
/*   returns 0 if succesfull                                       */
/*   returns -1 if failure                                         */
/*******************************************************************/
int init_mouse6d(port)
int port;
{
 int ret;

 ret=com_install(port);  
 if(ret) {
   printf("init_mouse6d: could not open mouse serial port\n");
   return(-1);
  }
  
 /* set up the serial port */
 com_raise_dtr();
 com_set_speed(19200);
 com_set_parity(COM_NONE,1);
 return(0);
}
/************************************************************/
/*                                                          */
/*  Routine to return mouse diagnostics                     */
/*  returns two diagnostic status bytes in one lone word or */
/*  -1 if it can't read anything from the mouse.            */
/*                                                          */
/************************************************************/
long mouse_diags()
{
 unsigned char d1,d2;
 unsigned long diag;

 /* Send the diagnostic command */
 com_tx(m_diag[0]);
 com_tx(m_diag[1]);
 /* wait for the mouse to perform diags */
 delay(200);
 /* Make sure we've got some data */
 if(com_rx_empty()) return(-1);
 /* read the data bytes */
 d1=com_rx();
 d2=com_rx();
 /* put the two status bytes into one long word */
 diag=(0x0000ffff & (d1<<8 | d2));
 return (diag);
}

/********************************************************/
/*                                                      */
/*  This routine reads Logtech mouse data into an       */
/*  an internal buffer                                  */
/*  returns 0 if succesfull or -1 if failure            */
/*                                                      */
/********************************************************/
int read_mouse6d()
{
 int i,j;
 unsigned char dum;

 /* make sure there's no extra data in the receive buffer */ 
 while(!com_rx_empty()) dum=com_rx();

 /* demand a report from the mouse */
 com_tx(m_report[0]);
 com_tx(m_report[1]);
 /* normaly we would go off and do some work here */
 delay(25);
 for(i=0;i<16;i++) {
   j=0;
    while(com_rx_empty()) {
     j++;
     if(j > 5000) {  /* if we get to many retries we exit */
       return (-1);
     }
    }
    mbuff[i]=com_rx(); /* read data into internal buffer */
 }
 return (0);
}

/**********************************************************/
/*                                                        */
/*  This routine is used to dump out data in the internal */
/*  buffer                                                */
/*                                                        */
/**********************************************************/
void print_mouse6d()
{
 printf("Buttons    %x\n",mbuff[0]);
 printf("X          %x %x %x\n",mbuff[1],mbuff[2],mbuff[3]);
 printf("Y          %x %x %x\n",mbuff[4],mbuff[5],mbuff[6]);
 printf("Z          %x %x %x\n",mbuff[7],mbuff[8],mbuff[9]);
 printf("Pitch      %x %x\n",mbuff[10],mbuff[11]);
 printf("Yaw        %x %x\n",mbuff[12],mbuff[13]);
 printf("Roll       %x %x\n",mbuff[14],mbuff[15]);
}

/**************************************************************/
/*                                                            */
/* This routine returns the mouse data it takes pointes to    */
/* x,y,z,roll, pitch,yaw, and the button/status data          */
/* returns 0 if succesful and -1 if failure                   */
/*                                                            */
/**************************************************************/
int get_mouse6d(x,y,z,roll,pitch,yaw,buttons)
int *x,*y,*z,*roll,*pitch,*yaw;
int *buttons;
{
 int stat;

 /* Read the data into the internal buffer */
 stat=read_mouse6d();
 if(stat != 0) return(-1);

 *x=0;
 *x|=((mbuff[1]<<14) | (mbuff[2]<<7) | mbuff[3]);
 if(mbuff[1] & 0x40)  *x|=POSITION_MASK;

 *y=0;
 *y|=(mbuff[4]<<14) | (mbuff[5]<<7) | mbuff[6];
 if(mbuff[4] & 0x40) *y|=POSITION_MASK;

 *z=0;
 *z|=(mbuff[7]<<14) | (mbuff[8]<<7) | mbuff[9];
 if(mbuff[7] & 0x40) *z|=POSITION_MASK;

 *pitch= ((mbuff[10]<<7) | mbuff[11]);
 if(mbuff[10] & 0x40) {
  *pitch|=ROTATION_MASK;
  *pitch+=ROTATION_OFFSET;
 }

 *yaw=(mbuff[12]<<7) | mbuff[13];
 if(mbuff[12] & 0x40) {
   *yaw|=ROTATION_MASK;
   *yaw+=ROTATION_OFFSET;
 }


 *roll=(mbuff[14]<<7) | mbuff[15];
 if(mbuff[14] & 0x40) {
   *roll|= ROTATION_MASK;
   *roll+=ROTATION_OFFSET;
 }

 *buttons=mbuff[0];
 return(0);
}
/*************************************************************/
/*                                                           */
/*  Shut down Logitech mouse processing                      */
/*                                                           */
/*************************************************************/
void close_mouse6d()
{
 com_lower_dtr();
 com_deinstall();
}


/*******************************************************************/
/*                                                                 */
/* Routine to initialize the tractor serail port                   */
/* returns 0 is succesful -1 if failure                            */
/*                                                                 */
/*******************************************************************/
int init_tractor(port) 
int port;
{
/*  use the bioscom function here instead of IBMCOM.C since */
/*  the tractor I/O is so simple                            */   

 tractorID=port-1;  /* bioscom starts port numbering at 0 */
 bioscom(SETUP_PORT,D8_BITS|NO_PARITY|BAUD110,tractorID);
 return(0);
}

/*****************************************************************/
/*                                                               */
/* Routine to pulse tractor                                      */
/*                                                               */
/*****************************************************************/
void pulse_tractor()
{
 static clock_t last=0;  /* last time the tractor was pulsed */
 clock_t now;

 now=clock();
 /*
   If it's been less than 1/2 sec since the tractor was pulsed
   this call will be ignored.  The tractor is only capable of responding
   about twice a second
*/
 if((now-last) < CLK_TCK/2) return;
 bioscom(XMT_BYTE,'P',tractorID);
 last=now;  /* update the last pulse time */
}


